//------------ Dallas Semiconductor Corporation Copyright 1994-1996 ---------------
//
//  Copyright (C) 1994, 1995, 1996 Dallas Semiconductor Corporation. 
//  All rights Reserved. Printed in U.S.A.
//  This software is protected by copyright laws of
//  the United States and of foreign countries.
//  This material may also be protected by patent laws of the United States 
//  and of foreign countries.
//  This software is furnished under a license agreement and/or a
//  nondisclosure agreement and may only be used or copied in accordance
//  with the terms of those agreements.
//  The mere transfer of this software does not imply any licenses
//  of trade secrets, proprietary technology, copyrights, patents,
//  trademarks, maskwork rights, or any other form of intellectual
//  property whatsoever. Dallas Semiconductor retains all ownership rights.
//
//  SKEY.C  - Main module for the SKEY.DLL to be used by the DS1425 VB demo.
//           It calls on the functions in SACWD400.DLL to work on the 
//           DS1410E
//   
//  Version 4.00
//

// includes
#include <stdio.h>
#include <string.h> 
#include "sacwd400.h"                          
                           
// type definitions   
typedef unsigned int  uint; 
typedef unsigned char uchar; 

    
// functions from SACWD400.DLL
extern uint  far pascal setup(uint,sa_struct far *);
extern uint  far pascal next(sa_struct far *);
extern uint  far pascal first(sa_struct far *);
extern uint  far pascal access(sa_struct far *);
extern uint  far pascal databyte(uint,sa_struct far *);
extern uint  far pascal gndtest(sa_struct far *);
extern uchar far *far pascal romdata(sa_struct far *);          
extern uchar far pascal keyopen(void);
extern void  far pascal keyclose(void);

// functions to be exported                
int _export far pascal SmartPortSearch(void);
int _export far pascal GetKList(uchar far *);      
int _export far pascal ReadScratch(int, uchar far *);                      
int _export far pascal ReadSubkey(int, int, uchar far *, uchar far *, uchar far *);     
int _export far pascal SetPassID(int, int, uchar far *, uchar far *); 
int _export far pascal WriteScratch(int, uchar far *);
int _export far pascal WriteSubkey(int, int, uchar far *, uchar far *);
int _export far pascal WEP(int);

// global variables           
sa_struct gb;       // state variable structure            
uchar krom[50][8];  // ROM list of devices found on port
                                             
                                             
//---------------------------------------------------------------------------
// Search the the ports to find the port with a DS1410E.
// Returns the port number 1-3 or 0 if no port found.
//
int far pascal SmartPortSearch(void)
{
   int portnum=1,rt=0;
   
   // check each port until the DS1410E is found or out of ports
   do 
   {   
      // attempt to setup the port
      if (setup(portnum,&gb))
      {
         if (keyopen())   
            if(gndtest(&gb))
               if (first(&gb))
                  rt = portnum;
         keyclose();
      } 
   }
   while ((++portnum <= 3) && (rt == 0));

   return rt;
}                                             
                                             
//---------------------------------------------------------------------------
// Search the 1-Wire and returns a list of the ROM codes on the 1-Wire.
// The return value is 0 if no brick or no devices and > 0 to indicate
// the number of devices found.  The DS1425s on the list will be 
// highlighed. The buffer returned can be parsed by using the return
// value and the fact that each entry has the same length of ?????.
// The 'pt' value represents the 1-Wire port number.  The 'B' indicates
// the brick type.
//
int far pascal GetKList(uchar far *buf)                            
{
   uint flag;
   int cnt=0,i;                        
   uchar far *ROM;
    
   // open the port            
   if (keyopen())  
   { 
      // get a pointer to the rom buffer        
      ROM = romdata(&gb);           
                 
      // find first device   
      flag = first(&gb);
      while(flag)
      {        
         // a the new device to the buffer
         if ((ROM[0] & 0x7F) == 0x02)
            sprintf(&buf[cnt*19],"-> %02X%02X%02X%02X%02X%02X%02X%02X",
                    ROM[7],ROM[6],ROM[5],ROM[4],ROM[3],ROM[2],ROM[1],ROM[0]);
         else
            sprintf((char far *)(&buf[cnt*19]),"   %02X%02X%02X%02X%02X%02X%02X%02X",
                    ROM[7],ROM[6],ROM[5],ROM[4],ROM[3],ROM[2],ROM[1],ROM[0]);
           
         // record the rom in global memory       
         for (i = 0; i < 8; i++)
            krom[cnt][i] = ROM[i];
         
         // count number of devices found    
         cnt++;                          
                
         flag = next(&gb);
      }               
      // release the port
      keyclose();
   }
      
   // zero terminate the buffer
   buf[cnt*19] = 0;   
      
   return cnt;
}

  
//---------------------------------------------------------------------------
// Read the scratchpad of the 'num' device.
//
int far pascal ReadScratch(int num, uchar far *buf)
{
   int i,done=0,rt=1;
   uchar b;
   uchar far *ROM;

   // open the port            
   if (keyopen())  
   { 
      // select the 'num' device by loading rom buffer
      ROM = romdata(&gb);           
      for (i = 0; i < 8; i++)
         ROM[i] = krom[num][i];           
      
      // loop until read is the same or access fail
      do
      {
         if (access(&gb) == 1)
         {
            databyte(0x69,&gb);    // read scratch command
            databyte(0xC0,&gb);    // starting address
            databyte(0x3F,&gb);    // compliment of address
            done = 1;         
            for (i = 0; i <= 0x3F; i++)
            {
               b = databyte(0xFF,&gb);
               if (b != buf[i])
               {
                  done = 0;
                  buf[i] = b;
               }
            }
         }
         else        
         {
            rt = 0;     
            done = 1;
         }
      }
      while (done == 0);
      // release the port
      keyclose();
   }    
   else
      rt = 0;

   // zero terminate the buffer
   buf[0x40] = 0;
   
   return rt;   
}  


//---------------------------------------------------------------------------
// Read the 'sbk' subkey of 'num' device with 'ps' password and place the 
// data in 'buf'.
//
int far pascal ReadSubkey(int num, int sbk, uchar far *ps, 
                                  uchar far *Id, uchar far *buf)
{
   int i,done,cnt=0,rt=1;
   uchar b;
   uchar far *ROM;

   // open the port            
   if (keyopen())  
   { 
      // select the 'num' device by loading rom buffer
      ROM = romdata(&gb);           
      for (i = 0; i < 8; i++)
         ROM[i] = krom[num][i];           
      
      // loop until read is the same or access fail
      do
      {
         if (access(&gb) == 1)
         {
            databyte(0x66,&gb);    // read subkey command
            databyte(0x10 | (sbk << 6),&gb);    // starting address
            databyte(~(0x10 | (sbk << 6)),&gb);    // compliment of address 
            // loop to read the id
            done = 1;
            for (i = 0; i < 8; i++)
            {
               b = databyte(0xFF,&gb);
               if (b != Id[i])
               {
                  done = 0;
                  Id[i] = b;
               }
            }            
            // loop to write the password
            for (i = 0; i < 8; i++)
               databyte(ps[i],&gb); 
            // loop to read the data     
            for (i = 0; i < 0x30; i++)
            {
               b = databyte(0xFF,&gb);
               if (b != buf[i])
               {
                  done = 0;
                  buf[i] = b;
               }
            }
         }
         else           
         {
            rt = 0;
            done = 1;
         }
      }
      while (done == 0 && cnt++ < 2);
      // release the port
      keyclose();
   }
   else
      rt = 0;

   // zero terminate the buffer
   buf[0x30] = 0;
   Id[8] = 0;
                   
   // return result
   if (cnt >= 2) rt = -1;
   return rt;                
}     

                           
//---------------------------------------------------------------------------
// Set the 'num' devices subkey number 'sbk' to id 'id' and password 'ps'.
//
int far pascal SetPassID(int num, int sbk, uchar far *id, uchar far *ps) 
{
   int i,done=0,cnt=0,rt=1;
   uchar tmpid[9],buf[64];
   uchar far *ROM;

   // open the port            
   if (keyopen())  
   { 
      // select the 'num' device by loading rom buffer
      ROM = romdata(&gb);           
      for (i = 0; i < 8; i++)
         ROM[i] = krom[num][i];           
      
      // loop until read is the same or access fail
      do
      {
         // try to set the id/password
         if (access(&gb) == 1)
         {
            databyte(0x5A,&gb);    // set password command
            databyte(sbk << 6,&gb);     // starting address
            databyte(~(sbk << 6),&gb);    // compliment of address 
            // loop to read the id
            for (i = 0; i < 8; i++)
               tmpid[i] = databyte(0xFF,&gb);
            // loop to write back the id
            for (i = 0; i < 8; i++)
               databyte(tmpid[i],&gb); 
            // loop to write the new id
            for (i = 0; i < 8; i++)
               databyte(id[i],&gb); 
            // loop to write the new password
            for (i = 0; i < 8; i++)
               databyte(ps[i],&gb); 
         }
         else                            
         { 
            rt = 0;
            break;
         }
         
         // attempt to read the subkey to check if pass/id set, data cleared
         if (ReadSubkey(num,sbk,ps,tmpid,buf) == 1)
         {
            for (i = 0; i < 0x30; i++)  // check data
               if (buf[i] != 0)
                  break;  
            done = (i == 0x30) ? 1 : 0;  
            for (i = 0; i < 8; i++)     // check id
               if (id[i] != tmpid[i])
                  break;  
            done = ((done == 1) && (i == 8)) ? 1 : 0;
         }
      }
      while (done == 0 && cnt++ < 2);
      // release the port
      keyclose();
   }
   else
      rt = 0;
   
   // return result        
   if (cnt >= 2) rt = -1; 
   return rt;
}     


//---------------------------------------------------------------------------
//.Write the 'num' devices scratchpad with data in 'buf'
//
int far pascal WriteScratch(int num, uchar far *buf)
{
   int i,done=0,cnt=0,rt=1;
   uchar tbuf[65];
   uchar far *ROM;

   // open the port            
   if (keyopen())  
   { 
      // select the 'num' device by loading rom buffer
      ROM = romdata(&gb);           
      for (i = 0; i < 8; i++)
         ROM[i] = krom[num][i];           
      
      // loop until read is the same or access fail
      do
      {
         // write data
         if (access(&gb) == 1)
         {
            databyte(0x96,&gb);    // write scratch command
            databyte(0xC0,&gb);    // starting address
            databyte(0x3F,&gb);    // compliment of address 
            // loop to write the data
            for (i = 0; i < 64; i++)
               databyte(buf[i],&gb);
         }
         else              
         {
            done = 1;
            rt = 0;    
            break;
         }
   
         // read back data to check
         if (ReadScratch(num,tbuf) == 1)
         {  
            for (i = 0; i < 64; i++)
               if (tbuf[i] != buf[i])
                  break;  
            if (i == 64)  
               done = 1;
         }
      }
      while (done == 0 && cnt++ < 2);
      // release the port
      keyclose();
   }
   else
      rt = 0;
                  
   // return result               
   if (cnt >= 2) rt = -1; 
   return rt;
}     


//---------------------------------------------------------------------------
// Write the 'num' devices subkey number 'sbk' with a password of 'ps'
// the data in 'buf'
//
int far pascal WriteSubkey(int num, int sbk, uchar far *ps, uchar far *buf)
{
   int i,done=0,cnt=0,rt=1;
   uchar tbuf[65],tmpid[9];
   uchar far *ROM;

   // open the port            
   if (keyopen())  
   { 
      // select the 'num' device by loading rom buffer
      ROM = romdata(&gb);           
      for (i = 0; i < 8; i++)
         ROM[i] = krom[num][i];           
      
      // loop until read is the same or access fail
      do
      {
         // write data
         if (access(&gb) == 1)
         {
            databyte(0x99,&gb);    // write subkey command
            databyte(0x10 | (sbk << 6),&gb);    // starting address
            databyte(~(0x10 | (sbk << 6)),&gb); // compliment of address 
            // loop to read the id
            for (i = 0; i < 8; i++)
               databyte(0xFF,&gb);
            // loop to write the password
            for (i = 0; i < 8; i++)
               databyte(ps[i],&gb);
            // loop to write the data
            for (i = 0; i < 0x30; i++)
               databyte(buf[i],&gb);
         }
         else              
         {
            done = 1;
            rt = 0;
            break;
         }
   
         // read back data to check
         if (ReadSubkey(num,sbk,ps,tmpid,tbuf) == 1)
         {  
            for (i = 0; i < 0x30; i++)
               if (tbuf[i] != buf[i])
                  break;  
            if (i == 0x30)  
               done = 1;
         }
      }
      while (done == 0 && cnt++ < 2);
      // release the port
      keyclose();
   }
   else
      rt = 0;
                  
   // return result  
   if (cnt >= 2) rt = -1; 
   return rt;                
}     


//---------------------------------------------------------------------------
// Exit procedure
//
int far pascal WEP(int ExtType)
{
   return 1;
}      