#include "DebugClient.hpp"

#include "logger.hpp"

std::string toString(bool value)
{
	return value ? "true" : "false";
}

/**
* Creates an XML string that describes the options supported by the debugger.
*
* @param options The options structure that describes the options supported
* by the debugger.
*
* @return The XML string that describes the options.
**/
std::string getDebuggerOptionsString(const DebuggerOptions& options)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	std::string optionsString = "<options>";
	
	if (!options.canDetach)
	{
		optionsString += "<option name=\"detach\" value=\"false\" />";
	}
	
	if (!options.canTerminate)
	{
		optionsString += "<option name=\"terminate\" value=\"false\" />";
	}
	
	if (!options.canMemmap)
	{
		optionsString += "<option name=\"memmap\" value=\"false\" />";
	}
	
	if (!options.canMultithread)
	{
		optionsString += "<option name=\"multithread\" value=\"false\" />";
	}
	
	if (!options.canValidMemory)
	{
		optionsString += "<option name=\"validmemory\" value=\"false\" />";
	}
	
	if (!options.canSoftwareBreakpoint)
	{
		optionsString += "<option name=\"softwareBreakpoints\" value=\"false\" />";
	}
	
	if (options.canHalt)
	{
		optionsString += "<option name=\"halt\" value=\"true\" />";
	}
	
	if (options.haltBeforeCommunicating)
	{
		optionsString += "<option name=\"haltBeforeCommunicating\" value=\"true\" />";
	}
	
	if (options.breakpointCount != -1)
	{
		optionsString += "<option name=\"breakpointCount\" value=\"" + zylib::zycon::toString(options.breakpointCount) + "\" />";
	}
	
	if (!options.hasStack)
	{
		optionsString += "<option name=\"hasStack\" value=\"true\" />";
	}
	
	optionsString += "<option name=\"pageSize\" value=\"" + zylib::zycon::toString(options.pageSize) + "\" />";

	// Note: above code is crappy: options should always be set *explicitly*, otherwise correct parsing depends on the default values in the source code!
	optionsString += "<option name=\"canBreakOnModuleLoad\" value=\"" + zylib::zycon::toBoolString(options.canBreakOnModuleLoad) + "\" />";
	optionsString += "<option name=\"canBreakOnModuleUnload\" value=\"" + zylib::zycon::toBoolString(options.canBreakOnModuleUnload) + "\" />";
	optionsString += "<option name=\"canTraceCount\" value=\"" + zylib::zycon::toBoolString(options.canTraceCount) + "\" />";

	// build exception list
	for (std::vector<DebugException>::const_iterator cit = options.exceptions.begin(); cit != options.exceptions.end(); ++cit)
	{
		optionsString += "<option name=\"exception\" exceptionName=\"" + cit->exceptionName + "\" exceptionCode=\"" + zylib::zycon::toString(cit->exceptionCode) +  + "\" handlingAction=\"" + zylib::zycon::toString(cit->handlingAction) + "\" />";
	}

	optionsString += "</options>";
	
	return optionsString;
}

/**
* Creates a register description from a list of register description objects.
*
* @param registers The list of register description objects.
*
* @return The generated register description string.
**/
std::string getRegisterString(const std::vector<RegisterDescription>& registers)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	std::string regString = "<registers>";
	
	for (std::vector<RegisterDescription>::const_iterator Iter = registers.begin(); Iter != registers.end(); ++Iter)
	{
		regString += "<register ";
		
		regString += "name=\"" + Iter->name + "\" ";
		regString += "size=\"" + zylib::zycon::toString(Iter->size) + "\" ";
		regString += "editable=\"" + zylib::zycon::toString(Iter->editable) + "\"";
		
		regString += "/>";
	}
	
	regString += "</registers>";

	msglog->log(LOG_ALL, "Register String created: %s", regString.c_str());
	
	return regString;
}

/**
* Creates an XML string that describes the threads of the target process.
*
* @param tids The threads of the target process.
*
* @return An XML string that describes the threads of the target process.
**/
std::string getThreadIdString(unsigned int activeThread, const std::vector<Thread>& tids)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	const char* STATE[] = {"Running", "Suspended"};
	
	std::string tidString = "<threads>";
	
	for (std::vector<Thread>::const_iterator Iter = tids.begin(); Iter != tids.end(); ++Iter)
	{
		tidString += "<thread tid=\"" + zylib::zycon::toString(Iter->tid) + "\" state=\"" + STATE[Iter->state] + "\"";
		
		if (Iter->tid == activeThread)
		{
			tidString += " active=\"" + toString(true) + "\" ";
		}
		
		tidString += "/>";
	}
	
	tidString += "</threads>";
	
	msglog->log(LOG_ALL, "Thread String created: %s", tidString.c_str());
	
	return tidString;
}

std::string getModulesString(const std::vector<Module>& modules)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	std::string modulesString = "<modules>";
	
	for (std::vector<Module>::const_iterator Iter = modules.begin(); Iter != modules.end(); ++Iter)
	{
		modulesString += "<module name=\"" + Iter->name + "\" path=\"" + Iter->path + "\" address=\"" + zylib::zycon::toString(Iter->baseAddress) + "\" size=\"" + zylib::zycon::toString(Iter->size) + "\" />";
	}
	
	modulesString += "</modules>";
	
	msglog->log(LOG_ALL, "Modules String created: %s", modulesString.c_str());
	
	return modulesString;
}

/**
* Creates an XML string that describes the architecture size of the target CPU.
*
* @param tids The size of the target memory.
*
* @return An XML string that describes the target architecture size.
**/
std::string getAddressSizeString(unsigned int size)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	std::string addressSizeString = "<size>" + zylib::zycon::toString(size) + "</size>";
	
	msglog->log(LOG_ALL, "Address Size String created: %s", addressSizeString.c_str());
	
	return addressSizeString;
}

/**
* Creates an XML string that gives information about the target process and the debug
* client.
*
* @param options The debugger options that are available in the debug client.
* @param addressSize The size of the architecture.
* @param registerNames The names of the registers of the target platform.
*
* @return A NaviError code that describes whether the operation was successful or not.
**/
std::string getInformationString(const DebuggerOptions& options, unsigned int addressSize, const std::vector<RegisterDescription>& registerNames)
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	std::string infoString = "<info>";
	
	infoString += getDebuggerOptionsString(options);
	infoString += getRegisterString(registerNames);
	infoString += getAddressSizeString(addressSize);
	
	infoString += "</info>";
	
	msglog->log(LOG_ALL, "Created Info String %s", infoString.c_str());
	
	return infoString;
}