#include "TcpTransport.hpp"

#include <iostream>

#include <zycon/src/zycon.h>

#include "../defs.hpp"
#include "../logger.hpp"

#include "../conns/GenericSocketFunctions.hpp"

/**
* Determines whether data is available from the gdbserver or not.
*
* @return True, if data is available. False, otherwise.
**/
bool TcpTransport::hasData() const
{
	return SocketFunctions::hasData(gdbSocket);
}

/**
* Opens the connection to the gdbclient.
*
* @return A NaviError code that indicates whether the operation succeeded or not.
**/
NaviError TcpTransport::open()
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	gdbSocket = SocketFunctions::connect(host, port);
	
	return gdbSocket ? NaviErrors::SUCCESS : NaviErrors::COULDNT_CONNECT_TO_GDBSERVER;
}

/**
* Closes the connection to the gdbserver.
*
* @return A NaviError code that indicates whether the operation succeeded or not.
**/
NaviError TcpTransport::close()
{
	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	SocketFunctions::close(gdbSocket);
	
	return NaviErrors::SUCCESS;
}

/**
* Sends data to the gdbserver.
*
* @param buffer The data to send.
* @param size The size of the buffer.
*
* @return A NaviError code that indicates whether the operation succeeded or not.
**/
NaviError TcpTransport::send(const char* buffer, unsigned int size) const
{
	return SocketFunctions::send(gdbSocket, buffer, size) == size ? NaviErrors::SUCCESS : NaviErrors::SEND_ERROR;
}

/**
* Reads data from the gdbserver.
*
* @param buffer The buffer the data is written to.
* @param size The number of bytes to read from the gdbserver.
*
* @return A NaviError code that indicates whether the operation succeeded or not.
**/
NaviError TcpTransport::read(char* buffer, unsigned int size) const
{
//	msglog->log(LOG_ALL, "Entering %s", __FUNCTION__);
	
	return SocketFunctions::read(gdbSocket, buffer, size) ? NaviErrors::SUCCESS : NaviErrors::CONNECTION_ERROR;
}
