package com.zynamics.binnavi.standardplugins.callresolver;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import BinNavi.API.disassembly.Address;
import BinNavi.API.disassembly.Function;
import BinNavi.API.disassembly.Module;
import BinNavi.Database.CTableNames;

/**
 * Contains code for finding the indirect call instructions of a module.
 */
public final class IndirectCallFinder
{
	/**
	 * Returns the direct function call addresses for a given module.
	 *
	 * @param module The module whose direct function call addresses are returned.
	 *
	 * @return The direct function call addresses of the module.
	 */
	private static Set<Address> getDirectFunctionCalls(final Module module)
	{
		final Set<Address> set = new HashSet<Address>();

		final String query = "select " + CTableNames.INSTRUCTIONS_TABLE + ".address" +
				" from " + CTableNames.INSTRUCTIONS_TABLE + "" +
				" join " + CTableNames.ADDRESS_REFERENCES_TABLE + " on " + CTableNames.INSTRUCTIONS_TABLE + ".address = " + CTableNames.ADDRESS_REFERENCES_TABLE + ".address AND " + CTableNames.INSTRUCTIONS_TABLE + ".module_id = " + CTableNames.ADDRESS_REFERENCES_TABLE + ".module_id" +
				" where type = 'call_direct' AND " + CTableNames.ADDRESS_REFERENCES_TABLE + ".module_id = " + module.getId();

		try
		{
			final ResultSet resultSet = module.getDatabase().executeQuery(query);

			try
			{
				while (resultSet.next())
				{
					set.add(new Address(resultSet.getLong("address")));
				}
			}
			finally
			{
				resultSet.close();
			}
		}
		catch (final SQLException exception)
		{
			exception.printStackTrace();
		}

		return set;
	}

	/**
	 * Returns information about all indirect call instructions of a module.
	 *
	 * @param module The module whose indirect call instructions are found.
	 *
	 * @return A list of indirect call information.
	 */
	public static List<IndirectCall> find(final Module module)
	{
		final Set<Address> importedFunctionCalls = getDirectFunctionCalls(module);

		final Map<Address, Function> functionMap = new HashMap<Address, Function>();

		for (final Function function : module.getFunctions())
		{
			functionMap.put(function.getAddress(), function);
		}

		final String callMnemonics =
				"'call', " + // x86
				"'bal', 'bgezal', 'bgezall', 'bltzal', 'bltzall', 'jal', 'jalr', " + // MIPS
				"'bl', 'blx', " + // ARM
				"'bcctrl', 'bcctr'" // PowerPC
		;

		final String query = "SELECT " + CTableNames.FUNCTIONS_TABLE + ".address as faddress, " + CTableNames.INSTRUCTIONS_TABLE + ".address" +
				" from " + CTableNames.FUNCTIONS_TABLE + "" +
				" join " + CTableNames.FUNCTION_VIEWS_TABLE + " on " + CTableNames.FUNCTIONS_TABLE + ".address = " + CTableNames.FUNCTION_VIEWS_TABLE + ".function and " + CTableNames.FUNCTIONS_TABLE + ".module_id = " + CTableNames.FUNCTION_VIEWS_TABLE + ".module_id" +
				" join " + CTableNames.NODES_TABLE + " on " + CTableNames.FUNCTION_VIEWS_TABLE + ".view_id = " + CTableNames.NODES_TABLE + ".view_id" +
				" join " + CTableNames.CODENODE_INSTRUCTIONS_TABLE + " on " + CTableNames.NODES_TABLE + ".id = " + CTableNames.CODENODE_INSTRUCTIONS_TABLE + ".node_id" +
				" join " + CTableNames.INSTRUCTIONS_TABLE + " on " + CTableNames.INSTRUCTIONS_TABLE + ".address = " + CTableNames.CODENODE_INSTRUCTIONS_TABLE + ".address AND " + CTableNames.INSTRUCTIONS_TABLE + ".module_id = " + CTableNames.CODENODE_INSTRUCTIONS_TABLE + ".module_id" +
				" join " + CTableNames.OPERANDS_TABLE + " on " + CTableNames.INSTRUCTIONS_TABLE + ".address = " + CTableNames.OPERANDS_TABLE + ".address AND " + CTableNames.INSTRUCTIONS_TABLE + ".module_id = " + CTableNames.OPERANDS_TABLE + ".module_id" +
				" join " + CTableNames.EXPRESSION_TREE_TABLE + "_mapping on " + CTableNames.OPERANDS_TABLE + ".expression_tree_id = " + CTableNames.EXPRESSION_TREE_TABLE + "_mapping.tree_id AND " + CTableNames.EXPRESSION_TREE_TABLE + "_mapping.module_id = " + CTableNames.FUNCTIONS_TABLE + ".module_id" +
				" join " + CTableNames.EXPRESSION_TREE_TABLE + " on " + CTableNames.EXPRESSION_TREE_TABLE + ".id = " + CTableNames.EXPRESSION_TREE_TABLE + "_mapping.tree_node_id AND " + CTableNames.EXPRESSION_TREE_TABLE + ".module_id = " + CTableNames.FUNCTIONS_TABLE + ".module_id" +
				" where " + CTableNames.FUNCTIONS_TABLE + ".module_id = " + module.getId() + " and mnemonic in (" + callMnemonics + ") and (" + CTableNames.EXPRESSION_TREE_TABLE + ".type = 'register' or " + CTableNames.EXPRESSION_TREE_TABLE + ".type = 'dereference')" +
				" group by faddress, address";

		final List<IndirectCall> addresses = new ArrayList<IndirectCall>();

		try
		{
			final ResultSet resultSet = module.getDatabase().executeQuery(query);

			try
			{
				while (resultSet.next())
				{
					final Address address = new Address(resultSet.getLong("address"));

					if (importedFunctionCalls.contains(address))
					{
						continue;
					}

					final Address faddress = new Address(resultSet.getLong("faddress"));

					final Function function = functionMap.get(faddress);

					addresses.add(new IndirectCall(module, function, address));
				}
			}
			finally
			{
				resultSet.close();
			}

			return addresses;
		}
		catch (final SQLException exception)
		{
			exception.printStackTrace();

			return new ArrayList<IndirectCall>();
		}
	}
}
