package com.zynamics.binnavi.standardplugins.pathfinder;

import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.JComponent;
import javax.swing.JMenuItem;

import BinNavi.API.disassembly.BasicBlock;
import BinNavi.API.disassembly.CouldntLoadDataException;
import BinNavi.API.disassembly.Function;
import BinNavi.API.disassembly.Module;
import BinNavi.API.disassembly.ModuleListenerAdapter;
import BinNavi.API.disassembly.View;
import BinNavi.API.helpers.IProgressThread;
import BinNavi.API.helpers.Logger;
import BinNavi.API.helpers.MessageBox;
import BinNavi.API.helpers.ProgressDialog;
import BinNavi.API.plugins.IModuleMenuPlugin;
import BinNavi.API.plugins.PluginInterface;

import com.zynamics.binnavi.standardplugins.utils.GuiHelper;

/**
 * The path finder plugin can be used to find all paths between two basic
 * blocks of an executable file.
 *
 * The plugin is an IModuleMenuPlugin. This means that the plugin extends
 * the context menu of module nodes in the project tree.
 */
public final class PathfinderPlugin implements IModuleMenuPlugin
{
	private PluginInterface m_pluginInterface;

	/**
	 * Shows the pathfinding dialog that is used to select the start block
	 * and the end block of the pathfinding operation.
	 *
	 * @param module The target module of the pathfinding operation.
	 */
	private void showPathfindingDialog(final Module module)
	{
		assert module.isLoaded() : "Internal Error: Target module is not loaded";

		final PathfindingDialog dlg = new PathfindingDialog(m_pluginInterface.getMainWindow().getFrame(), module);

		GuiHelper.centerChildToParent(m_pluginInterface.getMainWindow().getFrame(), dlg, true);

		dlg.setVisible(true);

		if (dlg.wasCancelled())
		{
			return;
		}

		final BasicBlock sourceBlock = dlg.getStartBlock();
		final BasicBlock targetBlock = dlg.getEndBlock();

		final Function firstFunction = dlg.getStartFunction();
		final Function secondFunction = dlg.getEndFunction();

		final CreationThread creationThread = new CreationThread(module, sourceBlock, targetBlock, firstFunction, secondFunction);

		ProgressDialog.show(m_pluginInterface.getMainWindow().getFrame(), "Creating path ...", creationThread);

		if (creationThread.threwException() == false && creationThread.getCreatedView() == null)
		{
			MessageBox.showInformation(m_pluginInterface.getMainWindow().getFrame(), "There is no path between the two selected blocks");
		}
		else
		{
			new Thread()
			{
				@Override
				public void run()
				{
					PluginInterface.instance().showInNewWindow(creationThread.getCreatedView());
				}
			}.start();
		}
	}

	@Override
	public List<JComponent> extendModuleMenu(final List<Module> modules)
	{
		// This function is used to extend the context menu of module
		// nodes in the project tree or in tables of the main window
		// where modules are listed.

		// The module list given as the parameter contains a list of modules.
		// In case the context menu of a module node is created, this list
		// contains exactly one module. In case the context menu of a
		// table is created, the list contains the corresponding modules of the selected
		// rows of the table.

		final List<JComponent> menus = new ArrayList<JComponent>();

		if (modules.size() == 1)
		{
			// The pathfinding functionality is only offered when the list
			// contains just a single module. This means that either a node
			// of the project tree was clicked or just one module is selected
			// in the modules table.

			final Module targetModule = modules.get(0);

			menus.add(new JMenuItem(new PathfindingAction(targetModule)));
		}

		return menus;
	}

	@Override
	public String getDescription()
	{
		return "Finds paths between basic blocks of an executable";
	}

	@Override
	public long getGuid()
	{
		return 4523525670943L;
	}

	@Override
	public String getName()
	{
		return "Pathfinding Plugin";
	}

	@Override
	public void init(final PluginInterface pluginProvider)
	{
		// Nothing to do here => The context menu of module
		// nodes is created in extendPluginMenu

		m_pluginInterface = pluginProvider;
	}

	@Override
	public void unload()
	{
		// Not used yet
	}

	/**
	 * This class is used to update the state of the created context menu
	 * from disabled to enabled once the target module is loaded.
	 */
	private static class ActionUpdater extends ModuleListenerAdapter
	{
		/**
		 * The context menu action to be enabled.
		 */
		private final AbstractAction m_action;

		/**
		 * Creates a new updater object that keeps module state and menu
		 * state synchronized.
		 *
		 * @param action The context menu action to be enabled.
		 */
		public ActionUpdater(final AbstractAction action)
		{
			m_action = action;
		}

		@Override
		public void loadedModule(final Module module)
		{
			// Once the target module is loaded it is possible
			// to enable the context menu to provide the
			// pathfinding functionality for the target module.

			m_action.setEnabled(true);
		}
	}

	/**
	 * This thread is used to create the path.
	 */
	private class CreationThread implements IProgressThread
	{
		private final Module module;
		private final BasicBlock sourceBlock;
		private final BasicBlock targetBlock;
		private final Function firstFunction;
		private final Function secondFunction;
		private View view;
		private boolean threwException = true;

		public CreationThread(final Module module, final BasicBlock sourceBlock, final BasicBlock targetBlock, final Function firstFunction, final Function secondFunction)
		{
			this.module = module;
			this.sourceBlock = sourceBlock;
			this.targetBlock = targetBlock;
			this.firstFunction = firstFunction;
			this.secondFunction = secondFunction;
		}

		@Override
		public boolean close()
		{
			return false;
		}

		public View getCreatedView()
		{
			return view;
		}

		@Override
		public void run()
		{
			try
			{
				view = PathFinder.createPath(module, sourceBlock, targetBlock, firstFunction, secondFunction);

				threwException = false;
			}
			catch (final CouldntLoadDataException e)
			{
				Logger.logException(e);
				MessageBox.showException(m_pluginInterface.getMainWindow().getFrame(), e, "Could not create path");
			}
			catch(final Exception e)
			{
				Logger.logException(e);
				MessageBox.showException(m_pluginInterface.getMainWindow().getFrame(), e, "Could not create path");
			}
		}

		public boolean threwException()
		{
			return threwException;
		}
	}

	/**
	 * This class provides the action that is performed when
	 * the user clicks the context menu entry created by this
	 * plugin.
	 */
	private class PathfindingAction extends AbstractAction
	{
		private static final long serialVersionUID = 6818235709619366218L;

		/**
		 * Target module of the pathfinding operation.
		 */
		private final Module m_module;

		private final ActionUpdater m_updater = new ActionUpdater(this);

		/**
		 * Creates a new pathfinding action object.
		 *
		 * @param module Target module of the pathfinding operation.
		 */
		public PathfindingAction(final Module module)
		{
			super("Pathfinder");

			m_module = module;

			// Pathfinding only works when the target module is loaded.
			// If the target module is not loaded, the context menu
			// must be disabled.
			setEnabled(module.isLoaded());

			// Since the state of the module can change, it might be
			// necessary to update the state of the context menu later.
			m_module.addListener(m_updater);
		}

		@Override
		public void actionPerformed(final ActionEvent e)
		{
			showPathfindingDialog(m_module);
		}
	}
}
