DROP TYPE IF EXISTS "address_reference_type";

CREATE TYPE "address_reference_type" AS ENUM (
    'conditional_true',
    'conditional_false',
    'unconditional',
    'switch',
    'call_direct',
    'call_indirect',
    'call_virtual',
    'data',
    'data_string'
);

DROP TYPE IF EXISTS "architecture_type";

CREATE TYPE "architecture_type" AS ENUM (
    'x86-32',
    'x86-64',
    'ARM-32',
    'PowerPC-32',
    'PowerPC-64',
    'MIPS-32',
    'MIPS-64',
    'GENERIC-32',
    'GENERIC-64',
    'REIL',
    'RREIL'
);

DROP TYPE IF EXISTS "edge_type";

CREATE TYPE "edge_type" AS ENUM (
    'jump_conditional_true',
    'jump_conditional_false',
    'jump_unconditional',
    'jump_switch',
    'jump_conditional_true_loop',
    'jump_conditional_false_loop',
    'jump_unconditional_loop',
    'enter_inlined_function',
    'leave_inlined_function',
    'inter_module',
    'inter_addressspace_edge',
    'textnode_edge',
    'dummy'
);

DROP TYPE IF EXISTS "function_type";

CREATE TYPE "function_type" AS ENUM (
    'normal',
    'library',
    'import',
    'thunk',
    'adjustor_thunk'
);

DROP TYPE IF EXISTS "node_type";

CREATE TYPE "node_type" AS ENUM (
    'code',
    'function',
    'group',
    'text'
);

DROP TYPE IF EXISTS "tag_type";

CREATE TYPE "tag_type" AS ENUM (
    'view_tag',
    'node_tag'
);

DROP TYPE IF EXISTS "view_type";

CREATE TYPE "view_type" AS ENUM (
    'native',
    'non-native'
);

SET default_tablespace = '';
SET default_with_oids = false;

CREATE TABLE "bn_address_references" (
    "module_id" integer NOT NULL,
    "address" bigint NOT NULL,
    "position" integer NOT NULL,
    "expression_id" integer NOT NULL,
    "type" "address_reference_type" NOT NULL,
    "target" bigint NOT NULL
);

CREATE TABLE "bn_address_spaces" (
    "id" integer NOT NULL,
    "project_id" integer NOT NULL,
    "debugger_id" integer,
    "name" text NOT NULL,
    "description" text NOT NULL,
    "creation_date" timestamp without time zone DEFAULT now() NOT NULL,
    "modification_date" timestamp without time zone DEFAULT '1970-01-01 00:00:00'::timestamp without time zone NOT NULL
);

CREATE SEQUENCE "bn_address_spaces_id_seq"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE "bn_address_spaces_id_seq" OWNED BY "bn_address_spaces"."id";

CREATE TABLE "bn_code_nodes" (
    "module_id" integer NOT NULL,
    "node_id" integer NOT NULL,
    "parent_function" bigint,
    "comment" text
);

CREATE TABLE "bn_codenode_instructions"  (
    "module_id" integer NOT NULL,
    "node_id" integer NOT NULL,
    "position" integer NOT NULL,
    "address" bigint NOT NULL,
    "comment" text
);

CREATE TABLE "bn_data_parts"  (
    "module_id" integer NOT NULL,
    "part_id" integer NOT NULL,
    "data" bytea
);

CREATE TABLE "bn_debuggers" (
    "id" integer NOT NULL,
    "name" text NOT NULL,
    "host" text NOT NULL,
    "port" integer NOT NULL
);

CREATE SEQUENCE "bn_debuggers_id_seq"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE "bn_debuggers_id_seq" OWNED BY "bn_debuggers"."id";

CREATE TABLE "bn_edge_paths" (
    "edge_id" integer NOT NULL,
    "position" integer NOT NULL,
    "x" double precision NOT NULL,
    "y" double precision NOT NULL
);

CREATE TABLE "bn_edges" (
    "id" integer NOT NULL,
    "source_node_id" integer NOT NULL,
    "target_node_id" integer NOT NULL,
    "comment" text,
    "x1" double precision NOT NULL,
    "y1" double precision NOT NULL,
    "x2" double precision NOT NULL,
    "y2" double precision NOT NULL,
    "type" edge_type NOT NULL,
    "color" integer NOT NULL,
    "visible" boolean NOT NULL,
    "selected" boolean NOT NULL
);

CREATE SEQUENCE "bn_edges_id_seq"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE "bn_edges_id_seq" OWNED BY "bn_edges"."id";

CREATE TABLE "bn_expression_substitutions" (
    "module_id" integer NOT NULL,
    "address" bigint NOT NULL,
    "position" integer NOT NULL,
    "expression_id" integer NOT NULL,
    "replacement" text NOT NULL
);

CREATE TABLE "bn_expression_tree" (
    "module_id" integer NOT NULL,
    "id" integer NOT NULL,
    "type" integer NOT NULL,
    "symbol" character varying(256),
    "immediate"  bigint,
    "position" integer NOT NULL,
    "parent_id" integer
);

CREATE TABLE "bn_expression_tree_ids" (
    "module_id" integer NOT NULL,
    "id" integer NOT NULL
);

CREATE TABLE "bn_expression_tree_mapping" (
    "module_id" integer NOT NULL,
    "tree_id" integer NOT NULL,
    "tree_node_id" integer NOT NULL
);

CREATE TABLE "bn_function_nodes" (
    "module_id" integer NOT NULL,
    "node_id" integer NOT NULL,
    "function"  bigint NOT NULL,
    "comment" text
);

CREATE TABLE "bn_function_views" (
    "module_id" integer NOT NULL,
    "view_id" integer NOT NULL,
    "function"  bigint NOT NULL
);

CREATE TABLE "bn_functions" (
    "module_id" integer NOT NULL,
    "address" bigint NOT NULL,
    "name" text NOT NULL,
    "original_name" text NOT NULL,
    "type" "function_type" NOT NULL,
    "description" text NOT NULL,
    "comment" text,
    "parent_module_name" text,
    "parent_module_id" integer,
    "parent_module_function" integer
);

CREATE TABLE "bn_global_edge_comments" (
    "src_module_id" integer NOT NULL,
    "dst_module_id" integer NOT NULL,
    "src_address" bigint NOT NULL,
    "dst_address" bigint NOT NULL,
    "comment" text NOT NULL
);

CREATE TABLE "bn_global_node_comments" (
    "module_id" integer NOT NULL,
    "address" bigint NOT NULL,
    "comment" text NOT NULL
);

CREATE TABLE "bn_group_nodes" (
    "node_id" integer NOT NULL,
    "comment" text,
    "collapsed" boolean NOT NULL
);

CREATE TABLE "bn_instructions" (
    "module_id" integer NOT NULL,
    "address" bigint NOT NULL,
    "mnemonic" character varying(32) NOT NULL,
    "comment" text,
    "data" bytea NOT NULL,
    "native" boolean NOT NULL,
    "architecture" architecture_type NOT NULL
);

CREATE TABLE "bn_module_settings" (
    "module_id" integer NOT NULL,
    "name" character varying(255) NOT NULL,
    "value" text NOT NULL
);

CREATE TABLE "bn_module_traces" (
    "module_id" integer NOT NULL,
    "trace_id" integer NOT NULL
);

CREATE TABLE "bn_module_views" (
    "module_id" integer NOT NULL,
    "view_id" integer NOT NULL
);

CREATE TABLE "bn_modules" (
    "id" integer NOT NULL,
    "raw_module_id" integer,
    "name" text NOT NULL,
    "description" text NOT NULL,
    "md5" character(32) NOT NULL,
    "sha1" character(40) NOT NULL,
    "debugger_id" integer,
    "image_base" bigint DEFAULT 0 NOT NULL,
    "file_base" bigint DEFAULT 0 NOT NULL,
    "import_time" timestamp without time zone DEFAULT now() NOT NULL,
    "modification_date" timestamp without time zone DEFAULT '1970-01-01 00:00:00'::timestamp without time zone NOT NULL,
    "data" bytea,
    "stared" boolean DEFAULT false NOT NULL,
    "initialization_state" integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE "bn_modules_id_seq"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE "bn_modules_id_seq" OWNED BY "bn_modules"."id";

CREATE TABLE "bn_nodes" (
    "id" integer NOT NULL,
    "view_id" integer NOT NULL,
    "parent_id" integer,
    "type" node_type NOT NULL,
    "x" double precision NOT NULL,
    "y" double precision NOT NULL,
    "width" double precision NOT NULL,
    "height" double precision NOT NULL,
    "color" integer NOT NULL,
    "bordercolor" integer DEFAULT 0 NOT NULL,
    "selected" boolean NOT NULL,
    "visible" boolean NOT NULL
);

CREATE SEQUENCE "bn_nodes_id_seq"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE "bn_nodes_id_seq" OWNED BY "bn_nodes"."id";

CREATE TABLE "bn_nodes_spacemodules" (
    "module_id" integer NOT NULL,
    "node" integer NOT NULL,
    "address_space" integer
);

CREATE TABLE "bn_operands" (
    "module_id" integer NOT NULL,
    "address" bigint NOT NULL,
    "expression_tree_id" integer NOT NULL,
    "position" integer NOT NULL
);

CREATE TABLE "bn_project_debuggers" (
    "project_id" integer NOT NULL,
    "debugger_id" integer NOT NULL
);

CREATE TABLE "bn_project_settings" (
    "project_id" integer NOT NULL,
    "name" character varying(255) NOT NULL,
    "value" text NOT NULL
);

CREATE TABLE "bn_project_traces" (
    "project_id" integer NOT NULL,
    "trace_id" integer NOT NULL
);

CREATE TABLE "bn_project_views" (
    "project_id" integer NOT NULL,
    "view_id" integer NOT NULL
);

CREATE TABLE "bn_projects" (
    "id" integer NOT NULL,
    "name" text NOT NULL,
    "description" text NOT NULL,
    "creation_date" timestamp without time zone DEFAULT now() NOT NULL,
    "modification_date" timestamp without time zone DEFAULT '1970-01-01 00:00:00'::timestamp without time zone NOT NULL,
    "stared" boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE "bn_projects_id_seq"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE "bn_projects_id_seq" OWNED BY "bn_projects"."id";

CREATE TABLE "bn_space_modules" (
    "module_id" integer NOT NULL,
    "address_space_id" integer NOT NULL,
    "image_base" bigint NOT NULL
);

CREATE TABLE "bn_tagged_nodes" (
    "node_id" integer NOT NULL,
    "tag_id" integer NOT NULL
);

CREATE TABLE "bn_tagged_views" (
    "view_id" integer NOT NULL,
    "tag_id" integer NOT NULL
);

CREATE TABLE "bn_tags" (
    "id" integer NOT NULL,
    "parent_id" integer,
    "name" text NOT NULL,
    "description" text NOT NULL,
    "type" "tag_type" NOT NULL
);

CREATE SEQUENCE "bn_tags_id_seq"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE "bn_tags_id_seq" OWNED BY "bn_tags"."id";

CREATE TABLE "bn_text_nodes" (
    "node_id" integer NOT NULL,
    "text" text
);

CREATE TABLE "bn_trace_event_values" (
    "trace_id" integer NOT NULL,
    "position" integer NOT NULL,
    "register_name" character varying(50) NOT NULL,
    "register_value" bigint NOT NULL,
    "memory_value" bytea NOT NULL
);

CREATE TABLE "bn_trace_events" (
    "module_id" integer,
    "trace_id" integer NOT NULL,
    "position" integer NOT NULL,
    "tid" integer NOT NULL,
    "address" bigint NOT NULL,
    "type" integer
);

CREATE TABLE "bn_traces" (
    "id" integer NOT NULL,
    "view_id" integer NOT NULL,
    "name" text NOT NULL,
    "description" text NOT NULL
);

CREATE SEQUENCE "bn_traces_id_seq"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE "bn_traces_id_seq" OWNED BY "bn_traces"."id";

CREATE TABLE "bn_view_settings" (
    "view_id" integer NOT NULL,
    "name" character varying(255) NOT NULL,
    "value" text NOT NULL
);

CREATE TABLE "bn_views" (
    "id" integer NOT NULL,
    "type" view_type NOT NULL,
    "name" text NOT NULL,
    "description" text NOT NULL,
    "creation_date" timestamp without time zone DEFAULT now() NOT NULL,
    "modification_date" timestamp without time zone DEFAULT '1970-01-01 00:00:00'::timestamp without time zone NOT NULL,
    "stared" boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE "bn_views_id_seq"
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE "bn_views_id_seq" OWNED BY "bn_views"."id";

ALTER TABLE "bn_address_spaces" ALTER COLUMN "id" SET DEFAULT nextval('bn_address_spaces_id_seq'::regclass);
ALTER TABLE "bn_debuggers" ALTER COLUMN "id" SET DEFAULT nextval('bn_debuggers_id_seq'::regclass);
ALTER TABLE "bn_edges" ALTER COLUMN "id" SET DEFAULT nextval('bn_edges_id_seq'::regclass);
ALTER TABLE "bn_modules" ALTER COLUMN "id" SET DEFAULT nextval('bn_modules_id_seq'::regclass);
ALTER TABLE "bn_nodes" ALTER COLUMN "id" SET DEFAULT nextval('bn_nodes_id_seq'::regclass);
ALTER TABLE "bn_projects" ALTER COLUMN "id" SET DEFAULT nextval('bn_projects_id_seq'::regclass);
ALTER TABLE "bn_tags" ALTER COLUMN "id" SET DEFAULT nextval('bn_tags_id_seq'::regclass);
ALTER TABLE "bn_traces" ALTER COLUMN "id" SET DEFAULT nextval('bn_traces_id_seq'::regclass);
ALTER TABLE "bn_views" ALTER COLUMN "id" SET DEFAULT nextval('bn_views_id_seq'::regclass);

ALTER TABLE ONLY "bn_address_references"
    ADD CONSTRAINT "bn_address_references_pkey" PRIMARY KEY ("module_id", "address", "position", "expression_id", "type", "target");

ALTER TABLE ONLY "bn_address_spaces"
    ADD CONSTRAINT "bn_address_spaces_pkey" PRIMARY KEY ("id");

ALTER TABLE ONLY "bn_code_nodes"
    ADD CONSTRAINT "bn_code_nodes_pkey" PRIMARY KEY ("node_id");

ALTER TABLE ONLY "bn_codenode_instructions" 
    ADD CONSTRAINT "bn_codenode_instructions_pkey" 
    PRIMARY KEY ("node_id", "position");

ALTER TABLE ONLY "bn_data_parts" 
    ADD CONSTRAINT "bn_data_parts_pkey" 
    PRIMARY KEY ("module_id", "part_id");

ALTER TABLE ONLY "bn_debuggers"
    ADD CONSTRAINT "bn_debuggers_pkey" 
    PRIMARY KEY ("id");

ALTER TABLE ONLY "bn_edge_paths"
    ADD CONSTRAINT "bn_edge_paths_pkey" 
    PRIMARY KEY ("edge_id", "position");

ALTER TABLE ONLY "bn_edges"
    ADD CONSTRAINT "bn_edges_pkey" 
    PRIMARY KEY ("id");

ALTER TABLE ONLY "bn_expression_substitutions"
    ADD CONSTRAINT "bn_expression_substitutions_pkey" 
    PRIMARY KEY ("module_id", "address", "position", "expression_id");

ALTER TABLE ONLY "bn_expression_tree_ids"
    ADD CONSTRAINT "bn_expression_tree_ids_pkey" 
    PRIMARY KEY ("module_id", "id");

ALTER TABLE ONLY "bn_expression_tree_mapping"
    ADD CONSTRAINT "bn_expression_tree_mapping_pkey" 
    PRIMARY KEY ("module_id", "tree_id", "tree_node_id");

ALTER TABLE ONLY "bn_expression_tree"
    ADD CONSTRAINT "bn_expression_tree_pkey" 
	PRIMARY KEY ("module_id", "id");

ALTER TABLE ONLY "bn_function_nodes"
    ADD CONSTRAINT "bn_function_nodes_pkey" 
	PRIMARY KEY ("node_id");

ALTER TABLE ONLY "bn_function_views"
    ADD CONSTRAINT "bn_function_views_pkey" 
	PRIMARY KEY ("view_id");

ALTER TABLE ONLY "bn_functions"
    ADD CONSTRAINT "bn_functions_pkey" 
	PRIMARY KEY ("module_id", "address");

ALTER TABLE ONLY "bn_global_edge_comments"
    ADD CONSTRAINT "bn_global_edge_comments_pkey" 
	PRIMARY KEY ("src_module_id", "dst_module_id", "src_address", "dst_address");

ALTER TABLE ONLY "bn_global_node_comments"
    ADD CONSTRAINT "bn_global_node_comments_pkey" 
	PRIMARY KEY ("module_id", "address");

ALTER TABLE ONLY "bn_group_nodes"
    ADD CONSTRAINT "bn_group_nodes_pkey" 
	PRIMARY KEY ("node_id");

ALTER TABLE ONLY "bn_instructions"
    ADD CONSTRAINT "bn_instructions_pkey" 
	PRIMARY KEY ("module_id", "address");

ALTER TABLE ONLY "bn_module_settings"
    ADD CONSTRAINT "bn_module_settings_pkey" 
	PRIMARY KEY ("module_id", "name");

ALTER TABLE ONLY "bn_module_traces"
    ADD CONSTRAINT "bn_module_traces_pkey" 
	PRIMARY KEY ("module_id", "trace_id");

ALTER TABLE ONLY "bn_module_views"
    ADD CONSTRAINT "bn_module_views_pkey" 
	PRIMARY KEY ("view_id");

ALTER TABLE ONLY "bn_modules"
    ADD CONSTRAINT "bn_modules_pkey" 
	PRIMARY KEY ("id");

ALTER TABLE ONLY "bn_nodes"
    ADD CONSTRAINT "bn_nodes_pkey" 
	PRIMARY KEY ("id");

ALTER TABLE ONLY "bn_nodes_spacemodules"
    ADD CONSTRAINT "bn_nodes_spacemodules_pkey" 
	PRIMARY KEY ("node");

ALTER TABLE ONLY "bn_operands"
    ADD CONSTRAINT "bn_operands_pkey" 
	PRIMARY KEY ("module_id", "address", "position");

ALTER TABLE ONLY "bn_project_settings"
    ADD CONSTRAINT "bn_project_settings_pkey" 
	PRIMARY KEY ("project_id", "name");

ALTER TABLE ONLY "bn_project_traces"
    ADD CONSTRAINT "bn_project_traces_pkey" 
	PRIMARY KEY ("project_id", "trace_id");

ALTER TABLE ONLY "bn_project_views"
    ADD CONSTRAINT "bn_project_views_pkey" 
	PRIMARY KEY ("view_id");

ALTER TABLE ONLY "bn_projects"
    ADD CONSTRAINT "bn_projects_pkey" 
	PRIMARY KEY ("id");

ALTER TABLE ONLY "bn_space_modules"
    ADD CONSTRAINT "bn_space_modules_pkey" 
	PRIMARY KEY ("address_space_id", "module_id");

ALTER TABLE ONLY "bn_tagged_nodes"
    ADD CONSTRAINT "bn_tagged_nodes_pkey" 
	PRIMARY KEY ("node_id", "tag_id");

ALTER TABLE ONLY "bn_tagged_views"
    ADD CONSTRAINT "bn_tagged_views_pkey" 
	PRIMARY KEY ("view_id", "tag_id");

ALTER TABLE ONLY "bn_tags"
    ADD CONSTRAINT "bn_tags_pkey" 
	PRIMARY KEY ("id");

ALTER TABLE ONLY "bn_text_nodes"
    ADD CONSTRAINT "bn_text_nodes_pkey" 
	PRIMARY KEY ("node_id");

ALTER TABLE ONLY "bn_trace_event_values"
    ADD CONSTRAINT "bn_trace_event_values_pkey" 
	PRIMARY KEY ("trace_id", "position", "register_name");

ALTER TABLE ONLY "bn_trace_events"
    ADD CONSTRAINT "bn_trace_events_pkey" 
	PRIMARY KEY ("trace_id", "position");

ALTER TABLE ONLY "bn_traces"
    ADD CONSTRAINT "bn_traces_pkey" 
	PRIMARY KEY ("id");

ALTER TABLE ONLY "bn_view_settings"
    ADD CONSTRAINT "bn_view_settings_pkey" 
	PRIMARY KEY ("view_id", "name");

ALTER TABLE ONLY "bn_views"
    ADD CONSTRAINT "bn_views_pkey" 
	PRIMARY KEY ("id");

CREATE INDEX "bn_nodes_spacemodules_address_space_idx" 
  ON "bn_nodes_spacemodules" 
  USING btree 
  ("address_space");

CREATE INDEX "bn_expression_substitutions_module_id_idx"
  ON "bn_expression_substitutions"
  USING btree
  ("module_id");

CREATE INDEX "bn_address_references_module_id_idx" 
  ON "bn_address_references" 
  USING btree 
  ("module_id");
CREATE INDEX "bn_address_references_type_idx" 
  ON "bn_address_references" 
  USING btree 
  ("type");
CREATE INDEX "bn_address_references_target_idx" 
  ON "bn_address_references" 
  USING btree 
  ("target");

CREATE INDEX "bn_codenode_instructions_address_idx" 
  ON "bn_codenode_instructions"  
  USING btree 
  ("address");

CREATE INDEX "bn_edges_source_node_id_idx" 
  ON "bn_edges" 
  USING btree 
  ("source_node_id");
CREATE INDEX "bn_edges_target_node_id_idx" 
  ON "bn_edges" 
  USING btree 
  ("target_node_id");

CREATE INDEX "bn_function_nodes_function_idx" 
  ON "bn_function_nodes" 
  USING btree 
  ("function");
CREATE INDEX "bn_function_nodes_module_id_idx" 
  ON "bn_function_nodes" 
  USING btree 
  ("module_id");
CREATE INDEX "bn_function_nodes_module_id_function_idx" 
  ON "bn_function_nodes" 
  USING btree 
  ("module_id", "function" );

CREATE INDEX "bn_function_views_module_id_idx" 
  ON "bn_function_views" 
  USING btree 
  ("module_id");

CREATE INDEX "bn_nodes_view_id_type_idx" 
  ON "bn_nodes" 
  USING btree 
  ("view_id", "type");

CREATE INDEX "bn_project_debuggers_debugger_id_idx" 
  ON "bn_project_debuggers" 
  USING btree 
  ("debugger_id");

CREATE INDEX "bn_trace_events_module_id_idx" 
  ON "bn_trace_events" 
  USING btree 
  ("module_id");

CREATE INDEX "bn_tags_parent_id_idx" 
  ON "bn_tags" 
  USING btree 
  ("parent_id");

CREATE INDEX "bn_functions_parent_module_id_parent_module_function_idx" 
  ON "bn_functions" 
  USING btree 
  ("parent_module_id", "parent_module_function");
CREATE INDEX "bn_functions_module_id_idx" 
  ON "bn_functions" 
  USING btree 
  ("module_id");
CREATE INDEX "bn_functions_address_idx" 
  ON "bn_functions" 
  USING btree 
  ("address");


CREATE INDEX "bn_project_views_project_id_idx" 
  ON "bn_project_views" 
  USING btree 
  ("project_id");

CREATE INDEX "bn_modules_raw_module_id_idx" 
  ON "bn_modules" 
  USING btree 
  ("raw_module_id");

CREATE INDEX "bn_tagged_views_tag_id_idx" 
  ON "bn_tagged_views" 
  USING btree 
  ("tag_id");

CREATE INDEX "bn_project_traces_trace_id_idx" 
  ON "bn_project_traces" 
  USING btree 
  ("trace_id");

CREATE INDEX "bn_views_type_idx" 
  ON "bn_views" 
  USING btree 
  ("type");

CREATE INDEX "bn_traces_view_id_idx" 
  ON "bn_traces" 
  USING btree 
  ("view_id");

CREATE INDEX "fki_bn_code_nodes_module_id_fkey"
  ON "bn_code_nodes"
  USING btree
  ("module_id");

ALTER TABLE ONLY "bn_address_spaces"
    ADD CONSTRAINT "bn_address_spaces_debugger_id_fkey" 
	FOREIGN KEY ("debugger_id") 
	REFERENCES "bn_debuggers"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_address_spaces"
    ADD CONSTRAINT "bn_address_spaces_project_id_fkey" 
	FOREIGN KEY ("project_id") 
	REFERENCES "bn_projects"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_code_nodes"
    ADD CONSTRAINT "bn_code_nodes_node_id_fkey" 
	FOREIGN KEY ("node_id") 
	REFERENCES "bn_nodes"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_code_nodes"
    ADD CONSTRAINT "bn_code_nodes_module_id_fkey" 
        FOREIGN KEY ("module_id")
        REFERENCES "bn_modules" ("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_codenode_instructions" 
    ADD CONSTRAINT "bn_codenode_instructions_node_id_fkey" 
	FOREIGN KEY ("node_id") 
	REFERENCES "bn_nodes"("id") ON DELETE CASCADE;  

ALTER TABLE ONLY "bn_data_parts" 
    ADD CONSTRAINT "bn_data_parts_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_edge_paths"
    ADD CONSTRAINT "bn_edge_paths_edge_id_fkey" 
	FOREIGN KEY ("edge_id") 
	REFERENCES "bn_edges"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_edges"
    ADD CONSTRAINT "bn_edges_source_node_id_fkey" 
	FOREIGN KEY ("source_node_id") 
	REFERENCES "bn_nodes"("id") ON DELETE CASCADE;

ALTER TABLE ONLY "bn_edges"
    ADD CONSTRAINT "bn_edges_target_node_id_fkey" 
	FOREIGN KEY ("target_node_id") 
	REFERENCES "bn_nodes"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_expression_tree"
    ADD CONSTRAINT "bn_expression_tree_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_function_nodes"
    ADD CONSTRAINT "bn_function_nodes_node_id_fkey" 
	FOREIGN KEY ("node_id") 
	REFERENCES "bn_nodes"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_function_views"
    ADD CONSTRAINT "bn_function_views_view_id_fkey" 
	FOREIGN KEY ("view_id") 
	REFERENCES "bn_views"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_functions"
    ADD CONSTRAINT "bn_functions_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_global_node_comments"
    ADD CONSTRAINT "bn_global_node_comments_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_group_nodes"
    ADD CONSTRAINT "bn_group_nodes_node_id_fkey" 
	FOREIGN KEY ("node_id") 
	REFERENCES "bn_nodes"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_instructions"
    ADD CONSTRAINT "bn_instructions_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_module_settings"
    ADD CONSTRAINT "bn_module_settings_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_module_traces"
    ADD CONSTRAINT "bn_module_traces_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_module_traces"
    ADD CONSTRAINT "bn_module_traces_trace_id_fkey" 
	FOREIGN KEY ("trace_id") 
	REFERENCES "bn_traces"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_module_views"
    ADD CONSTRAINT "bn_module_views_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_module_views"
    ADD CONSTRAINT "bn_module_views_view_id_fkey" 
	FOREIGN KEY ("view_id") 
	REFERENCES "bn_views"("id") ON DELETE CASCADE;

ALTER TABLE ONLY "bn_modules"
    ADD CONSTRAINT "bn_modules_debugger_id_fkey" 
	FOREIGN KEY ("debugger_id") 
	REFERENCES "bn_debuggers"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_modules"
    ADD CONSTRAINT "bn_modules_raw_module_id_fkey" 
	FOREIGN KEY ("raw_module_id") 
	REFERENCES "modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_nodes"
    ADD CONSTRAINT "bn_nodes_parent_id_fkey" 
	FOREIGN KEY ("parent_id") 
	REFERENCES "bn_group_nodes"("node_id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_nodes_spacemodules"
    ADD CONSTRAINT "bn_nodes_spacemodules_address_space_fkey" 
	FOREIGN KEY ("address_space") 
	REFERENCES "bn_address_spaces"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_nodes_spacemodules"
    ADD CONSTRAINT "bn_nodes_spacemodules_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_nodes_spacemodules"
    ADD CONSTRAINT "bn_nodes_spacemodules_node_fkey" 
	FOREIGN KEY ("node") 
	REFERENCES "bn_nodes"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_project_debuggers"
    ADD CONSTRAINT "bn_project_debuggers_debugger_id_fkey" 
	FOREIGN KEY ("debugger_id") 
	REFERENCES "bn_debuggers"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_project_debuggers"
    ADD CONSTRAINT "bn_project_debuggers_project_id_fkey" 
	FOREIGN KEY ("project_id") 
	REFERENCES "bn_projects"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_project_settings"
    ADD CONSTRAINT "bn_project_settings_project_id_fkey" 
	FOREIGN KEY ("project_id") 
	REFERENCES "bn_projects"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_project_traces"
    ADD CONSTRAINT "bn_project_traces_project_id_fkey" 
	FOREIGN KEY ("project_id") 
	REFERENCES "bn_projects"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_project_traces"
    ADD CONSTRAINT "bn_project_traces_trace_id_fkey" 
	FOREIGN KEY ("trace_id") 
	REFERENCES "bn_traces"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_project_views"
    ADD CONSTRAINT "bn_project_views_project_id_fkey" 
	FOREIGN KEY ("project_id") 
	REFERENCES "bn_projects"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_project_views"
    ADD CONSTRAINT "bn_project_views_view_id_fkey" 
	FOREIGN KEY ("view_id") 
	REFERENCES "bn_views"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_space_modules"
    ADD CONSTRAINT "bn_space_modules_address_space_id_fkey" 
	FOREIGN KEY ("address_space_id") 
	REFERENCES "bn_address_spaces"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_space_modules"
    ADD CONSTRAINT "bn_space_modules_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_tagged_nodes"
    ADD CONSTRAINT "bn_tagged_nodes_node_id_fkey" 
	FOREIGN KEY ("node_id") 
	REFERENCES "bn_nodes"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_tagged_nodes"
    ADD CONSTRAINT "bn_tagged_nodes_tag_id_fkey" 
    FOREIGN KEY ("tag_id") 
    REFERENCES "bn_tags"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_tagged_views"
    ADD CONSTRAINT "bn_tagged_views_tag_id_fkey" 
	FOREIGN KEY ("tag_id") 
	REFERENCES "bn_tags"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_tagged_views"
    ADD CONSTRAINT "bn_tagged_views_view_id_fkey" 
	FOREIGN KEY ("view_id") 
	REFERENCES "bn_views"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_tags"
    ADD CONSTRAINT "bn_tags_parent_id_fkey" 
	FOREIGN KEY ("parent_id") 
	REFERENCES "bn_tags"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_text_nodes"
    ADD CONSTRAINT "bn_text_nodes_node_id_fkey" 
	FOREIGN KEY ("node_id") 
	REFERENCES "bn_nodes"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_trace_events"
    ADD CONSTRAINT "bn_trace_events_module_id_fkey" 
	FOREIGN KEY ("module_id") 
	REFERENCES "bn_modules"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_trace_events"
    ADD CONSTRAINT "bn_trace_events_trace_id_fkey" 
	FOREIGN KEY ("trace_id") 
	REFERENCES "bn_traces"("id") ON DELETE CASCADE; 

ALTER TABLE ONLY "bn_view_settings"
    ADD CONSTRAINT "bn_view_settings_view_id_fkey" 
	FOREIGN KEY ("view_id") 
	REFERENCES "bn_views"("id") ON DELETE CASCADE; 
