/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-97 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su
 *                              FIDO:   2:5020/209
 *
 */

#ifndef _AREG_HPP
#define _AREG_HPP
#pragma pack(push, 1)           // IDA uses 1 byte alignments!

//
//      This file contains definitions of register architecture.
//      These definitions are not used by IDP modules anymore,
//      so don't use this file at all.
//      I hope that it will disappear in the future
//

#define MAXREG  22

class WorkReg {
  friend class reg;
  ushort val;
  ushort flag;
#define RF_DEF   0x0001         // Defined
#define RF_TYPE  0x0042         // Mask for type
#define         RFT_OFF 0x0002          // Offset
#define         RFT_SEG 0x0040          // Segment
#define         RFT_INT 0x0000          // Integer
#define         RFT_BYT 0x0042          // Byte
#define          RFB_HIGH 0x0004                // High byte defined
#define          RFB_LOW  0x0010                // Low  byte defined

#define RF_FORCE 0x0100         // Forced
#define RF_UNDEF 0x0200         // Undefined
#define RF_LUNDF 0x0400         // Undefined
#define RF_HUNDF 0x0800         // Undefined

public:

        ushort value    (void)  { return val; }
        ushort high     (void)  { return val >> 8;   }
        ushort low      (void)  { return val & 0xFF; }

        friend int isEmpty (WorkReg &r);
        friend int isDef   (WorkReg &r);
        friend int HisDef  (WorkReg &r);
        friend int LisDef  (WorkReg &r);
        friend int isAny   (WorkReg &r);
        friend int HisAny  (WorkReg &r);
        friend int LisAny  (WorkReg &r);
        friend int isForced(WorkReg &r);
        friend int isOff   (WorkReg &r);
        friend int isSeg   (WorkReg &r);
        friend int isInt   (WorkReg &r);
        friend int isByt   (WorkReg &r);

        void   undef    (void);
        void   Lundef   (void);
        void   Hundef   (void);

        void   doForce  (ushort x);
        void   doInt    (ushort x);
        void   doSeg    (ushort x);
        void   doOff    (void);
        void   doHigh   (uchar h);
        void   doLow    (uchar l);

        friend int operator==(WorkReg &r1,WorkReg &r2)
                { return r1.flag == r2.flag && r1.val == r2.val; }
        friend int operator!=(WorkReg &r1,WorkReg &r2)
                { return ! (r1 == r2); }

        char    *desc(char *name);
};

struct AbstractRegister {
  WorkReg *ptr;

        void     setLink(WorkReg *r)  { ptr   = r; }
        WorkReg *getLink(void)        { return ptr; }

        int isForced(void) { return ::isForced(*ptr); }
        int isEmpty (void) { return ::isEmpty(*ptr);  }

        virtual ushort       value (void) = 0;
        virtual void         undef (void) = 0;
        virtual void         doInt (ushort x) = 0;
        virtual void         doForce(ushort x) = 0;

        virtual int isDef   (void) = 0;
        virtual int HisDef  (void) = 0;
        virtual int LisDef  (void) = 0;
        virtual int isAny   (void) = 0;
        virtual int HisAny  (void) = 0;
        virtual int LisAny  (void) = 0;
        virtual int isOff   (void) = 0;
        virtual int isSeg   (void) = 0;
        virtual int isInt   (void) = 0;
        virtual int isByt   (void) = 0;
};

struct ByteReg : public AbstractRegister {

        virtual ushort       value (void) = 0;
        virtual void         undef (void) = 0;
        virtual void         doInt (ushort x) = 0;
        virtual int isDef   (void) = 0;
        virtual int isAny   (void) = 0;
        virtual int HisDef  (void)      { return 0; }
        virtual int LisDef  (void)      { return 0; }

        void doForce(ushort)    { }
        int HisAny  (void) { return 0; }
        int LisAny  (void) { return 0; }
        int isOff  (void)  { return 0; }
        int isSeg  (void)  { return 0; }
        int isInt  (void)  { return 0; }
        int isByt  (void)  { return 1; }
};

struct HighReg : public ByteReg {

        ushort       value (void) { return ptr->high(); }
                 int isDef (void) { return ::HisDef(*ptr); }
        void         undef (void) { ptr->Hundef(); }
        void         doInt (ushort x){ ptr->doHigh(char(x)); }
                 int isAny (void) { return ::HisAny(*ptr); }

};

struct LowReg : public ByteReg {

        ushort       value (void) { return ptr->low(); }
                 int isDef (void) { return ::LisDef(*ptr); }
        void         undef (void) { ptr->Lundef(); }
        void         doInt (ushort x){ ptr->doLow(char(x)); }
                 int isAny (void) { return ::LisAny(*ptr); }
};

struct WordReg : public AbstractRegister {

        ushort       value (void) { return ptr->value(); }
                 int isDef (void) { return ::isDef(*ptr); }
        void         undef (void) { ptr->undef(); }
        void         doInt (ushort x){ ptr->doInt(x); }
        void         doForce(ushort x) { ptr->doForce(x); }
                 int isAny (void) { return ::isAny(*ptr); }
        int HisAny  (void) { return ::HisAny(*ptr); }
        int LisAny  (void) { return ::LisAny(*ptr); }
        int HisDef  (void) { return ::HisDef(*ptr); }
        int LisDef  (void) { return ::LisDef(*ptr); }
        int isOff   (void) { return ::isOff(*ptr);  }
        int isSeg   (void) { return ::isSeg(*ptr); }
        int isInt   (void) { return ::isInt(*ptr); }
        int isByt   (void) { return 0; }
};

#define maxRegSet       7               // Max register set index

WorkReg *regLink(WorkReg *set);         // Link Abstract registers
                                        // to given set
                                        // Done autom-lly by LoadRegs()


#pragma pack(pop)
#endif // _AREG_HPP
