/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-97 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su
 *                              FIDO:   2:5020/209
 *
 */

#ifndef BYTES_HPP
#define BYTES_HPP

#include <llong.hpp>    // for long long
#include <idp.hpp>      // ph.nbits
#include <lines.hpp>
#include <segment.hpp>  // for is_visible_segm()
#pragma pack(push, 1)   // IDA uses 1 byte alignments!

typedef ulong enum_t;   // #include <enum.hpp>


//
//      This file contains functions that deal with individual byte
//      characteristics.
//      Each byte of a program being disassembled is represented by a 32-bit
//      quantity. We will call it 'flags'. Structure of flags is represented
//      below. You are not allowed to inspect individual bits of flags
//      and modify them directly. Use special function to inspect and/or
//      modify flags.
//      Flags are kept in a virtual array file (*.id1)
//      Addresses (ea) are all 32-bit quantities.
//
//--------------------------------------------------------------------------
// ida virtual memory configuration:
// (not reachable from modules)
extern ushort ida_vpagesize;    // Virtual memory page size       (*.id1)
extern ushort ida_vpages;       // Size of virtual memory window  (*.id1)
extern ushort ida_npagesize;    // Name pointers page size        (*.nam)
extern ushort ida_npages;       // Number of name pointer pages   (*.nam)

//--------------------------------------------------------------------------
// initialize virtual array (*.id1)
//     input_size - input file size (used only if database didn't exist before)

void FlagsInit(ulong input_size);


// terminate virtual array

void FlagsTerm(void);


// make virtual array empty.
// returns: 1-ok, 0-failure(never should occur)

bool FlagsReset(void);


// allocate flags for address range (startEA..endEA)
// startEA should be lower than endEA.
// endEA does not belong to the range.
// exit with an error message if not enough disk space
// return 0 if ok
// otherwise return error code

int FlagsEnable(ea_t startEA,ea_t endEA);


// deallocate flags for address range (startEA..endEA)
// startEA should be lower than endEA.
// endEA does not belong to the range.
// exit with an error message if not enough disk space (this may occur too)
// return 0 if ok
// otherwise return error code

int FlagsDisable(ea_t startEA,ea_t endEA);


// Get next address in the program. (i.e. address which has flags)
// Return BADADDR if no such address exist.

ida_export ea_t nextaddr(ea_t ea);


// Get previous address in the program.
// Return BADADDR if no such address exist.

ida_export ea_t prevaddr(ea_t ea);


// Get start of next contiguous addresses in the program.
// (a block of addresses without holes)
// Return BADADDR if next chunk doesn't exist.

ida_export ea_t nextchunk(ea_t ea);


// Get start of previous contiguous addresses in the program.
// Return BADADDR if previous chunk doesn't exist.

ida_export ea_t prevchunk(ea_t ea);


// Get start of the contiguous address block containing 'ea'.
// Return BADADDR if 'ea' doesn't belong to the program.

ida_export ea_t chunkstart(ea_t ea);


// Get size of the contiguous address block containing 'ea'.
// Return 0 if 'ea' doesn't belong to the program.

ida_export ulong chunksize(ea_t ea);

// Search for a hole in the addressing space of the program.
// bottom - address to start searching
// size   - size of desired block
// step   - bit mask for the start of hole (0xF would align hole to a paragraph)
//          if 'step' is negative, the bottom address with be aligned
//          otherwise the kernel will try to use it as is and align it
//          only the hole is too small.
// Return start of the hole or BADADDR

ida_export ea_t freechunk(ea_t bottom,ulong size,long step);


// Find next address with a flag satisfying the function 'testf'.
// Start searching from address 'ea'+1 and inspect bytes up to 'maxea'.
// maxea is not included in the search range
// Return the found address or BADADDR.

ida_export ea_t nextthat(ea_t ea,ea_t maxea,bool (*testf)(flags_t flags));


// Find previous address with a flag satisfying the function 'testf'.
// Start searching from address 'ea'+1 and inspect bytes up to 'minea'.
// minea is included in the search range
// Return the found address or BADADDR.

ida_export ea_t prevthat(ea_t ea,ea_t minea,bool (*testf)(flags_t flags));


// Get start of previous defined item. Return BADADDR if none exist.
// minea is included in the search range

ida_export ea_t prev_head(ea_t ea, ea_t minea);


// Get start of next defined item. Return BADADDR if none exist.
// maxea is not included in the search range

ida_export ea_t next_head(ea_t ea, ea_t maxea);


// Get address of previous non-tail byte. Return BADADDR if none exist.

ida_export ea_t prev_not_tail(ea_t ea);


// Get address of next non-tail byte. Return BADADDR if none exist.

ida_export ea_t next_not_tail(ea_t ea);


// Get address of the head of the item (see definition at the end of
// this file)

inline ea_t get_item_head(ea_t ea);


// Adjust the address and get the nearest visible address
// (i.e. an adress which will appear in the disassembly)
// This function returns BADADDR only if no addresses are valid

ida_export ea_t adjust_visea(ea_t ea);


// Get previous visible address
// Returns BADADDR if none exists.

ida_export ea_t prev_visea(ea_t ea);


// Get next visible address
// Returns BADADDR if none exists.

ida_export ea_t next_visea(ea_t ea);


// Is an address the first visible address?

ida_export bool is_first_visea(ea_t ea);


// Is an address the last visible address?

ida_export bool is_last_visea(ea_t ea);


// Is the address visible on the screen (not hidden)?

bool is_visible_finally(ea_t ea); // do we need to show anything
                                  // at this address?

inline bool is_finally_visible_item(ea_t ea) // is instruction visible?
 { return (inf.s_cmtflg & SW_SHHID_ITEM) != 0 || is_visible_item(ea); }

inline bool is_finally_visible_func(func_t *pfn) // is function visible?
 { return (inf.s_cmtflg & SW_SHHID_FUNC) != 0 || is_visible_func(pfn); }

inline bool is_finally_visible_segm(segment_t *s) // is segment visible?
 { return (inf.s_cmtflg & SW_SHHID_SEGM) != 0 || is_visible_segm(s); }


// this function is only for the kernel
void invalidate_visea_cache(void);

// Get end address of the item at 'ea'. The returned address
// doesn't belong to the current item. Unexplored bytes are counted as
// 1 byte entities.

ida_export ea_t get_item_end(ea_t ea);
#define itemEnd get_item_end            // for compatibility


// Calculate maximal reasonable end address of a new item.
// This function will limit the item with the current segment bounds.
//      ea      - linear address.
//      how     - when to stop the search. A combination of ITEM_END...
// returns: end of new item. If it is not possible to create an item,
//          it will return 'ea'.

#define ITEM_END_FIXUP  0x0001          // stop at the first fixup
#define ITEM_END_INITED 0x0002          // if  isLoaded(ea): stop if uninitialized byte is encountered
                                        // if !isLoaded(ea): stop if   initialized byte is encountered
#define ITEM_END_NAME   0x0004          // stop at the first named location
#define ITEM_END_XREF   0x0008          // stop at the first referenced location
ida_export ea_t calc_max_item_end(ea_t ea, int how=15);



// Get size of item (instruction/data) in bytes. Unexplored bytes have
// length of 1 byte. This function never returns 0.

inline ulong getSize(ea_t ea) { return get_item_end(ea) - ea; }


// Flush virtual array to disk

void fluFlags(void);


// Is the specified address 'ea' present in the program?

ida_export bool isEnabled(ea_t ea);


// Get flags value for address 'ea'. If address is not present in the program,
// return 0.

ida_export flags_t getFlags(ea_t ea);


// Get flag of the item at 'ea' even if it is a tail byte of some
// array or structure. This function is used to get flags of structure members
// or array elements.
//      from    - linear address of the instruction which refers to 'ea'
//      n       - number of operand which refers to 'ea'
//      ea      - the referenced address
//      appzero - meaningful only if the name refers to a structure.
//                append a struct field name if the field offset is zero?
// returns: flags or 0 (if failed)

ida_export flags_t get_item_flag(ea_t from, int n, ea_t ea, bool appzero);


// Modify flags value for address 'ea'.  (ONLY KERNEL MAY USE THIS FUNCTION!)

ida_export void setFlags(ea_t ea,flags_t flags);


// Set the specified bits of flags (ONLY KERNEL MAY USE THIS FUNCTION!)

ida_export void setFlbits(ea_t EA,flags_t bits);


// Clear the specified bits of flags (ONLY KERNEL MAY USE THIS FUNCTION!)

ida_export void clrFlbits(ea_t EA,flags_t bits);


// Maximum address allowed to use in the program being disassebled.
// This is obsolete, don't use it!

#define MAXADDR         0xFF000000L             // Max allowed address in the
                                                // IDA (excluded)


//--------------------------------------------------------------------------
//
//      Here we define structure of flags.
//      Low 8 bits contain value of corresponding byte of the program.
//      The next bit is set if the byte is initialized
//

#define MS_VAL  0x000000FFL             // Mask for byte value
#define FF_IVL  0x00000100L             // Byte has value ?

// Do flags contain byte value?

inline  bool     hasValue(flags_t F)  { return (F & FF_IVL) != 0; }


// Get byte value from flags

inline  uchar   byteValue(flags_t F) { return ((uchar)(F & MS_VAL)); }


// Delete byte value from flags. The corresponding byte becomes
// uninitialized.

inline  void    delValue(ea_t ea) { clrFlbits(ea,FF_IVL); }


// Does the specified address have a byte value (is initialized?)

inline bool isLoaded     (ea_t ea) { return hasValue(getFlags(ea)); }


// Get one byte (8-bit) of the program at 'ea'
// This function works only for 8bit byte processors.

ida_export uchar get_byte(ea_t ea);


// Get one word (16-bit) of the program at 'ea'
// This function takes into account order of bytes specified in inf.mf
// This function works only for 8bit byte processors.

ida_export ushort get_word(ea_t ea);

// Get one dword (32-bit) of the program at 'ea'
// This function takes into account order of bytes specified in inf.mf
// This function works only for 8bit byte processors.

ida_export ulong get_long(ea_t ea);


// Get one qword (64-bit) of the program at 'ea'
// This function takes into account order of bytes specified in inf.mf
// This function works only for 8bit byte processors.

ida_export ulonglong get_qword(ea_t ea);


// Get one wide byte of the program at 'ea'
// Some processors may access more than 8bit quantity at an address.
// These processors have 32-bit byte organization from the IDA's point of view.

ida_export ulong get_full_byte(ea_t ea);


// Get one wide word (64-bit) of the program at 'ea'
// Some processors may access more than 8bit quantity at an address.
// These processors have 32-bit byte organization from the IDA's point of view.
// This function takes into account order of bytes specified in inf.mf

ida_export ulonglong get_full_word(ea_t ea);


// Get two wide words (128-bit) of the program at 'ea'
// Some processors may access more than 8bit quantity at an address.
// These processors have 32-bit byte organization from the IDA's point of view.
// This function takes into account order of bytes specified in inf.mf
// Note: this function works incorrectly if ph.nbits > 16

ida_export ulonglong get_full_long(ea_t ea);


// Get 8 bits of the program at 'ea'
// The main usage of this function is to iterate range of bytes.
// Here is an example:
//      ulong v;
//      int nbit = 0;
//      for ( .... ) {
//        uchar byte = get_8bit(ea,v,nbit);
//        ...
//      }
// 'ea' is incremented each time when a new byte is read.
// In the above example, it will be incremented in the first loop iteration.

ida_export uchar get_8bit(ea_t &ea,ulong &v,int &nbit);


// Get 16bits of the program at 'ea'
// If the current processor has 16-bit byte, return 1 byte (getFullByte())
// Otherwise return getWord()

ida_export ulong get_16bit(ea_t ea);


// Get not more than 32bits of the program at 'ea'
// Depending on the ph.nbits, returns 32 bit value
//   if ( nbits <= 8 ) return get_long(ea);
//   if ( nbits <= 16) return get_full_word(ea);
//   return get_full_byte(ea);

ida_export ulong get_32bit(ea_t ea);


// Get original byte value (that was before patching)
// This function works for wide byte processors too.

ida_export ulong get_original_byte(ea_t ea);


// Get original word value (that was before patching)
// This function works for wide byte processors too.
// This function takes into account order of bytes specified in inf.mf

ida_export ulonglong get_original_word(ea_t ea);


// Get original long value (that was before patching)
// This function works for wide byte processors too.
// This function takes into account order of bytes specified in inf.mf

ida_export ulonglong get_original_long(ea_t ea);


// Set value of one byte of the program.
// This function works for wide byte processors too.
//      ea - linear address
//      x  - byte value

ida_export void put_byte(ea_t ea,ulong x);

// Set value of one word of the program.
// This function takes into account order of bytes specified in inf.mf
// This function works for wide byte processors too.
//      ea - linear address
//      x  - word value

ida_export void put_word(ea_t ea,ulonglong x);


// Set value of one dword of the program.
// This function takes into account order of bytes specified in inf.mf
// This function works for wide byte processors too.
//      ea - linear address
//      x  - dword value

ida_export void put_long(ea_t ea,ulonglong x);


// Set value of one qword (8 bytes) of the program.
// This function takes into account order of bytes specified in inf.mf
// This function works for wide byte processors too.
//      ea - linear address
//      x  - qword value

ida_export void put_qword(ea_t ea,ulonglong x);


// Patch a byte of the program. The original value of the byte is saved
// and can be obtained by get_original_byte() function.
// This function works for wide byte processors too.

ida_export void patch_byte(ea_t ea,ulong x);


// Patch a word of the program. The original value of the word is saved
// and can be obtained by get_original_word() function.
// This function works for wide byte processors too.
// This function takes into account order of bytes specified in inf.mf

ida_export void patch_word(ea_t ea,ulonglong x);


// Patch a dword of the program. The original value of the dword is saved
// and can be obtained by get_original_long() function.
// This function DOESN'T work for wide byte processors.
// This function takes into account order of bytes specified in inf.mf

ida_export void patch_long(ea_t ea,ulonglong x);


// Add a value to one byte of the program.
// This function works for wide byte processors too.
//      ea     - linear address
//      value  - byte value

ida_export void add_byte(ea_t ea,ulong value);


// Add a value to one word of the program.
// This function works for wide byte processors too.
// This function takes into account order of bytes specified in inf.mf
//      ea     - linear address
//      value  - byte value

ida_export void add_word(ea_t ea,ulonglong value);


// Add a value to one dword of the program.
// This function works for wide byte processors too.
// This function takes into account order of bytes specified in inf.mf
//      ea     - linear address
//      value  - byte value
// Note: this function works incrrectly if ph.nbits > 16

ida_export void add_long(ea_t ea,ulonglong value);


// Get the specified number of bytes of the program into the buffer.
//       ea - linear address
//       buf - buffer to hold bytes
//       size - size of buffer in normal 8-bit bytes (sizeof(buf))
// returns: 1-ok
//          0-failure
//            (out of program address space or uninited bytes, for example)

ida_export bool get_many_bytes(ea_t ea,void *buf,int size);


// Modify the specified number of bytes of the program.
// This function does not save the original values of bytes.
// (this function is for the kernel only)
//       ea - linear address
//       buf - buffer with new values of bytes
//       size - size of buffer in normal 8-bit bytes (sizeof(buf))
// See also patch_many_bytes() function.

ida_export void put_many_bytes(ea_t ea,const void *buf,int size);


// Patch the specified number of bytes of the program.
// Original values of bytes are saved and are available with get_original...()
// functions.
// (this function is for the kernel only)
//       ea - linear address
//       buf - buffer with new values of bytes
//       size - size of buffer in normal 8-bit bytes (sizeof(buf))
// See also put_many_bytes() function.

ida_export void patch_many_bytes(ea_t ea,const void *buf,int size);


//-------------------------------------------------------------------------
//      Each byte of the program may be in one of four states:
//        - unexplored
//        - start of instruction
//        - start of data
//        - second, third (tail) byte of instruction or data.
//      Initially all bytes of the program are unexplored.
//      IDA modifies flags and doing so converts bytes to instructions
//      and data.

#define MS_CLS  0x00000600L             // Mask for typing
#define FF_CODE 0x00000600L             // Code ?
#define FF_DATA 0x00000400L             // Data ?
#define FF_TAIL 0x00000200L             // Tail ?
#define FF_UNK  0x00000000L             // Unknown ?

// Does flag denote start of an instruction?

inline  bool     isCode(flags_t F)  { return (F & MS_CLS) == FF_CODE; }


// Does flag denote start of data?

inline  bool     isData(flags_t F)  { return (F & MS_CLS) == FF_DATA; }


// Does flag denote tail byte?

inline  bool     isTail(flags_t F)  { return (F & MS_CLS) == FF_TAIL; }


// Does flag denote unexplored byte?

inline  bool     isUnknown(flags_t F){return (F & MS_CLS) == FF_UNK;  }


// Does flag denote start of instruction OR data?

inline  bool     isHead(flags_t F)  { return (F & FF_DATA) != 0; }


// Convert item (instruction/data) to unexplored bytes
//      expand - propogate undefined items, for example removing an instruction
//               removes all references to the next instruction, then plan
//               to convert to unexplored the next instruction too.

ida_export void do_unknown(ea_t ea, bool expand);
#define doUnknown do_unknown // temporary - for compatibility

// Convert range of addresses to unexplored bytes
//      ea     - starting linear address
//      size   - number of bytes in the area to be undefined
//      expand - propogate undefined items, for example removing an instruction
//               removes all references to the next instruction, then plan
//               to convert to unexplored the next instruction too.

ida_export void do_unknown(ea_t ea, size_t size, bool expand);


//-------------------------------------------------------------------------
// Manual instructions (they are used to completely override an automatically
// generated instruction by a user specified string).

// Is the instruction overridden?
//      ea - linear address of the instruction or data item

ida_export bool is_manual_insn(ea_t ea);        // Is the instruction overridden?


// Retrieve the user-specified string for the manual instruction
//      ea       - linear address of the instruction or data item
//      buf      - the input buffer for the string
//                 if NULL, the buffer will be allocated in heap
//      bufsize  - size of the buffer if buf != NULL
//                 bufsize may be NULL
// returns: NULL - no user-specified string

ida_export char *get_manual_insn(ea_t ea,char *buf,ulong bufsize);
                                // Get the user-specified instruction string


// Set manual instruction string
//      ea - linear address of the instruction or data item
//      manual_insn -   ""   - delete manual string
//                      NULL - do nothing
// If "manual_insn" is equal to an empty string, then the
// manual instruction string will be deleted

ida_export void set_manual_insn(ea_t ea,const char *manual_insn); // Set user-speficied string


//-------------------------------------------------------------------------
//
//      Flags keep information common to all four states of bytes.
//      This information will not be automatically discarded during
//      transitions between different states.
//

#define MS_COMM  0x000FF800L            // Mask of common bits
#define FF_COMM  0x00000800L            // Has comment ?
#define FF_REF   0x00001000L            // has references
#define FF_LINE  0x00002000L            // Has next or prev lines ?
#define FF_NAME  0x00004000L            // Has name ?
#define FF_LABL  0x00008000L            // Has dummy name?
#define FF_FLOW  0x00010000L            // Exec flow from prev instruction
#define FF_SIGN0 0x00020000L            // Inverted signness of operand 0
#define FF_SIGN1 0x00040000L            // Inverted signness of operand 1
#define FF_VAR   0x00080000L            // is variable byte?

// Does the previous instruction exist and pass execution flow to the current
// byte?

inline  bool isFlow (flags_t F)     { return (F & FF_FLOW) != 0; }


// Is the current byte marked as variable? (this is not used by IDA itself and
// may become obsolete, please don't use it)

inline  bool isVar  (flags_t F)     { return (F & FF_VAR ) != 0; }


// Does the current byte have additional anterior or posterior lines?

inline  bool hasExtra(flags_t F)    { return (F & FF_LINE) != 0; }


// Does the current byte have an indented comment?

inline  bool has_cmt(flags_t F)     { return (F & FF_COMM) != 0; }


// Does the current byte have cross-references to it?

inline  bool hasRef (flags_t F)     { return (F & FF_REF)  != 0; }


// Does the current byte have non-trivial (non-dummy) name?

inline  bool has_name(flags_t F)    { return (F & FF_NAME) != 0; }


// Does the current byte have dummy (auto-generated, with special prefix) name?

#define FF_ANYNAME      (FF_LABL|FF_NAME)
inline  bool has_dummy_name(flags_t F){ return (F & FF_ANYNAME) == FF_LABL; }


// Does the current byte have auto-generated (no special prefix) name?

inline  bool has_auto_name(flags_t F){ return (F & FF_ANYNAME) == FF_ANYNAME; }


// Does the current byte have any name?

inline  bool has_any_name(flags_t F){ return (F & FF_ANYNAME) != 0; }


// Does the current byte have user-specified name?

inline  bool has_user_name(flags_t F){ return (F & FF_ANYNAME) == FF_NAME; }


// signness deals with the form of operands of the current instruction/data.
// inverted signness means the following:
//    if the bit is clear       |then when the bit is set
//    and the output is         |the output should be:
//    ------------              |----------
//    unsigned                  |signed
//    signed                    |unsigned
//

// Should signness of the first operand inverted during output?

inline  bool inverted_signness0(flags_t F){ return (F & FF_SIGN0) != 0; }


// Should signness of other operands inverted during output?

inline  bool inverted_signness1(flags_t F){ return (F & FF_SIGN1) != 0; }


// Should signness of n-th operand inverted during output?
// allowed values of n: 0-first operand, 1-other operands

bool inverted_signness(flags_t F,int n);


// Toggle signness of the first operand

inline void toggle_signness0(ea_t ea) { setFlags(ea,getFlags(ea)^FF_SIGN0); }


// Toggle signness of the other operands

inline void toggle_signness1(ea_t ea) { setFlags(ea,getFlags(ea)^FF_SIGN1); }


// Toggle signness of n-th operand
// allowed values of n: 0-first operand, 1-other operands

inline bool toggle_signness(ea_t ea,int n)
  { if ( n == 0 ) toggle_signness0(ea); else toggle_signness1(ea); return 1;}


// get bits for signness (this function is used by kernel only)

inline flags_t signflag(void) { return FF_SIGN0|FF_SIGN1; }


// Work with binary negation of operands
// ash.bnot should be defined in the idp module in order to work
// with this functions

bool is_bnot(ea_t ea,int n);            // should negate the operand?
bool set_bnot(ea_t ea,int n);           // set negation bit
bool clr_bnot(ea_t ea,int n);           // clear negation bit
inline bool toggle_bnot(ea_t ea, int n)
{
  return (is_bnot(ea,n) ? clr_bnot : set_bnot)(ea,n);
}

// Set the variableness of an address
// The variable addresses are marked with '*' in the disassembly
//      ea    - linear address
//      isvar - 0: clear 'variable' mark
//              1: set 'variable' mark

void doVar(ea_t ea,bool isvar=true);


//-------------------------------------------------------------------------
//
//      Type of instruction/data operands is kept in flags too.
//      IDA keeps 2 type values:
//        - type of the first operand
//        - type of other operands (I will call this 'type of second operand'
//          althrough it is apllied to third, fourth operands too)
//      For data bytes, only the first type is used (i.e. all elements of
//      an array have the same type). Type of second operand is not used for
//      data bytes.
//

#define MS_0TYPE 0x00F00000L            // Mask for 1st arg typing
#define FF_0VOID 0x00000000L            // Void (unknown)?
#define FF_0NUMH 0x00100000L            // Hexadecimal number?
#define FF_0NUMD 0x00200000L            // Decimal number?
#define FF_0CHAR 0x00300000L            // Char ('x')?
#define FF_0SEG  0x00400000L            // Segment?
#define FF_0OFF  0x00500000L            // Offset?
#define FF_0NUMB 0x00600000L            // Binary number?
#define FF_0NUMO 0x00700000L            // Octal number?
#define FF_0ENUM 0x00800000L            // Enumeration?
#define FF_0FOP  0x00900000L            // Forced operand?
#define FF_0STRO 0x00A00000L            // Struct offset?
#define FF_0STK  0x00B00000L            // Stack variable?

#define MS_1TYPE 0x0F000000L            // Mask for 2nd arg typing
#define FF_1VOID 0x00000000L            // Void (unknown)?
#define FF_1NUMH 0x01000000L            // Hexadecimal number?
#define FF_1NUMD 0x02000000L            // Decimal number?
#define FF_1CHAR 0x03000000L            // Char ('x')?
#define FF_1SEG  0x04000000L            // Segment?
#define FF_1OFF  0x05000000L            // Offset?
#define FF_1NUMB 0x06000000L            // Binary number?
#define FF_1NUMO 0x07000000L            // Octal number?
#define FF_1ENUM 0x08000000L            // Enumeration?
#define FF_1FOP  0x09000000L            // Forced operand?
#define FF_1STRO 0x0A000000L            // Struct offset?
#define FF_1STK  0x0B000000L            // Stack variable?

// Is the first operand defined? Initially operand has no type (i.e. void)

inline bool isDefArg0(flags_t F)    { return (F & MS_0TYPE) != FF_0VOID; }


// Is the second operand defined? Initially operand has no type (i.e. void)

inline bool isDefArg1(flags_t F)    { return (F & MS_1TYPE) != FF_1VOID; }


// Is the first operand offset? (example: push offset xxx)

inline bool isOff0(flags_t F)       { return (F & MS_0TYPE) == FF_0OFF;  }


// Is the second operand offset? (example: mov ax, offset xxx)

inline bool isOff1(flags_t F)       { return (F & MS_1TYPE) == FF_1OFF;  }


// Is the first operand character constant? (example: push 'a')

inline bool isChar0(flags_t F)      { return (F & MS_0TYPE) == FF_0CHAR; }


// Is the second operand character constant? (example: mov al, 'a')

inline bool isChar1(flags_t F)      { return (F & MS_1TYPE) == FF_1CHAR; }


// Is the first operand segment selector? (example: push seg seg001)

inline bool isSeg0(flags_t F)       { return (F & MS_0TYPE) == FF_0SEG;  }


// Is the second operand segment selector? (example: mov dx, seg dseg)

inline bool isSeg1(flags_t F)       { return (F & MS_1TYPE) == FF_1SEG;  }


// Is the first operand a symbolic constant (enum member)?

inline bool isEnum0(flags_t F)      { return (F & MS_0TYPE) == FF_0ENUM;  }


// Is the second operand a symbolic constant (enum member)?

inline bool isEnum1(flags_t F)      { return (F & MS_1TYPE) == FF_1ENUM;  }


// Is the first operand manually defined? (for kernel only)

inline bool isFop0(flags_t F)       { return (F & MS_0TYPE) == FF_0FOP;  }


// Is the second operand manually defined? (for kernel only)

inline bool isFop1(flags_t F)       { return (F & MS_1TYPE) == FF_1FOP;  }


// Is the first operand an offset within a struct?

inline bool isStroff0(flags_t F)    { return (F & MS_0TYPE) == FF_0STRO;  }


// Is the second operand an offset within a struct?

inline bool isStroff1(flags_t F)    { return (F & MS_1TYPE) == FF_1STRO;  }


// Is the first operand a stack variable?

inline bool isStkvar0(flags_t F)    { return (F & MS_0TYPE) == FF_0STK;  }


// Is the second operand a stack variable?

inline bool isStkvar1(flags_t F)    { return (F & MS_1TYPE) == FF_1STK;  }


// Is the first or second operand a number (i.e. binary,octal,decimal or hex?)

ida_export bool isNum0(flags_t F);
ida_export bool isNum1(flags_t F);

//-------------------------------------------------------------------------
//
//      The following 2 masks are used with operand numbers
//
#define OPND_OUTER      0x80            // outer offset base (combined with
                                        //  operand number)
                                        // used only in set,get,del_offset()
                                        // functions
#define OPND_MASK       0x07            // mask for operand number
#define OPND_ALL        OPND_MASK       // all operands

// Examine flags for the specified operand type.
// 'n' may be 0         - first operand
//            1         - second operand
//            OPND_ALL  - both operands: function returns 1 if the first
//                        OR the second operand satisfies the condition

ida_export bool isOff(flags_t F,int n);             // is offset?
ida_export bool isChar(flags_t F,int n);            // is character constant?
ida_export bool isSeg(flags_t F,int n);             // is segment?
ida_export bool isDefArg(flags_t F,int n);          // is defined?
ida_export bool isEnum(flags_t F,int n);            // is enum?
ida_export bool isFop(flags_t F,int n);             // is forced operand? (use is_forced_operand())
ida_export bool isStroff(flags_t F,int n);          // is struct offset?
ida_export bool isStkvar(flags_t F,int n);          // is stack variable?
ida_export bool isNum(flags_t F,int n);             // is number (bin,oct,dec,hex)?
ida_export bool isVoid(ea_t ea,flags_t F,int n);   // is 'void' operand?


// (internal function) change type of operand(s)
//      ea   - linear address
//      type - new flag value (should be got from charflag(), numflag() and
//             similar functions. see below.)
//      n    - number of operand (0,1,-1)
// returns: 1-ok, 0-failed (applied to a tail byte)

bool set_op_type(ea_t ea,flags_t type,int n);


// (internal function) destroys old type information and returns
// new flags value. this function is called from the kernel only.
//      ea   - linear address
//      type - new flag value (should be got from ....flag() functions)
//      n    - number of operand (0,1,-1)

flags_t typeflag(ea_t ea,flags_t oldflag,flags_t type,int n);


// set operand type 'segment'
// if applied to unexplored bytes, converts them to 16/32bit word data
//      ea   - linear address
//      n    - number of operand (0,1,-1)
// returns 1-ok, 0-failure

ida_export bool op_seg(ea_t ea,int n);


// set operand type 'enum'
// if applied to unexplored bytes, converts them to 16/32bit word data
//      ea   - linear address
//      n    - number of operand (0,1,-1)
//      id   - id of enum
// returns 1-ok, 0-failure

ida_export bool op_enum(ea_t ea,int n,enum_t id);


// get enum id of 'enum' operand
//      ea   - linear address
//      n    - number of operand (0,1,-1)
// returns: id of enum or BADNODE

ida_export enum_t get_enum_id(ea_t ea,int n);


// set operand type 'struct offset'
// if applied to unexplored bytes, converts them to 16/32bit word data
//      ea   - linear address
//      n    - number of operand (0,1,-1)
//      path - structure path (strpath). see nalt.hpp for more info.
//      path_len - length of the structure path
// returns 1-ok, 0-failure

ida_export bool op_stroff(ea_t ea, int n, const tid_t *path, int path_len);


// set struct path for operand
//      ea   - linear address
//      n    - number of operand (0,1,-1)
//      path - structure path (strpath). see nalt.hpp for more info.
//      path_len - length of the structure path
// returns 1-ok, 0-failure

ida_export bool set_stroff_path(ea_t ea, int n, const tid_t *path, int plen);


// get struct path of operand
//      ea   - linear address
//      n    - number of operand (0,1,-1)
//      path - buffer for structure path (strpath). see nalt.hpp for more info.
// returns: length of strpath

ida_export int get_stroff_path(ea_t ea, int n, tid_t *path);


// delete struct path information
//      ea   - linear address
//      n    - number of operand (0,1,-1)
// returns: true

ida_export bool del_stroff_path(ea_t ea, int n);


// set operand type 'stack variable'
//      ea   - linear address
//      n    - number of operand (0,1,-1)
// should be applied to an instruction within a function
// returns 1-ok, 0-failure

ida_export bool op_stkvar(ea_t ea,int n);


// Set forced operand
//      ea - linear address
//      n  - number of operand (0,1,2)
//      op - text of operand
// returns: 1-ok, 0-failure

ida_export bool set_forced_operand(ea_t ea,int n,const char *op);


// Get forced operand
//      ea - linear address
//      n  - number of operand (0,1,2)

ida_export char *get_forced_operand(ea_t ea,int n);


// Is operand manually defined?
//      ea - linear address
//      n  - number of operand (0,1,2)

ida_export bool is_forced_operand(ea_t ea,int n);


//-------------------------------------------------------------------------
//      Type information bits for flags
//      Values of these functions are used as input to set_op_type() function

inline flags_t charflag(void) { return FF_1CHAR|FF_0CHAR; }
inline flags_t offflag (void) { return FF_1OFF |FF_0OFF ; }
inline flags_t enumflag(void) { return FF_1ENUM|FF_0ENUM; }
inline flags_t stroffflag(void) { return FF_1STRO|FF_0STRO; }
inline flags_t stkvarflag(void) { return FF_1STK|FF_0STK; }
inline flags_t segflag (void) { return FF_1SEG |FF_0SEG ; }
ida_export flags_t numflag(void); // returns number of default base (bin,oct,dec,hex)
                                // the following 4 functions return number
                                // flag of the specified base, regardless
                                // of the current default base for the current
                                // processor. Better to use numflag(), than to
                                // use these functions:
inline flags_t hexflag (void) { return FF_1NUMH|FF_0NUMH; }
inline flags_t decflag (void) { return FF_1NUMD|FF_0NUMD; }
inline flags_t octflag (void) { return FF_1NUMO|FF_0NUMO; }
inline flags_t binflag (void) { return FF_1NUMB|FF_0NUMB; }

// the following functions set operand type.
// if they are applied to unexplored bytes, they convert them
//                              no segment   : fail
//                              16bit segment: to 16bit word data
//                              32bit segment: to dword
//      ea   - linear address
//      n    - number of operand (0,1,-1)
// return 1-ok, 0-failure

inline bool op_chr(ea_t ea,int n) { return set_op_type(ea,charflag(),n); }
inline bool op_num(ea_t ea,int n) { return set_op_type(ea,numflag (),n); }
inline bool op_hex(ea_t ea,int n) { return set_op_type(ea,hexflag (),n); }
inline bool op_dec(ea_t ea,int n) { return set_op_type(ea,decflag (),n); }
inline bool op_oct(ea_t ea,int n) { return set_op_type(ea,octflag (),n); }
inline bool op_bin(ea_t ea,int n) { return set_op_type(ea,binflag (),n); }

// Remove type information about the specified operand from the database
//      ea   - linear address
//      n    - number of operand (0,1,-1)
// returns: 1

ida_export bool noType(ea_t ea,int n);

// Get default base of number for the current processor
// returns: 2,8,10,16

ida_export int getDefaultRadix(void);


// Get radix of the operand, in: flags
//      F    - flags
//      n    - number of operand (0,1,-1)
// returns: 2,8,10,16
// if the operand is not a number, returns getDefaultRadix()

ida_export int getRadix(flags_t F,int n);


// Get radix of the operand, in: address
//      ea   - linear address
//      n    - number of operand (0,1,-1)
// returns: 2,8,10,16
// if the operand is not a number, returns getDefaultRadix()

ida_export int getRadixEA(ea_t ea,int n);


//-------------------------------------------------------------------------
//
//      Bits for DATA bytes
//      They specify type of data (byte, word, dword, etc)
//

#define DT_TYPE 0xF0000000L             // Mask for DATA typing

#define FF_BYTE     0x00000000L         // byte
#define FF_WORD     0x10000000L         // word
#define FF_DWRD     0x20000000L         // double word
#define FF_QWRD     0x30000000L         // quadro word
#define FF_TBYT     0x40000000L         // tbyte
#define FF_ASCI     0x50000000L         // ASCII ?
#define FF_STRU     0x60000000L         // Struct ?
#define FF_OWRD     0x70000000L         // octaword (16 bytes)
#define FF_FLOAT    0x80000000L         // float
#define FF_DOUBLE   0x90000000L         // double
#define FF_PACKREAL 0xA0000000L         // packed decimal real
#define FF_ALIGN    0xB0000000L         // alignment directive

// The following functions return bit masks to be used as input for doData()

inline flags_t byteflag(void) { return FF_DATA|FF_BYTE; }
inline flags_t wordflag(void) { return FF_DATA|FF_WORD; }
inline flags_t dwrdflag(void) { return FF_DATA|FF_DWRD; }
inline flags_t qwrdflag(void) { return FF_DATA|FF_QWRD; }
inline flags_t owrdflag(void) { return FF_DATA|FF_OWRD; }
inline flags_t tbytflag(void) { return FF_DATA|FF_TBYT; }
inline flags_t asciflag(void) { return FF_DATA|FF_ASCI; }
inline flags_t struflag(void) { return FF_DATA|FF_STRU; }
inline flags_t alignflag(void) { return FF_DATA|FF_ALIGN; }
inline flags_t floatflag(void) { return FF_DATA|FF_FLOAT; }
inline flags_t doubleflag(void) { return FF_DATA|FF_DOUBLE; }
inline flags_t packrealflag(void) { return FF_DATA|FF_PACKREAL; }


// Convert to data array
// If 'ea' is unexplored, then define array of bytes
//      ea   - linear address
//      size - size of array in bytes. should be dividable to size of
//             current item.
//      tid  - type id. If the current item is a structure instance,
//             then tid is structure id. Otherwise should be BADNODE.
// return 1-ok, 0-failure

ida_export bool doData(ea_t ea,ulong size,tid_t tid);


// Convert to data (byte,word,dword,etc)
//      ea   - linear address
//      dataflag - type of data. Value of function byteflag(), wordflag(), etc.
//      size - size of array in bytes. should be dividable to size of
//             one item of the specified type.
//      tid  - type id. If the specified type is a structure,
//             then tid is structure id. Otherwise should be BADNODE.
// return 1-ok, 0-failure

ida_export bool doData(ea_t ea,flags_t dataflag,ulong size,tid_t tid);


// Convert to specified data type. (convenience functions)
//      ea   - linear address
//      size - size of array in bytes. should be dividable to size of
//             one item of the specified type.
// return 1-ok, 0-failure

ida_export bool doByte(ea_t ea,ulong size);             // 1-byte
ida_export bool doWord(ea_t ea,ulong size);             // 2-byte
ida_export bool doDwrd(ea_t ea,ulong size);             // 4-byte
ida_export bool doQwrd(ea_t ea,ulong size);             // 8-byte
ida_export bool doOwrd(ea_t ea,ulong size);             // 16-byte
ida_export bool doTbyt(ea_t ea,ulong size);             // ?-byte (defined in IDP)
ida_export bool doFloat(ea_t ea,ulong size);            // 4-byte
ida_export bool doDouble(ea_t ea,ulong size);           // 8-byte
ida_export bool doPackReal(ea_t ea,ulong size); // ?-byte (defined in IDP)
ida_export bool doASCI(ea_t ea,ulong size);              // don't use this! use make_ascii_string()
ida_export bool doStruct(ea_t ea,ulong size,tid_t tid);
ida_export bool doAlign(ea_t ea,ulong length,int alignment);
                                                // alignment: 0 or 2..32
                                                // if it is 0, is will be calculated
ida_export int calc_min_align(ulong length);    // returns: 1..32
ida_export int calc_max_align(ea_t endea);     // returns: 0..32
ida_export bool do16bit(ea_t ea,ulong length);  // convert to 16-bit quanztity
                                                // (depending on the byte size)


// Check flags for the specified data type

inline bool isByte  (flags_t F)   { return isData(F) && (F & DT_TYPE) == FF_BYTE; }
inline bool isWord  (flags_t F)   { return isData(F) && (F & DT_TYPE) == FF_WORD; }
inline bool isDwrd  (flags_t F)   { return isData(F) && (F & DT_TYPE) == FF_DWRD; }
inline bool isQwrd  (flags_t F)   { return isData(F) && (F & DT_TYPE) == FF_QWRD; }
inline bool isOwrd  (flags_t F)   { return isData(F) && (F & DT_TYPE) == FF_OWRD; }
inline bool isTbyt  (flags_t F)   { return isData(F) && (F & DT_TYPE) == FF_TBYT; }
inline bool isFloat (flags_t F)   { return isData(F) && (F & DT_TYPE) == FF_FLOAT; }
inline bool isDouble(flags_t F)   { return isData(F) && (F & DT_TYPE) == FF_DOUBLE; }
inline bool isPackReal(flags_t F) { return isData(F) && (F & DT_TYPE) == FF_PACKREAL; }
inline bool isASCII (flags_t F)   { return isData(F) && (F & DT_TYPE) == FF_ASCI; }
inline bool isStruct(flags_t F)   { return isData(F) && (F & DT_TYPE) == FF_STRU; }
inline bool isAlign(flags_t F)    { return isData(F) && (F & DT_TYPE) == FF_ALIGN; }


// determine maximum length of ascii string
//      ea           - starting address
//      strtype      - string type. ASCSTR_... constant (see nalt.hpp)
//      ignore_heads - 1: don't stop if another data item is encountered
//                     0: take into account other data items

ida_export size_t get_max_ascii_length(ea_t ea, long strtype, bool ignore_heads=false);


// Get contents of ascii string
// This function returns the displayed part of the string
// It works even if the string is not created in the database yet.
//      ea      - linear address of the string
//      len     - length of the string
//      type    - type of the string
//      buf     - output buffer
//      bufsize - size of output buffer
// returns 1-ok, 0-ascii string is too long and was truncated

bool get_ascii_contents(ea_t ea, size_t len, long type, char *buf, size_t bufsize);


// convert to ascii string and give a meaningful name
// 'start' may be higher than 'end', the kernel will swap them in this case
//      start - starting address
//      end   - length of the string
//              if 0, then get_max_ascii_length() will be used
//              to determine the length
//      strtype - string type. ASCSTR_... constant (see nalt.hpp)
// returns: 1-ok,0-failure

ida_export bool make_ascii_string(ea_t start, size_t len, long strtype);


// print the string type name
//      long - the string type
//      buf  - the output buffer. 20 bytes is enough
// returns: ptr to buf

char *print_ascii_string_type(char *buf, long strtype);


// Get additional information about a type
//      ea      - linear address of item
//      n       - number of operand, 0 or 1
//      flags   - flags of the item
//      buf     - buffer to receive the result. may not be NULL
// Returns: NULL - no additional type information

ida_export typeinfo_t *get_typeinfo(ea_t ea, int n, flags_t flags, typeinfo_t *buf);


// Set additional information about a data type
//      ea      - linear address of item
//      n       - number of operand, 0 or 1
//      flags   - flags of the item
//      ti      - additional type information
// This function is a low level one. Only the kernel should use it.
// Returns: 1-ok, 0-ti is NULL while it shouldn't be NULL

bool set_typeinfo(ea_t ea, int n, flags_t flag, const typeinfo_t *ti);


// Delete additional information about a data type
//      ea      - linear address of item
//      n       - number of operand, 0 or 1
//      flags   - flags of the item
// This function is a low level one. Only the kernel should use it.

void del_typeinfo(ea_t ea, flags_t flag);
void del_operand_typeinfo(ea_t ea, int n, flags_t flag);


// get size of data type specified in flags 'F'.
//      F    - flags
//      ti   - additional information about the data type. For example,
//             if the current item is a structure instance,
//             then ti->tid is structure id. Otherwise is ignored (may be NULL).
// returns:     byte - 1
//              word - 2
//              etc.
// If flags doesn't specify a data, then return 1

ida_export ulong get_data_type_size(flags_t flags,const typeinfo_t *ti);
inline ulong get_data_type_size(ea_t ea, flags_t flags)
{
  typeinfo_t ti;
  return get_data_type_size(flags, get_typeinfo(ea, 0, flags, &ti));
}


// Can define item (instruction/data) of the specified 'length'
// starting at 'ea'?
// Returns: 1-yes, 0-no
// This function may return 0 if:
//      - something was defined at 'ea' already
//      - a new item would cross segment boundaries
//      - a new item would overlap with existing items
// NOTE: this function converts to unexplored all encountered data items
//       with fixup information. Should be fixed in the future.

ida_export bool can_define_item(ea_t ea,ulong length);


//-------------------------------------------------------------------------
//
//      Bits for CODE bytes
//      They specify characteristics of instructions
//

#define MS_CODE 0xF0000000L
#define FF_FUNC 0x10000000L             // function start?
//              0x20000000L             // not used
#define FF_IMMD 0x40000000L             // Has Immediate value ?
#define FF_JUMP 0x80000000L             // Has jump table or switch_info?

// Convert to an instruction
// (internal function, should not be used in modules, never)
// use ua_code() instead.
//      ea     - linear address
//      length - length of instruction
// Returns: 1-ok, 0-failure

ida_export bool doCode(ea_t ea,int length);


// Has immediate value?

inline bool isImmd(flags_t F)       { return isCode(F) && (F & FF_IMMD) != 0; }


// Is function start?

inline bool  isFunc(flags_t F)      { return isCode(F) && (F & FF_FUNC) != 0; }


// Set 'has immediate operand' flag

ida_export void doImmd(ea_t ea);


// Clear 'has immediate operand' flag

ida_export void noImmd(ea_t ea);


//-------------------------------------------------------------------------
//
//      Bits for TAIL bytes
//      Modules don't change or use tail bytes, so all the following
//      definitions are for the kernel only.
//

#define MS_TAIL 0xFFF00000LU            // Mask of tail byte bits
#define TL_TSFT (4*5)                   // number of bits to shift to get
                                        // tail offsets
#define TL_TOFF MS_TAIL                 // Offset to next NOT-tail byte
#define MAX_TOFF (TL_TOFF>>TL_TSFT)     // Max offset can be written to flags

inline ushort gettof(flags_t F) { return ushort((F & TL_TOFF) >> TL_TSFT); }

//--------------------------------------------------------------------------
//      I N D E N T E D  C O M M E N T S
//--------------------------------------------------------------------------

// Set an indented comment
//      ea     - linear address
//      comm   - comment string (max len MAXSTR)
//      rptble - is repeatable?
// returns: 1-ok, 0-failure

ida_export bool set_cmt(ea_t ea,const char *comm,bool rptble);


// Get an indented comment
//      ea     - linear address
//      rptble - get repeatable comment?
// returns: comment or NULL

ida_export char *get_cmt(ea_t ea,bool rptble);


// Get a repeatable comment of any type (indented or function comment)
// This function is used to display an indented comment if no regular
// (non-repeatable) comment is present.
//      ea     - linear address. may point to tail byte, the function
//               will find start of the item
// returns: comment or NULL

ida_export char *get_rpt_cmt(ea_t ea);


// Get any indented comment (regular or repeatable indented or function)
// This function is used to display an indented comment for an item.
// It looks for a regular comment and calls get_rpt_cmt() if it is not found.
//      ea      - linear address
//      cmttype - will contain color of the comment. The color depends on
//                the type of the comment.
// returns: comment or NULL

ida_export char *get_any_cmt(ea_t ea,color_t *cmttype);


// Append to an indented comment
// Creates a new comment if none exists.
// Appends a newline character and the specified string otherwise.
//      ea      - linear address
//      str     - comment string to append
//      rptble  - append to repeatable comment?
// returns: 1-ok, 0-failure

ida_export bool append_cmt(ea_t ea,const char *str,bool rptble);


// Delete an indented comment attached to an instruction
// when the instruction is deleted

extern bool del_code_comments;


//--------------------------------------------------------------------------
//      S E A R C H  F U N C T I O N S
//--------------------------------------------------------------------------
// Find forward a byte with the specified value
//      ea         - linear address
//      size       - number of bytes to inspect
//      value      - value to find
//      sense_case - case sensitive search
// returns: address of byte or BADADDR

ida_export ea_t find_byte(ea_t sEA,ulong size,uchar value,bool sense_case);


// Find backward a byte with the specified value
//      ea         - linear address (it is not inspected)
//      size       - number of bytes to inspect
//      value      - value to find
//      sense_case - case sensitive search
// returns: address of byte or BADADDR

ida_export ea_t find_byter(ea_t sEA,ulong size,uchar value,bool sense_case);


// search for a string in the program
//      startEA - linear address, start of area to search
//      endEA   - linear address, end of area to search (inclusive)
//      image   - stream of bytes to search
//      mask    - array of 1/0 bytes, it's length is 'len'. 1 means to perform
//                the comparision of the corresponding byte. 0 means not to perform.
//                if mask == NULL, then all bytes of 'image' will be compared.
//      len     - length of srchStr in bytes
//      step    - direction of search
//      flags   - search flags. combination of:
//                  BIN_SEARCH_CASE    - case sensitive
//                  BIN_SEARCH_NOCASE  - case insensitive
//                  BIN_SEARCH_NOBREAK - don't check for Ctrl-Break
// returns BADADDR (if pressed Ctrl-Break or not found) or string address.

ida_export ea_t bin_search(
                 ea_t startEA,         // area to search
                 ea_t endEA,
                 const uchar *image,   // string to search
                 const uchar *mask,    // comparision mask
                 int len,              // length of string to search
                 int step,             // direction:
#define BIN_SEARCH_FORWARD      1
#define BIN_SEARCH_BACKWARD     (-1)
                 int flags);
#define BIN_SEARCH_CASE         0x01
#define BIN_SEARCH_NOCASE       0x00
#define BIN_SEARCH_NOBREAK      0x02


// Compare 'len' bytes of the program starting from 'ea' with 'image'.
//      ea         - linear address
//      image      - bytes to compare with
//      len        - length of block to compare in bytes.
//      sense_case - case-sensitive comparison?
//      mask       - array of 1/0 bytes, it's length is 'len'. 1 means to perform
//                   the comparision of the corresponding byte. 0 means not to perform.
//                   if mask == NULL, then all bytes of 'image' will be compared.
// Returns: 1- equal, 0-no

ida_export bool equal_bytes(ea_t ea,const uchar *image,const uchar *mask,
                                                        int len,bool sense_case);


//--------------------------------------------------------------------------
ida_export extern const ulong power2[32];    // Powers of 2, from 2^0 to 2^31
ida_export extern const ulong lowbits[33];   // Low-order bits, from 0 to 32


//--------------------------------------------------------------------------
//      L O W  L E V E L  F U N C T I O N S (kernel only)
//--------------------------------------------------------------------------

inline  void    doFlow  (ea_t ea)      { setFlbits(ea, FF_FLOW);  }
inline  void    noFlow  (ea_t ea)      { clrFlbits(ea, FF_FLOW);  }

void    doRef   (ea_t ea);
void    noRef   (ea_t ea);

inline  void    doExtra (ea_t ea)      { setFlbits(ea, FF_LINE); }
inline  void    noExtra (ea_t ea)      { clrFlbits(ea, FF_LINE); }

//--------------------------------------------------------------------------
ulong coagulate(ea_t ea);
bool coagulate_dref(ea_t From,ea_t To,flags_t F, bool may_define);

//--------------------------------------------------------------------------
inline ea_t get_item_head(ea_t ea)
{
  if ( isTail(getFlags(ea)) ) ea = prev_not_tail(ea);
  return ea;
}

#ifndef BYTES_SOURCE    // undefined bit masks so no one can use them directly
#undef MS_VAL
#undef FF_IVL
#undef MS_CLS
#undef FF_CODE
#undef FF_DATA
#undef FF_TAIL
#undef FF_UNK
#undef MS_COMM
#undef FF_COMM
#undef FF_REF
#undef FF_LINE
#undef FF_NAME
#undef FF_LABL
#undef FF_ANYNAME
#undef FF_FLOW
#undef FF_SIGN0
#undef FF_SIGN1
#undef FF_VAR
#undef MS_0TYPE
#undef FF_0VOID
#undef FF_0NUMH
#undef FF_0NUMD
#undef FF_0CHAR
#undef FF_0SEG
#undef FF_0OFF
#undef FF_0NUMB
#undef FF_0NUMO
#undef FF_0ENUM
#undef FF_0FOP
#undef FF_0STRO
#undef FF_0STK
#undef FF_0HIGH
#undef MS_1TYPE
#undef FF_1VOID
#undef FF_1NUMH
#undef FF_1NUMD
#undef FF_1CHAR
#undef FF_1SEG
#undef FF_1OFF
#undef FF_1NUMB
#undef FF_1NUMO
#undef FF_1ENUM
#undef FF_1FOP
#undef FF_1STRO
#undef FF_1STK
#undef FF_1HIGH
#undef DT_TYPE
#undef FF_BYTE
#undef FF_WORD
#undef FF_DWRD
#undef FF_QWRD
#undef FF_FLOAT
#undef FF_DOUBLE
#undef FF_TBYT
#undef FF_PACKREAL
#undef FF_ASCI
#undef FF_STRU
#undef FF_ALIGN
#undef MS_CODE
#undef FF_FUNC
#undef FF_IMMD
//#undef FF_JUMP
#undef MS_TAIL
#undef TL_TSFT
#undef TL_TOFF
#undef MAX_TOFF
#endif // BYTES_SOURCE

#pragma pack(pop)
#endif // BYTES_HPP
