/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-98 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@datarescue.com, ig@estar.msk.su
 *
 *
 */

#ifndef _DISKIO_HPP
#define _DISKIO_HPP

//
//      This file contains file I/O functions for IDA.
//      You should not use standard C file I/O functions in modules.
//      Use functions from this header, pro.h and fpro.h instead.
//
//      Also this file declares a call_system() function.
//

#include <stdio.h>
#include <help.h>
#include <llong.hpp>

//-------------------------------------------------------------------------
//      F I L E   N A M E   M A N I P U L A T I O N
//-------------------------------------------------------------------------

// Set file name extension if none exist
// This function appends extension to a file name.
// It won't change file name if extension already exists
//      file - file name to modify
//      ext  - extension with or without '.'
// returns: 'file'

char*   FileExt(char *file,const char *ext);    // Set EXT if no ext


// Set file name extension unconditionally
//      outbuf  - buffer to hold answer. if NULL, 'file' is modified
//      file    - file name
//      ext     - new extension
// returns: pointer to new file name

char*   SetFileExt(char *outbuf,char *file,const char *ext); // Always set ext


// Get pointer to extension of file name
//      file - file name
// returns: pointer to file name or NULL if extension doesn't exist

char*   hasExt  (const char *file);             // has Extension ?


//-------------------------------------------------------------------------
//      S E A R C H   F O R   F I L E S
//-------------------------------------------------------------------------


// Get IDA directory

const char *idadir(void);


// Seach for IDA system file
// This function searches for a file in
//  1. ida directory
//  2. current directory
//  3. in the PATH
//      filename - name of file to search
// returns: NULL-not found, otherwise - pointer to full file name in
// a static storage

char *getsysfile(const char *filename);         // Search for IDA system file


// enumerate files in the specified directory
// while func() returns 0
//      path    - directory to enumerate files in
//      fname   - mask of file names to enumerate
//      func    - callback function called for each file
//                      file - full file name (with path)
//                      ud   - user data
//                if 'func' returns non-zero value, the enumeration
//                is stopped and full path of the current file
//                is returned to the caller.
//      ud      - user data. this pointer will be passed to
//                the callback function
// return name of file (ptr to static) or NULL if callback never returned
// non-zero value

char *enumerate_files(       const char *path,
                             const char *fname,
                             int (*func)(const char *file,void *ud),
                             void *ud);


// enumerate IDA system files
//      fname   - mask of file names to enumerate
//      func    - callback function called for each file
//                      file - full file name (with path)
//                      ud   - user data
//                if 'func' returns non-zero value, the enumeration
//                is stopped and full path of the current file
//                is returned to the caller.
//      ud      - user data. this pointer will be passed to
//                the callback function
// return name of file (ptr to static) or NULL if callback never returned
// non-zero value

char *enumerate_system_files(const char *fname,
                             int (*func)(const char *file,void *ud),
                             void *ud);


//-------------------------------------------------------------------------
//      O P E N / R E A D / W R I T E / C L O S E   F I L E S
//-------------------------------------------------------------------------

//      There are two sets of "open file" functions.
//      The first set tries to open a file and returns: success or failure
//      The second set is "open or die": if the file cannot be opened
//      then the function will display an error message and exit.


// Open a new file for write in text mode, deny write
// If a file exists, it will be removed.
// returns: NULL-failure

FILE *fopenWT(const char *file);


// Open a new file for write in binary mode, deny read/write
// If a file exists, it will be removed.
// returns: NULL-failure

FILE *fopenWB(const char *file);


// Open a file for read in text mode, deny write
// returns: NULL-failure

FILE *fopenRT(const char *file);


// Open a file for read in binary mode, deny write
// returns: NULL-failure

FILE *fopenRB(const char *file);


// Open a file for read/write in binary mode, deny write
// returns: NULL-failure

FILE *fopenM(const char *file);


// Open a file for append in text mode, deny none
// returns: NULL-failure

FILE *fopenA(const char *file);


// Open a file for read in binary mode or die, deny write
// If a file cannot be opened, this function displays a message and exits.

FILE *openR(const char *file);


// Open a file for read in text mode or die, deny write
// If a file cannot be opened, this function displays a message and exits.

FILE *openRT(const char *file);


// Open a file for read/write in binary mode or die, deny read/write
// If a file cannot be opened, this function displays a message and exits.

FILE *openM(const char *file);


// Open a new file for write in binary mode or die, deny read/write
// If a file exists, it will be removed.
// If a file cannot be opened, this function displays a message and exits.

FILE *ecreate(const char *file);


// Open a new file for write in text mode or die, deny read/write
// If a file exists, it will be removed.
// If a file cannot be opened, this function displays a message and exits.

FILE *ecreateT(const char *file);


// Close a file or die.
// If a file cannot be closed, this function displays a message and exits.

void eclose(FILE *fp);


// Read from file or die.
//      fp   - pointer to file
//      buf  - buffer to read in
//      size - number of bytes to read
// If a read error occurs, this function displays a message and exits.

void eread(FILE *fp,void *buf,unsigned size);


// Write to file or die.
//      fp   - pointer to file
//      buf  - buffer to write
//      size - number of bytes to write
// If a write error occurs, this function displays a message and exits.

void ewrite(FILE *fp,const void *buf,unsigned size);


// Position in the file or die.
//      fp   - pointer to file
//      pos  - absolute position in the file
// If an error occurs, this function displays a message and exits.

void eseek(FILE *fp,long pos);


//-------------------------------------------------------------------------
//      F I L E   S I Z E   /   D I S K   S P A C E
//-------------------------------------------------------------------------

// Get length of file in bytes
//      fp   - pointer to file

ulong efilelength(FILE *fp);


// Change size of file or die.
//      fp   - pointer to file
//      size - new size of file
// If the file is expanded, it is expanded with zero bytes.
// If an error occurs, this function displays a message and exits.

void chsize(FILE *fp,ulong size);


// Get free disk space in bytes
//      drive - number of drive
//               0 - current drive
//               1 - drive 'A' and so on

ulonglong getdspace(int drive);                 // get free disk space
                                                // 0 - current drive


// Get free disk space in bytes
//      path - name of any file on the disk to get information about

ulonglong getdspace(const char *path);


//-------------------------------------------------------------------------
//      I / O  P O R T  D E F I N I T I O N S  F I L E
//-------------------------------------------------------------------------
struct ioport_t
{
  int address;          // address of the port
  char *name;           // name of the port
  void *userdata;       // arbitrary data. initialized by NULL.
};

// read i/o port definitions from a config file
//      _numports - place to put the number of ports
//      file      - config file name
//      default_device - contains device name to load. If default_device[0] == 0
//                  then the default device is determined by .default directive
//                  in the config file
//                  sizeof(default_device) should be at least MAXSTR.
//      callback  - callback to call when the input line can't be parsed normally
//                    line - input line to parse
//                  returns error message. if NULL, then the line is parsed ok.
// returns: array of ioports
// The format of the input file:
// ; each device definition begins with a line like this:
// ;
// ;       .devicename
// ;
// ;  after it go the port definitions in this format:
// ;
// ;       portname        address
// ;
// ; lines beginning with a space are ignored.
// ; comment lines should be started with ';' character.
// ;
// ; the default device is specified at the start of the file
// ;
// ;       .default device_name
// ;
// ; all lines non conforming to the format are passed to the callback function

ioport_t *read_ioports(int *_numports,
                       const char *file,
                       char *default_device,
                       const char *callback(const ioport_t *ports, int numports, const char *line)=NULL);

// Allow the user to choose the ioport device
//      file      - config file name
//      device    - in: contains default device name. If default_device[0] == 0
//                  then the default device is determined by .default directive
//                  in the config file
//                  out: the selected device name
//                  sizeof(device) should be at least MAXSTR.
// returns: true  - the user selected a device, its name is in 'device'
//          false - the selection was cancelled

bool choose_ioport_device(const char *file, char *device);



// Find ioport in the array of ioports
const char *find_ioport(const ioport_t *ports, int numports, int address);


// Free ioports array
void free_ioports(ioport_t *ports, int numports);


//-------------------------------------------------------------------------
//      S Y S T E M  S P E C I F I C  C A L L S
//-------------------------------------------------------------------------

// Execute a operating system command
// This function suspends the interface (Tvision), runs the command
// and redraw the screen.
//      command - command to execute. If NULL, a shell is activated
// Returns: the error code returned by system() call.

int call_system(const char *command);


// Work with priorities

ulong get_thread_priority(void);        // 0 - something wrong (can't happen
                                        // under OS/2)
int   set_thread_priority(ushort pclass,long delta);


//-------------------------------------------------------------------------
//      K E R N E L   O N L Y   F U N C T I O N S
//-------------------------------------------------------------------------

// The following definitions are for the kernel only:

void    checkdspace(unsigned long space);       // if there is less than
                                                // 'space' on the current disk
                                                // warns the user
int     lowdiskgo(unsigned long space);         // if there is less than
                                                // 'space' on the current disk
                                                // asks user to confirm
                                                // returns 1 - continue


extern char **ida_argv;
extern char *exename;


#endif // _DISKIO_HPP
