/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-99 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su
 *                              FIDO:   2:5020/209
 *
 *      Enums and bitfields
 *      Bitfields will be abbreviated as "bf".
 *
 */

#ifndef _ENUM_HPP
#define _ENUM_HPP

#include <stdio.h>

#include <help.h>
#include <nalt.hpp>

//--------------------------------------------------------------------------
// enumerations and bitfields are represented as enum_t.

// internal declaration, you shouldn't use this:
// (except of type names like enum_t, const_t, bmask_t)

ida_export extern netnode enums;        // main netnode to access all enums
                                        // should not be used directly!
                                        // altval(idx)-1 -> enum_t
                                        // altval('R',enum_t)-1 -> idx
                                        // altval(-1)  -> qty
#define ENUM_REVERSE    'Y'
#define ENUM_SELMEMS    'Z'     // temporary array for choose_enum_by_value
#define ENUM_QTY_IDX    ulong(-1)
#define ENUM_FLG_IDX    ulong(-3)
#define ENUM_FLAGS      ulong(-5)
#define ENUM_FLAGS_IS_BF  0x00000001
#define ENUM_FLAGS_HIDDEN 0x00000002

typedef ulong enum_t;   // enum_t is simple netnode
                        // name(), cmt() and rptcmt() are kept in the usual form
                        // memqty is kept in altval(-1)
                        // Bitfields have altval(-5) set to 1.
                        // enums: all members are kept in altval('E',member_value)-1
                        // bf: altval('M') keeps ptrs to nodes by bitmasks
typedef ulong bmask_t;  // a bit mask is 32 bits
                        //     if a mask is one-bit, then ptr to const_t is kept immediately in altval('M')
const bmask_t DEFMASK = bmask_t(-1);

                        //     otherwise, masknode.altval('E') keeps ptrs to const_t by values
                        //     (the same as with enums)
#define ENUM_MASKS      'm'
#define ENUM_MEMBERS    'E'
typedef ulong const_t;  // members of enums are kept as netnodes
                        // name(), cmt() and rptcmt() are kept in the usual form
                        // altval(-2) is enum_t
#define CONST_ENUM      ulong(-2)
#define CONST_VALUE     ulong(-3)
#define CONST_BMASK     ulong(-6)

inline void     init_enums(void) { enums.create("$ enums"); }
inline void     save_enums(void) {}
inline void     term_enums(void) {}

bool set_enum_flag(enum_t id, ulong bit, bool hidden);  // internal function
//--------------------------------------------------------------------------
// QUERIES - PUBLIC DECLARATIONS

// get number of declared enum_t types
inline ulong    get_enum_qty(void)      { return enums.altval(ENUM_QTY_IDX); }

// get enum by its ordinal number (0..n)
inline enum_t   getn_enum(ulong n)      { return (n==ENUM_QTY_IDX) ? BADNODE : enums.altval(n)-1; }

// get serial number of enum.
// the serial number determines the place of the enum in the enum window
inline ulong    get_enum_idx(enum_t id) { return enums.altval(id,ENUM_REVERSE)-1; }

// get enum by name
enum_t          get_enum(const char *name);

// is enum a bitfield?
// (otherwise - plain enum, no bitmasks except of DEFMASK are allowed)
inline bool     is_bf(enum_t id)                        { return (netnode(id).altval(ENUM_FLAGS) & ENUM_FLAGS_IS_BF) != 0; }

// is enum collapsed?
inline bool     is_enum_hidden(enum_t id)               { return (netnode(id).altval(ENUM_FLAGS) & ENUM_FLAGS_HIDDEN) != 0; }
inline bool     set_enum_hidden(enum_t id, bool hidden) { return set_enum_flag(id, ENUM_FLAGS_HIDDEN, hidden); }

// get name of enum
inline char*    get_enum_name(enum_t id)                { return netnode(id).name();  }

// get enum comment
inline char*    get_enum_cmt(enum_t id,bool repeatable) { return netnode(id).supval(repeatable != 0); }

// get number of member of the enum
inline ulong    get_enum_size(enum_t id)                { return netnode(id).altval(ENUM_QTY_IDX);    }

// get flags determining the representation of the enum
// (currently they define the numeric base: octal, decimal, hex, bin) and signness
inline flags_t  get_enum_flag(enum_t id)                { return netnode(id).altval(ENUM_FLG_IDX);    }


// get symbolic constant (a enum member) by its name
const_t         get_const(const char *name);

// get value of a constant
inline ulong    get_const_value(const_t id)             { return netnode(id).altval(CONST_VALUE); }

// get the parent enum of a constant
inline enum_t   get_const_enum(const_t id)              { return netnode(id).altval(CONST_ENUM)-1; }

// get bitmask of a constant
inline bmask_t  get_const_bmask(const_t id)             { return netnode(id).altval(CONST_BMASK)-1; }

// find a constant by enum, value and bitmask
const_t         get_const(enum_t id,ulong value,bmask_t mask=DEFMASK);

// get access to all used bitmasks in the enum
inline bmask_t  get_first_bmask(enum_t id)              { return netnode(id).alt1st(ENUM_MASKS); }
inline bmask_t  get_last_bmask(enum_t id)               { return netnode(id).altlast(ENUM_MASKS); }
inline bmask_t  get_next_bmask(enum_t id,bmask_t bmask) { return netnode(id).altnxt(bmask,ENUM_MASKS); }
inline bmask_t  get_prev_bmask(enum_t id,bmask_t bmask) { return netnode(id).altprev(bmask,ENUM_MASKS); }

// get access to all constants with the specified bitmask in the enum
ulong           get_first_const(enum_t id, bmask_t bmask=DEFMASK);
ulong           get_last_const(enum_t id, bmask_t bmask=DEFMASK);
ulong           get_next_const(enum_t id, ulong value, bmask_t bmask=DEFMASK);
ulong           get_prev_const(enum_t id, ulong value, bmask_t bmask=DEFMASK);

// get name of constant by const_t
inline char*    get_const_name(const_t id)              { return netnode(id).name();  }

// get comment of constant
inline char*    get_const_cmt(const_t id,bool repeatable){ return netnode(id).supval(repeatable != 0); }

//--------------------------------------------------------------------------
// manipulation

// add new enum type
// if idx==BADADDR then add as the last idx
enum_t          add_enum(ulong idx,const char *name,flags_t flag);
void            del_enum(enum_t id);
bool            set_enum_idx(enum_t id,ulong idx);

// set 'bitfield' bit of enum (i.e. convert it to a bitfield)
bool            set_enum_bf(enum_t id, bool bf);

bool            set_enum_name(enum_t id,const char *name);
bool            set_enum_cmt(enum_t id,const char *cmt,bool repeatable);
inline bool     set_enum_flag(enum_t id,flags_t flag)     { return netnode(id).altset(ENUM_FLG_IDX,flag); }

// this function returns error code (0 is ok)
int             add_const(enum_t id,const char *name,ulong value, bmask_t bmask=DEFMASK);
#define CONST_ERROR_NAME  1     // already have member with this name (bad name)
#define CONST_ERROR_VALUE 2     // already have member with this value
#define CONST_ERROR_ENUM  3     // bad enum id
#define CONST_ERROR_MASK  4     // bad bmask
#define CONST_ERROR_ILLV  5     // bad bmask and value combination (~bmask & value != 0)

bool            del_const(enum_t id,ulong value,bmask_t bmask=DEFMASK);
inline bool     set_const_name(const_t id,const char *name)               { return set_enum_name(id,name);            }
inline bool     set_const_cmt(const_t id,const char *cmt,bool repeatable) { return set_enum_cmt(id, cmt, repeatable); }

// kernel helper functions:

long            print_all_enums(FILE *fp);
inline enum_t   get_selected_enum(ulong n) { return enums.altval(n,ENUM_SELMEMS); }
inline void     add_selected_enum(ulong &idx,enum_t id) { enums.altset(idx++,id,ENUM_SELMEMS); }
inline void     unmark_selected_enums(void){ enums.altdel_any(ENUM_SELMEMS); }

// can we use the specified bitmask in the enum?
bool is_good_bmask(enum_t id, bmask_t bmask);

inline bool is_one_bit_mask(bmask_t mask)
{
  return (mask & (mask-1)) == 0;
}

// get bitmask node if bf-scheme is used
// otherwise returns BADNODE
inline netnode get_bmask_node(enum_t id,bmask_t bmask)
{
  if ( bmask == DEFMASK || !is_bf(id) ) return BADNODE;
  return netnode(id).altval(bmask, ENUM_MASKS)-1;
}

inline bool set_enum_flag(enum_t id, ulong bit, bool set)
{
  if ( id == BADNODE ) return false;
  netnode n(id);
  ulong f = n.altval(ENUM_FLAGS);
  setflag(f, bit, set);
  return n.altset(ENUM_FLAGS, f);
}

// work with the bitmask name & comment
inline bool     set_bmask_name(enum_t id,bmask_t bmask, const char *name) { return get_bmask_node(id,bmask).rename(name); }
inline char*    get_bmask_name(enum_t id,bmask_t bmask)                   { return get_bmask_node(id,bmask).name();       }
inline bool     set_bmask_cmt(enum_t id,bmask_t bmask,const char *cmt,bool repeatable) { return set_enum_cmt(get_bmask_node(id,bmask), cmt, repeatable); }
inline char*    get_bmask_cmt(enum_t id,bmask_t bmask,bool repeatable)                 { return get_bmask_node(id,bmask).supval(repeatable != 0); }


#ifndef ENUM_SOURCE
#define enums   dont_use_enums_directly         // prohibit access to 'enums'
                                                // netnode
#endif

#endif // _ENUM_HPP
