/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-97 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su
 *                              FIDO:   2:5020/209
 *
 */

#ifndef FIXUP_HPP
#define FIXUP_HPP

//
//      This file contains functions that deal with fixup information
//      A loader should setup fixup information using set_fixup()
//      function.
//

#include <netnode.hpp>
#include <segment.hpp>
#pragma pack(push, 1)           // IDA uses 1 byte alignments!

// Fixup information structure

struct fixup_data_t {
  uchar type;                   // fixup type
#define FIXUP_MASK      0xF
#define FIXUP_BYTE      FIXUP_OFF8 // 8-bit offset.
#define FIXUP_OFF8      0       // 8-bit offset.
#define FIXUP_OFF16     1       // 16-bit offset.
#define FIXUP_SEG16     2       // 16-bit base--logical segment base (selector).
#define FIXUP_PTR32     3       // 32-bit long pointer (16-bit base:16-bit
                                // offset).
#define FIXUP_OFF32     4       // 32-bit offset.
#define FIXUP_PTR48     5       // 48-bit pointer (16-bit base:32-bit offset).
#define FIXUP_HI8       6       // high  8 bits of 16bit offset
#define FIXUP_HI16      7       // high 16 bits of 32bit offset
#define FIXUP_LOW8      8       // low   8 bits of 16bit offset
#define FIXUP_LOW16     9       // low  16 bits of 32bit offset
#define FIXUP_HI22      0xA     // high 22 bits of 32bit offset (SPARC)
#define FIXUP_LOW10     0xB     // low  10 bits of 32bit offset (SPARC)
#define FIXUP_OFF64     0xC     // 64-bit offset
#define FIXUP_REL       0x10    // fixup is relative to the linear address
                                // specified in the 3d parameter to set_fixup()
#define FIXUP_SELFREL   0x0     // self-relative?
                                //   - disallows the kernel to convert operands
                                //      in the first pass
                                //   - this fixup is used during output
                                // This type of fixups is not used anymore.
                                // Anyway you can use it for commenting purposes
                                // in the loader modules
#define FIXUP_EXTDEF    0x20    // target is a location (otherwise - segment)
#define FIXUP_UNUSED    0x40    // fixup is ignored by IDA
                                //   - disallows the kernel to convert operands
                                //   - this fixup is not used during output
#define FIXUP_CREATED   0x80    // fixup was not present in the input file
  ushort sel;           // target selector
  ulong  off;           // target offset
  long displacement;    // target displacement
};

extern ida_export netnode near fixup_node;

// Set fixup information. You should fill fixup_data_t and call this
// function and the kernel will remember information in the database.
// Use this function if FIXUP_REL bit is clear.

void set_fixup(ulong source,fixup_data_t *fp);


// Set fixup information. You should fill fixup_data_t and call this
// function and the kernel will remember information in the database.
// Use this function if FIXUP_REL bit is set.

void set_fixup(ulong source,fixup_data_t *fd,ulong offset_base);


// Delete fixup information.

inline void del_fixup(ulong source) { fixup_node.supdel(source); }


// Get fixup information

inline fixup_data_t *get_fixup(ulong source)
        { return (fixup_data_t *)fixup_node.supval(source); }


// Enumerate addresses with fixup information:

inline ulong get_first_fixup_ea(void)    { return fixup_node.sup1st(); }
inline ulong get_next_fixup_ea(ulong ea) { return fixup_node.supnxt(ea); }
inline ulong get_prev_fixup_ea(ulong ea) { return fixup_node.supprev(ea); }


// Use fixup information for an address.
//      item_ea  - start address of item to modify
//      fixup_ea - address of fixup record
//      n        - number of operand. may be 0, 1, 2
// This function converts item_ea flags to offsets/segments.
// For undefined bytes, you may set item_ea == fixup_ea. In this case this
// function will create an item (byte, word, dword) there.
// Returns:
//      0 - no fixup at fixup_ea or it has FIXUP_NOUSED flag
//      1 - ok, the fixup information was applied

int apply_fixup(ulong item_ea, ulong fixup_ea, int n);


// Get base of fixup for set_offset() function

ulong get_fixup_base(ulong source,fixup_data_t *fd);


//--------------------------------------------------------------------------
inline ulong get_fixup_extdef_ea(fixup_data_t *fd) {
  return (fd != NULL && (fd->type & FIXUP_EXTDEF) != 0) 
        ? toEA(ask_selector(fd->sel),fd->off)
        : BADADDR;
}

//--------------------------------------------------------------------------
inline ushort get_fixup_segdef_sel(fixup_data_t *fd) {
  return (fd != NULL && (fd->type & FIXUP_EXTDEF) == 0) 
        ? fd->sel
        : BADSEL;
}

ulong  get_fixup_extdef_ea (ulong ea);
ushort get_fixup_segdef_sel(ulong ea);

char *get_fixup_desc(ea_t source, fixup_data_t *fdp,char *buf);


// Does the speficied address range contain any fixup information?

int contains_fixups(ea_t ea,size_t size);


// Convert information about the fixup records in the database to
// the new format.

void convert_fixups(void);


#pragma pack(pop)
#endif // FIXUP_HPP
