/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-98 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su, ig@datarescue.com
 *                              FIDO:   2:5020/209
 *
 */

#ifndef _IDP_HPP
#define _IDP_HPP

//
//      This file contains definition of the interface to IDP modules
//      The interface consists of 2 structures:
//        - definition of target assembler      name: ash
//        - definition of current processor     name: ph
//      These structures contain information about processor features,
//      function pointers, etc.

#include <stdio.h>

#include <segment.hpp>
#include <nalt.hpp>
#include <funcs.hpp>
#include <ua.hpp>
struct member_t;                // #include <struct.hpp>
struct mvm_t;                   // #include <micro.hpp>

#pragma pack(push, 1)           // IDA uses 1 byte alignments!

//      Versions history:
// 1                    There was no version control
// 2    27.02.96        Version control added
// 3    01.03.96        AS_HEXF5 is added
//                      int r50_to_asc(ushort *r, char *p, int k);
//                      int asc_to_r50(char *p, ushort *r, int k);
//                      AddSegment(ushort base,ulong start,ulong end,char *sclass);
//                      Pointer to preline().
//                      Pointer to XlatAsciiOutput
// 4    21.03.96        strspn(), putLong()
// 5    22.03.96        a_curip added
// 6    01.04.96        loader() added
// 7    05.04.96        translate() added
//                      ASB_BINF1 added
// 8    27.04.96        getFop/doFop functions
// 9    11.05.96        some rearrangments of database and API
// 10   14.06.96        many functions are removed.
// 11   23.06.96        netnode function interface is changed
// 12   08.07.96        get_item_flag() function is added
// 13   04.08.96        dataSeg() function changed
//                      maybe_code_sequence() function added
// 14   09.08.96        tbyte_size added
// 15   24.08.96        FF_FOP flags are moved
// 16   02.09.96        is_switch() function is added
// 17   06.09.96        changes for java
// 18   10.09.96        many fields are renamed
// 19   30.09.96        extract_address function
// 20   04.10.96        loader function has additional parameter: filename
// 21   05.10.96        loader function has not additional parameter: filename
// 22   23.11.96        codeSeg() function has addition parameter: opnum
// 23   16.12.96        ASO_OCTF4 added
// 24   24.12.96        ASB_BINF4 added
// 25   03.01.97        set_fixup() is not inline function anymore
// 26   25.01.97        some api functions are changed (bin_search)
// 27   07.02.97        Segment::Algin and Comb() are modified
// 28   04.03.97        Color functions
// 29   12.04.97        API is documented and clarified
// 30   25.05.97        Stack pointer register is added.
// 31   05.06.97        Stack variable definition lines are processor specific
// 32   06.07.97        "public" keyword is added
// 33   24.07.97        "align" keyword is added
// 34   12.09.97        get_name_base_ea() is added
// 35   07.01.98        micro virtual machine is added
// 36   04.02.98        new functions are added
// 37   21.03.98        32-bit bytes are added
//                      byte get/put functions are renamed
// 38   27.05.98        AS_ALIGN2 is added
// 39   26.10.98        REF_LOW, REF_HIGH format strings are added
// 40   14.11.98	The interface functions are renumbered,
//                      ph.loader is removed
//                      ph.notify is changed
// 41   10.12.98        get_name_expr() has additional arg
// 42   24.01.99        interface of snprintf() is changed
// 43   20.03.99        ph.is_jump_func is added
// 44   08.04.99        get_nlist_name is added
// 45   30.05.99        typeinfo calls are added
// 46   03.06.99        different bit widths for code and data
// 47   20.08.99        modifications for the gui version
// 48   04.11.99        CBuilder v4.0 is used to compile IDA
// 49   08.02.00        API is changed
// 50   18.02.00        Register variables are added
// 51   28.03.00        API is changed
// 52   08.05.00        API is changed
// 53   08.05.00        API is changed

#define         IDP_INTERFACE_VERSION   53

//-----------------------------------------------------------------------

class AbstractRegister;
class WorkReg;

struct rginfo {         // this structure is used only when detailed
                        // information on processor register is needed.
                        // Actually is used only for 80x86 processors.
  AbstractRegister  *low;
  AbstractRegister *high;
  AbstractRegister *word;
};

//-----------------------------------------------------------------------
// Interface to the function that is used during checking of manual operands.
typedef struct {        // checkarg_preline() parameter block
    int     *ind;
    char    *prefix;
    char    *seg;
    char    *reg;
    char    *offset;
} s_preline;

//-----------------------------------------------------------------------
typedef struct {        // structure used to describe byte streams
                        // (for "ret" instruction and empirics)
  uchar len;
  uchar *bytes;
} bytes_t;

//-----------------------------------------------------------------------
// IDA uses internal representation of processor instructions.
// Definition of all internal instructions are kept in special arrays.
// One of such arrays describes instruction names are features.

typedef struct {        // structure used in ins.cpp
                        // names and features of all instructions
  const char *name;
  ushort feature;
#define CF_STOP 0x0001  // Instruction doesn't pass execution to the
                        // next instruction
#define CF_CALL 0x0002  // CALL instruction (should make a procedure here)
#define CF_CHG1 0x0004  // The instruction modifies the first operand
#define CF_CHG2 0x0008  // The instruction modifies the second operand
#define CF_CHG3 0x0010  // The instruction modifies the third operand
#define CF_USE1 0x0020  // The instruction uses value of the first operand
#define CF_USE2 0x0040  // The instruction uses value of the second operand
#define CF_USE3 0x0080  // The instruction uses value of the third operand
#define CF_JUMP 0x0100  // The instruction passes execution using indirect
                        // jump or call (thus needs additional analysis)
#define CF_SHFT 0x0200  // Bit-shift instruction (shl,shr...)
#define CF_HLL  0x0400  // Instruction may be present in a high level
                        // language function.
} instruc_t;

int InstrIsSet(int icode,int bit);      // does the specified instruction
                                        // have the specified feature?

//=====================================================================
//
//      This structure describes the target assembler.
//      An IDP module may have several target assemblers.
//      In this case you should create a structure for each supported
//      assembler.
//
struct asm_t
{
  ulong flag;                           // Assembler features:
#define AS_OFFST      0x00000001L       // offsets are 'offset xxx' ?
#define AS_COLON      0x00000002L       // create colons after data names ?
#define AS_UDATA      0x00000004L       // can use '?' in data directives

#define AS_2CHRE      0x00000008L       // double char constants are: "xy
#define AS_NCHRE      0x00000010L       // char constants are: 'x
#define AS_N2CHR      0x00000020L       // can't have 2 byte char consts

                                        // ASCII directives:
#define AS_1TEXT      0x00000040L       //   1 text per line, no bytes
#define AS_NHIAS      0x00000080L       //   no characters with high bit
#define AS_NCMAS      0x00000100L       //   no commas in ascii directives

#define AS_HEXFM      0x00000E00L       // format of hex numbers:
#define ASH_HEXF0     0x00000000L       //   34h
#define ASH_HEXF1     0x00000200L       //   h'34
#define ASH_HEXF2     0x00000400L       //   34
#define ASH_HEXF3     0x00000600L       //   0x34
#define ASH_HEXF4     0x00000800L       //   $34
#define ASH_HEXF5     0x00000A00L       //   <^R   > (radix)
#define AS_DECFM      0x00003000L       // format of dec numbers:
#define ASD_DECF0     0x00000000L       //   34
#define ASD_DECF1     0x00001000L       //   #34
#define ASD_DECF2     0x00002000L       //   34.
#define ASD_DECF3     0x00003000L       //   .34
#define AS_OCTFM      0x0001C000L       // format of octal numbers:
#define ASO_OCTF0     0x00000000L       //   123o
#define ASO_OCTF1     0x00004000L       //   0123
#define ASO_OCTF2     0x00008000L       //   123
#define ASO_OCTF3     0x0000C000L       //   @123
#define ASO_OCTF4     0x00010000L       //   o'123
#define ASO_OCTF5     0x00014000L       //   123q
#define AS_BINFM      0x000E0000L       // format of binary numbers:
#define ASB_BINF0     0x00000000L       //   010101b
#define ASB_BINF1     0x00020000L       //   ^B010101
#define ASB_BINF2     0x00040000L       //   %010101
#define ASB_BINF3     0x00060000L       //   0b1010101
#define ASB_BINF4     0x00080000L       //   b'1010101
#define ASB_BINF5     0x000A0000L       //   b'1010101'

#define AS_UNEQU      0x00100000L       // replace underfined data items
                                        // with EQU (for ANTA's A80)
#define AS_ONEDUP     0x00200000L       // One array definition per line
#define AS_NOXRF      0x00400000L       // Disable xrefs during the output file generation
#define AS_XTRNTYPE   0x00800000L       // Assembler understands type of extrn
                                        // symbols as ":type" suffix
#define AS_RELSUP     0x01000000L       // Checkarg: 'and','or','xor' operations
                                        // with addresses are possible
#define AS_LALIGN     0x02000000L       // Labels at "align" keyword
                                        // are supported.
#define AS_NOCODECLN  0x04000000L       // don't create colons after code names
#define AS_NOTAB      0x08000000L       // Disable tabulation symbols during the output file generation
#define AS_NOSPACE    0x10000000L       // No spaces in expressions
#define AS_ALIGN2     0x20000000L       // .align directive expects an exponent rather than a power of 2
                                        // (.align 5 means to align at 32byte boundary)
#define AS_ASCIIC     0x40000000L       // ascii directive accepts C-like
                                        // escape sequences (\n,\x01 and similar)
#define AS_ASCIIZ     0x80000000L       // ascii directive inserts implicit
                                        // zero byte at the end

  ushort uflag;                         // user defined flags (local only for IDP)
                                        // you may define and use your own bits
  const char *name;                     // Assembler name (displayed in menus)
  help_t help;                          // Help screen number, 0 - no help
  const char **header;                  // array of automatically generated header lines
                                        // they appear at the start of disassembled text
  const ushort *badworks;               // array of unsupported instructions
                                        // (array of cmd.itype, zero terminated)
  const char *origin;                   // org directive
  const char *end;                      // end directive
  const char *cmnt;                     // comment string
  char ascsep;                          // ASCII string delimiter
  char accsep;                          // ASCII char constant delimiter
  const char *esccodes;                 // ASCII special chars
                                        // (they can't appear in character and
                                        // ascii constants)
//
//      Data representation (db,dw,...):
//
  const char *a_ascii;                  // ASCII string directive
  const char *a_byte;                   // byte directive
  const char *a_word;                   // word directive
  const char *a_dword;                  // NULL if not allowed
  const char *a_qword;                  // NULL if not allowed
  const char *a_oword;                  // NULL if not allowed
  const char *a_float;                  // float;  4bytes; NULL if not allowed
  const char *a_double;                 // double; 8bytes; NULL if not allowed
  const char *a_tbyte;                  // long double;    NULL if not allowed
  const char *a_packreal;               // packed decimal real NULL if not allowed
  const char *a_dups;                   // array keyword. the following
                                        // sequences may appear:
                                        //      #h - header
                                        //      #d - size
                                        //      #v - value
                                        //      #s(b,w,l,q,f,d,o) - size specifiers
                                        //                        for byte,word,
                                        //                            dword,qword,
                                        //                            float,double,oword
  const char *a_bss;                    // uninitialized data directive
                                        // should include '%s' for the
                                        // size of data
  const char *a_equ;                    // 'equ' Used if AS_UNEQU is set
  const char *a_seg;                    // 'seg ' prefix (example: push seg seg001)

//
//  Pointer to checkarg_preline() function. If NULL, checkarg won't be called.
//  'checkarg_operations' is used by checkarg()
//
  int (*checkarg_preline)(char *argstr, s_preline *S);
  char *(*checkarg_atomprefix)(char *operand,void *res); // if !NULL, is called before each atom
  const char **checkarg_operations;

//
// translation to use in character and string constants.
// usually 1:1, i.e. trivial translation (may specify NULL)
//
  const uchar *XlatAsciiOutput;         // If specified, must be 256 chars long
  const char *a_curip;                  // current IP (instruction pointer)
                                        // symbol in assembler
  void (*func_header)(func_t *,char *buf); // function header line
                                        // if NULL, then function headers
                                        // are displayed as normal lines
  void (*func_footer)(func_t *);        // function footer line
                                        // if NULL, then a comment line
                                        // is displayed
  const char *a_public;                 // "public" name keyword
  const char *a_weak;                   // "weak"   name keyword
  const char *a_extrn;                  // "extrn"  name keyword
  const char *a_comdef;                 // "comm" (communal variable)
//
// Get name of type of item at ea or id.
// (i.e. one of: byte,word,dword,near,far,etc...)
//
  const char *(*get_type_name)(flags_t flag,ea_t id);

  const char *a_align;                  // "align" keyword

// Left and right braces used in complex expressions

  char lbrace;
  char rbrace;

  const char *mod;      // %  mod     assembler time operation
  const char *band;     // &  bit and assembler time operation
  const char *bor;      // |  bit or  assembler time operation
  const char *xor;      // ^  bit xor assembler time operation
  const char *bnot;     // ~  bit not assembler time operation
  const char *shl;      // << shift left assembler time operation
  const char *shr;      // >> shift right assembler time operation
  const char *a_sizeof; // size of type

};

//=====================================================================
//
//      This structure describes a a processor module (IDP)
//      An IDP file may have only one such a structure, called LPH.
//      The kernel will copy it to 'ph' structure and use 'ph'.
//

struct processor_t
{
  ushort version;                       // Expected kernel version,
                                        //   should be IDP_INTERFACE_VERSION
  int   id;                             // IDP id
#define PLFM_386        0               // Intel 80x86
#define PLFM_Z80        1               // 8085, Z80
#define PLFM_I860       2               // Intel 860
#define PLFM_8051       3               // 8051
#define PLFM_TMS        4               // TMS320C5x
#define PLFM_6502       5               // 6502
#define PLFM_PDP        6               // PDP11
#define PLFM_68K        7               // Motoroal 680x0
#define PLFM_JAVA       8               // Java
#define PLFM_6800       9               // Motorola 68xx
#define PLFM_RESEVRED   10              // (not implemented)
#define PLFM_88K        11              // (not implemented) Motorola 88000
#define PLFM_MIPS       12              // MIPS
#define PLFM_ARM        13              // Advanced RISC Machines
#define PLFM_TMSC6      14              // TMS320C6x
#define PLFM_PPC        15              // PowerPC
#define PLFM_80196      16              // Intel 80196
#define PLFM_Z8         17              // Z8
#define PLFM_SH         18              // Hitachi SH
#define PLFM_IS         19              // (not ready) InstallShield (krk@mail.com)
#define PLFM_AVR        20              // Atmel 8-bit RISC processor(s)
#define PLFM_H8         21              // Hitachi H8/300, H8/2000
#define PLFM_PIC        22              // Microchip's PIC
#define PLFM_SPARC      23              // SPARC
#define PLFM_ALPHA      24              // DEC Alpha
#define PLFM_HPPA       25              // Hewlett-Packard PA-RISC
#define PLFM_H8500      26              // Hitachi H8/500
#define PLFM_TRICORE    27              // Tasking Tricore
#define PLFM_DSP56K     28              // Motorola DSP5600x

                                        // Numbers above 0x8000 are reserved
                                        // for third-party modules

  ushort flag;                          // Processor features
#define PR_SEGS       0x0001            // has segment registers?
#define PR_USE32      0x0002            // supports 32-bit addressing?
#define PR_DEFSEG32   0x0004            // segments are 32-bit by default
#define PR_RNAMESOK   0x0008            // allow to user register names for
                                        // location names
//#define PR_DB2CSEG    0x0010            // .byte directive in code segments
//                                        // should define even number of bytes
//                                        // (used by AVR processor)
#define PR_ADJSEGS    0x0020            // IDA may adjust segments moving
                                        // their starting/ending addresses.
#define PR_DEFNUM     0x00C0            // default number representation:
#define PRN_HEX       0x0000            //      hex
#define PRN_OCT       0x0040            //      octal
#define PRN_DEC       0x0080            //      decimal
#define PRN_BIN       0x00C0            //      binary
#define PR_WORD_INS   0x0100            // instruction codes are grouped
                                        // 2bytes in binrary line prefix
#define PR_NOCHANGE   0x0200            // The user can't change segments
                                        // and code/data attributes
                                        // (display only)
#define PR_ASSEMBLE   0x0400            // Module has a built-in assembler
                                        // and understands IDP_ASSEMBLE
#define PR_ALIGN      0x0800            // All data items should be aligned
                                        // properly
#define PR_TYPEINFO   0x1000            // ph.get_default_enum_size is valid
#define PR_USE64      0x2000            // supports 64-bit addressing?
                                        // Currently IDA doesn't have full 64bit support
#define PR_SGROTHER   0x4000            // the segment registers don't contain
                                        // the segment selectors, something else
#define PR_STACK_UP   0x8000            // the stack grows up

  int has_segregs(void)         { return (flag & PR_SEGS)     != 0; }
  int SegmentUse32Default(void) { return (flag & PR_DEFSEG32) != 0; }
  bool use32(void)              { return (flag & (PR_USE64|PR_USE32)) != 0; }
  bool use64(void)              { return (flag & PR_USE64)    != 0; }
  bool ti(void)                 { return (flag & PR_TYPEINFO) != 0; }
  bool stkup(void)              { return (flag & PR_STACK_UP) != 0; }

  int cnbits;                           // number of bits in a byte
                                        // for code segments
                                        // (usually 8)
                                        // IDA supports values up to 32
  int dnbits;                           // number of bits in a byte
                                        // for non-code segments
                                        // (usually 8)
                                        // IDA supports values up to 32
//
// Number of 8bit bytes required to hold one byte of the target processor
//
  int cbsize(void) { return (cnbits+7)/8; }     // for code segments
  int dbsize(void) { return (dnbits+7)/8; }     // for other segments

                                        // IDP module may support several compatible
                                        // processors. The following arrays define
                                        // processor names:
  char  **psnames;                      // short processor names (NULL terminated)
                                        // Each name should be shorter than 9 characters
  char  **plnames;                      // long processor names (NULL terminated)
                                        // No restriction on name lengthes.
  asm_t **assemblers;                   // pointer to array of target
                                        // assembler definitions. You may
                                        // change this array when current
                                        // processor is changed.
                                        // (NULL terminated)

//
// Callback function. IDP module can take appropriate
// actions when some events occurs in the kernel.
//
  enum idp_notify
  {
        init = 0,               // The IDP module is just loaded
        term,                   // The IDP module is being unloaded
        newprc,                 // Before changing proccesor type
                                // arg - int processor number in the
                                //       array of processor names
                                // return 1-ok,0-prohibit
        newasm,                 // Before setting a new assembler
                                // arg = int asmnum
        newfile,                // A new file is loaded (already)
                                // arg - char * input file name
        oldfile,                // An old file is loaded (already)
                                // arg - char * input file name
        newseg,                 // A new segment is about to be created
                                // arg = segment_t *
                                // return 1-ok, 0-segment should not be created
        assemble,               // Assemble an instruction
                                // (display a warning if an error is found)
                                // args:
                                //  ea_t ea -  linear address of instrucition
                                //  ea_t cs -  cs of instruction
                                //  ea_t ip -  ip of instruction
                                //  bool use32 - is 32bit segment?
                                //  const char *line - line to assemble
                                //  uchar *bin - pointer to output opcode buffer
                                // returns size of the instruction in bytes
        makemicro,              // Generate microcode for the instruction
                                // in 'cmd' structure.
                                // arg - mblock_t *
                                // returns MICRO_... error codes
        outlabel,               // The kernel is going to generate an instruction
                                // label line or a function header
                                // args:
                                //   ea_t ea -
                                //   const char *colored_name -
                                // If returns 0, then the kernel should
                                // not generate the label
        rename,                 // The kernel is going to rename a byte
                                // args:
                                //   ea_t ea
                                //   const char *new_name
                                // If returns 0, then the kernel should
                                // not rename it
        may_show_sreg,          // The kernel wants to display the segment registers
                                // in the messages window.
                                // arg - ea_t current_ea
                                // if this function returns 0
                                // then the kernel will not show
                                // the segment registers.
                                // (assuming that the module have done it)
        closebase,              // The database will be closed now
        load_idasgn,            // FLIRT signature have been loaded
                                // for normal processing (not for
                                // recognition of startup sequences)
                                // arg - const char *short_sig_name
                                // returns: nothing
        coagulate,              // Try to define some unexplored bytes
                                // This notification will be called if the
                                // kernel tried all possibilities and could
                                // not find nothing more useful than to
                                // convert to array of bytes.
                                // The module can help the kernel and convert
                                // the bytes into something more useful.
                                // arg:
                                //      ea_t start_ea
                                // returns: number of converted bytes + 1
        auto_empty,             // Info: the analysis queue is empty
                                // args: none
                                // returns: none
        func_bounds,            // find_func_bounds() finished its work
                                // The module may fine tune the function bounds
                                // args: int *possible_return_code
                                //       func_t *pfn
                                //       ea_t max_func_end_ea (from the kernel's point of view)
                                // returns: none
        may_be_func,            // can a function start here?
                                // arg: none, the instruction is in 'cmd'
                                // returns: probability 0..100
                                // 'cmd' structure is filled upon the entrace
                                // the idp module is allowed to modify 'cmd'
        is_sane_insn,           // is the instruction sane for the current file type?
                                // arg:  int no_crefs
                                // 1: the instruction has no code refs to it.
                                //    ida just tries to convert unexplored bytes
                                //    to an instruction (but there is no other
                                //    reason to convert them into an instruction)
                                // 0: the instruction is created because
                                //    of some coderef, user request or another
                                //    weighty reason.
                                // The instruction is in 'cmd'
                                // returns: 1-ok, 0-no, the instruction isn't
                                // likely to appear in the program
        is_jump_func,           // is the function a trivial "jump" function?
                                // args:  const func_t *pfn
                                //        ea_t *jump_target
                                // returns: 1-no, 2-yes, see jump_target
        gen_regvar_def,         // generate register variable definition line
                                // args:  regvar_t *v
                                // returns: 0-ok
        setsgr,                 // The kernel has changed a segment register value
                                // args:  ea_t startEA
                                //        ea_t endEA
                                //        int regnum
                                //        sel_t value
                                //        uchar tag (SR_... values)
                                // returns: 1-ok, 0-error
        set_compiler,           // The kernel has changed the compiler information
                                // (inf.cc structure)

                                // The codes below will be called only if
                                // PR_TYPEINFO is set
        cmangle_name=500,       // Mangle/unmangle a C symbol name
                                // const til_t *ti    - pointer to til
                                // const char *name   - name of symbol
                                // const type_t *type - type of symbol. If NULL then it will try to guess.
                                // char *outbuf       - output buffer
                                // size_t bufsize     - size of the output buffer
                                // bool mangle        - true-mangle, false-unmangle
                                // cm_t cc            - real calling convention for VOIDARG functions
                                // returns: true if success
        guess_memory_model,     // Try to guess the memory model
                                // args:    none
                                // returns: CM_N_... (see typeinf.hpp)
        based_ptr,              // get prefix and size of 'segment based' ptr
                                // type (something like char _ss *ptr)
                                // see description in typeinf.hpp
                                // args:  unsigned ptrt
                                //        const char **ptrname (output arg)
                                // returns: size of type
        max_ptr_size,           // get maximal size of a pointer in bytes
                                // args:  none
                                // returns: max possible size of a pointer
        align_size,             // get alignment delta for a structure field
                                // args:  size_t cur_tot_size - the structure size calculated so far
                                //                              (for unions: 0)
                                //        size_t elem_size    - size of the current field
                                //                              if elem_size == 0, the the final alignment for
                                //                              the whole structure should be calculated
                                //        size_t algn         - the structure alignment (1,2,4,8...)
                                //        cm_t cm             - the current calling convention and model
                                // returns: number of padding bytes required _before_ the field
                                //          if no padding is required, returns 0
        get_default_enum_size,  // get default enum size
                                // args:  cm_t cm
                                // returns: sizeof(enum)
        calc_arglocs,           // calculate function argument locations
                                // args:    const type_t **type
                                //          ulong *arglocs - the result array
                                //          int maxn       - number of elements in arglocs
                                // returns: int number_of_arguments
        use_stkarg_name,        // use information about a stack argument
                                // args:    ea_t ea        - address of the push instruction which
                                //                           pushes the function argument into the stack
                                //          const char *name - the function argument name
                                // returns: true - ok, false - failed, the kernel will create
                                //          a comment with the argument name for the instruction
        loader=1000,            // this code and higher ones are reserved
                                // for the loaders.
                                // the arguments and the return values are
                                // defined by the loaders
  };
  int   (*notify)(idp_notify msgid, ...); // Various notifications for the idp

//
// The following functions generate portions of the disassembled text.
//
  void  (*header)(void);                // function to produce start of disassembled text
  void  (*footer)(void);                // function to produce end of disassembled text

  void  (*segstart)(ea_t ea);          // function to produce start of segment
  void  (*segend)  (ea_t ea);          // function to produce end of segment

  void  (*assumes) (ea_t ea);          // function to produce assume directives
                                        // when segment register value changes
                                        // if your processor has no segment
                                        // registers, you may define it as NULL

// Analyse one instruction and fill 'cmd' structure.
// cmd.ea contains address of instruction to analyse.
// Return length of the instruction in bytes, 0 if instruction can't be decoded.
// This function shouldn't change the database, flags or anything else.
// All these actions should be performed only by u_emu() function.

  int   (*u_ana)   (void);

//
// Emulate instruction, create cross-references, plan to analyse
// subsequent instructions, modify flags etc. Upon entrance to this function
// all information about the instruction is in 'cmd' structure.
// Return length of the instruction in bytes.

  int   (*u_emu)   (void);

// Generate text representation of an instruction in 'cmd' structure.
// This function shouldn't change the database, flags or anything else.
// All these actions should be performed only by u_emu() function.

  void  (*u_out)   (void);

// Generate text representation of an instructon operand.
// This function shouldn't change the database, flags or anything else.
// All these actions should be performed only by u_emu() function.
// The output text is placed in the buffer pointer by 'u_line' pointer.
// Returns: 1-ok, 0-operand is hidden.

  int   (*u_outop) (op_t &op);


// Generate text represenation of data items
// This function MAY change the database and create cross-references, etc.

  void  (*d_out)   (ea_t ea);          // disassemble data

// Compare instruction operands.
// Return 1-equal,0-not equal operands.
// This pointer function may be NULL.

  int   (*cmp_opnd)(op_t &op1,op_t &op2);// returns 1 - equal operands

// Can the operand have a type as offset, segment, decimal, etc.
// (for example, a register AX can't have a type, the user can't change its
// representation. see bytes.hpp for information about types and flags)
// This pointer function may be NULL.

  int   (*can_have_type)(op_t &op);

//
//      Processor register information:
//

  int   regsNum;                        // number of registers
  char  **regNames;                     // their names

// The following pointers may be NULL:

  AbstractRegister &(*getreg)(int regnum); // Get register value.
                                        // may be NULL. If specified, will be
                                        // used in the determining predefined
                                        // comment based on register value

  int   rFiles;                         // number of register files
  char  **rFnames;                      // register names for files
  rginfo *rFdescs;                      // description of registers
  WorkReg *CPUregs;                     // pointer to CPU registers

// Segment register information (use virtual CS and DS registers if your
// processor doesn't have segment registers):

  int   regFirstSreg;                   // number of first segment register
  int   regLastSreg;                    // number of last segment register
  int   segreg_size;                    // size of a segment register in bytes

// You should define 2 virtual segment registers for CS and DS.
// Let's call them rVcs and rVcs.

  int   regCodeSreg;                    // number of CS register
  int   regDataSreg;                    // number of DS register

//
//      Empirics
//

  bytes_t *codestart;                   // Array of typical code start sequences
                                        // This array is used when a new file
                                        // is loaded to find code starts.
                                        // This array is terminated with
                                        // a zero length item.
  bytes_t *retcodes;                    // Array of 'return' instruction opcodes
                                        // This array is used to determine
                                        // form of autogenerated locret_...
                                        // labels.
                                        // This array should be defined
                                        // The last item of it should be { 0, NULL }

//
//      Instruction set
//

  int   instruc_start;                  // icode of the first instruction
  int   instruc_end;                    // icode of the last instruction + 1
  instruc_t *instruc;                   // Array of instructions
  int   (*is_far_jump)(int icode);      // is indirect far jump or call instruction?

//
//      Translation function for offsets
//      Currently used in the offset display functions
//      to calculate the referenced address
//
  ulong (*translate)(ulong base,ulong offset);

//
//      Size of long double (tbyte) for this processor
//      (meaningful only if ash.a_tbyte != NULL)
//
  size_t tbyte_size;

//
//      Floating point -> IEEE
// error codes returned by this function (load/store):
#define REAL_ERROR_FORMAT  -1 // not supported format for current .idp
#define REAL_ERROR_RANGE   -2 // number too big (small) for store (mem NOT modifyed)
#define REAL_ERROR_BADDATA -3 // illegal real data for load (IEEE data not filled)
//
  int (*realcvt)(void *m, ushort *e, ushort swt);

//
// Number of digits in floating numbers after the decimal point.
// If an element of this array equals 0, then the corresponding
// floating point data is not used for the processor.
// This array is used to align numbers in the output.
//      real_width[0] - number of digits for short floats (only PDP-11 has it)
//      real_width[1] - number of digits for "float"
//      real_width[2] - number of digits for "double"
//      real_width[3] - number of digits for "long double"
// Example: IBM PC module has { 0,7,15,19 }
//
  char real_width[4];

//
//  Find 'switch' idiom
//      fills 'si' structure with information and returns 1
//      returns 0 if switch is not found.
//      input: 'cmd' structure is correct.
//      this function may use and modify 'cmd' structure
//
  int (*is_switch)(switch_info_t *si);

//
//  Generate map file. If this pointer is NULL, the kernel itself
//  will create the map file.
//  This function returns number of lines in output file.
//  0 - empty file, -1 - write error
//
  long (*gen_map_file)(FILE *fp);

//
//  Extract address from a string. Returns BADADDR if can't extract.
//  Returns BADADDR-1 if kernel should use standard algorithm.
//
  ea_t (*extract_address)(ea_t ea,const char *string,int x);

//
//  Check whether the operand is relative to stack pointer
//  This function is used to determine how to output a stack variable
//  (if it returns 0, then the operand is relative to frame pointer)
//  This function may be absent. If it is absent, then all operands
//  are sp based by default.
//  Define this function only if some stack references use frame pointer
//  instead of stack pointer.
//  returns: 1 - yes, 0 - no
//
   int (*is_sp_based)(op_t &x);

//
//  Create a function frame for a newly created function.
//  Set up frame size, its attributes etc.
//  This function may be absent.
//
   int (*create_func_frame)(func_t *pfn);


// Get size of function return address in bytes
//      pfn - pointer to function structure, can't be NULL
// If this functin is absent, the kernel will assume
//      4 bytes for 32-bit function
//      2 bytes otherwise

   int (*get_frame_retsize)(func_t *pfn);


//
//  Generate stack variable definition line
//  If this function is NULL, then the kernel will create this line itself.
//  Default line is
//              varname = type ptr value
//  where 'type' is one of byte,word,dword,qword,tbyte
//
   void (*gen_stkvar_def)(char *buf,const member_t *mptr,long v);

// Generate text representation of an item in a special segment
// i.e. absolute symbols, externs, communal definitions etc.
// returns: 1-overflow, 0-ok

   int (*u_outspec)(ea_t ea,uchar segtype);

// Icode of return instruction. It is ok to give any of possible return
// instructions

   int icode_return;

// Set IDP-specific option
//      keyword - keyword encoutered in IDA.CFG file
//                if NULL, then a dialog form should be displayed
//      value_type - type of value of the keyword
#define IDPOPT_STR 1    // string constant (char *)
#define IDPOPT_NUM 2    // number (ulong *)
#define IDPOPT_BIT 3    // bit, yes/no (int *)
#define IDPOPT_FLT 4    // float (double *)
//      value   - pointer to value
// returns:
#define IDPOPT_OK       NULL            // ok
#define IDPOPT_BADKEY   ((char*)1)      // illegal keyword
#define IDPOPT_BADTYPE  ((char*)2)      // illegal type of value
#define IDPOPT_BADVALUE ((char*)3)      // illegal value (bad range, for example)
//      otherwise return pointer to an error message

  const char *(*set_idp_options)(const char *keyword,int value_type,const void *value);

//      Is the instruction created only for alignment purposes?
//      returns: number of bytes in the instruction

  int (*is_align_insn)(ea_t ea);

//      Micro virtual machine description
//      If NULL, IDP doesn't support microcodes.

  mvm_t *mvm;

};

// The following two structures contain information about the current
// processor and assembler.

extern "C" ida_export processor_t ph;   // Current processor
extern "C" ida_export asm_t ash;        // Current assembler

int str2regf(const char *p);    // -1 - error. Returns word reg number
int str2reg(const char *p);     // -1 - error. Returns any reg number

ida_export void  intel_data(ea_t ea);   // kernel function to display data items
                                        // and undefined bytes
                                        // This function should be used to
                                        // display data.
ida_export int   gen_spcdef(ea_t ea,uchar segtype);
                                        // generate declaration for item
                                        // in a special segment
                                        // return: 1-overflow, 0-ok
ida_export int   gen_extern(ea_t ea,const char *name);
                                        // generate declaration of extern symbol
                                        // return: 1-overflow, 0-ok
ida_export int   gen_abssym(ea_t ea,const char *name);
                                        // generate declaration of absolute symbol
                                        // return: 1-overflow, 0-ok
ida_export int   gen_comvar(ea_t ea,const char *name);
                                        // generate declaration of communal variable
                                        // return: 1-overflow, 0-ok

// Set target processor type
//      procname - name of processor type
//      level    - the power of request:
//        SETPROC_COMPAT - search for the processor type in the current module
//        SETPROC_ALL    - search for the processor type in all modules
//                         only if there were not calls with SETPROC_USER
//        SETPROC_USER   - search for the processor type in all modules
//                         and prohibit level SETPROC_USER
//        SETPROC_FATAL  - can be combined with previous bits.
//                         means that if the processor type can't be
//                         set, IDA should display an error message and exit.
// Returns: NULL - failed, otherwise path of file with processor module

#define SETPROC_COMPAT  0
#define SETPROC_ALL     1
#define SETPROC_USER    2

#define SETPROC_FATAL   0x80

char *set_processor_type(const char *procname,int level);


// Get name of the current processor module
//      buf -  the output buffer, should be at least QMAXFILE length
// The name is derived from the file name.
// For example, for IBM PC the module is named "pc.w32" (windows version)
// Then the module name is "PC" (uppercased)
// If no processor module is loaded, this function will return NULL

char *get_idp_name(char *buf);


// Unload the processor module.
// This function is for the kernel only.

void free_processor_module(void);


// Set target assembler
//      asmnum - number of assembler in the current IDP module

void set_target_assembler(short asmnum);


// Read IDA.CFG file and configure IDA for the current processor
// This is an internal kernel function.
// It should not be used in modules.

void read_config_file(const char *idppath);


// get number of bits in a byte at the given address
// the result depends on the segment type
// if the address doesn't belong to a segment, this function
// returns ph.dnbits()

int nbits(ea_t ea);


// get number of bytes required to store a byte at the given address

inline int bytesize(ea_t ea)
          { return (nbits(ea)+7)/8; }


#pragma pack(pop)
#endif // _IDP_HPP
