/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-97 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su
 *                              FIDO:   2:5020/209
 *
 */

#ifndef _LOADER_HPP
#define _LOADER_HPP

#include <netnode.hpp>
#pragma pack(push, 1)           // IDA uses 1 byte alignments!

//
//      This file contains:
//        - definitions of IDP, LDR, PLUGIN module interfaces.
//        - functions to load files into the database
//        - functions to generate output files
//        - high level functions to work with the database (open, save, close)
//
//      The LDR interface consists of one structure loader_t
//      The IDP interface consists of one structure processor_t (see idp.hpp)
//      The PLUGIN interface consists of one structure plugin_t
//
//      Modules can't use standard FILE* functions.
//      They must use functions from <fpro.h>
//
//      Modules can't use standard memory allocation functions.
//      They must use functions from <pro.h>
//
//      The first exported entry to the module should point the
//      the appropriate structure. (loader_t for LDR module, for example)
//

//----------------------------------------------------------------------
//              DEFINITION OF LDR MODULES
//----------------------------------------------------------------------

#define MAX_FILE_FORMAT_NAME    64

struct loader_t {
  ushort version;       // api version, should be IDP_INTERFACE_VERSION
  ulong flags;          // loader flags
#define LDRF_RELOAD  0x0001     // loader recognizes NEF_RELOAD flag  

//
//      check input file format. if recognized, then return 1
//      and fill 'fileformatname'.
//      otherwise return 0
//      This function will be called many times till it returns !=0.
//      'n' parameter will be incremented after each call.
//      Initially, n==0 for each loader.
//      This function may return a unique file format number instead of 1.
//      To get this unique number, please contact the author.
//
//      If the return value is ORed with ACCEPT_FIRST, then this format
//      should be placed first in the "load file" dialog box
//
#define ACCEPT_FIRST    0x8000

  int (*accept_file)(FILE *fp,char fileformatname[MAX_FILE_FORMAT_NAME],int n);
//
//      load file into the database.
//              fp       - pointer to file positioned at the start of the file
//              fileformatname - name of type of the file (it was returned
//                               by the accept_file)
//              neflags  - user-supplied flags. They determine how to
//                         load the file. See bit definitions in core.hpp
//
  void (*load_file)(FILE *fp,ushort neflags,const char *fileformatname);
#define NEF_SEGS        0x0001            // Create segments
#define NEF_RSCS        0x0002            // Load resources
#define NEF_NAME        0x0004            // Rename entries
#define NEF_MAN         0x0008            // Manual load
#define NEF_FILL        0x0010            // Fill segment gaps
#define NEF_IMPS        0x0020            // Create imports section
#define NEF_TIGHT       0x0040            // Don't align segments (OMF)
#define NEF_FIRST       0x0080            // This is the first file loaded
                                          // into the database.
#define NEF_CODE        0x0100            // for load_binary_file:
                                          //   load as a code segment
#define NEF_RELOAD      0x0200            // reload the file at the same place:
                                          //   don't create segments
                                          //   don't create fixup info
                                          //   don't import segments
                                          //   etc
                                          // load only the bytes into the base.
                                          // a loader should have LDRF_RELOAD
                                          // bit set
//
//      create output file from the database.
//      this function may be absent.
//      if fp == NULL, then this function returns:
//                      0 - can't create file of this type
//                      1 - ok, can create file of this type
//      if fp != NULL, then this function should create the output file
//
  int (*save_file)(FILE *fp,const char *fileformatname);
};

//----------------------------------------------------------------------
// LDR module file name extensions:

#ifdef __DOS16__
#define LOADER_EXT "ldd"
#endif
#ifdef __DOS32__
#define LOADER_EXT "ldx"
#endif
#ifdef __OS2__
#define LOADER_EXT "ldo"
#endif
#ifdef __NT__
#define LOADER_EXT "ldw"
#endif

#define LOADER_DLL "*." LOADER_EXT

//----------------------------------------------------------------------
//      Functions for the UI to load files
//----------------------------------------------------------------------

struct load_info_t      // List of loaders
{
  load_info_t *next;
  char dllname[QMAXPATH];
  char ftypename[MAX_FILE_FORMAT_NAME];
  filetype_t ftype;
  int pri;              // 1-place first, 0-normal priority
};

// Build list of potential loaders

load_info_t *build_loaders_list(const char *filename);


// Free the list of loaders

void free_loaders_list(load_info_t *list);


// Get name of loader used to load the input file into the database
// If no external loader was used, returns NULL
// Otherwise returns the loader file name without the extension
// (for example, for PE files we will get "PE")

char *get_loader_name(void);


// Load a binary file into the database.
// This function usually is called from ui.
//      original_file_name - the name of input file as is
//                           (if the input file is from library, then
//                            this is the name in the library)
//      real_file_name     - the name of file which contains data
//                           to be loaded. usually this name is the same
//                           as 'original_file_name' but for library modules
//                           this argument contains the name of a temporary
//                           file created to extract the module from the
//                           library
//      _neflags           - see NEF_.. constants. For the first file
//                           the flag NEF_FIRST must be set.
// Returns: 1-ok, 0-failed

int load_binary_file(const char *original_file_name,
                     const char *extracted_file_name,
                     ushort _neflags,
                     ulong fileoff,
                     ulong basepara,
                     ulong binoff);


// Load a non-binary file into the database.
// This function usually is called from ui.
//      original_file_name - the name of input file as is
//                           (if the input file is from library, then
//                            this is the name in the library)
//      real_file_name     - the name of file which contains data
//                           to be loaded. usually this name is the same
//                           as 'original_file_name' but for library modules
//                           this argument contains the name of a temporary
//                           file created to extract the module from the
//                           library
//      sysdlldir          - a directory with system dlls.
//      _neflags           - see NEF_.. constants. For the first file
//                           the flag NEF_FIRST must be set.
//      loader             - pointer to load_info_t structure.
//                           If the current IDP module has ph.loader != NULL
//                           then this argument is ignored.
// Returns: 1-ok, 0-failed

int load_nonbinary_file(const char *file,
                        const char *temp_file,
                        const char *sysdlldir,
                        ushort _neflags,
                        load_info_t *loader);


//--------------------------------------------------------------------------
// Output file types:

typedef int ofile_type_t;
const ofile_type_t
  OFILE_MAP = 0,        // MAP file
  OFILE_EXE = 1,        // Executable file
  OFILE_IDC = 2,        // IDC file
  OFILE_LST = 3,        // Disassembly listing
  OFILE_ASM = 4,        // Assembly
  OFILE_DIF = 5;        // Difference


//------------------------------------------------------------------
// Generate an output file
//      type  - type of output file.
//      fp    - the output file handle
//      ea1   - start address. For some file types this argument is ignored
//      ea2   - end address. For some file types this argument is ignored
//      flags - bit combination of GENFLG_...
// returns: number of the generated lines.
//          -1 if an error occured
//          ofile_exe: 0-can't generate exe file, 1-ok

int gen_file(ofile_type_t otype, FILE *fp, ulong ea1, ulong ea2, int flags);

// 'flag' is a combination of the following:
#define GENFLG_MAPSEG  0x0001          // map: generate map of segments
#define GENFLG_MAPNAME 0x0002          // map: include dummy names
#define GENFLG_MAPDMNG 0x0004          // map: demangle names
#define GENFLG_IDCTYPE 0x0008          // idc: gen only information about types
#define GENFLG_ASMTYPE 0x0010          // asm&lst: gen information about types too


//----------------------------------------------------------------------
//      Helper functions for the loaders & ui
//----------------------------------------------------------------------

// Load portion of file into the database
// This function will include (ea1..ea2) into the addressing space of the
// program (make it enabled)
//      fp         - pointer ot file
//      pos        - position in the file
//      (ea1..ea2) - range of destination linear addresses
//      patchable  - should the kernel remember correspondance of
//                   file offsets to linear addresses.
// returns: 1-ok,0-read error, a warning is displayed

int file2base(FILE *fp,long pos,ulong ea1,ulong ea2,int patchable);

#define FILEREG_PATCHABLE       1       // means that the input file may be
                                        // patched (i.e. no compression,
                                        // no iterated data, etc)
#define FILEREG_NOTPATCHABLE    0       // the data is kept in some encoded
                                        // form in the file.


// Load database from the memory.
//      memptr     - pointer to buffer with bytes
//      (ea1..ea2) - range of destination linear addresses
//      fpos       - position in the input file the data is taken from.
//                   if == -1, then no file position correspond to the data.
// This function works for wide byte processors too.
// returns: always 1

int mem2base(const void *memptr,ulong ea1,ulong ea2,long fpos);


// Unload database to a binary file.
//      fp         - pointer to file
//      pos        - position in the file
//      (ea1..ea2) - range of source linear addresses
// This function works for wide byte processors too.
// returns: 1-ok(always), write error leads to immediate exit

int base2file(FILE *fp,long pos,ulong ea1,ulong ea2);


// Load information from DBG file into the database
//      fp    - handler to opened file. If NULL, then fname is checked
//      fname - name of file to load (only if fp == NULL)
// Returns: 1-ok, 0-error (message is displayed)

int load_dbg_module(FILE *fp, const char *fname);


// Add long comment at inf.minEA:
//      Input file:     ....
//      File format:    ....
// This function should be called only from the loader to describe the input file.

void create_filename_cmt(void);


// Get the input file type
// This function can recognize libraries and zip files.

filetype_t      get_basic_file_type(const char *file);


// Get name of the current file type
// The current file type is kept in inf.filetype.
//      buf - buffer for the file type name.
// This function never returns NULL.

char *get_file_type_name(char *buf);


//----------------------------------------------------------------------
//      Work with IDS files: read and use information from them
//
// This structure is used in import_module():

struct impinfo_t {
  const char *dllname;
  void (*func)(ulong num, const char *name,ulong node);
  ulong node;
};

// Find and import DLL module.
//      module  - name of DLL
//      windir  - system directory with dlls
//      modnode - node with information about imported entries
//                imports by ordinals:
//                  altval(ord) contains linear address
//                imports by name:
//                  supval(ea) contains the imported name
//                Either altval or supval arrays may be absent.
//                The node should never be deleted.
//      importer- callback function (may be NULL):
//                check dll module
//              call 'func' for all exported entries in the file
//                      fp - pointer to file opened in binary mode.
//                      file position is 0.
//                      ud - pointer to impinfo_t structure
//              this function checks that 'dllname' match the name of module
//              if not, it returns 0
//              otherwise it calls 'func' for each exported entry
//              if dllname==NULL then 'func' will be called with num==0 and name==dllname
//              and returns:
//                      0 - dllname doesn't match, should continue
//                      1 - ok
//      ostype  - type of operating system (subdir name)
//                NULL means the IDS directory itself (not recommended)

void import_module(const char *module,
                   const char *windir,
                   netnode modnode,
                   int (*importer)(FILE *fp,impinfo_t *ii),
                   const char *ostype);


// Get full path of IDS file
// (this function is for the kernel only)

char *get_ids_filename(const char *idsname,const char *ostype);


// Load and apply IDS file
//      fname - name of file to apply
// This function loads the specified IDS file and applies it to the database
// If the program imports functions from a module with the same name
// as the name of the ids file being loaded, then only functions from this
// module will be affected. Otherwise (i.e. when the program does not import
// a module with this name) any function in the program may be affected.
// Returns:
//      1 - ok
//      0 - some error (a message is displayed)
//          if the ids file does not exist, no message is displayed

int load_ids_module(char *fname);


//----------------------------------------------------------------------
//              DEFINITION OF PLUGIN MODULES
//----------------------------------------------------------------------
// A plugin is a module in plugins subdirectory which can be called by
// pressing a hotkey. It usually performs an action asked by the user.

struct plugin_t
{
  int version;          // should be equal to IDP_INTERFACE_VERSION
  int flags;            // features of the plugin:
#define PLUGIN_MOD  0x0001      // Plugin changes the database.
                                // IDA won't call the plugin if
                                // the processor prohibited any changes
                                // by setting PR_NOCHANGES in processor_t.
#define PLUGIN_DRAW 0x0002      // IDA should redraw everything after calling
                                // plugin
#define PLUGIN_SEG  0x0004      // Plugin may be applied only if the
                                // current address belongs to a segment
  int (*init)(void);    // initialize
#define PLUGIN_OK    0  // Plugin agrees to work with the current database
#define PLUGIN_SKIP  1  // Plugin doesn't want to be loaded
  void (*term)(void);   // terminate. This function will be called
                        // when the user will exit IDA. May be NULL.
  void (*run)(int arg); // invoke plugin
  char *comment;        // long comment about the plugin
                        // it could appear in the status line
                        // or as a hint
  char *help;           // multiline help about the plugin
  char *wanted_name;    // the preferred short name of the plugin
  char *wanted_hotkey;  // the preferred hotkey to run the plugin
};

//--------------------------------------------------------------------------
// PLUGIN module file name extensions:

#ifdef __DOS32__
#define PLUGIN_EXT  "pld"
#endif
#ifdef __OS2__
#define PLUGIN_EXT  "pl2"
#endif
#ifdef __NT__
#define PLUGIN_EXT  "plw"
#endif

#define PLUGIN_DLL "*." PLUGIN_EXT

//----------------------------------------------------------------------
// LOW LEVEL DLL LOADING FUNCTIONS
// Only the kernel should use these functions!

#define LNE_MAXSEG      10      // Max number of segments

#if 0
extern char dlldata[4096];      // Reserved place for DLL data
#define DLLDATASTART    0xA0    // Absolute offset of dlldata
extern char ldrdata[64];        // Reserved place for LOADER data
#define LDRDATASTART    (DLLDATASTART+sizeof(dlldata)) // Absolute offset of ldrdata
#endif

typedef void *idadll_t[LNE_MAXSEG];

int load_dll(const char *file,idadll_t dllmem,void **dcb);
                                // memptr - allocated segments
                                //          segment 1 (data) isn't allocated
                                // dcb - ptr first entry of DLL
                                // Returns 0 - ok, else:
#define RE_NOFILE       1       /* No such file */
#define RE_NOTIDP       2       /* Not IDP file */
#define RE_NOPAGE       3       /* Can't load: bad segments */
#define RE_NOLINK       4       /* No linkage info */
#define RE_BADRTP       5       /* Bad relocation type */
#define RE_BADORD       6       /* Bad imported ordinal */
#define RE_BADATP       7       /* Bad relocation atype */
#define RE_BADMAP       8       /* DLLDATA offset is invalid */

           void load_dll_or_die(const char *file,idadll_t dllmem,void **entry);
ida_export int  load_dll_or_say(const char *file,idadll_t dllmem,void **entry);

void free_dll(idadll_t dllmem);

// The processor description string should be at the offset 0x80 of the IDP file
// (the string consists of the processor types separated by colons)

#define IDP_DESC_START   0x80
#define IDP_DESC_END    0x200

char *get_idp_desc(const char *file,char *buf); // Get IDP module name

//--------------------------------------------------------------------------
// IDP module file name extensions:

#ifdef __DOS16__
#define IDP_EXT  "idp"
#endif
#ifdef __DOS32__
#define IDP_EXT  "d32"
#endif
#ifdef __OS2__
#define IDP_EXT  "dll"
#endif
#ifdef __NT__
#define IDP_EXT  "w32"
#endif

#define IDP_DLL "*." IDP_EXT

//------------------------------------------------------------------------

//
//      work with file regions (for patching)
//
void init_fileregions(void);                            // called by the kernel
void term_fileregions(void);                            // called by the kernel
inline void save_fileregions(void) {}                   // called by the kernel
void add_fileregion(ulong ea1,ulong ea2,long fpos);     // called by the kernel

// Get offset in the input file which corresponds to the given ea
// If the specified ea can't be mapped into the input file offset,
// return -1.

long get_fileregion_offset(ea_t ea);


// Get linear address which corresponds to the specified input file offset.
// If can't be found, then return BADADDR

ea_t get_fileregion_ea(long offset);


//------------------------------------------------------------------------
// Generate an exe file (unload the database in binary form)
//      fp  - the output file handle
//            if fp == NULL then returns:
//                      1 - can generate an executable file
//                      0 - can't generate an executable file
// returns: 1-ok, 0-failed

int gen_exe_file(FILE *fp);


//------------------------------------------------------------------------
// Reload the input file
// This function reloads the byte values from the input file
// It doesn't modify the segmentation, names, comments, etc.
//      file - name of the input file
//             if file == NULL then returns:
//                      1 - can reload the input file
//                      0 - can't reload the input file
// returns: 1-ok, 0-failed

int reload_file(const char *file);


//------------------------------------------------------------------------
// Generate an IDC file (unload the database in text form)
//      fp        - the output file handle
//      onlytypes - if true then generate idc about enums, structs and
//                  other type definitions used in the program
// returns: 1-ok, 0-failed

int gen_idc_file(FILE *fp, bool onlytypes);


//------------------------------------------------------------------------
// Output to a file all lines controlled by place 'pl'
// This function is for the kernel only.

class place_t;
long print_all_places(FILE *fp,place_t *pl);


//--------------------------------------------------------------------------
// Get name of the input file

inline char *get_input_file_path(void) { return RootNode.value(); } // full path
char *get_root_filename(void);                  // file name only


//--------------------------------------------------------------------------
//
//      KERNEL ONLY functions & data
//

#define FILE_FORMAT_NAME_SUPIDX 1       // RootNode index for file format name
                                        // for loader modules

ida_export extern char *input_file;     // full path to the input file
ida_export extern char database_idb[];  // full path of IDB file
           extern char database_id0[];  // full path of original ID0 file (not exported)
ida_export bool is_database_ext(const char *ext); // check the file extension

extern ulong ida_database_memory;       // Database buffer size in bytes.

ida_export extern int database_flags;   // close_database() will:
#define DBFL_KILL       0x01            // delete unpacked database
#define DBFL_COMP       0x02            // compress database

char *LookupOldBase(void);                      // NULL-no old base, database
                                                // name otherwise
int check_database(const char *file);           // 0-no,1-yes
int open_database(int isnew,const char *file,ulong input_size);
                                                // returns 'waspacked'
int flush_buffers(void);                        // Flush buffers to disk
void save_database(void);                       // Flush buffers and make
                                                // a copy of database
void close_database(void);                      // see database_flags
bool compress_btree(const char *btree_file);    // (internal) compress .ID0

bool get_input_file_from_archive(char *filename, char **temp_file_ptr);

#pragma pack(pop)
#endif
