/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-97 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su
 *                              FIDO:   2:5020/209
 *
 */

#ifndef _PRO_H
#define _PRO_H

//
//      This is the first header included in IDA project.
//      It defines the most common types, functions and data.
//      Also, it tries to make system dependent definitions.
//

#include <stdlib.h>     /* size_t, NULL, memory */
#include <string.h>     /* str... */
#include <fcntl.h>      /* O_... */
#include <stdarg.h>
#include <stddef.h>
#include <assert.h>
#ifdef __BORLANDC__
#  include <io.h>         /* open, ... */
#  include <dir.h>        /* mkdir */
#else
#ifdef __WATCOMC__
#  include <io.h>         /* open, ... */
#  include <direct.h>     /* mkdir */
#else
#  include <unistd.h>     /* open, ... */
#  include <sys/stat.h>   /* mkdir */
#endif
#endif
#include <sys/types.h>  /* uchar, ... */

#ifndef __NOT_ONLY_PRO_FUNCS__
#define __ONLY_PRO_FUNCS__    // force usage of wrapper functions
#endif

/*==================================================*/
#ifdef __cplusplus
#define C_INCLUDE       extern "C" {
#define C_INCLUDE_END   }
#else
#define C_INCLUDE
#define C_INCLUDE_END
#endif

/*==================================================*/

#define __MF__  0               // Byte sex of our platform
                                // (Most significant byte First)
                                // 0 - little endian (Intel 80x86)
                                // 1 - big endian (PowerPC)

/*==================================================*/
/* Macro to avoid of message 'Parameter ... is never used' */
#if defined(__BORLANDC__) && !defined(__WIN32__) || defined(__WATCOMC__)
#define qnotused(x)     (x=x)
#else
#define qnotused(x)
#endif

/*==================================================*/
#ifdef __WATCOMC__
#define CONST_CAST(x)   (x)
#else
#define CONST_CAST(x)   const_cast<x>
#endif

/*==================================================*/
#if defined(__IDP__) && defined(__WIN32__)
#define ida_export      __declspec(dllimport)
#else
#define ida_export
#endif

/*==================================================*/
#if (defined(__WATCOMC__) && (__WATCOMC__ < 1100)) || defined(__DOS16__)

typedef int bool;
const bool false = 0;
const bool true = 1;

#endif

/*==================================================*/
/* error codes */
/*--------------------------------------------------*/

#define eOk        0    /* No error             */
#define eOS        1    /* OS error, see errno  */
#define eDiskFull  2    /* Disk Full            */
#define eReadError 3    /* Read Error           */

typedef int error_t;

/*--------------------------------------------------*/
/* internal code of last error occured              */
/* see err.h for error handling functions           */
extern ida_export error_t qerrno;

/*==================================================*/
/* type of OS */
/*--------------------------------------------------*/
typedef enum
{
   osMSDOS,
   osAIX_RISC,
   osOS2,
   osNT
} ostype_t;

/*--------------------------------------------------*/
extern ostype_t ostype;

/*==================================================*/
/* memory */
/*--------------------------------------------------*/
ida_export void *qalloc( size_t size );
ida_export void *qrealloc( void *alloc, size_t newsize );
ida_export void *qcalloc( size_t nitems, size_t itemsize );
ida_export void qfree( void *alloc );
ida_export char *qstrdup( const char *string );
#define qnew(t)        ((t*)qalloc(sizeof(t)))
#define qnewarray(t,n) ((t*)qcalloc((n),sizeof(t)))
#define qnumber(a)     (sizeof(a)/sizeof((a)[0]))

#ifdef __ONLY_PRO_FUNCS__
#define malloc          dont_use_this_function
#define calloc          dont_use_this_function
#define realloc         dont_use_this_function
#define free            dont_use_this_function
#define strdup          dont_use_this_function
#endif
/*==================================================*/
/* file name */
/*--------------------------------------------------*/
/* maximum number of characters in path and file specification */
#ifdef __MSDOS__
#define QMAXPATH        _MAX_PATH
#define QMAXFILE        (_MAX_FNAME+_MAX_EXT)
#else
#define QMAXPATH        255     /* POSIX standard */
#define QMAXFILE        255     /* POSIX standard is 14 */
#endif
/*--------------------------------------------------*/
/* construct 'path' from component's list terminated by NULL, return 'path'.
   if 'path'==NULL then pointer to static array will be returned,
   qmakepath( NULL, NULL ) just returns pointer to static array.
*/
ida_export char *qmakepath( char *path, const char *s1, ... );
/*--------------------------------------------------*/
/* split 'path' into 'dir' and 'file' parts, you may specify NULL
   as 'dir'/'file' parameters. 'path' may be changed.
   return file part.
*/
ida_export char *qsplitpath( char *path, char **dir, char **file );
/*--------------------------------------------------*/
/* construct filename from base name and extension, return 'file'.
   if 'file'==NULL then pointer to static array will be returned,
   qmakefile( NULL, NULL, NULL ) just returns pointer to static array.
*/
ida_export char *qmakefile( char *file, const char *base, const char *ext );
/*--------------------------------------------------*/
/* split filename to base name and extension, you may specify NULL
   as 'base'/'ext' parameters. 'file' may be changed.
   return base part.
*/
ida_export char *qsplitfile( char *file, char **base, char **ext );
ida_export int qisabspath(const char *file);
ida_export const char *basename(const char *path);
char *qmake_full_path(char *dst, const char *src);

/* Delimiter of directory lists */
#ifdef __UNIX__
#define DELIMITER       ":"     /* Unix   */
#else
#define DELIMITER       ";"     /* MS DOS */
#endif

/*==================================================*/
/* input/output */
/*--------------------------------------------------*/
#if !defined(__MSDOS__) && !defined(__OS2__) && !defined(__NT__)
#define O_BINARY        0
#endif

#ifndef SEEK_SET
#define SEEK_SET        0
#define SEEK_CUR        1
#define SEEK_END        2
#endif
/*--------------------------------------------------*/
/* you should use these functions for file i/o                */
/* they do the same as their counterparts from Clib.          */
/* the only difference is that they set 'qerrno' variable too */

ida_export int  qopen( const char *file, int mode );     /* open existing file */
ida_export int  qcreate( const char *file, int stat );   /* create new file with O_RDWR */
ida_export int  qread( int h, void *buf, size_t n );
ida_export int  qwrite( int h, const void *buf, size_t n );
ida_export long qtell( int h );
ida_export long qseek( int h, long offset, int whence );
ida_export int  qclose( int h );
ida_export long qfilesize( char *fname );
ida_export long qfilelength( int h );
ida_export int  qchsize( int h, unsigned long fsize );
ida_export int  qfileexist( const char *file );
ida_export int  qmkdir( const char *file, int mode );
#define qfileexist(p)   ( access( (p), 0 ) == 0 )

#ifdef __ONLY_PRO_FUNCS__
#define open            dont_use_this_function
// netnode::create clashes with the following definition
//#define create          dont_use_this_function
#define read            dont_use_this_function
#define write           dont_use_this_function
#define tell            dont_use_this_function
#define lseek           dont_use_this_function
#define close           dont_use_this_function
#define filelength      dont_use_this_function
#define mkdir           dont_use_this_function
#define system          dont_use_this_function
#define min             dont_use_this_function
#define max             dont_use_this_function
#define atexit          dont_use_this_function
#define exit            dont_use_this_function
#endif

/*==================================================*/
/* strings */
/*--------------------------------------------------*/
#ifndef __BORLANDC__
ida_export char *stpcpy( char *dest, const char *src );
ida_export char *strlwr( char *s );
ida_export char *strupr( char *s );
#endif
/*--------------------------------------------------*/
ida_export char *strleft( char *s, int len );
ida_export char *strright( char *s, int len );
ida_export char *strcenter( char *s, int len );
ida_export char *strpad( char *s, int len );
ida_export char *strpadc( char *s, int len, char c );
ida_export char *strprepadc( char *s, int len, char c );
ida_export char *strclip( char *s );
ida_export char *strclipc( char *s, char clipchar );
ida_export char *strpreclip( char *s );
ida_export char *strfill(char *s, char chr, int count);
ida_export char *strcompact(char *string);

/*--------------------------------------------------*/
#if defined(__BORLANDC__) && __BORLANDC__ >= 0x0540 
#define SNPRINTF_PRESENT
#include <stdio.h>              // snprintf is here
#endif

#if defined(__GNUC__)
#define SNPRINTF_PRESENT
#endif

#if defined(__MSC_VER__)
#define SNPRINTF_PRESENT
#define snprintf _snprintf
#define vsnprintf _vsnprintf
#endif

#if !defined(SNPRINTF_PRESENT)
ida_export int snprintf(char *buffer, size_t n, const char *format, ...);
ida_export int vsnprintf(char *buffer, size_t n, const char *format, va_list va);
#endif

/*==================================================*/
/* Replace all entries of 'char1' by 'char2' in string
   'str'
*/
ida_export char *strrpl( char *str, int char1, int char2 );

/*==================================================*/
/* system call with parameters substitution */
/*--------------------------------------------------*/
ida_export int qsystem( const char *fmt, ... );
ida_export void qexit(int code);
ida_export void qatexit(void (*func)(void));

/*==================================================*/
/* uchar, ... */
/*--------------------------------------------------*/
#if defined(__MSDOS__) || defined(__OS2__) || defined(__NT__)
typedef unsigned char  uchar;
typedef unsigned short ushort;
typedef unsigned int   uint;
typedef unsigned long  ulong;

#ifdef __BORLANDC__
#define __HAS_LONGLONG__
#endif
#include <llong.hpp>

typedef          char   int8;
typedef signed   char   sint8;
typedef unsigned char   uint8;
typedef          short  int16;
typedef unsigned short  uint16;
typedef          long   int32;
typedef unsigned long   uint32;
typedef longlong        int64;
typedef ulonglong       uint64;

inline bool can_place32(uint64 a) { return a == (uint64)(uint32)low(a); }
inline bool can_place32(int64 a)  { return a == ( int64)( int32)low(a); }

#endif

/*==================================================*/
/* universal min, max */
/*--------------------------------------------------*/
#define qmin(a,b) ( (a) < (b)? (a): (b) )
#define qmax(a,b) ( (a) > (b)? (a): (b) )

/*==================================================*/
/* Add-ins for 2/4 byte read/writes.
        h - file handle
        res - value read from file
        size - size of value in bytes (1,2,4)
        mostfirst - is MSB first? (0/1)

   All these functions return 0 - Ok */

ida_export int readbytes(int h,ulong *res,int size,int mostfirst);
ida_export int writebytes(int h,ulong l,int size,int mostfirst);

ida_export int read2bytes(int h,ushort *res,int mostfirst);
#define read4bytes(h,res,mostfirst)     readbytes(h,res,4,mostfirst)
#define write2bytes(h,l,mostfirst)      writebytes(h,l,2,mostfirst)
#define write4bytes(h,l,mostfirst)      writebytes(h,l,4,mostfirst)

/*==================================================*/
#ifdef __cplusplus
inline ulong swap32(ulong x)
{ return (x>>24) | (x<<24) | ((x>>8) & 0x0000FF00L) | ((x<<8) & 0x00FF0000L); }
inline ushort swap16(ushort x)
{ return ushort((x<<8) | (x>>8)); }
#else
#define swap32(x) ulong((x>>24) | (x<<24) | ((x>>8) & 0x0000FF00L) | ((x<<8) & 0x00FF0000L))
#define swap16(x) ushort((x<<8) | (x>>8))
#endif

#endif /* _PRO_H */
