/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-97 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su
 *                              FIDO:   2:5020/209
 *
 */

#ifndef _QUEUE_HPP
#define _QUEUE_HPP

//
//      This file contains functions that deal with the list of problems.
//      IDA maintains list of problems. It inserts suspicious addresses
//      to this list. Unfortunately, user-interface with this list is not
//      ideal yet.
//      There are several problem lists.
//      An address may be inserted to any list. You are free to use any list.
//      The kernel simply maintains these lists, no additional processing
//      is done. Addresses in the lists are kept sorted.

//      The following problem lists exist:

typedef uchar qtype_t;
const qtype_t
  Q_noBase  =  1,  // Can't find offset base
  Q_noName  =  2,  // Can't find name
  Q_noFop   =  3,  // Can't find forced op
  Q_noComm  =  4,  // Can't find comment           !!!! not used anymore !!!
  Q_noRef   =  5,  // Can't find references
  Q_jumps   =  6,  // Jump by table                !!!! ignored
  Q_disasm  =  7,  // Can't disasm
  Q_head    =  8,  // Already head
  Q_noValid =  9,  // Exec flows beyond limits
  Q_lines   = 10,  // Too many lines
  Q_badstack= 11,  // Failed to trace the value of the stack pointer
  Q_att     = 12,  // Attention! Probably erroneous situation.
  Q_final   = 13,  // Decision to convert to instruction/data is made by IDA
  Q_rolled  = 14,  // The decision made by IDA was wrong and rolled back
  Q_Qnum    = 15;  // Number of qtypes


// Insert an address to a list of problems.
// Display a message saying about the problem (except of Q_att,Q_final)
// Q_jumps is temporarily ignored.
//      ea   - linear address
//      type - problem queue type

void  QueueMark(qtype_t type,ea_t ea);


// Get an address from any problem list.
// The address is not removed from the list. The kernel returns an address
// from a list with the smallest type value.
//      type - problem queue type the address is got from
// returns: linear address or BADADDR

ea_t QueueGet(qtype_t *type);


// Get an address from the specified problem list
// The address is not removed from the list.
//      type  - problem queue type
//      lowea - the returned address will be higher or equal
//              than the specified address
// returns: linear address or BADADDR

ea_t QueueGet(qtype_t type,ea_t lowea);


// Remove an address from a problem list
//      ea   - linear address
//      type - problem queue type

void  QueueDel(qtype_t type,ea_t ea);


// Remove an address from all problem lists
//      ea   - linear address

void  QueueDel(ea_t ea);


// Get queue problem description string

const char *get_long_queue_name(qtype_t type);
const char *get_short_queue_name(qtype_t type);


// The kernel only functions:

       void init_queue(void);
inline void save_queue(void) {}
       void term_queue(void);

#endif  //  _QUEUE_HPP
