/*
 *              Class "Virtual Array of Longs"
 *
 *              Written by Ilfak Guilfanov, 1991
 *              Latest modifications at 1997
 *              Copyright (c), All Rights Reserved.
 */

#ifndef VARRAY_HPP
#define VARRAY_HPP

#include <pro.h>
#include <vm.hpp>
#pragma pack(push, 1)   // IDA uses 1 byte alignments!

implementVM(ulong);

class vaptr_t;                  // internal classes
class vaheader_t;

class Varray {

public:

        Varray(void)    { Pages = NULL; }
        ~Varray(void)   { vclose(); }

        int     linkTo  (const char *file,int Psize,int PoolSize);
                                        // Psize - how many longs
                                        // PoolSize - how many pages
                                        // if file doesn't exist, it will be
                                        // created
        void    vclose   (void);
//
//      Address space functions
//
        int     enable  (ulong start,ulong end);        // 0-ok,else error
        int     disable (ulong start,ulong end);        // 0-ok,else error
        bool    enabled (const ulong addr)              // is enabled ?
                                { return getoff(addr) != 0; }
        ulong nextaddr  (ulong addr);   // get next enabled addr
                                                        // if not exist, returns -1
        ulong prevaddr  (ulong addr);   // get prev enabled addr
                                                        // if not exist, returns -1
        ulong prevchunk (ulong addr);   // return prev chunk last addr
                                                        // if not exist, returns -1
        ulong nextchunk (ulong addr);   // return next chunk first addr
                                                        // if not exist, returns -1
        ulong size      (ulong addr);   // return size of the chunk
                                                        // addr may be any address in the chunk
        ulong getstart  (ulong addr);   // get start address of the chunk
                                                        // if illegal address given, returns -1
//
//      Read/Write functions
//
        long    vread   (ulong num)     { return *Raddr(num); }
        void    vwrite  (ulong num,long val) { *Waddr(num)  =  val; }
        void    setbits (ulong num,long bit) { *Waddr(num) |=  bit; }
        void    clrbits (ulong num,long bit) { *Waddr(num) &= ~bit; }
        ulong*  Waddr   (ulong num);    // return address for the 'num', mark page as dirty
        ulong*  Raddr   (ulong num);    // return address for the 'num'
        void    vflush  (void)          { Pages->vflush(); }

        void    memset  (ulong start,ulong size,ulong x);
        void    memcpy  (ulong start,ulong size,Varray &src,ulong srcstart);
        ulong memcmp(ulong start,ulong size,Varray &v2,ulong v2start);
                                                // returns -1 - if equal
                                                // else address where mismatches
        ulong memscn (ulong start,ulong size,ulong x);
        ulong memtst (ulong start,ulong size,bool (*test)(ulong));
        ulong memtstr(ulong start,ulong size,bool (*test)(ulong));

        ulong  *vread    (ulong start,      ulong *buffer,ulong size);
        void    vwrite   (ulong start,const ulong *buffer,ulong size);

        void    shright (ulong from);           // Shift right tail of array
        void    shleft  (ulong from);           // Shift left  tail of array

//
//      Sorted array functions
//
        ulong bsearch(ulong ea);        // Binary search
                                        // Returns index to >= ea
                                        // Attention: this func may return
                                        // pointer past array !
        bool addsorted(ulong ea);// Add an element to a sorted array.
                                        // If element exists, return 0
                                        // else 1
        bool delsorted(ulong ea);// Del element from a sorted array
                                        // If doesn't exist, return 0
                                        // else return 1
private:
  declareVM(ulong) *Pages;
  vaheader_t *header;                   // Header page
  ulong lastnum;                        // Last accessed address
  vaptr_t *lastvp;                      //  vptr of it
  ulong lastoff;                        //  offset of it
  ushort lastPage;
  ulong *lPage;
  uint psize;                           // how many items can be put into a page

//
// scan the Varray forward. call 'perform' for each page. If perform()
// returns >=0, then this is an index in the page, return address with this index.
// if perform() < 0, continue...
// Returns the address calculated by the index returned by perform().
//      perform() args: page - pointer to the page
//                      s    - size of the page
// 'change' - will perform() change the pages
//
  ulong vascan(ulong start,ulong size,int (*perform)(ulong *page,int s),bool change);
  ulong vascanr(ulong start,ulong size,int (*perform)(ulong *page,int s),bool change);

  ulong getoff(ulong addr);
  int movePages(ulong offset,
                ushort op,
                ushort np,
                int changecur);

};

#pragma pack(pop)
#endif // VARRAY_HPP
