/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-98 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su, ig@datarescue.com
 *                              FIDO:   2:5020/209
 *
 */

#include "i51.hpp"
#include <entry.hpp>
#include <srarea.hpp>

//--------------------------------------------------------------------------
processor_subtype_t ptype;
ea_t intmem = 0;
ea_t sfrmem = 0;

static char * RegNames[] =
{
  "A", "AB", "B",
  "R0", "R1", "R2",  "R3",  "R4",  "R5",  "R6",  "R7",
  "R8", "R9", "R10", "R11", "R12", "R13", "R14", "R15",
  "WR0",  "WR2",  "WR4",  "WR6",  "WR8",  "WR10", "WR12", "WR14",
  "WR16", "WR18", "WR20", "WR22", "WR24", "WR26", "WR28", "WR30",
  "DR0",  "DR4",  "DR8",  "DR12", "DR16", "DR20", "DR24", "DR28",
  "DR32", "DR36", "DR40", "DR44", "DR48", "DR52", "DPX",  "SPX",
  "DPTR","C", "PC",
  "cs","ds"
};

//----------------------------------------------------------------------

predefined_t far iregs[] = {
  { prc_51,  0x80, 0, "P0"     ,  NULL },
  { prc_51,  0x81, 0, "SP"     ,  NULL },
  { prc_51,  0x82, 0, "DPL"    ,  NULL },
  { prc_51,  0x83, 0, "DPH"    ,  NULL },
  { prc_251, 0x84, 0, "DPXL"   ,  NULL },
  { prc_51,  0x87, 0, "PCON"   ,  "Power Control" },
  { prc_51,  0x88, 0, "TCON"   ,  "Timer Control" },
  { prc_51,  0x89, 0, "TMOD"   ,  "Timer Mode" },
  { prc_51,  0x8A, 0, "TL0"    ,  "Timer0 - Low Byte" },
  { prc_51,  0x8B, 0, "TL1"    ,  "Timer1 - Low Byte" },
  { prc_51,  0x8C, 0, "TH0"    ,  "Timer0 - High Byte" },
  { prc_51,  0x8D, 0, "TH1"    ,  "Timer1 - High Byte" },
  { prc_930, 0x8F, 0, "FADDR"  ,  "Function Address Register" },
  { prc_51,  0x90, 0, "P1"     ,  NULL },
  { prc_930, 0x97, 0, "HADDR"  ,  "Hub Address Register" },
  { prc_51,  0x98, 0, "SCON"   ,  "Serial Port Control" },
  { prc_51,  0x99, 0, "SBUF"   ,  "Serial Port Data" },
  { prc_930, 0x9A, 0, "HPPWR"  ,  "Hub Port Power Control" },
  { prc_51,  0xA0, 0, "P2"     ,  NULL },
  { prc_930, 0xA1, 0, "HIE"    ,  "Hub Interrupt Enable Register" },
  { prc_930, 0xA2, 0, "FIE"    ,  "USB Function Interrupt Enable Register" },
  { prc_930, 0xA3, 0, "FIE1"   ,  "USB Function Interrupt Enable Register" },
  { prc_51,  0xA6, 0, "WDTRST" ,  "WatchDog Timer Reset Register" },
  { prc_251, 0xA7, 0, "WCON"   ,  "Wait State Control Register" },
  { prc_51,  0xA8, 0, "IE"     ,  "Interrupt Enable" },
  { prc_51,  0xA9, 0, "SADDR"  ,  "Slave Address" },
  { prc_930, 0xAE, 0, "HSTAT"  ,  "Hub Status and Configuration" },
  { prc_51,  0xB0, 0, "P3"     ,  NULL },
  { prc_930, 0xB1, 0, "IEN1"   ,  "Interrupt Enable 1" },
  { prc_930, 0xB2, 0, "IPL1"   ,  "Interrupt Priority Low 1" },
  { prc_930, 0xB3, 0, "IPH1"   ,  "Interrupt Priority High 1"  },
  { prc_251, 0xB7, 0, "IPH0"   ,  "Interrupt Priority High" },
  { prc_251, 0xB8, 0, "IPL0"   ,  "Interrupt Priority Low" },
  { prc_51,  0xB8, 0, "IP"     ,  "Interrupt Priority" },
  { prc_51,  0xB9, 0, "SADEN"  ,  "Slave Address Mask" },
  { prc_251, 0xBE, 0, "SPH"    ,  NULL },
  { prc_930, 0xC0, 0, "FIFLG"  ,  "Interrupt Flag Register" },
  { prc_930, 0xC1, 0, "FIFLG1" ,  "Interrupt Flag Register 1" },
  { prc_930, 0xC7, 0, "EPCONFIG", "Endpoint Configuration Register" },
  { prc_51,  0xC8, 0, "T2CON"  ,  "Timer2 Control" },
  { prc_51,  0xC9, 0, "T2MOD"  ,  "Timer2 Mode" },
  { prc_51,  0xCA, 0, "RCAP2L" ,  "Timer2 Reload Value - Low Byte" },
  { prc_51,  0xCB, 0, "RCAP2H" ,  "Timer2 Reload Value - High Byte" },
  { prc_51,  0xCC, 0, "TL2"    ,  "Timer2 - Low Byte" },
  { prc_51,  0xCD, 0, "TH2"    ,  "Timer2 - High Byte" },
  { prc_930, 0xCF, 0, "HPCON"  ,  "Hub Port Control" },
  { prc_51,  0xD0, 0, "PSW"    ,  NULL },
  { prc_251, 0xD1, 0, "PSW1"   ,  NULL },
  { prc_930, 0xD2, 0, "SOFL"   ,  "Start of Frame Low Register" },
  { prc_930, 0xD3, 0, "SOFH"   ,  "Start of Frame High Register" },
  { prc_930, 0xD4, 0, "HPINDEX",  "Hub Port Index Register" },
  { prc_930, 0xD5, 0, "HPSC"   ,  "Hub Port Status Change" },
  { prc_930, 0xD7, 0, "HPSTAT" ,  "Hub Port Status" },
  { prc_51,  0xD8, 0, "CCON"   ,  "PCA Control" },
  { prc_51,  0xD9, 0, "CMOD"   ,  "PCA Mode" },
  { prc_51,  0xDA, 0, "CCAPM0" ,  "PCA Mode Module0" },
  { prc_51,  0xDB, 0, "CCAPM1" ,  "PCA Mode Module1" },
  { prc_51,  0xDC, 0, "CCAPM2" ,  "PCA Mode Module2" },
  { prc_51,  0xDD, 0, "CCAPM3" ,  "PCA Mode Module3" },
  { prc_51,  0xDE, 0, "CCAPM4" ,  "PCA Mode Module4" },
  { prc_930, 0xDF, 0, "PCON1"  ,  NULL },
  { prc_51,  0xE0, 0, "ACC"    ,  NULL },
  { prc_930, 0xE1, 0, "EPCON"  ,  "Endpoint Control Register" },
  { prc_930, 0xE2, 0, "RXSTAT" ,  "Endpoint Receive Status" },
  { prc_930, 0xE3, 0, "RXDAT"  ,  "Receive FIFO Data" },
  { prc_930, 0xE4, 0, "RXCON"  ,  "Receive FIFO Control" },
  { prc_930, 0xE5, 0, "RXFLG"  ,  "Receive FIFO Flag" },
  { prc_930, 0xE6, 0, "RXCNTL" ,  "Receive FIFO Byte Count Low" },
  { prc_930, 0xE7, 0, "RXCNTH" ,  "Receive FIFO Byte Count High" },
  { prc_930, 0xE8, 0, "HIFLG"  ,  "Hub Interrupt Flag Register" },
  { prc_51,  0xE9, 0, "CL"     , "PCA Counter Low Byte" },
  { prc_51,  0xEA, 0, "CCAP0L" , "PCA Module0 Capture Value - Low Byte" },
  { prc_51,  0xEB, 0, "CCAP1L" , "PCA Module1 Capture Value - Low Byte" },
  { prc_51,  0xEC, 0, "CCAP2L" , "PCA Module2 Capture Value - Low Byte" },
  { prc_51,  0xED, 0, "CCAP3L" , "PCA Module3 Capture Value - Low Byte" },
  { prc_51,  0xEE, 0, "CCAP4L" , "PCA Module4 Capture Value - Low Byte" },
  { prc_51,  0xF0, 0, "B"      ,  NULL },
  { prc_930, 0xF1, 0, "EPINDEX",  "Endpoint Index Register" },
  { prc_930, 0xF2, 0, "TXSTAT" ,  "Endpoint Transmit Statis" },
  { prc_930, 0xF3, 0, "TXDAT"  ,  "Transmit FIFO Data" },
  { prc_930, 0xF4, 0, "TXCON"  ,  "Transmit FIFO Control" },
  { prc_930, 0xF5, 0, "TXFLG"  ,  "Transmit FIFO Flag" },
  { prc_930, 0xF6, 0, "TXCNTL" ,  "Transmit FIFO Byte Count Low" },
  { prc_930, 0xF7, 0, "TXCNTH" ,  "Transmit FIFO Byte Count High" },
  { prc_51,  0xF9, 0, "CH"     , "PCA Counter High Byte" },
  { prc_51,  0xFA, 0, "CCAP0H" , "PCA Module0 Capture Value - High Byte" },
  { prc_51,  0xFB, 0, "CCAP1H" , "PCA Module1 Capture Value - High Byte" },
  { prc_51,  0xFC, 0, "CCAP2H" , "PCA Module2 Capture Value - High Byte" },
  { prc_51,  0xFD, 0, "CCAP3H" , "PCA Module3 Capture Value - High Byte" },
  { prc_51,  0xFE, 0, "CCAP4H" , "PCA Module4 Capture Value - High Byte" },
  { prc_51,  0x00, 0, NULL     ,  NULL }
};

predefined_t far ibits[] =
{
  { prc_251, 0x87, 0, "IDL"    ,  "PCON.0 - Idle Mode Bit"                   },
  { prc_251, 0x87, 1, "PD"     ,  "PCON.1 - Power Down Mode Bit"             },
  { prc_251, 0x87, 2, "GF0"    ,  "PCON.2 - General Purpose Flag"            },
  { prc_251, 0x87, 3, "GF1"    ,  "PCON.3 - General Purpose Flag 1"          },
  { prc_251, 0x87, 4, "POF"    ,  "PCON.4 - Power Off Flag"                  },
  { prc_930, 0x87, 5, "LC"     ,  "PCON.5 - Low CLock Mode Enable"           },
  { prc_251, 0x87, 6, "SMOD0"  ,  "PCON.6 - SCON.7 Select"                   },
  { prc_251, 0x87, 7, "SMOD1"  ,  "PCON.7 - Double Baud Rate Bit"            },
  { prc_51,  0x88, 0, "IT0"    ,  "TCON.0 - EXT. INTERRUPT 0 TYPE"           },
  { prc_51,  0x88, 1, "IE0_"   ,  "TCON.1 - EXT. INTERRUPT 0 EDGE FLAG"      },
  { prc_51,  0x88, 2, "IT1"    ,  "TCON.2 - EXT. INTERRUPT 1 TYPE"           },
  { prc_51,  0x88, 3, "IE1"    ,  "TCON.3 - EXT. INTERRUPT 1 EDGE FLAG"      },
  { prc_51,  0x88, 4, "TR0"    ,  "TCON.4 - TIMER 0 ON/OFF CONTROL"          },
  { prc_51,  0x88, 5, "TF0"    ,  "TCON.5 - TIMER 0 OVERFLOW FLAG"           },
  { prc_51,  0x88, 6, "TR1"    ,  "TCON.6 - TIMER 1 ON/OFF CONTROL"          },
  { prc_51,  0x88, 7, "TF1"    ,  "TCON.7 - TIMER 1 OVERFLOW FLAG"           },
  { prc_251, 0x89, 0, "T0_M0"  ,  "TMOD.0 - Timer 0 Mode Select"             },
  { prc_251, 0x89, 1, "T0_M1"  ,  "TMOD.1 - Timer 0 Mode Select"             },
  { prc_251, 0x89, 2, "T0_CT"  ,  "TMOD.3 - Timer 0 Counter/Timer Select"    },
  { prc_251, 0x89, 3, "T0_GATE",  "TMOD.4 - Timer 0 Gate"                    },
  { prc_251, 0x89, 4, "T1_M0"  ,  "TMOD.0 - Timer 1 Mode Select"             },
  { prc_251, 0x89, 5, "T1_M1"  ,  "TMOD.1 - Timer 1 Mode Select"             },
  { prc_251, 0x89, 6, "T1_CT"  ,  "TMOD.3 - Timer 1 Counter/Timer Select"    },
  { prc_251, 0x89, 7, "T1_GATE",  "TMOD.4 - Timer 1 Gate"                    },
  { prc_51,  0x98, 0, "RI"     ,  "SCON.0 - RECEIVE INTERRUPT FLAG"          },
  { prc_51,  0x98, 1, "TI"     ,  "SCON.1 - TRANSMIT INTERRUPT FLAG"         },
  { prc_51,  0x98, 2, "RB8"    ,  "SCON.2 - RECEIVE BIT 8"                   },
  { prc_51,  0x98, 3, "TB8"    ,  "SCON.3 - TRANSMIT BIT 8"                  },
  { prc_51,  0x98, 4, "REN"    ,  "SCON.4 - RECEIVE ENABLE"                  },
  { prc_51,  0x98, 5, "SM2"    ,  "SCON.5 - SERIAL MODE CONTROL BIT 2"       },
  { prc_51,  0x98, 6, "SM1"    ,  "SCON.6 - SERIAL MODE CONTROL BIT 1"       },
  { prc_51,  0x98, 7, "SM0"    ,  "SCON.7 - SERIAL MODE CONTROL BIT 0"       },
  { prc_930, 0x9A, 1, "HPPWR.1",  "Port Power Control Register for USB Port 1" },
  { prc_930, 0x9A, 2, "HPPWR.2",  "Port Power Control Register for USB Port 2" },
  { prc_930, 0x9A, 3, "HPPWR.3",  "Port Power Control Register for USB Port 3" },
  { prc_930, 0x9A, 4, "HPPWR.4",  "Port Power Control Register for USB Port 4" },
  { prc_930, 0xA1, 0, "HTXE0"  ,  "HIE.0" },
  { prc_930, 0xA1, 1, "HRXE0"  ,  "HIE.1" },
  { prc_930, 0xA2, 0, "FTXIE0" ,  "FIE.0 - Function Interrupt Transmit Enable 0" },
  { prc_930, 0xA2, 1, "FRXIE0" ,  "FIE.1 - Function Interrupt Receive Enable 0" },
  { prc_930, 0xA2, 2, "FTXIE1" ,  "FIE.2 - Function Interrupt Transmit Enable 1" },
  { prc_930, 0xA2, 3, "FRXIE1" ,  "FIE.3 - Function Interrupt Receive Enable 1" },
  { prc_930, 0xA2, 4, "FTXIE2" ,  "FIE.4 - Function Interrupt Transmit Enable 2" },
  { prc_930, 0xA2, 5, "FRXIE2" ,  "FIE.5 - Function Interrupt Receive Enable 2" },
  { prc_930, 0xA2, 6, "FTXIE3" ,  "FIE.6 - Function Interrupt Transmit Enable 3" },
  { prc_930, 0xA2, 7, "FRXIE3" ,  "FIE.7 - Function Interrupt Receive Enable 3" },
  { prc_930, 0xA3, 0, "FTXIE4" ,  "FIE1.0 - Function Interrupt Transmit Enable 4" },
  { prc_930, 0xA3, 1, "FRXIE4" ,  "FIE1.1 - Function Interrupt Receive Enable 4" },
  { prc_930, 0xA3, 2, "FTXIE5" ,  "FIE1.2 - Function Interrupt Transmit Enable 5" },
  { prc_930, 0xA3, 3, "FRXIE5" ,  "FIE1.3 - Function Interrupt Receive Enable 5" },
  { prc_251, 0xA7, 0, "RTWE"   ,  "WCON.0 - RealTime WAIT Enable"            },
  { prc_251, 0xA7, 1, "RTWCE"  ,  "WCON.1 - RealTime WAIT CLOCK Enable"      },
  { prc_51,  0xA8, 0, "EX0"    ,  "IE.0 - EXTERNAL INTERRUPT 0 ENABLE"       },
  { prc_51,  0xA8, 1, "ET0"    ,  "IE.1 - TIMER 0 INTERRUPT ENABLE"          },
  { prc_51,  0xA8, 2, "EX1"    ,  "IE.2 - EXTERNAL INTERRUPT 1 ENABLE"       },
  { prc_51,  0xA8, 3, "ET1"    ,  "IE.3 - TIMER 1 INTERRUPT ENABLE"          },
  { prc_51,  0xA8, 4, "ES"     ,  "IE.4 - SERIAL PORT INTERRUPT ENABLE"      },
  { prc_51,  0xA8, 5, "ET2"    ,  "IE.5 - TIMER 2 INTERRUPT ENABLE"          },
  { prc_51,  0xA8, 6, "EC"     ,  "IE.6 - PCA INTERRUPT ENABLE"              },
  { prc_51,  0xA8, 7, "EA"     ,  "IE.7 - GLOBAL INTERRUPT ENABLE"           },
  { prc_930, 0xAE, 1, "OVI"    ,  "HSTAT.1 - Latest Overcurrent Indicator" },
  { prc_930, 0xAE, 3, "OVISC"  ,  "HSTAT.3 - Hub Overcurrent Indicator Status Change" },
  { prc_930, 0xAE, 4, "EP1EN"  ,  "HSTAT.4 - Hub End Point 1 Enable" },
  { prc_930, 0xAE, 5, "EP1STL" ,  "HSTAT.5 - Hub End Point 1 Stall Field" },
  { prc_930, 0xAE, 6, "HRWUPE" ,  "HSTAT.6 - Hub Remote Wake-up Enable Bit" },
  { prc_51,  0xB0, 0, "RXD"    ,  "P3.0 - SERIAL PORT RECEIVE INPUT"         },
  { prc_51,  0xB0, 1, "TXD"    ,  "P3.1 - SERIAL PORT TRANSMIT OUTPUT"       },
  { prc_51,  0xB0, 2, "INT0"   ,  "P3.2 - EXTERNAL INTERRUPT 0 INPUT"        },
  { prc_51,  0xB0, 3, "INT1"   ,  "P3.3 - EXTERNAL INTERRUPT 1 INPUT"        },
  { prc_51,  0xB0, 4, "T0"     ,  "P3.4 - TIMER 0 COUNT INPUT"               },
  { prc_51,  0xB0, 5, "T1"     ,  "P3.5 - TIMER 1 COUNT INPUT"               },
  { prc_51,  0xB0, 6, "WR"     ,  "P3.6 - WRITE CONTROL FOR EXT. MEMORY"     },
  { prc_51,  0xB0, 7, "RD"     ,  "P3.7 - READ CONTROL FOR EXT. MEMORY"      },
  { prc_930, 0xB1, 0, "ESOF"   ,  "IEN1.0 - Enable USB Hub/Start of Frame" },
  { prc_930, 0xB1, 1, "EF"     ,  "IEN1.1 - Enable Function" },
  { prc_930, 0xB1, 2, "ESR"    ,  "IEN1.2 - Enable Suspend/Resume" },
  { prc_51,  0xB8, 0, "PX0"    ,  "IP.0 - EXTERNAL INTERRUPT 0 PRIORITY"     },
  { prc_51,  0xB8, 1, "PT0"    ,  "IP.1 - TIMER 0 PRIORITY"                  },
  { prc_51,  0xB8, 2, "PX1"    ,  "IP.2 - EXTERNAL INTERRUPT 1 PRIORITY"     },
  { prc_51,  0xB8, 3, "PT1"    ,  "IP.3 - TIMER 1 PRIORITY"                  },
  { prc_51,  0xB8, 4, "PS"     ,  "IP.4 - SERIAL PORT PRIORITY"              },
  { prc_51,  0xB8, 5, "PT2"    ,  "IP.5 - TIMER 2 PRIORITY"                  },
  { prc_51,  0xB8, 6, "PC"     ,  "IP.6 - PCA PRIORITY"                      },
  { prc_930, 0xC0, 0, "FTXD0"  ,  "FIFLG.0 - Function Transmit Done Flag. Endpoint 0" },
  { prc_930, 0xC0, 1, "FRXD0"  ,  "FIFLG.1 - Function Receive Done Flag. Endpoint 0" },
  { prc_930, 0xC0, 2, "FTXD1"  ,  "FIFLG.2 - Function Transmit Done Flag. Endpoint 1" },
  { prc_930, 0xC0, 3, "FRXD1"  ,  "FIFLG.3 - Function Receive Done Flag. Endpoint 1" },
  { prc_930, 0xC0, 4, "FTXD2"  ,  "FIFLG.4 - Function Transmit Done Flag. Endpoint 2" },
  { prc_930, 0xC0, 5, "FRXD2"  ,  "FIFLG.5 - Function Receive Done Flag. Endpoint 2" },
  { prc_930, 0xC0, 6, "FTXD3"  ,  "FIFLG.6 - Function Transmit Done Flag. Endpoint 3" },
  { prc_930, 0xC0, 7, "FRXD3"  ,  "FIFLG.7 - Function Receive Done Flag. Endpoint 3" },
  { prc_930, 0xC1, 0, "FTXD4"  ,  "FIFLG1.0 - Function Transmit Done Flag. Endpoint 4" },
  { prc_930, 0xC1, 1, "FRXD4"  ,  "FIFLG1.1 - Function Receive Done Flag. Endpoint 4" },
  { prc_930, 0xC1, 2, "FTXD5"  ,  "FIFLG1.2 - Function Transmit Done Flag. Endpoint 5" },
  { prc_930, 0xC1, 3, "FRXD5"  ,  "FIFLG1.3 - Function Receive Done Flag. Endpoint 5" },
  { prc_930, 0xC7, 0, "SIXEPPEN", "EPCONFIG.0 - Six Endpoint Pair Enable"   },
  { prc_51,  0xC8, 0, "CPRL2"  ,  "T2CON.0 - Capture/Reload"                 },
  { prc_51,  0xC8, 1, "CT2"    ,  "T2CON.1 - Timer2 Counter/Timer"           },
  { prc_51,  0xC8, 2, "TR2"    ,  "T2CON.2 - Timer2 Run"                     },
  { prc_51,  0xC8, 3, "EXEN2"  ,  "T2CON.3 - Timer2 External Enable"         },
  { prc_51,  0xC8, 4, "TCLK"   ,  "T2CON.4 - Transmit Clock"                 },
  { prc_51,  0xC8, 5, "RCLK"   ,  "T2CON.5 - Receive Clock"                  },
  { prc_51,  0xC8, 6, "EXF2"   ,  "T2CON.6 - Timer2 External Flag"           },
  { prc_51,  0xC8, 7, "TF2"    ,  "T2CON.7 - Timer2 Overflow Flag"           },
  { prc_51,  0xC9, 0, "DCEN"   ,  "T2MOD.0 - Down Count Enable Bit"          },
  { prc_51,  0xC9, 1, "T2OE"   ,  "T2MOD.1 - Timer 2 Output Enable Bit"      },
  { prc_930, 0xCF, 0, "HPCON.0",  "Encoded Hub Port Control Commons"         },
  { prc_930, 0xCF, 1, "HPCON.1",  "Encoded Hub Port Control Commons"         },
  { prc_930, 0xCF, 2, "HPCON.2",  "Encoded Hub Port Control Commons"         },
  { prc_51,  0xD0, 0, "P"      ,  "PSW.0 - ACCUMULATOR PARITY FLAG"          },
  { prc_51,  0xD0, 1, "UD"     ,  "PSW.1 - User Defined Flag"                },
  { prc_51,  0xD0, 2, "OV"     ,  "PSW.2 - Overflow"                         },
  { prc_51,  0xD0, 3, "RS0"    ,  "PSW.3 - Register Bank Select 0"           },
  { prc_51,  0xD0, 4, "RS1"    ,  "PSW.4 - REGISTER BANK SELECT 1"           },
  { prc_51,  0xD0, 5, "F0"     ,  "PSW.5 - FLAG 0"                           },
  { prc_51,  0xD0, 6, "AC"     ,  "PSW.6 - Auxiallary Carry"                 },
  { prc_51,  0xD0, 7, "C"      ,  "PSW.7 - Carry"                            },
  { prc_251, 0xD1, 1, "Z1"     ,  "PSW1.1 - Zero"                            },
  { prc_251, 0xD1, 2, "OV1"    ,  "PSW1.2 - Overflow"                        },
  { prc_251, 0xD1, 3, "RS01"   ,  "PSW1.3 - Register Bank Select"            },
  { prc_251, 0xD1, 4, "RS11"   ,  "PSW1.4 - Register Bank Select"            },
  { prc_251, 0xD1, 5, "N1"     ,  "PSW1.5 - Negative"                        },
  { prc_251, 0xD1, 6, "AC1"    ,  "PSW1.6 - Auxillary Carry"                 },
  { prc_251, 0xD1, 7, "C1"     ,  "PSW1.7 - Carry"                           },
  { prc_930, 0xD3, 4, "FTLOCK" ,  "SOFH.4 - Frame Timer Locked"              },
  { prc_930, 0xD3, 5, "SOFIE"  ,  "SOFH.5 - SOF - Interrupt Enable"          },
  { prc_930, 0xD3, 6, "ASOF"   ,  "SOFH.6 - Any SOF"                         },
  { prc_930, 0xD3, 7, "SOFACK" ,  "SOFH.7 - SOF Token Received OK"           },
  { prc_930, 0xD4, 0, "HPIDX.0",  "Port Index Select"                        },
  { prc_930, 0xD4, 1, "HPIDX.1",  "Port Index Select"                        },
  { prc_930, 0xD4, 2, "HPIDX.2",  "Port Index Select"                        },
  { prc_930, 0xD5, 0, "PCSC"   ,  "HPSC.0 - Port Connect Status Change" },
  { prc_930, 0xD5, 1, "PESC"   ,  "HPSC.1 - Port Enable Status Change" },
  { prc_930, 0xD5, 2, "PSSC"   ,  "HPSC.2 - Port Suspend Status Change" },
  { prc_930, 0xD5, 4, "RSTSC"  ,  "HPSC.4 - Reset Status Change" },
  { prc_930, 0xD7, 0, "PCSTAT" ,  "HPSTAT.0 - Port Connect Status" },
  { prc_930, 0xD7, 1, "PESTAT" ,  "HPSTAT.1 - Port Enable Status" },
  { prc_930, 0xD7, 2, "PSSTAT" ,  "HPSTAT.2 - Port Suspend Status" },
  { prc_930, 0xD7, 3, "PRSTAT" ,  "HPSTAT.3 - Port Reset Status" },
  { prc_930, 0xD7, 4, "PPSTAT" ,  "HPSTAT.4 - Port Power Status" },
  { prc_930, 0xD7, 5, "LSSTAT" ,  "HPSTAT.5 - Low Speed Device Attach Status" },
  { prc_930, 0xD7, 6, "DMSTAT" ,  "HPSTAT.6 - Dm STATUS" },
  { prc_930, 0xD7, 7, "DPSTAT" ,  "HPSTAT.7 - Dp STATUS" },
  { prc_51,  0xD8, 0, "CCF0"   ,  "CCON.0 - Capture Flag PCA Module0"        },
  { prc_51,  0xD8, 1, "CCF1"   ,  "CCON.1 - Capture Flag PCA Module1"        },
  { prc_51,  0xD8, 2, "CCF2"   ,  "CCON.2 - Capture Flag PCA Module2"        },
  { prc_51,  0xD8, 3, "CCF3"   ,  "CCON.3 - Capture Flag PCA Module3"        },
  { prc_51,  0xD8, 4, "CCF4"   ,  "CCON.4 - Capture Flag PCA Module4"        },
  { prc_51,  0xD8, 6, "CR"     ,  "CCON.6 - PCA Timer Run Flag"              },
  { prc_51,  0xD8, 7, "CF"     ,  "CCON.7 - PCA Timer Overflow Flag"         },
  { prc_251, 0xD9, 0, "ECF"    ,  "CMOD.0 - PCA Timer/Counter Interrupt Enable" },
  { prc_251, 0xD9, 1, "CPS0"   ,  "CMOD.1 - PCA Timer/Counter Input Select"  },
  { prc_251, 0xD9, 2, "CPS1"   ,  "CMOD.2 - PCA Timer/Counter Input Select"  },
  { prc_251, 0xD9, 6, "WDTE"   ,  "CMOD.6 - Watchdog Timer Enable"           },
  { prc_251, 0xD9, 7, "CIDL"   ,  "CMOD.7 - PCA Timer/Counter Idle Control"  },
  { prc_251, 0xDA, 0, "ECCF0"  ,  "Enable CCF0 interrupt"                    },
  { prc_251, 0xDB, 0, "ECCF1"  ,  "Enable CCF1 interrupt"                    },
  { prc_251, 0xDC, 0, "ECCF2"  ,  "Enable CCF2 interrupt"                    },
  { prc_251, 0xDD, 0, "ECCF3"  ,  "Enable CCF3 interrupt"                    },
  { prc_251, 0xDE, 0, "ECCF4"  ,  "Enable CCF4 interrupt"                    },
  { prc_251, 0xDA, 1, "BWM0"   ,  "Pulse with modulation mode"               },
  { prc_251, 0xDB, 1, "BWM1"   ,  "Pulse with modulation mode"               },
  { prc_251, 0xDC, 1, "BWM2"   ,  "Pulse with modulation mode"               },
  { prc_251, 0xDD, 1, "BWM3"   ,  "Pulse with modulation mode"               },
  { prc_251, 0xDE, 1, "BWM4"   ,  "Pulse with modulation mode"               },
  { prc_251, 0xDA, 2, "TOG0"   ,  "Toggle"                                   },
  { prc_251, 0xDB, 2, "TOG1"   ,  "Toggle"                                   },
  { prc_251, 0xDC, 2, "TOG2"   ,  "Toggle"                                   },
  { prc_251, 0xDD, 2, "TOG3"   ,  "Toggle"                                   },
  { prc_251, 0xDE, 2, "TOG4"   ,  "Toggle"                                   },
  { prc_251, 0xDA, 3, "MAT0"   ,  "Match"                                    },
  { prc_251, 0xDB, 3, "MAT1"   ,  "Match"                                    },
  { prc_251, 0xDC, 3, "MAT2"   ,  "Match"                                    },
  { prc_251, 0xDD, 3, "MAT3"   ,  "Match"                                    },
  { prc_251, 0xDE, 3, "MAT4"   ,  "Match"                                    },
  { prc_251, 0xDA, 4, "CAPN0"  ,  "Capture mode (negative)"                  },
  { prc_251, 0xDB, 4, "CAPN1"  ,  "Capture mode (negative)"                  },
  { prc_251, 0xDC, 4, "CAPN2"  ,  "Capture mode (negative)"                  },
  { prc_251, 0xDD, 4, "CAPN3"  ,  "Capture mode (negative)"                  },
  { prc_251, 0xDE, 4, "CAPN4"  ,  "Capture mode (negative)"                  },
  { prc_251, 0xDA, 5, "CAPP0"  ,  "Capture mode (positive)"                  },
  { prc_251, 0xDB, 5, "CAPP1"  ,  "Capture mode (positive)"                  },
  { prc_251, 0xDC, 5, "CAPP2"  ,  "Capture mode (positive)"                  },
  { prc_251, 0xDD, 5, "CAPP3"  ,  "Capture mode (positive)"                  },
  { prc_251, 0xDE, 5, "CAPP4"  ,  "Capture mode (positive)"                  },
  { prc_251, 0xDA, 6, "ECOM0"  ,  "Compare modes"                            },
  { prc_251, 0xDB, 6, "ECOM1"  ,  "Compare modes"                            },
  { prc_251, 0xDC, 6, "ECOM2"  ,  "Compare modes"                            },
  { prc_251, 0xDD, 6, "ECOM3"  ,  "Compare modes"                            },
  { prc_251, 0xDE, 6, "ECOM4"  ,  "Compare modes"                            },
  { prc_930, 0xDF, 0, "GSUS"   ,  "PCON1.0 - Global Suspend Bit"             },
  { prc_930, 0xDF, 1, "GRSM"   ,  "PCON1.1 - Global Resume Bit"              },
  { prc_930, 0xDF, 2, "RWU"    ,  "PCON1.2 - Remote Wake Up Bit"             },
  { prc_930, 0xDF, 3, "URST"   ,  "PCON1.3 - USB Reset Flag"                 },
  { prc_930, 0xDF, 4, "URDIS"  ,  "PCON1.4 - USB Reset Disable"              },
  { prc_930, 0xE1, 0, "TXEPEN" ,  "EPCON.0 - Transmit Endpoint Enable"       },
  { prc_930, 0xE1, 1, "TXOE"   ,  "EPCON.1 - Transmit Output Enable"         },
  { prc_930, 0xE1, 2, "RXEPEN" ,  "EPCON.2 - Receive Endpoint Enable"        },
  { prc_930, 0xE1, 3, "RXOE"   ,  "EPCON.3 - Receive Input Enable"           },
  { prc_930, 0xE1, 4, "RXSPM"  ,  "EPCON.4 - Receive Single Packet Mode"     },
  { prc_930, 0xE1, 5, "STLEP"  ,  "EPCON.5 - Control Endpoint"               },
  { prc_930, 0xE1, 6, "TXSTL"  ,  "EPCON.6 - Stall Transmit Endpoint"        },
  { prc_930, 0xE1, 7, "RXSTL"  ,  "EPCON.7 - Stall Receive Endpoint"         },
  { prc_930, 0xE2, 0, "RXACK"  ,  "RXSTAT.0 - Receive Acknowledged"          },
  { prc_930, 0xE2, 1, "RXERR"  ,  "RXSTAT.1 - Receive Error"                 },
  { prc_930, 0xE2, 2, "RXVOID" ,  "RXSTAT.2 - Receive Void Condition"        },
  { prc_930, 0xE2, 3, "RXSOVW" ,  "RXSTAT.3 - Receive Data Sequence OverWrite Bit" },
  { prc_930, 0xE2, 4, "EDOVW"  ,  "RXSTAT.4 - End OverWrite Flag"            },
  { prc_930, 0xE2, 5, "STOVW"  ,  "RXSTAT.5 - Start OverWrite Flag"          },
  { prc_930, 0xE2, 6, "RXSETUP",  "RXSTAT.6 - Received Setup Token"          },
  { prc_930, 0xE2, 7, "RXSEQ"  ,  "RXSTAT.7 - Receiver Endpoint Sequence Bit" },
  { prc_930, 0xE4, 0, "REVWP"  ,  "RXCON.0 - Reverse Write Pointer"          },
  { prc_930, 0xE4, 1, "ADVWM"  ,  "RXCON.1 - Advance Write Marker"           },
  { prc_930, 0xE4, 2, "ARM"    ,  "RXCON.2 - Auto Receive Management"        },
  { prc_930, 0xE4, 3, "RXISO"  ,  "RXCON.3 - Receive Isochoronous Data Type" },
  { prc_930, 0xE4, 4, "RXFFRC" ,  "RXCON.4 - FIFO Read Complete"             },
  { prc_930, 0xE4, 5, "RXWS"   ,  "RXCON.5 - Receive FIFO Wait-state Read"   },
  { prc_930, 0xE4, 7, "RXCLR"  ,  "RXCON.7 - Clear Receive FIFO"             },
  { prc_930, 0xE5, 0, "RXOVF"  ,  "RXFLG.0 - Receive FIFO Overrun Flag"      },
  { prc_930, 0xE5, 1, "RXURF"  ,  "RXFLG.1 - Receive FIFO Underrun Flag"     },
  { prc_930, 0xE5, 2, "RXFULL" ,  "RXFLG.2 - Receive FIFO Full Flag"         },
  { prc_930, 0xE5, 3, "RXEMP"  ,  "RXFLG.3 - Receive FIFO Empty Flag"        },
  { prc_930, 0xE5, 6, "RXFIF0" ,  "RXFLG.6 - Receive FIFO Index Flag"        },
  { prc_930, 0xE5, 7, "RXFIF1" ,  "RXFLG.7 - Receive FIFO Index Flag"        },
  { prc_930, 0xE8, 0, "HTXD0"  ,  "HIFLG.0 - Hub Transmit Done. Endpoint 0"  },
  { prc_930, 0xE8, 1, "HRXD0"  ,  "HIFLG.1 - Hub Receive Done. Endpoint 0"   },
  { prc_930, 0xF1, 0, "ENINX0" ,  "EPINDIX.0 - Endpoint Index"               },
  { prc_930, 0xF1, 1, "ENINX1" ,  "EPINDIX.1 - Endpoint Index"               },
  { prc_930, 0xF1, 2, "ENINX2" ,  "EPINDIX.2 - Endpoint Index"               },
  { prc_930, 0xF1, 7, "HORF"   ,  "EPINDIX.7 - Hub/Function Bit"             },
  { prc_930, 0xF3, 0, "TXDAT.0",  "TXDAT.0 - Hub Status Change"              },
  { prc_930, 0xF3, 1, "TXDAT.1",  "TXDAT.1 - Port 1 Status Change"           },
  { prc_930, 0xF3, 2, "TXDAT.2",  "TXDAT.2 - Port 2 Status Change"           },
  { prc_930, 0xF3, 3, "TXDAT.3",  "TXDAT.3 - Port 3 Status Change"           },
  { prc_930, 0xF3, 4, "TXDAT.4",  "TXDAT.4 - Port 4 Status Change"           },
  { prc_930, 0xF4, 0, "REVRP"  ,  "RXCON.0 - Reverse Read Pointer"           },
  { prc_930, 0xF4, 1, "ADVRM"  ,  "RXCON.1 - Advance Read Marker"            },
  { prc_930, 0xF4, 2, "ATM"    ,  "RXCON.2 - Auto Transmit Management"       },
  { prc_930, 0xF4, 3, "TXISO"  ,  "RXCON.3 - Transmit Isochoronous Data Type"},
  { prc_930, 0xF4, 5, "FFSZ0"  ,  "RXCON.5 - FIFO Size"                      },
  { prc_930, 0xF4, 6, "FFSZ1"  ,  "RXCON.6 - FIFO Size"                      },
  { prc_930, 0xF4, 7, "TXCLR"  ,  "RXCON.7 - Transmit Clear"                 },
  { prc_930, 0xF5, 0, "TXOVF"  ,  "TXFLG.0 - Transmit FIFO Overrun Flag"     },
  { prc_930, 0xF5, 1, "TXURF"  ,  "TXFLG.1 - Transmit FIFO Underrun Flag"    },
  { prc_930, 0xF5, 2, "TXFULL" ,  "TXFLG.2 - Transmit FIFO Full Flag"        },
  { prc_930, 0xF5, 3, "TXEMP"  ,  "TXFLG.3 - Transmit FIFO Empty Flag"       },
  { prc_930, 0xF5, 6, "TXFIF0" ,  "TXFLG.6 - Transmit FIFO Index Flag"       },
  { prc_930, 0xF5, 7, "TXFIF1" ,  "TXFLG.7 - Transmit FIFO Index Flag"       },
  { prc_930, 0xF2, 0, "TXACK"  ,  "TXSTAT.0 - Transmit Acknowledged"         },
  { prc_930, 0xF2, 1, "TXERR"  ,  "TXSTAT.1 - Transmit Error"                },
  { prc_930, 0xF2, 2, "TXVOID" ,  "TXSTAT.2 - Transmit Void Condition"       },
  { prc_930, 0xF2, 3, "TXSOVW" ,  "TXSTAT.3 - Transmit Data Sequence OverWrite Bit" },
  { prc_930, 0xF2, 4, "TXFLUSH",  "TXSTAT.4 - Transmit FIFO Packet FLushed"  },
  { prc_930, 0xF2, 7, "TXSEQ"  ,  "TXSTAT.7 - Transmit Endpoint Sequence Bit" },
  { prc_51,  0x00, 0, NULL     ,  NULL }
};

//----------------------------------------------------------------------
int IsPredefined(const char *name)
{
  predefined_t *ptr;

  for ( ptr = ibits; ptr->name != NULL; ptr++ )
    if ( strcmp(ptr->name,name) == 0 ) return 1;

  for ( ptr = iregs; ptr->name != NULL; ptr++ )
    if ( strcmp(ptr->name,name) == 0 ) return 1;

  return 0;
}

//----------------------------------------------------------------------
predefined_t *GetPredefined(predefined_t *ptr,int addr,int bit)
{
  for ( ; ptr->name != NULL; ptr++ )
  {
    if ( ptr->proc > ptype ) continue;
    if ( addr == ptr->addr && bit == ptr->bit )
      return ptr;
  }
  return NULL;
}

typedef struct
{
  char proc;
  char off;
  char *name;
  char *cmt;
} entry_t;

static entry_t entries[] =
{
  { prc_51,  0x03, "extint0", "External interrupt 0 (INT0 / EX0)" },
  { prc_51,  0x0B, "timint0", "Timer interrupt 0 (TIM0)" },
  { prc_51,  0x13, "extint1", "External interrupt 1 (INT1 / EX1)" },
  { prc_51,  0x1B, "timint1", "Timer interrupt 1 (TIM1)" },
  { prc_51,  0x23, "serint",  "Serial port interrupt (SERIAL)" },
  { prc_51,  0x2B, "timint2", "Timer interrupt 2 (TIM2) (52 or higher)" },
  { prc_51,  0x33, "pcaint",  "PCA (programmable counter array) interrupt\n(only 51f or higher)" },
  { prc_930, 0x43, "usbhub",  "USB Hub/SOF (isochronous end point) (only 930)" },
  { prc_930, 0x4B, "usbfun",  "USB Function (non-isochronous end point) (only 930)" },
  { prc_930, 0x53, "usbglb",  "USB Global Suspend/Resume and USB Reset (only 930)" },
  { prc_251, 0x7B, "trapint", "TRAP (program interrupt) (only 251 or 930)" }
};

//----------------------------------------------------------------------
// Get linear address of a special segment
//      sel - selector of the segment
static long specialSeg(ushort sel)
{
  segment_t *s = get_segm_by_sel(sel);
  if ( s != NULL )
  {
    if ( s->type != SEG_IMEM )          // is the segment type correct? - no
    {
      s->type = SEG_IMEM;               // fix it
      s->update();
    }
    return s->startEA;
  }
  return BADADDR;
}

//----------------------------------------------------------------------
static ea_t AdditionalSegment(int size,int offset,char *name)
{
  segment_t s;
  s.startEA = (ptype > prc_51)
                   ? (inf.maxEA + 0xF) & ~0xF
                   : freechunk(0, size, 0xF);
  s.endEA   = s.startEA + size;
  s.sel     = allocate_selector((s.startEA-offset) >> 4);
  s.type    = SEG_IMEM;                         // internal memory
  add_segm(&s, name, NULL, ADDSEG_NOSREG|ADDSEG_OR_DIE);
  return s.startEA - offset;
}

//----------------------------------------------------------------------
// The kernel event notifications
// Here you may take desired actions upon some kernel events

static int notify(processor_t::idp_notify msgid, ...)
{
  static int first_time = 1;
  va_list va;
  va_start(va, msgid);

  switch(msgid)
  {
    case ph.init:
      inf.mf = 1;       // Set a big endian mode of the IDA kernel
      break;

    case ph.newfile:
      {
        segment_t *sptr = getnseg(0);
        if ( sptr != NULL )
        {
          if ( sptr->startEA-get_segm_base(sptr) == 0 )
          {
            inf.beginEA = sptr->startEA;
            inf.startIP = 0;
            for ( int i=0; i < qnumber(entries); i++ )
            {
              if ( entries[i].proc > ptype ) continue;
              ea_t ea = inf.beginEA+entries[i].off;
              if ( isEnabled(ea) && get_byte(ea) != 0xFF )
              {
                add_entry(ea, ea, entries[i].name, 1);
                set_cmt(ea, entries[i].cmt, 1);
              }
            }
          }
        }
        segment_t *scode = getnseg(0);
        set_segm_class(scode, "CODE");

        intmem = AdditionalSegment(256, 0, "INTMEM");
        sfrmem = AdditionalSegment(128, 128, "SFR");
        if ( ptype > prc_51 )
        {
          AdditionalSegment(0x10000-256-128, 256+128, "RAM");
          if ( scode != NULL )
          {
            ea_t align = (scode->endEA + 0xFFF) & ~0xFFF;
            if ( getseg(align-7) == scode )     // the code segment size is
            {                                   // multiple of 4K or near it
              uchar b0 = get_byte(align-8);
              // 251:
              //  0  : 1-source, 0-binary mode
              //  6,7: must be 1s
              // 82930:
              //  0  : 1-source, 0-binary mode
              //  7  : must be 1s
              uchar b1 = get_byte(align-7);
              // 251
              //  0: eprommap 0 - FE2000..FE4000 is mapped into 00E000..100000
              //              1 - .............. is not mapped ...............
              //  1: must be 1
              //  3:
              //  2: must be 1
              //  4: intr 1 - upon interrupt PC,PSW are pushed into stack
              //          0 - upon interrupt only PC is pushed into stack
              //  5: must be 1
              //  6: must be 1
              //  7: must be 1
              // 82930:
              //  3: must be 1
              //  5: must be 1
              //  6: must be 1
              //  7: must be 1
                msg("b0=%x b1=%x\n", b0, b1);
//              if ( (b0 & 0x80) == 0x80 && (b1 & 0xEA) == 0xEA )
              {                         // the init bits are correct
                char pname[sizeof(inf.procName)+1];
                inf.get_proc_name(pname);
                char ntype = (b0 & 1) ? 's' : 'b';
                char *ptr = tail(pname)-1;
                if ( ntype != *ptr
                  && askyn_c(1,
                       "The input file seems to be for the %s mode of the processor. "
                       "Do you want to change the current processor type?",
                       ntype == 's' ? "source" : "binary") > 0
                   )
                {
                  *ptr = ntype;
                  first_time = 1;
                  set_processor_type(pname, SETPROC_COMPAT);
                }
              }
            }
          }
        }

        // the default data segment will be INTMEM
        set_default_dataseg(getseg(intmem)->sel);


        predefined_t *ptr;
        for ( ptr=iregs; ptr->name != NULL; ptr++ )
        {
          ea_t ea = sfrmem + ptr->addr;
          ea_t oldea = get_name_ea(BADADDR, ptr->name);
          if ( oldea != ea )
          {
            if ( oldea != BADADDR ) del_global_name(oldea);
            doUnknown(ea, 1);
            set_name(ea, ptr->name, SN_NOLIST);
          }
          if ( ptr->cmt != NULL ) set_cmt(ea,ptr->cmt, 1);
        }
      }
      break;

    case ph.oldfile:
      sel_t sel;
      if ( atos("INTMEM",&sel) ) intmem = specialSeg(sel);
      if ( atos("SFR",&sel) ) sfrmem = specialSeg(sel) - 0x80;
      break;

    case ph.newseg:
        // make the default DS point to INTMEM
        // (8051 specific issue)
      {
        segment_t *newseg = va_arg(va, segment_t *);
        segment_t *intseg = getseg(intmem);
        if ( intseg != NULL )
          newseg->defsr[rVds-ph.regFirstSreg] = intseg->sel;
      }
      break;

    case ph.newprc:
      {
        processor_subtype_t prcnum = processor_subtype_t(va_arg(va, int));
        if ( !first_time && prcnum != ptype )
        {
          warning("Sorry, it is not possible to change" // (this is 8051 specific)
                  " the processor mode on the fly."
                  " Please reload the input file"
                  " if you want to change the processor.");
          return 0;
        }
        first_time = 0;
        ptype = prcnum;
      }
      break;

    default:
      break;
  }
  va_end(va);

  return(1);
}

//-----------------------------------------------------------------------
//      Checkarg data. Common for all assemblers. Not good.
//
//      What is a checkarg?
//        It is a possibilty to compare the value of a manually entered
//        operand against its original value.
//        Checkarg is currently implemented for IBM PC, 8051, and PDP-11
//        processors. Other processor are unlikely to be supported.
//      You may just get rid of checkarg and replace the pointers to it
//      in the 'LPH' structure by NULLs.
//
//-----------------------------------------------------------------------
static const char *operdim[15] = {  //    15
     "(", ")", "!", "-", "+", "%",
     "\\", "/", "*", "&", "|", "^", "<<", ">>", NULL};

inline int pere(int c) { return(c&0xFF); }

static int preline(char *argstr, s_preline *S)
{
    char    *pc;
    int     *ind;
    char    *prefix;
    char    *seg;
    char    *reg;
    char    *offset;

    ind     = S->ind;
    prefix  = S->prefix;
    seg     = S->seg;
    reg     = S->reg;
    offset  = S->offset;
    *ind = 0;
    *prefix = '\0';
    *reg = '\0';
    *seg = '\0';
    *offset = '\0';

    pc = argstr;
    if (*pc == '#') ++*ind, pc++;
    stpcpy(offset, pc);

    return(0);

} /* preline */

//
//              Definitions of the target assemblers
//              8051 has unusually many of them.
//

//-----------------------------------------------------------------------
//                   ASMI
//-----------------------------------------------------------------------
static asm_t asmi = {
  AS_COLON | ASH_HEXF3 | AS_1TEXT | AS_NCHRE | ASO_OCTF1 | AS_RELSUP,
  UAS_PSAM | UAS_NOSEG | UAS_AUBIT | UAS_PBIT | UAS_PBYTNODEF | UAS_NOENS,
  "ASMI",
  0,
  NULL,         // no headers
  NULL,         // no bad instructions
  ".equ $, ",
  ".end",

  ";",          // comment string
  '"',          // string delimiter
  '\'',         // char delimiter
  "\\\"'",      // special symbols in char and string constants

  ".text",      // ascii string directive
  ".byte",      // byte directive
  ".word",      // word directive
  NULL,         // dword  (4 bytes)
  NULL,         // qword  (8 bytes)
  NULL,         // oword  (16 bytes)
  NULL,         // float  (4 bytes)
  NULL,         // double (8 bytes)
  NULL,         // tbyte  (10/12 bytes)
  NULL,         // packed decimal real
  NULL,         // arrays (#h,#d,#v,#s(...)
  ".byte 0xFF;(array %s)", // uninited arrays
  ".equ",       // equ
  NULL,         // seg prefix
  preline, NULL, operdim,
  NULL,
  "$",
  NULL,		// func_header
  NULL,		// func_footer
  NULL,		// public
  NULL,		// weak
  NULL,		// extrn
  NULL,		// comm
  NULL,		// get_type_name
  NULL,		// align
  '(', ')',	// lbrace, rbrace
  "%",    // mod
  "&",    // and
  "|",    // or
  "^",    // xor
  "!",    // not
  "<<",   // shl
  ">>",   // shr
  NULL,   // sizeof
};

//-----------------------------------------------------------------------
//                   8051 Macro Assembler   -   Version 4.02a
//                Copyright (C) 1985 by 2500 A.D. Software, Inc.
//-----------------------------------------------------------------------
static asm_t adasm = {
  AS_COLON | ASH_HEXF0 ,
  UAS_PBIT | UAS_SECT,
  "8051 Macro Assembler by 2500 A.D. Software",
  0,
  NULL,         // no headers
  NULL,         // no bad instructions
  "org",
  "end",

  ";",          // comment string
  '"',          // string delimiter
  '\'',         // char delimiter
  "\\\"'",      // special symbols in char and string constants

  "db",         // ascii string directive
  "db",         // byte directive
  "dw",         // word directive
  "long",       // dword  (4 bytes)
  NULL,         // qword  (8 bytes)
  NULL,         // oword  (16 bytes)
  NULL,         // float  (4 bytes)
  NULL,         // double (8 bytes)
  NULL,         // tbyte  (10/12 bytes)
  NULL,         // packed decimal real
  NULL,         // arrays (#h,#d,#v,#s(...)
  "ds %s",      // uninited arrays
  "reg",        // equ
  NULL,         // seg prefix
  preline, NULL, operdim,
  NULL,
  "$",
  NULL,		// func_header
  NULL,		// func_footer
  NULL,		// public
  NULL,		// weak
  NULL,		// extrn
  NULL,		// comm
  NULL,		// get_type_name
  NULL,		// align
  '(', ')',	// lbrace, rbrace
  NULL,    // mod
  NULL,    // and
  NULL,    // or
  NULL,    // xor
  NULL,    // not
  NULL,    // shl
  NULL,    // shr
  NULL,    // sizeof
};

//-----------------------------------------------------------------------
//      PseudoSam
//-----------------------------------------------------------------------
static const char *ps_headers[] = {
".code",
NULL };

static asm_t pseudosam = {
  AS_COLON | ASH_HEXF1 | AS_N2CHR,
  UAS_PBIT | UAS_PSAM | UAS_SELSG,
  "PseudoSam by PseudoCode",
  0,
  ps_headers,
  NULL,
  ".org",
  ".end",

  ";",          // comment string
  '"',          // string delimiter
  '\'',         // char delimiter
  "\\\"'",      // special symbols in char and string constants

  ".db",        // ascii string directive
  ".db",        // byte directive
  ".dw",        // word directive
  NULL,         // dword  (4 bytes)
  NULL,         // qword  (8 bytes)
  NULL,         // oword  (16 bytes)
  NULL,         // float  (4 bytes)
  NULL,         // double (8 bytes)
  NULL,         // tbyte  (10/12 bytes)
  NULL,         // packed decimal real
  NULL,         // arrays (#h,#d,#v,#s(...)
  ".rs %s",     // uninited arrays
  ".equ",       // equ
  NULL,         // seg prefix
  preline, NULL, operdim,
  NULL,
  "$",
  NULL,		// func_header
  NULL,		// func_footer
  NULL,		// public
  NULL,		// weak
  NULL,		// extrn
  NULL,		// comm
  NULL,		// get_type_name
  NULL,		// align
  '(', ')',	// lbrace, rbrace
  NULL,    // mod
  NULL,    // and
  NULL,    // or
  NULL,    // xor
  NULL,    // not
  NULL,    // shl
  NULL,    // shr
  NULL,    // sizeof
};

//-----------------------------------------------------------------------
//      Cross-16 assembler definiton
//-----------------------------------------------------------------------
static const char *cross16_headers[] = {
"cpu \"8051.tbl\"",
NULL };

static asm_t cross16 = {
  AS_COLON | ASH_HEXF0 | AS_NHIAS,
  UAS_PBIT | UAS_NOSEG | UAS_NOBIT | UAS_EQCLN,
  "Cross-16 by Universal Cross-Assemblers",
  0,
  cross16_headers,
  NULL,
  "org",
  "end",

  ";",          // comment string
  '"',          // string delimiter
  '\0',         // char delimiter (no char consts)
  "\\\"'",      // special symbols in char and string constants

  "dfb",        // ascii string directive
  "dfb",        // byte directive
  "dwm",        // word directive
  NULL,         // dword  (4 bytes)
  NULL,         // qword  (8 bytes)
  NULL,         // oword  (16 bytes)
  NULL,         // float  (4 bytes)
  NULL,         // double (8 bytes)
  NULL,         // tbyte  (10/12 bytes)
  NULL,         // packed decimal real
  NULL,         // arrays (#h,#d,#v,#s(...)
  "dfs %s",     // uninited arrays
  "equ",        // Equ
  NULL,         // seg prefix
  preline, NULL, operdim,
  NULL,
  "$",
  NULL,		// func_header
  NULL,		// func_footer
  NULL,		// public
  NULL,		// weak
  NULL,		// extrn
  NULL,		// comm
  NULL,		// get_type_name
  NULL,		// align
  '(', ')',	// lbrace, rbrace
  NULL,    // mod
  NULL,    // and
  NULL,    // or
  NULL,    // xor
  NULL,    // not
  NULL,    // shl
  NULL,    // shr
  NULL,    // sizeof
};

//-----------------------------------------------------------------------
//      8051 Cross-Assembler by MetaLink Corporation
//-----------------------------------------------------------------------
static asm_t mcross = {
  AS_COLON | ASH_HEXF0 | AS_NHIAS,
  UAS_NOSEG | UAS_CDSEG | UAS_AUBIT | UAS_NODS | UAS_NOENS,
  "8051 Cross-Assembler by MetaLink Corporation",
  0,
  NULL,
  NULL,
  "org",
  "end",

  ";",          // comment string
  '\'',         // string delimiter
  '\0',         // char delimiter (no char consts)
  "\\\"'",      // special symbols in char and string constants

  "db",         // ascii string directive
  "db",         // byte directive
  "dw",         // word directive
  NULL,         // dword  (4 bytes)
  NULL,         // qword  (8 bytes)
  NULL,         // oword  (16 bytes)
  NULL,         // float  (4 bytes)
  NULL,         // double (8 bytes)
  NULL,         // tbyte  (10/12 bytes)
  NULL,         // packed decimal real
  NULL,         // arrays (#h,#d,#v,#s(...)
  "ds %s",      // uninited arrays
  "equ",        // Equ
  NULL,         // seg prefix
  preline, NULL, operdim,
  NULL,
  "$",
  NULL,		// func_header
  NULL,		// func_footer
  NULL,		// public
  NULL,		// weak
  NULL,		// extrn
  NULL,		// comm
  NULL,		// get_type_name
  NULL,		// align
  '(', ')',	// lbrace, rbrace
  NULL,    // mod
  NULL,    // and
  NULL,    // or
  NULL,    // xor
  NULL,    // not
  NULL,    // shl
  NULL,    // shr
  NULL,    // sizeof
};

//-----------------------------------------------------------------------
//      TASM assembler definiton
//-----------------------------------------------------------------------
static const char *tasm_headers[] = {
".msfirst",
NULL };

static asm_t tasm = {
  AS_COLON | AS_N2CHR | AS_1TEXT,
  UAS_PBIT | UAS_NOENS | UAS_EQCLN | UAS_NOSEG,
  "Table Driven Assembler (TASM) by Speech Technology Inc.",
  0,
  tasm_headers,
  NULL,
  ".org",
  ".end",

  ";",          // comment string
  '"',          // string delimiter
  '\'',         // char delimiter
  "\\\"'",      // special symbols in char and string constants

  ".text",      // ascii string directive
  ".db",        // byte directive
  ".dw",        // word directive
  NULL,         // dword  (4 bytes)
  NULL,         // qword  (8 bytes)
  NULL,         // oword  (16 bytes)
  NULL,         // float  (4 bytes)
  NULL,         // double (8 bytes)
  NULL,         // tbyte  (10/12 bytes)
  NULL,         // packed decimal real
  NULL,         // arrays (#h,#d,#v,#s(...)
  ".block %s",  // uninited arrays
  ".equ",
  NULL,         // seg prefix
  preline, NULL, operdim,
  NULL,
  "$",
  NULL,		// func_header
  NULL,		// func_footer
  NULL,		// public
  NULL,		// weak
  NULL,		// extrn
  NULL,		// comm
  NULL,		// get_type_name
  NULL,		// align
  '(', ')',	// lbrace, rbrace
  NULL,    // mod
  "and",   // and
  "or",    // or
  NULL,    // xor
  "not",   // not
  NULL,    // shl
  NULL,    // shr
  NULL,    // sizeof
};

static asm_t *asms[] = { &asmi, &adasm, &pseudosam, &cross16, &mcross, &tasm, NULL };
//-----------------------------------------------------------------------
// The short and long names of the supported processors
// The short names must match
// the names in the module DESCIPTION in the makefile

static char *shnames[] =
{
  "8051",
  "80251b",
  "80251s",
  "80930b",
  "80930s",
  NULL
};

static char *lnames[] =
{
  "Intel 8051",
  "Intel 80251 in binary mode",
  "Intel 80251 in source mode",
  "Intel 80930 in binary mode",
  "Intel 80930 in source mode",
  NULL
};

//--------------------------------------------------------------------------
// Opcodes of "return" instructions. This information will be used in 2 ways:
//      - if an instruction has the "return" opcode, its autogenerated label
//        will be "locret" rather than "loc".
//      - IDA will use the first "return" opcode to create empty subroutines.

static uchar retcode_1[] = { 0x22 };
static uchar retcode_2[] = { 0x32 };

static bytes_t retcodes[] = {
 { sizeof(retcode_1), retcode_1 },
 { sizeof(retcode_2), retcode_2 },
 { 0, NULL }                            // NULL terminated array
};

//-----------------------------------------------------------------------
//      Processor Definition
//-----------------------------------------------------------------------
extern "C" processor_t LPH = {
  IDP_INTERFACE_VERSION, // version
  PLFM_8051,             // id
  PR_RNAMESOK,           // can use register names for byte names
  8,                            // 8 bits in a byte for code segments
  8,                            // 8 bits in a byte for other segments

  shnames,              // array of short processor names
                        // the short names are used to specify the processor
                        // with the -p command line switch)
  lnames,               // array of long processor names
                        // the long names are used to build the processor type
                        // selection menu

  asms,                 // array of target assemblers

  notify,               // the kernel event notification callback

  header,               // generate the disassembly header
  footer,               // generate the disassembly footer

  segstart,             // generate a segment declaration (start of segment)
  std_gen_segm_footer,  // generate a segment footer (end of segment)

  NULL,                 // generate 'assume' directives

  ana,                  // analyse an instruction and fill the 'cmd' structure
  emu,                  // emulate an instruction

  out,                  // generate a text representation of an instruction
  outop,                // generate a text representation of an operand
  i51_data,             // generate a text representation of a data item
  NULL,                 // compare operands
  NULL,                 // can an operand have a type?

  qnumber(RegNames),    // Number of registers
  RegNames,             // Regsiter names
  NULL,                 // get abstract register

  0,                    // Number of register files
  NULL,                 // Register file names
  NULL,                 // Register descriptions
  NULL,                 // Pointer to CPU registers

  rVcs,rVds,
  0,                    // size of a segment register
  rVcs,rVds,

  NULL,                 // No known code start sequences
  retcodes,

  0,I51_last,
  Instructions
};
