/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-97 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su
 *                              FIDO:   2:5020/209
 *
 */

#ifndef _PRO_H
#define _PRO_H

//
//      This is the first header included in IDA project.
//      It defines the most common types, functions and data.
//      Also, it tries to make system dependent definitions.
//

#include <stdlib.h>     /* size_t, NULL, memory */
#include <fcntl.h>      /* O_... */
#include <stdarg.h>
#include <stddef.h>
#include <assert.h>
#include <ctype.h>
#if defined(__BORLANDC__)
#  include <io.h>         /* open, ... */
#  include <dir.h>        /* mkdir */
#  ifdef __NT__
#    include <alloc.h>
#  endif
#  include <new.h>
#define WIN32_LEAN_AND_MEAN
#else
#if defined(__WATCOMC__) || defined(_MSC_VER)
#  include <io.h>         /* open, ... */
#  include <direct.h>     /* mkdir */
#  include <limits.h>
#else
#  include <unistd.h>     /* open, ... */
#  include <sys/stat.h>   /* mkdir */
#endif
#endif
#include <sys/types.h>  /* uchar, ... */
#include <sys/stat.h>

#if defined(__WATCOMC__) || !defined(__cplusplus)
#include <string.h>
#else
#define STL_SUPPORT_PRESENT
#include <string>
using std::string;
#endif

#pragma pack(push, 1)
/*==================================================*/
#ifdef __cplusplus
#define C_INCLUDE       extern "C" {
#define C_INCLUDE_END   }
#else
#define C_INCLUDE
#define C_INCLUDE_END
#endif

/*==================================================*/
#if !defined(__OS2__) && !defined(__MSDOS__) && !defined(__NT__) && !defined(__LINUX__)
#error "Please define one of: __NT__, __OS2__, __MSDOS__, __LINUX__"
#endif

#if defined(__LINUX__) && defined(__BORLANDC__)
#define __KYLIX__
#endif

/*==================================================*/
#ifndef MAXSTR
#define MAXSTR 1024
#endif

// Some NT functions require __cdecl calling convention
#ifdef __NT__
#define NT_CDECL __cdecl
#else
#define NT_CDECL
#endif

/*==================================================*/

#define __MF__  0               // Byte sex of our platform
                                // (Most significant byte First)
                                // 0 - little endian (Intel 80x86)
                                // 1 - big endian (PowerPC)

/*==================================================*/
/* Macro to avoid of message 'Parameter ... is never used' */
#if defined(__BORLANDC__) && !defined(__NT__) || defined(__WATCOMC__)
#define qnotused(x)     (x=x)
#elif defined(__BORLANDC__)
#define qnotused(x)     &x
#else
#define qnotused(x)
#endif

// GNU C complains about some data types in va_arg because they are promoted to int
// and proposes to replace them by int.
#ifdef __GNUC__
#define va_argi(va, type)  (type)va_arg(va, int)
#else
#define va_argi(va, type)  va_arg(va, type)
#endif

/*==================================================*/
#ifdef __WATCOMC__
#define CONST_CAST(x)   (x)
#else
#define CONST_CAST(x)   const_cast<x>
#endif

/*==================================================*/
#if defined(__IDP__) && defined(__NT__) // for modules
#define idaapi          __stdcall
#define idaman          extern "C"
#define ida_export      idaapi
#define ida_export_data __declspec(dllimport)
#elif defined(__NT__)                   // for the kernel
#define idaapi          __stdcall
#define idaman          extern "C"
#define ida_export      idaapi
#define ida_export_data
#elif defined(__LINUX__)                // for linux
#define idaapi
#define idaman          extern "C"
#define ida_export      idaapi
#define ida_export_data
#else                                   // for watcom
#define idaapi
#define idaman          extern
#define ida_export
#define ida_export_data
#endif

/*==================================================*/
#if (defined(__WATCOMC__) && (__WATCOMC__ < 1100)) || defined(__DOS16__)

typedef int bool;
const bool false = 0;
const bool true = 1;

#endif

/*==================================================*/
/* uchar, ... */
/*--------------------------------------------------*/
// Linux C mode compiler already has these types defined
#if !defined(__LINUX__) || defined(__cplusplus)
typedef unsigned char  uchar;
typedef unsigned short ushort;
#if defined(__KYLIX__) // Borland Kylix has uint
using Qt::uint;
#else
typedef unsigned int   uint;
#endif
typedef unsigned long  ulong;
#endif

#include <llong.hpp>

typedef          char   int8;
typedef signed   char   sint8;
typedef unsigned char   uint8;
typedef          short  int16;
typedef unsigned short  uint16;
typedef          long   int32;
typedef unsigned long   uint32;
typedef longlong        int64;
typedef ulonglong       uint64;

inline bool can_place32(uint64 a) { return a == (uint64)(uint32)low(a); }
inline bool can_place32(int64 a)  { return a == ( int64)( int32)low(a); }

#ifdef __EA64__
typedef ulonglong ea_t;   // effective address
typedef ulonglong sel_t;  // segment selector
typedef ulonglong asize_t;// memory chunk size
typedef longlong adiff_t; // address difference
#ifdef __GNUC__
#define FMT_EA "ll"
#else
#define FMT_EA "L"
#endif
#else
typedef ulong ea_t;       // effective address
typedef ulong sel_t;      // segment selector
typedef ulong asize_t;    // memory chunk size
typedef long  adiff_t;    // address difference
#define FMT_EA "l"
#endif

typedef asize_t uval_t;   // unsigned value used by the processor
                          // for 32-bit processors, ulong
                          // for 64-bit processors, ulonglong
typedef adiff_t sval_t;   // nsigned value used by the processor
                          // for 32-bit processors, ulong
                          // for 64-bit processors, ulonglong
#define BADADDR ea_t(-1)  // this value is used for 'bad address'

/*==================================================*/
/* error codes */
/*--------------------------------------------------*/

#define eOk        0    /* No error             */
#define eOS        1    /* OS error, see errno  */
#define eDiskFull  2    /* Disk Full            */
#define eReadError 3    /* Read Error           */

typedef int error_t;

/*--------------------------------------------------*/
/* internal code of last error occured              */
/* see err.h for error handling functions           */
idaman error_t ida_export_data qerrno;

/*==================================================*/
/* type of OS */
/*--------------------------------------------------*/
typedef enum
{
   osMSDOS,
   osAIX_RISC,
   osOS2,
   osNT,
   osLINUX,
} ostype_t;

/*--------------------------------------------------*/
extern ostype_t ostype;

/*==================================================*/
/* memory */
/*--------------------------------------------------*/
idaman void *ida_export qalloc( size_t size );
idaman void *ida_export qrealloc( void *alloc, size_t newsize );
idaman void *ida_export qcalloc( size_t nitems, size_t itemsize );
idaman void  ida_export qfree( void *alloc );
idaman char *ida_export qstrdup( const char *string );
#define qnew(t)        ((t*)qalloc(sizeof(t)))
#define qnewarray(t,n) ((t*)qcalloc((n),sizeof(t)))

#define qnumber(a)     (sizeof(a)/sizeof((a)[0]))
#ifdef __GNUC__
#define qoffsetof(type, name) (((char *)&((type *)1)->name)-(char*)1)  // gcc complains about offsetof()
#else
#define qoffsetof offsetof
#endif

// Reverse memory block
// (the first byte is exchanged with the last bytes, etc.)
// analog of strrev() function
//      buf - pointer to buffer to reverse
//      size - size of buffer
// returns: pointer to buffer

idaman void *ida_export memrev(void *buf,unsigned int size);

#ifdef __GNUC__
idaman int ida_export memicmp(const void *x, const void *y, size_t size);
#endif

/*==================================================*/
/* file name */
/*--------------------------------------------------*/
/* maximum number of characters in path and file specification */
#if defined(__GNUC__) || defined(__KYLIX__)
#define QMAXPATH        PATH_MAX
#define QMAXFILE        PATH_MAX
#else
#define QMAXPATH        _MAX_PATH
#define QMAXFILE        (_MAX_FNAME+_MAX_EXT)
#endif
/*--------------------------------------------------*/
/* construct 'path' from component's list terminated by NULL, return 'path'.
   if 'path'==NULL then pointer to static array will be returned,
   qmakepath( NULL, NULL ) just returns pointer to static array.
*/
idaman char *ida_export vqmakepath( char *path, const char *s1, va_list );
inline char *qmakepath( char *path, const char *s1, ... )
{
  va_list va;
  va_start(va, s1);
  char *code = vqmakepath(path, s1, va);
  va_end(va);
  return code;
}

/*--------------------------------------------------*/
/* split 'path' into 'dir' and 'file' parts, you may specify NULL
   as 'dir'/'file' parameters. 'path' may be changed.
   return file part.
*/
idaman char *ida_export qsplitpath( char *path, char **dir, char **file );
/*--------------------------------------------------*/
/* construct filename from base name and extension, return 'file'.
   if 'file'==NULL then pointer to static array will be returned,
   qmakefile( NULL, NULL, NULL ) just returns pointer to static array.
*/
idaman char *ida_export qmakefile( char *file, const char *base, const char *ext );
/*--------------------------------------------------*/
/* split filename to base name and extension, you may specify NULL
   as 'base'/'ext' parameters. 'file' may be changed.
   return base part.
*/
idaman char *      ida_export qsplitfile( char *file, char **base, char **ext );
idaman int         ida_export qisabspath(const char *file);
idaman const char *ida_export qbasename(const char *path);
inline char *qbasename(char *path) { return (char *)qbasename((const char *)path); }
char *qmake_full_path(char *dst, const char *src);

/* Delimiter of directory lists */
#if defined(__UNIX__) || defined(__LINUX__)
#define DELIMITER       ":"     /* Unix   */
#else
#define DELIMITER       ";"     /* MS DOS */
#endif

/*==================================================*/
/* input/output */
/*--------------------------------------------------*/
#if !defined(__MSDOS__) && !defined(__OS2__) && !defined(__NT__) && !defined(_MSC_VER)
#define O_BINARY        0
#endif

#ifndef SEEK_SET
#define SEEK_SET        0
#define SEEK_CUR        1
#define SEEK_END        2
#endif
/*--------------------------------------------------*/
/* you should use these functions for file i/o                */
/* they do the same as their counterparts from Clib.          */
/* the only difference is that they set 'qerrno' variable too */

idaman int   ida_export qopen( const char *file, int mode );     /* open existing file */
idaman int   ida_export qcreate( const char *file, int stat );   /* create new file with O_RDWR */
idaman int   ida_export qread( int h, void *buf, size_t n );
idaman int   ida_export qwrite( int h, const void *buf, size_t n );
idaman long  ida_export qtell( int h );
idaman long  ida_export qseek( int h, long offset, int whence );
idaman int   ida_export qclose( int h );
idaman ulong ida_export qfilesize( const char *fname );
idaman ulong ida_export qfilelength( int h );
idaman int   ida_export qchsize( int h, unsigned long fsize );
idaman int   ida_export qmkdir( const char *file, int mode );
inline bool idaapi qfileexist(const char *file)
{
  struct stat sb;
  return stat(file, &sb) == 0;
}

/*==================================================*/
/* strings */
/*--------------------------------------------------*/
idaman char *ida_export stpcpyq( char *dest, const char *src );
#if !defined(__BORLANDC__)
#define stpcpy stpcpyq
#endif
#if !defined(__BORLANDC__) && !defined(_MSC_VER)
idaman char *ida_export strlwr( char *s );
idaman char *ida_export strupr( char *s );
#endif
#ifdef __LINUX__
#define strnicmp strncasecmp
#define stricmp  strcasecmp
#endif
/*--------------------------------------------------*/
idaman char *ida_export strleft( char *s, int len );
idaman char *ida_export strright( char *s, int len );
idaman char *ida_export strcenter( char *s, int len );
idaman char *ida_export strpad( char *s, int len );
idaman char *ida_export strpadc( char *s, int len, char c );
idaman char *ida_export strprepadc( char *s, int len, char c );
idaman char *ida_export strclip( char *s );
idaman char *ida_export strclipc( char *s, char clipchar );
idaman char *ida_export strpreclip( char *s );
idaman char *ida_export strfill(char *s, char chr, int count);
idaman char *ida_export strcompact(char *string);
// qstrncpy makes sure that there is a terminating zero
// nb: this function doesn't fill the whole buffer zeroes as strncpy does
idaman char *ida_export qstrncpy(char *dst, const char *src, size_t dstsize);
// qstpncpy returns pointer to the end of the destination
// FIXME: export it!
inline char *qstpncpy(char *dst, const char *src, size_t dstsize)
{
  qstrncpy(dst, src, dstsize);
  return strchr(dst, '\0');
}

/*--------------------------------------------------*/
// Our definitions of qsnprintf/qsscanf support one additional format specifier
//
//      %a              which corresponds to ea_t
//
// Usual optional fields like the width can be used too: %04a
// The width specifier will be doubled for 64-bit version
// These function return the number of characters _actually written_ to the output string
// (which is different from the snprintf)
idaman int ida_export qvsnprintf(char *buffer, size_t n, const char *format, va_list va);
inline int qsnprintf(char *buffer, size_t n, const char *format, ...)
{
  va_list va;
  va_start(va, format);
  int code = qvsnprintf(buffer, n, format, va);
  va_end(va);
  return code;
}

// append a string to the buf. end denotes the end of the buffer
inline int append_snprintf(char *buf, char *end, const char *format, ...)
{
  int code = 0;
  va_list va;
  va_start(va, format);
  buf = strchr(buf, '\0');
  int n = end - buf;
  if ( n > 0 )
    code = qvsnprintf(buf, n, format, va);
  va_end(va);
  return code;
}

idaman int ida_export qvsscanf(const char *input, const char *format, va_list va);
inline int qsscanf(const char *input, const char *format, ...)
{
  va_list va;
  va_start(va, format);
  int code = qvsscanf(input, format, va);
  va_end(va);
  return code;
}

/*==================================================*/
/* Replace all entries of 'char1' by 'char2' in string
   'str'
*/
idaman char *ida_export strrpl( char *str, int char1, int char2 );

/*==================================================*/
/* system call with parameters substitution */
/*--------------------------------------------------*/
idaman void ida_export qexit(int code);
idaman void ida_export qatexit(void (idaapi *func)(void));

/*==================================================*/
/* universal min, max */
/*--------------------------------------------------*/
#define qmin(a,b) ( (a) < (b)? (a): (b) )
#define qmax(a,b) ( (a) > (b)? (a): (b) )

//----------------------------------------------------------------------
// rotate left
template<class T> T qrotl(T value, T count)
{
  const size_t nbits = sizeof(T) * 8;
  count %= nbits;

  T high = value >> (nbits - count);
  value <<= count;
  value |= high;
  return value;
}

// rotate right
template<class T> T qrotr(T value, T count)
{
  const size_t nbits = sizeof(T) * 8;
  count %= nbits;

  T low = value << (nbits - count);
  value >>= count;
  value |= low;
  return value;
}

/* BCB6 treats multicharacter constant differently from old versions
   We are forced to abandon them (it is good because they are not portable anyway) */

#define MC2(c1, c2)          ushort(((c2)<<8)|c1)
#define MC3(c1, c2, c3)      ulong(((((c3)<<8)|(c2))<<8)|c1)
#define MC4(c1, c2, c3, c4)  ulong(((((((c4)<<8)|(c3))<<8)|(c2))<<8)|c1)

/*==================================================*/
/* Add-ins for 2/4 byte read/writes.
        h - file handle
        res - value read from file
        size - size of value in bytes (1,2,4)
        mostfirst - is MSB first? (0/1)

   All these functions return 0 - Ok */

idaman int ida_export readbytes(int h,ulong *res,int size,int mostfirst);
idaman int ida_export writebytes(int h,ulong l,int size,int mostfirst);

idaman int ida_export read2bytes(int h,ushort *res,int mostfirst);
#define read4bytes(h,res,mostfirst)     readbytes(h,res,4,mostfirst)
#define write2bytes(h,l,mostfirst)      writebytes(h,l,2,mostfirst)
#define write4bytes(h,l,mostfirst)      writebytes(h,l,4,mostfirst)

/*==================================================*/
#ifdef __cplusplus
inline ulong swap32(ulong x)
  { return (x>>24) | (x<<24) | ((x>>8) & 0x0000FF00L) | ((x<<8) & 0x00FF0000L); }
inline ushort swap16(ushort x)
  { return ushort((x<<8) | (x>>8)); }
#else
#define swap32(x) ulong((x>>24) | (x<<24) | ((x>>8) & 0x0000FF00L) | ((x<<8) & 0x00FF0000L))
#define swap16(x) ushort((x<<8) | (x>>8))
#endif

#ifdef __EA64__
#define swapea  swap64
#else
#define swapea  swap32
#endif

// swap 2 objects of the same type using memory copies
template <class T> inline void qswap(T &a, T &b)
{
  char temp[sizeof(T)];
  memcpy(&temp, &a, sizeof(T));
  memcpy(&a, &b, sizeof(T));
  memcpy(&b, &temp, sizeof(T));
}

// append a character to the buffer checking the buffer size
#define APPCHAR(buf, end, chr)                    \
  do                                              \
  {                                               \
    if ( buf < end )                              \
      *buf++ = chr;                               \
  } while ( 0 )

// append a zero byte to the buffer checking the buffer size
#define APPZERO(buf, end)                         \
  do                                              \
  {                                               \
    if ( buf >= end )                             \
      end[-1] = '\0';                             \
    else                                          \
      *buf = '\0';                                \
  } while ( 0 )

// append a string to the buffer checking the buffer size
#define APPEND(buf, end, name)                    \
  do                                              \
  {                                               \
    const char *__in = name;                      \
    while ( 1 )                                   \
    {                                             \
      if ( buf >= end )                           \
      {                                           \
        buf = end-1;                              \
        buf[0] = '\0';                            \
        break;                                    \
      }                                           \
      if ( (*buf = *__in++) == 0 )                \
        break;                                    \
      buf++;                                      \
    }                                             \
  } while ( 0 )

// append a string to the buffer checking the buffer size, max 'size' characters
// nb: the trailing zero might be absent in the output buffer!
#define NAPPEND(buf, end, block, size)            \
  do                                              \
  {                                               \
    const char *__in = block;                     \
    int __msize = size;                           \
    while ( --__msize >= 0 )                      \
    {                                             \
      if ( buf >= end )                           \
      {                                           \
        buf = end-1;                              \
        buf[0] = '\0';                            \
        break;                                    \
      }                                           \
      if ( (*buf = *__in++) == 0 )                \
        break;                                    \
      buf++;                                      \
    }                                             \
  } while ( 0 )

/*==================================================*/
// STL replacement templates
#ifdef __WATCOMC__ // since watcom doesn't support stl, implement simple vector right here

class exception {};
class bad_alloc : public exception {};

template <class T> class vector
{
  T *array;
  size_t n, alloc;
  vector<T> &assign(const vector<T> &x)
  {
    if ( x.n > 0 )
    {
      array = (T*)qalloc(x.alloc * sizeof(T));
      if ( array != NULL )
      {
        alloc = x.alloc;
        for ( int i=0; i < x.n; i++ )
          array[i] = x.array[i];
        n = x.n;
      }
    }
    return *this;
  }
public:
  vector(void) : array(NULL), n(0), alloc(0) {}
  vector(const vector<T> &x) : array(NULL), n(0), alloc(0) { assign(x); }
  ~vector(void) { clear(); }
  void push_back(const T &x)
  {
    if ( n >= alloc )
    {
      int m = qmax(8, alloc * 2);
      T *a = (T*)qrealloc(array, m * sizeof(T));
      if ( a == NULL ) throw bad_alloc();
      array = a;
      alloc = m;
    }
    new (array+n) T(x); // create a new element in the vector
    n++;
  }
  void pop_back(void)
  {
    if ( n > 0 )
      array[--n].~T();
  }
  int size(void) const { return n; }
  bool empty(void) const { return n == 0; }
  const T &operator[](int idx) const { return array[idx]; }
        T &operator[](int idx)       { return array[idx]; }
  void clear(void)
  {
    if ( n > 0 )
    {
      for ( int i=0; i < n; i++ )
        array[i].~T();
      qfree(array);
      array = NULL;
      alloc = 0;
      n = 0;
    }
  }
  vector<T> &operator=(const vector<T> &x)
  {
    clear();
    return assign(x);
  }
  void resize(size_t s, const T &x)
  {
    if ( s < n )
    {
      while ( n > s )
        array[--n].~T();
    }
    else if ( s >= alloc )
    {
      T *a = (T*)qrealloc(array, s * sizeof(T));
      if ( a == NULL ) throw bad_alloc();
      array = a;
      alloc = s;
    }
    while ( n < s )
    {
      new(array+n) T(x);
      n++;
    }
  }
};

class string    // implement simple string class
{
  vector<char> body;
public:
  string(void) {
  }
  string(const char *ptr)
  {
    size_t len = strlen(ptr) + 1;
    body.resize(len, '\0');
    memcpy(&body[0], ptr, len);
  }
  string(const char *ptr, size_t len)
  {
    body.resize(len, '\0');
    memcpy(&body[0], ptr, len);
  }
  size_t length(void) const { size_t l = body.size(); return l ? l - 1 : 0; }
  const char *c_str(void) const { return body.size() ? &body[0] : ""; }
  string &operator+=(char c)
  {
    body.resize(body.size()+1, '\0');
    size_t len = length();
    body[len]   = c;
    body[len+1] = '\0';
    return *this;
  }
  string &operator+=(const string &r)
  {
    size_t len = length();
    size_t rlen = r.length();
    body.resize(len+rlen+1, '\0');
    memcpy(&body[len], &r.body[0], rlen);
    body[len+rlen] = '\0';
    return *this;
  }
  bool operator==(const string &r)
  {
    return strcmp(&body[0], &r.body[0]) == 0;
  }
};

#endif // __WATCOMC__

#pragma pack(pop)
#endif /* _PRO_H */
