/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-97 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@estar.msk.su
 *                              FIDO:   2:5020/209
 *
 *      ARM Image File (AIF) Dumper
 *      ---------------------------
 *
 */

#define __NOT_ONLY_PRO_FUNCS__

#include <pro.h>
#include <fpro.h>
#include "aif.h"

static int mf = 0;
static ulong far magic[] = { ZERO_CODE };

//--------------------------------------------------------------------------
static void near error(const char *format,...) {
  va_list va;
  va_start(va,format);
  vfprintf(stderr,format,va);
  va_end(va);
//  fprintf(stderr,"\n");
  qexit(1);
}

//--------------------------------------------------------------------------
static ulong swap(ulong x) {
  union {
    unsigned long l;
    char c[4];
  } u;
  char chr;
  u.l = x;
  chr = u.c[3]; u.c[3] = u.c[0]; u.c[0] = chr;
  chr = u.c[2]; u.c[2] = u.c[1]; u.c[1] = chr;
  return u.l;
}

//--------------------------------------------------------------------------
//
//      check input file format. if recognized, then return 1
//      otherwise return 0
//
int is_aif_file(FILE *fp) {
  aif_header_t hd;
  if ( qfread(fp, &hd, sizeof(hd)) != sizeof(hd) ) return 0;
  if ( memcmp(hd.zero_code,magic,sizeof(magic)) != 0 ) {
    for ( int i=0; i < qnumber(hd.zero_code); i++ )
      hd.zero_code[i] = swap(hd.zero_code[i]);
    if ( memcmp(hd.zero_code,magic,sizeof(magic)) != 0 ) return 0;
  }
  return 1;
}

//--------------------------------------------------------------------------
static void near swap_section(section_t *s) {
  s->codestart = swap(s->codestart);
  s->datastart = swap(s->datastart);
  s->codesize  = swap(s->codesize);
  s->datasize  = swap(s->datasize);
  s->fileinfo  = swap(s->fileinfo);
  s->debugsize = swap(s->debugsize);
  s->name      = swap(s->name);
}

//--------------------------------------------------------------------------
static ushort near process_item(uchar *di,section_t *sect,ulong &offset) {
  ulong fw = *(ulong *)di;
  if ( mf ) fw = swap(fw);
  ushort len = fw >> 16;
//  printf("%2ld. %02x %lx\n",fw & 0xFFFF,len,offset);
  offset += len;
  switch ( fw & 0xFFFF ) {
    case AIF_DEB_SECT:  // section
      sect = (section_t *)di;
      if ( mf ) swap_section(sect);
      if ( sect->debugsize != 0 ) {
        offset -= len;
        len = sect->debugsize;
        offset += len;
      }
      printf("SECTION code=%08lx size=%08lx\n"
             "\tdata=%08lx size=%08lx\n"
             "\tdebugsize=%08lx debugversion=%d\n"
             "\tfileinfo=%08lx name=%08lx\n",
                sect->codestart,sect->codesize,
                sect->datastart,sect->datasize,
                sect->debugsize,sect->asdversion,
                sect->fileinfo,sect->name);
      switch ( sect->lang ) {
        case LANG_C:
          printf("\tC source level debugging data is present\n");
          break;
        case LANG_PASCAL:
          printf("\tPascal source level debugging data is present\n");
          break;
        case LANG_FORTRAN:
          printf("\tFortran-77 source level debugging data is present\n");
          break;
        case LANG_ASM:
          printf("\tARM assembler line number data is present\n");
          break;
      }
      if ( sect->lang == LANG_NONE ) {
        size_t nsyms = size_t(sect->name);
        dsym_t *ds = (dsym_t *)(sect+1);
        char *str = (char *)(ds+nsyms);
        for ( int i=0; i < nsyms; i++,ds++ ) {
          if ( ds->sym & ASD_16BITSYM ) continue;
          size_t off = size_t(ds->sym & ASD_SYMOFF);
          char *name = str + off;
          putchar('\t');
          if ( ds->sym & ASD_ABSSYM   ) printf("abs "); else printf("rel ");
          if ( ds->sym & ASD_GLOBSYM  ) printf("pub "); else printf("loc ");
          if ( ds->is_text()          ) printf("txt "); else printf("    ");
          if ( ds->is_data()          ) printf("dat "); else printf("    ");
          if ( ds->is_bss()           ) printf("bss "); else printf("    ");
          if ( ds->sym & ASD_16BITSYM ) printf("tmb "); else printf("    ");
          printf("%08lX %08lX %s\n",ds->sym,ds->value,name);
        }
      } else {
        char name[64];
        char *nptr = (char *)&sect->name;
        int len = *nptr++;
        if ( len >= sizeof(name) ) len = sizeof(name)-1;
        memcpy(name,nptr,len);
        name[len] = '\0';
        printf("\tName: \"%s\"\n",name);
      }
#if 0
      if ( sect->fileinfo != 0 ) {      // fileinfo is present?
        process_item(di+size_t(sect->fileinfo),sect);
      }
#endif
      break;
    case AIF_DEB_FDEF:  // procedure/function definition
      printf("procedure/function definition\n");
      break;
    case AIF_DEB_ENDP:  // endproc
      printf("endproc\n");
      break;
    case AIF_DEB_VAR:   // variable
      printf("variable\n");
      break;
    case AIF_DEB_TYPE:  // type
      printf("type\n");
      break;
    case AIF_DEB_STRU:  // struct
      printf("struct\n");
      break;
    case AIF_DEB_ARRAY: // array
      printf("array\n");
      break;
    case AIF_DEB_RANGE: // subrange
      printf("subrange\n");
      break;
    case AIF_DEB_SET:   // set
      printf("set\n");
      break;
    case AIF_DEB_FILE:  // fileinfo
      printf("fileinfo\n");
      break;
    case AIF_DEB_CENUM: // contiguous enumeration
      printf("contiguous enumeration\n");
      break;
    case AIF_DEB_DENUM: // discontiguous enumeration
      printf("discontiguous enumeration\n");
      break;
    case AIF_DEB_FDCL:  // procedure/function declaration
      printf("procedure/function declaration\n");
      break;
    case AIF_DEB_SCOPE: // begin naming scope
      printf("begin naming scope\n");
      break;
    case AIF_DEB_ENDS:  // end naming scope
      printf("end naming scope\n");
      break;
    case AIF_DEB_BITF:  // bitfield
      printf("bitfield\n");
      break;
    case AIF_DEB_MACRO: // macro definition
      printf("macro definition\n");
      break;
    case AIF_DEB_ENDM:  // macro undefinition
      printf("macro undefinition\n");
      break;
    case AIF_DEB_CLASS: // class
      printf("class\n");
      break;
    case AIF_DEB_UNION: // union
      printf("union\n");
      break;
    case AIF_DEB_FPMAP: // FP map fragment
      printf("FP map fragment\n");
      break;
    default:
      printf("unknown (0x%ld.)!!!\n",fw & 0xFFFF);
      break;
  }
  return len;
}

//--------------------------------------------------------------------------
//
//      load file into the database.
//
void dump_aif_file(FILE *fp) {
  int i;
  aif_header_t hd;
  qfread(fp,&hd,sizeof(hd));
  if ( memcmp(hd.zero_code,magic,sizeof(magic)) != 0 ) {
    ulong *ptr = (ulong *)&hd;
    for ( i=0; i < sizeof(hd)/sizeof(ulong); i++,ptr++ )
      *ptr = swap(*ptr);
    mf = 1;
  }
  if ( (hd.address_mode & 0xFF) != 32 ) {
    if ( (hd.address_mode & 0xFF) != 0 ) error("26-bit modules are not supported");
    printf("Old AIF format file...");
  }
  if ( hd.decompress_code != NOP ) error("Compressed modules are not supported");
  if ( hd.self_reloc_code != NOP ) error("Self-relocating modules are not supported");

  int isexec = is_bl(hd.entry_point);
  ulong offset = sizeof(aif_header_t);
  ulong start = hd.image_base;
  if ( isexec ) {
    start += sizeof(aif_header_t);
    hd.readonly_size -= sizeof(aif_header_t);
  }
  ulong end = start + hd.readonly_size;
  printf("CODE SEGMENT\n");
  printf("\tFile offset: %08lX\n",offset);
  printf("\tAddress    : %08lX\n",start);
  printf("\tSize       : %08lX\n",end-start);
  offset += hd.readonly_size;
  if ( hd.readwrite_size != 0 ) {
    start = (hd.address_mode & AIF_SEP_DATA) ? hd.data_base : end;
    end = start + hd.readwrite_size;
    printf("\nDATA SEGMENT\n");
    printf("\tFile offset: %08lX\n",offset);
    printf("\tAddress    : %08lX\n",start);
    printf("\tSize       : %08lX\n",end-start);
    offset += hd.readwrite_size;
  }
  if ( hd.zero_init_size != 0 ) {
    start = end;
    end = start + hd.zero_init_size;
    printf("\nBSS SEGMENT\n");
    printf("\tAddress    : %08lX\n",start);
    printf("\tSize       : %08lX\n",end-start);
  }

  if ( isexec )
    hd.entry_point = hd.image_base
                   + offsetof(aif_header_t,entry_point)
                   + ((hd.entry_point & ~BLMASK) << 2)
                   + 8;
  printf("\nENTRY POINT: %08lX\n",hd.entry_point);

  if ( hd.debug_size != 0 ) {
    printf("\nDebugging information is present (%ld bytes at file offset %08lX)\n",
                                                        hd.debug_size,offset);
    uchar *di = qnewarray(uchar,size_t(hd.debug_size));
    if ( di == NULL ) error("no memory");
    qfseek(fp,offset,SEEK_SET);
    qfread(fp,di,size_t(hd.debug_size));
    uchar *end = di + size_t(hd.debug_size);
    section_t *sect = NULL;
    while ( di < end ) {
      ushort len = process_item(di,sect,offset);
      di += len;
    }
  }

}

int main(int argc,char *argv[]) {
  if ( argc < 2 ) error("usage: dumpaif file");
  FILE *fp = fopen(argv[1],"rb");
  if ( fp == NULL ) error("can't open file '%'",argv[1]);
  if ( !is_aif_file(fp) ) error("%s is not AIF file",argv[1]);
  qfseek(fp,0,SEEK_SET);
  dump_aif_file(fp);
  fclose(fp);
  return 0;
}
