/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-99 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@datarescue.com
 *
 *
 */

#include "h8500.hpp"
#include <frame.hpp>

static int flow;
//------------------------------------------------------------------------
static void process_immediate_number(int n)
{
  doImmd(cmd.ea);
  if ( isDefArg(uFlag,n) ) return;
  switch ( cmd.itype )
  {
    case H8500_add_q:
    case H8500_bclr:
    case H8500_bnot:
    case H8500_bset:
    case H8500_btst:
      op_dec(cmd.ea, n);
      uFlag = getFlags(cmd.ea);
      break;
    case H8500_and:
    case H8500_or:
    case H8500_xor:
    case H8500_andc:
    case H8500_orc:
    case H8500_xorc:
      op_num(cmd.ea, n);
      uFlag = getFlags(cmd.ea);
      break;
  }
}

//----------------------------------------------------------------------
inline bool issp(int x)
{
  return x == SP;
}

inline bool isbp(int x)
{
  return x == FP;
}

//----------------------------------------------------------------------
bool is_sp_based(const op_t &x)
{
  return (x.type != o_displ || x.type != o_phrase) && issp(x.phrase);
}

//----------------------------------------------------------------------
static void add_stkpnt(int value)
{
  if ( value & 1 ) value++;
  add_auto_stkpnt(cmd.ea+cmd.size, value);
}

//----------------------------------------------------------------------
inline bool is_mov(int itype)
{
  return itype >= H8500_mov_g && itype <= H8500_mov_s;
}

//----------------------------------------------------------------------
static bool get_op_value(op_t &x, int *value)
{
  if ( x.type == o_imm )
  {
    *value = x.value;
    return true;
  }
  bool ok = false;
  if ( x.type == o_reg )
  {
    int reg = x.reg;
    insn_t saved = cmd;
    if ( decode_prev_insn(cmd.ea) != BADADDR
      && is_mov(cmd.itype)
      && cmd.Op1.type == o_imm
      && cmd.Op2.type == o_reg
      && cmd.Op2.reg  == reg )
    {
      *value = cmd.Op1.value;
      ok = true;
    }
    cmd = saved;
  }
  return ok;
}

//----------------------------------------------------------------------
static int calc_reglist_count(int regs)
{
  int count = 0;
  for ( int i=0; i < 8; i++,regs>>=1 )
    if ( regs & 1 ) count++;
  return count;
}

//----------------------------------------------------------------------
static void trace_sp(void)
{
  // @sp++
  if ( cmd.Op1.type == o_phrase
    && issp(cmd.Op1.reg)
    && cmd.Op1.phtype == ph_post )
  {
    int size = 2;
    if ( cmd.Op2.type == o_reglist ) size *= calc_reglist_count(cmd.Op2.reg);
    add_stkpnt(size);
    return;
  }

  // @--sp
  if ( cmd.Op2.type == o_phrase
    && issp(cmd.Op2.reg)
    && cmd.Op2.phtype == ph_pre )
  {
    int size = 2;
    if ( cmd.Op1.type == o_reglist ) size *= calc_reglist_count(cmd.Op2.reg);
    add_stkpnt(-size);
    return;
  }

  int v;
  switch ( cmd.itype )
  {
    case H8500_add_g:
    case H8500_add_q:
    case H8500_adds:
      if ( !issp(cmd.Op2.reg) ) return;
      if ( get_op_value(cmd.Op1, &v) ) add_stkpnt(v);
      break;
    case H8500_sub:
    case H8500_subs:
      if ( !issp(cmd.Op2.reg) ) return;
      if ( get_op_value(cmd.Op1, &v) ) add_stkpnt(-v);
      break;
  }
}

//----------------------------------------------------------------------
static void process_operand(op_t &x,int isAlt,int isload)
{
  switch ( x.type )
  {
    case o_reg:
    case o_phrase:
    case o_reglist:
      return;
    case o_imm:
      if ( !isload ) interr("emu");
      process_immediate_number(x.n);
      if ( isOff(uFlag, x.n) ) ua_add_off_drefs(x, dr_O);
      break;
    case o_displ:
      process_immediate_number(x.n);
      if ( isAlt ) break;
      if ( isOff(uFlag, x.n) )
      {
        ua_add_off_drefs(x, isload ? dr_R : dr_W);
        ea_t ea = calc_target(cmd.ea+x.offb, cmd.ea, x.n, x.addr);
        ua_dodata(ea, x.dtyp);
        if ( !isload ) doVar(ea);
      }
      // create stack variables if required
      if ( may_create_stkvars() && !isDefArg(uFlag, x.n) )
      {
        func_t *pfn = get_func(cmd.ea);
        if ( pfn != NULL
          && (issp(x.phrase)
              || isbp(x.phrase) && (pfn->flags & FUNC_FRAME) != 0) )
        {
          if ( ua_stkvar(x, x.addr) ) op_stkvar(cmd.ea, x.n);
        }
      }
      break;
    case o_near:
      {
        cref_t ftype = fl_JN;
        ea_t ea = calc_mem(x);
        if ( Instructions[cmd.itype].feature & CF_CALL )
        {
          func_t *pfn  = get_func(ea);
          if ( pfn != NULL && (pfn->flags & FUNC_NORET) != 0 ) flow = 0;
          ftype = fl_CN;
        }
        ua_add_cref(x.offb, ea, ftype);
      }
      break;
    case o_mem:
      {
        ea_t ea = calc_mem(x);
        ua_add_dref(x.offb, ea, isload ? dr_R : dr_W);
        ua_dodata(ea, x.dtyp);
        if ( !isload ) doVar(ea);
      }
      break;
    default:
      interr("emu");
  }
}


//----------------------------------------------------------------------
int emu(void)
{
  int Feature = Instructions[cmd.itype].feature;
  int flag1 = is_forced_operand(cmd.ea, 0);
  int flag2 = is_forced_operand(cmd.ea, 1);
  int flag3 = is_forced_operand(cmd.ea, 2);

  flow = ((Feature & CF_STOP) == 0);

  if ( Feature & CF_USE1 ) process_operand(cmd.Op1, flag1, 1);
  if ( Feature & CF_USE2 ) process_operand(cmd.Op2, flag2, 1);
  if ( Feature & CF_USE3 ) process_operand(cmd.Op3, flag3, 1);

  if ( Feature & CF_CHG1 ) process_operand(cmd.Op1, flag1, 0);
  if ( Feature & CF_CHG2 ) process_operand(cmd.Op2, flag2, 0);
  if ( Feature & CF_CHG3 ) process_operand(cmd.Op3, flag3, 0);

//
//      Determine if the next instruction should be executed
//
  if ( segtype(cmd.ea) == SEG_XTRN ) flow = 0;
  if ( flow ) ua_add_cref(0,cmd.ea+cmd.size,fl_F);


//
//      Handle SP modifications
//
  if ( should_trace_sp() && auto_state == AU_USED )
  {
    if ( !flow )
      recalc_spd(cmd.ea);     // recalculate SP register for the next insn
    else
      trace_sp();
  }

#if 0
//      Hanfle SP offsets like this:
//
//        addiu   $4, $29, 0x28+var_8
//
  if ( may_create_stkvars()
    && cmd.itype == MIPS_addiu
    && cmd.Op2.type == o_reg
    && cmd.Op2.reg  == R29
    && cmd.Op1.reg  != R29 )
        if ( ua_stkvar(cmd.Op3, cmd.Op3.value) ) op_stkvar(cmd.ea, 2);
#endif

  return 1;
}

//----------------------------------------------------------------------
int is_jump_func(const func_t * /*pfn*/, ea_t *jump_target)
{
  *jump_target = BADADDR;
  return 1; // means "no"
}

//----------------------------------------------------------------------
int may_be_func(void)           // can a function start here?
                                // arg: none, the instruction is in 'cmd'
                                // returns: probability 0..100
                                // 'cmd' structure is filled upon the entrace
                                // the idp module is allowed to modify 'cmd'
{
//  if ( cmd.itype == H8_push && isbp(cmd.Op1.reg) ) return 100;  // push.l er6
  return 0;
}

//----------------------------------------------------------------------
int is_sane_insn(int /*nocrefs*/)
{
  if ( cmd.itype == H8500_nop )
  {
    for ( int i=0; i < 8; i++ )
      if ( get_word(cmd.ea-i*2) != 0 ) return 1;
    return 0; // too many nops in a row
  }
  return 1;
}

//----------------------------------------------------------------------
int is_align_insn(ea_t ea)
{
  if ( !ua_ana0(ea) ) return 0;
  switch ( cmd.itype )
  {
    case H8500_nop:
      break;
    case H8500_mov_g:         // B/W Move data
    case H8500_mov_e:         // B   Move data
    case H8500_mov_i:         // W   Move data
    case H8500_mov_f:         // B/W Move data
    case H8500_mov_l:         // B/W Move data
    case H8500_mov_s:         // B/W Move data
    case H8500_or:
    case H8500_and:
      if ( cmd.Op1.type == cmd.Op2.type && cmd.Op1.reg == cmd.Op2.reg ) break;
    default:
      return 0;
  }
  return cmd.size;
}

//----------------------------------------------------------------------
int h8500_get_frame_retsize(func_t *pfn)
{
  return pfn->flags & FUNC_FAR ? 4 : 2;
}
