/*
 *      Interactive disassembler (IDA).
 *      Copyright (c) 1990-99 by Ilfak Guilfanov.
 *      ALL RIGHTS RESERVED.
 *                              E-mail: ig@datarescue.com
 *
 *
 */

#include "pic.hpp"
#include <frame.hpp>
#include <srarea.hpp>
#include <struct.hpp>

//----------------------------------------------------------------------
static void out_bad_address(ea_t addr)
{
  out_tagon(COLOR_ERROR);
  OutLong(addr, 16);
  out_tagoff(COLOR_ERROR);
  QueueMark(Q_noName, cmd.ea);
}

//----------------------------------------------------------------------
inline void outreg(int r)
{
  out_register(ph.regNames[r]);
}

//----------------------------------------------------------------------
ea_t calc_code_mem(ea_t ea)
{
  return toEA(cmd.cs, ea);
}

//----------------------------------------------------------------------
ea_t calc_data_mem(ea_t ea)
{
  return dataseg + map_port(ea);
}

//----------------------------------------------------------------------
bool outop(op_t &x)
{
  ea_t ea;
  switch ( x.type )
  {

    case o_void:
      return 0;

    case o_reg:
      outreg(x.reg);
      break;

    case o_imm:
      if ( is_bit_insn() )
      {
        const char *name = find_bit(cmd.Op1.addr, x.value);
        if ( name != NULL )
        {
          out_line(name, COLOR_IMPNAME);
          break;
        }
      }
      OutValue(x, OOFS_IFSIGN|OOFW_IMM);
      break;

    case o_mem:
      {
        ea = calc_data_mem(x.addr);
        const char *name = find_sym(x.addr);
        if ( name == NULL ) goto OUTNAME;
        out_addr_tag(ea);
        out_line(name, COLOR_IMPNAME);
      }
      break;

    case o_near:
      {
        ea = calc_code_mem(x.addr);
OUTNAME:
        const char *ptr = get_name_expr(cmd.ea+x.offb, x.n, ea, x.addr);
        if ( ptr == NULL )
          out_bad_address(x.addr);
        else
          OutLine(ptr);
      }
      break;

    default:
      error("interr: out");
      break;
  }
  return 1;
}

//----------------------------------------------------------------------
bool conditional_insn(void)
{
  if (isFlow(uFlag))
  {
    int code = get_full_byte(cmd.ea-1);
    switch (ptype)
    {
      case PIC12:
        if ((code & 0xFC0) == 0x2C0) return true;        // 0010 11df ffff DECFSZ  f, d           Decrement f, Skip if 0
        else if ((code & 0xFC0) == 0x3C0) return true;   // 0011 11df ffff INCFSZ  f, d           Increment f, Skip if 0
        else if ((code & 0xF00) == 0x600) return true;   // 0110 bbbf ffff BTFSC   f, b           Bit Test f, Skip if Clear
        else if ((code & 0xF00) == 0x700) return true;   // 0111 bbbf ffff BTFSS   f, b           Bit Test f, Skip if Set
        break;
      case PIC14:
        if ((code & 0x3F00) == 0x0B00) return true;      // 00 1011 dfff ffff DECFSZ  f, d        Decrement f, Skip if 0
        else if ((code & 0x3F00) == 0x0F00) return true; // 00 1111 dfff ffff INCFSZ  f, d        Increment f, Skip if 0
        else if ((code & 0x3C00) == 0x1800) return true; // 01 10bb bfff ffff BTFSC   f, b        Bit Test f, Skip if Clear
        else if ((code & 0x3C00) == 0x1C00) return true; // 01 11bb bfff ffff BTFSS   f, b        Bit Test f, Skip if Set
        break;
      case PIC16:
        break;
    }
  }
  return false;
}

//----------------------------------------------------------------------
void out(void)
{
  char buf[MAXSTR];
  init_output_buffer(buf, sizeof(buf));

  if ( conditional_insn() ) OutChar(' ');
  OutMnem();

  out_one_operand(0);
  if ( cmd.Op2.type != o_void )
  {
    out_symbol(',');
    OutChar(' ');
    out_one_operand(1);
    if ( cmd.Op3.type != o_void )
    {
      out_symbol(',');
      OutChar(' ');
      out_one_operand(2);
    }
  }

  if ( ( cmd.Op1.type == o_mem && cmd.Op1.addr == PIC16_INDF2 )
    || ( cmd.Op2.type == o_mem && cmd.Op2.addr == PIC16_INDF2 ) )
  {
    func_t *pfn  = get_func(cmd.ea);
    struc_t *sptr  = get_frame(pfn);
    if ( pfn != NULL && sptr != NULL )
    {
      member_t *mptr = get_member(sptr, pfn->frregs + pfn->frsize);
      if ( mptr != NULL )
      {
        char *name = get_member_name(mptr->id);
        if ( name != NULL )
        {
          OutChar(' ');
          out_line(ash.cmnt, COLOR_AUTOCMT);
          OutChar(' ');
          out_line(name, COLOR_LOCNAME);
        }
      }
    }
  }

  if ( isVoid(cmd.ea, uFlag, 0) ) OutImmChar(cmd.Op1);
  if ( isVoid(cmd.ea, uFlag, 1) ) OutImmChar(cmd.Op2);
  if ( isVoid(cmd.ea, uFlag, 2) ) OutImmChar(cmd.Op3);

  term_output_buffer();
  gl_comm = 1;
  MakeLine(buf);
}

//--------------------------------------------------------------------------
static void print_segment_register(int reg, sel_t value)
{
  if ( reg == ph.regDataSreg ) return;
  if ( value != BADSEL )
    gen_cmt_line("assume %s = %s", ph.regNames[reg], btoa(value));
  else
    gen_cmt_line("drop %s", ph.regNames[reg]);
}

//--------------------------------------------------------------------------
void assumes(ea_t ea)         // function to produce assume directives
{
  segreg_t *Darea  = getSRarea(ea);
  segment_t *Sarea = getseg(ea);
  if ( Sarea == NULL || Darea == NULL || !inf.s_assume ) return;

  for ( int i=ph.regFirstSreg; i <= ph.regLastSreg; i++ )
  {
    if ( i == ph.regCodeSreg ) continue;
    sel_t now  = getSR(ea, i);
    bool show = (ea == Sarea->startEA);
    if ( show || Darea->startEA == ea )
    {
      segreg_t *prev = getSRarea(ea-1);
      if ( show || (prev != NULL && getSR(prev->startEA, i) != now) )
        print_segment_register(i, now);
    }
  }
}

//--------------------------------------------------------------------------
void segstart(ea_t ea)
{
  segment_t *Sarea = getseg(ea);
  if ( is_spec_segm(Sarea->type) ) return;
  const char *sname = get_true_segm_name(Sarea);
  const char *sclas = get_segm_class(Sarea);

  printf_line(-1, COLSTR("%s %s (%s)", SCOLOR_AUTOCMT),
                  ash.cmnt,
                 strcmp(sclas,"CODE") == 0
                    ? ".text"
                    : strcmp(sclas,"BSS") == 0
                         ? ".bss"
                         : ".data",
                 sname);
  if ( Sarea->orgbase != 0 ) printf_line(-1, COLSTR("%s %s", SCOLOR_ASMDIR),
                                        ash.origin, btoa(Sarea->orgbase));
}

//--------------------------------------------------------------------------
void segend(ea_t) {
#if 0
  segment_t *s = getseg(ea-1);
  if ( is_spec_segm(s->type) ) return;
  printf_line(0,COLSTR(";%-*s ends",SCOLOR_AUTOCMT),inf.indent-2,get_segm_name(s));
#endif
}

//--------------------------------------------------------------------------
void header(void)
{
  gen_cmt_line("Processor       : %-8.8s", inf.procName);
  gen_cmt_line("Target assembler: %s", ash.name);
  printf_line(0,COLSTR("include \"P%s.INC\"",SCOLOR_ASMDIR),device);
//  gen_cmt_line("Byte sex        : %s", inf.mf ? "Big endian" : "Little endian");
  if ( ash.header != NULL )
    for ( const char **ptr=ash.header; *ptr != NULL; ptr++ )
      printf_line(0,COLSTR("%s",SCOLOR_ASMDIR),*ptr);
  MakeNull();
}

//--------------------------------------------------------------------------
void footer(void)
{
  char name[MAXSTR];
  get_colored_name(BADADDR, inf.beginEA, name, sizeof(name));
  const char *end = ash.end;
  if ( end == NULL )
    printf_line(-1,COLSTR("%s end %s",SCOLOR_AUTOCMT), ash.cmnt, name);
  else
    printf_line(-1,COLSTR("%s",SCOLOR_ASMDIR)
                  " "
                  COLSTR("%s %s",SCOLOR_AUTOCMT), ash.end, ash.cmnt, name);
}

//--------------------------------------------------------------------------
static void out_equ(bool indent, const char *name, uval_t off)
{
  if ( name != NULL )
  {
    gl_name = 0;
    char buf[MAXSTR];
    char *const end = buf + sizeof(buf);
    char *p = buf;
    if ( indent ) APPCHAR(p, end, ' ');
    APPEND(p, end, name);
    p = addblanks(buf, inf.indent-1);
    APPCHAR(p, end, ' ');
    p = tag_addstr(p, end, COLOR_KEYWORD, ash.a_equ);
    APPCHAR(p, end, ' ');
    tag_addstr(p, end, COLOR_NUMBER, btoa(off));
    MakeLine(buf, 0);
  }
}

//--------------------------------------------------------------------------
// output "equ" directive(s) if necessary
static int out_equ(ea_t ea)
{
  segment_t *s = getseg(ea);
  if ( s != NULL && s->type == SEG_IMEM && ash.a_equ != NULL )
  {
    char nbuf[MAXSTR];
    char *name = get_name(BADADDR, ea, nbuf, sizeof(nbuf));
    if ( name != NULL )
    {
      uval_t off = ea - get_segm_base(s);
      out_equ(false, name, off);
      const ioport_bit_t *bits = find_bits(off);
      if ( bits != NULL )
      {
        for ( int i=0; i < 8; i++ )
          out_equ(true, bits[i].name, i);
        MakeNull();
      }
    }
    else
    {
      MakeLine("");
    }
    return true;
  }
  return false;
}

//--------------------------------------------------------------------------
void data(ea_t ea)
{
  gl_name = 1;

  // the kernel's standard routine which outputs the data knows nothing
  // about "equ" directives. So we do the following:
  //    - try to output an "equ" directive
  //    - if we succeed, then ok
  //    - otherwise let the standard data output routine, intel_data()
  //        do all the job

  if ( !out_equ(ea) )
    intel_data(ea);
}
