/*
       IDA Pro remote debugger server
*/

#include <signal.h>

static bool verbose = false;

#define REMOTE_DEBUGGER
#define DEBUGGER_SERVER
#include "idarpc.cpp"

#ifdef __NT__
#define SYSTEM "Windows32"
#define socklen_t int
#endif

#ifdef __LINUX__
#define SYSTEM "Linux"
#ifdef LIBWRAP
extern "C" const char *check_connection(int);
#endif // LIBWRAP
#endif // __LINUX__

static SOCKET listen_socket = -1;
//--------------------------------------------------------------------------
void neterr(const char *module)
{
  int code = get_network_error();
  fprintf(stderr, "%s: %s\n", module, winerr(code));
  exit(1);
}

//--------------------------------------------------------------------------
static void NT_CDECL shutdown_gracefully(int signum)
{
  fprintf(stderr, "got signal #%d\n", signum);
  if ( listen_socket != -1 ) closesocket(listen_socket);
  if ( rpc_socket != -1 ) closesocket(rpc_socket);
  _exit(1);
}

//--------------------------------------------------------------------------
// debugger remote server
int main(int argc, char *argv[])
{
  int port_number = debugger_t::default_port_number;
  const char *password = NULL;
  printf("IDA " SYSTEM " remote debugger server. Version 1.0. Copyright Datarescue 2004\n");
  while ( argc > 1 && (argv[1][0] == '-' || argv[1][0] == '/'))
  {
    switch ( argv[1][1] )
    {
      case 'p':
        port_number = atoi(&argv[1][2]);
        break;
      case 'P':
        password = argv[1] + 2;
        break;
      case 'v':
        verbose = true;
        break;
      default:
        error("usage: ida_remote [switches]\n"
              "  -p...  port number\n"
              "  -P...  password\n"
              "  -v     verbose\n");
    }
    argv++;
    argc--;
  }
  signal(SIGINT, shutdown_gracefully);
  signal(SIGSEGV, shutdown_gracefully);
//  signal(SIGPIPE, SIG_IGN);
  init_sockets();
  listen_socket = socket(AF_INET, SOCK_STREAM, 0);
  if ( listen_socket == -1 )
    neterr("socket");

  setup_socket(listen_socket);

  struct sockaddr_in sa;
  memset(&sa, 0, sizeof(sa));
  sa.sin_family = AF_INET;
  sa.sin_port   = htons(port_number);
  if ( bind(listen_socket, (sockaddr *)&sa, sizeof(sa)) == SOCKET_ERROR )
    neterr("bind");
  if ( listen(listen_socket, SOMAXCONN) == SOCKET_ERROR )
    neterr("listen");
  printf("Listening to port #%u...\n", port_number);

  poll_debug_events = false;
  has_pending_event = false;

  while ( true )
  {
    sockaddr_in sa;
    socklen_t salen = sizeof(sa);
    rpc_socket = accept(listen_socket, (sockaddr *)&sa, &salen);
    if ( rpc_socket == -1 )
      neterr("accept");
#if defined(__LINUX__) && defined(LIBWRAP)
    const char *p;
    if((p = check_connection(rpc_socket)) != NULL) {
      fprintf(stderr,
              "ida-server CONNECTION REFUSED from %s (tcp_wrappers)\n", p);
      shutdown(rpc_socket, 2);
      close(rpc_socket);
      continue;
    }
#endif // defined(__LINUX__) && defined(LIBWRAP)

    printf("=========================================================\n"
           "Accepting incoming connection...\n");

    bool ok = true;
    string open = prepare_rpc_packet(RPC_OPEN);
    rpc_packet_t *rp = process_request(open, true);
    if ( password != NULL )
    {
      const uchar *answer = (uchar *)(rp+1);
      char *pass = extract_str(&answer);
      if ( strcmp(pass, password) != '\0' )
      {
        printf("Bad password\n");
        ok = false;
      }
    }
    qfree(rp);

    open = prepare_rpc_packet(RPC_OK);
    append_long(open, ok);
    send_request(open);

    if ( ok )
    {
      string cmd;
      qfree(process_request(cmd, true));
    }

    closesocket(rpc_socket);
    rpc_socket = -1;
    printf("Closing incoming connection...\n");
  }
}

