#include "debmod.h"
#include <err.h>
#include <diskio.hpp>
#include <kernwin.hpp>

static char *get_event_name(event_id_t id)
{
  switch ( id )
  {
    case NO_EVENT:       return "NO_EVENT";
    case THREAD_START:   return "THREAD_START";
    case STEP:           return "STEP";
    case SYSCALL:        return "SYSCALL";
    case WINMESSAGE:     return "WINMESSAGE";
    case PROCESS_DETACH: return "PROCESS_DETACH";
    case PROCESS_START:  return "PROCESS_START";
    case PROCESS_ATTACH: return "PROCESS_ATTACH";
    case PROCESS_SUSPEND: return "PROCESS_SUSPEND";
    case LIBRARY_LOAD:   return "LIBRARY_LOAD";
    case PROCESS_EXIT:   return "PROCESS_EXIT";
    case THREAD_EXIT:    return "THREAD_EXIT";
    case BREAKPOINT:     return "BREAKPOINT";
    case EXCEPTION:      return "EXCEPTION";
    case LIBRARY_UNLOAD: return "LIBRARY_UNLOAD";
    case INFORMATION:    return "INFORMATION";
    default:             return "???";
  }
}

//--------------------------------------------------------------------------
bool debmod_t::same_as_oldmemcfg(const meminfo_vec_t &areas)
{
  return old_areas == areas;
}

//--------------------------------------------------------------------------
void debmod_t::save_oldmemcfg(const meminfo_vec_t &areas)
{
  old_areas = areas;
}

//--------------------------------------------------------------------------
bool debmod_t::check_input_file_crc32(uint32 orig_crc)
{
  if ( orig_crc == 0 )
    return true; // the database has no crc

  linput_t *li = open_linput(input_file_path, false);
  if ( li == NULL )
    return false;

  uint32 crc = calc_file_crc32(li);
  close_linput(li);
  return crc == orig_crc;
}

//--------------------------------------------------------------------------
const exception_info_t *debmod_t::find_exception(int code)
{
  for ( size_t i=0; i < exceptions.size(); i++ )
  {
    if ( exceptions[i].code == (uint)code )
      return &exceptions[i];
  }
  return NULL;
}

//--------------------------------------------------------------------------
bool debmod_t::get_exception_name(int code, char *buf, size_t bufsize)
{
  const exception_info_t *ei = find_exception(code);
  if ( ei != NULL )
  {
    qstrncpy(buf, ei->name.c_str(), bufsize);
    return true;
  }
  qsnprintf(buf, bufsize, "%08lX", code);
  return false;
}

//----------------------------------------------------------------------
// Display a system error message. This function always returns false
bool debmod_t::deberr(const char *format, ...)
{
  if (!debug_debugger)
    return false;

  int code = get_system_specific_errno();
  va_list va;
  va_start(va, format);
  dvmsg(0, dmsg_ud, format, va);
  va_end(va);
  dmsg(": %s\n", winerr(code));
  return false;
}

//--------------------------------------------------------------------------
// used to debug the debugger
void debmod_t::debdeb(const char *format, ...)
{
  if (!debug_debugger)
    return;

  va_list va;
  va_start(va, format);
  dvmsg(0, dmsg_ud, format, va);
  va_end(va);
}

//--------------------------------------------------------------------------
debmod_t::debmod_t(void)
{
  input_file_path[0] = '\0';
  dmsg_ud = NULL;
  debug_debugger = false;
}

//--------------------------------------------------------------------------
void debmod_t::cleanup(void)
{
  input_file_path[0] = '\0';
  old_areas.clear();
  clear_debug_names();
  exceptions.clear();
}

//--------------------------------------------------------------------------
void idaapi debmod_t::dbg_set_exception_info(const exception_info_t *table, int qty)
{
  exceptions.clear();
  for ( int i=0; i < qty; i++ )
    exceptions.push_back(*table++);
}

//--------------------------------------------------------------------------
char *debmod_t::debug_event_str(const debug_event_t *ev)
{
  static char buf[MAXSTR];
  return debug_event_str(ev, buf, sizeof(buf));
}

//--------------------------------------------------------------------------
char *debmod_t::debug_event_str(const debug_event_t *ev, char *buf, size_t bufsize)
{
  char *ptr = buf;
  char *end = buf + bufsize;
  ptr += qsnprintf(ptr, end-ptr, "%s ea=%a",
    get_event_name(ev->eid),
    ev->ea);
  switch ( ev->eid )
  {
  case PROCESS_START:  // New process started
  case PROCESS_ATTACH: // Attached to running process
  case LIBRARY_LOAD:   // New library loaded
    ptr += qsnprintf(ptr, end-ptr, " base=%a size=%a rebase=%a name=%s",
      ev->modinfo.base,
      ev->modinfo.size,
      ev->modinfo.rebase_to,
      ev->modinfo.name);
    break;
  case PROCESS_EXIT:   // Process stopped
  case THREAD_EXIT:    // Thread stopped
    ptr += qsnprintf(ptr, end-ptr, " exit_code=%d", ev->exit_code);
    break;
  case BREAKPOINT:     // Breakpoint reached
    ptr += qsnprintf(ptr, end-ptr, " hea=%a kea=%a", ev->bpt.hea, ev->bpt.kea);
    break;
  case EXCEPTION:      // Exception
    ptr += qsnprintf(ptr, end-ptr, " code=%x can_cont=%d ea=%a info=%s",
      ev->exc.code,
      ev->exc.can_cont,
      ev->exc.ea,
      ev->exc.info);
    break;
  case LIBRARY_UNLOAD: // Library unloaded
  case INFORMATION:    // User-defined information
    APPCHAR(ptr, end, ' ');
    APPEND(ptr, end, ev->info);
    break;
  default:
    break;
  }
  qsnprintf(ptr, end-ptr, " pid=%d tid=%d handled=%d",
    ev->pid,
    ev->tid,
    ev->handled);
  return buf;
}

//--------------------------------------------------------------------------
name_info_t *debmod_t::get_debug_names()
{
  return &dn_names;
}

//--------------------------------------------------------------------------
void debmod_t::clear_debug_names()
{
  if ( dn_names.addrs.empty() )
    return;

  typedef qvector<char *> charptr_vec_t;
  charptr_vec_t::iterator it_end = dn_names.names.end();
  for ( charptr_vec_t::iterator it=dn_names.names.begin();it!=it_end;++it )
    qfree(*it);

  dn_names.names.clear();
  dn_names.addrs.clear();
}

//--------------------------------------------------------------------------
void debmod_t::save_debug_name(ea_t ea, const char *name)
{
  dn_names.addrs.push_back(ea);
  dn_names.names.push_back(qstrdup(name));
}
