#ifndef __DEBUGGER_MODULE__
#define __DEBUGGER_MODULE__

//
//
//      This is the base debmod_t class definition
//      From this class all debugger code must inherite and specialize
//
//      Some OS specific functions must be implemented:
//        bool init_subsystem();
//        bool term_subsystem();
//        debmod_t *create_debug_session();
//        int create_thread(thread_cb_t thread_cb, void *context);
//        bool lock_begin();
//        bool lock_end();
//

#include <pro.h>
#include <area.hpp>
#include <idd.hpp>
#include <deque>
#include "consts.h"

extern debugger_t debugger;

struct name_info_t
{
  qvector<ea_t> addrs;
  qvector<char *> names;
};

// Very simple class to store pending events
enum queue_pos_t
{
  IN_FRONT,
  IN_BACK
};

struct eventlist_t : public std::deque<debug_event_t>
{
private:
  bool synced;
public:
  // save a pending event
  void enqueue(const debug_event_t &ev, queue_pos_t pos)
  {
    if ( pos != IN_BACK )
      push_front(ev);
    else
      push_back(ev);
  }

  // retrieve a pending event
  bool retrieve(debug_event_t *event)
  {
    if ( empty() )
      return false;
    // get the first event and return it
    *event = front();
    pop_front();
    return true;
  }
};

typedef int ioctl_handler_t(
  class rpc_engine_t *rpc,
  int fn,
  const void *buf,
  size_t size,
  void **poutbuf,
  ssize_t *poutsize);

int send_ioctl(void *ud, int fn, const void *buf, size_t size, void **poutbuf, ssize_t *poutsize);

// Main class to represent a debugger module
class debmod_t
{
private:
  char *debug_event_str(const debug_event_t *ev, char *buf, size_t bufsize);

protected:
  char input_file_path[QMAXPATH];
  qvector<exception_info_t> exceptions;
  name_info_t dn_names;

  int send_ioctl(int fn, const void *buf, size_t size, void **poutbuf, ssize_t *poutsize)
  {
    return ::send_ioctl(dmsg_ud, fn, buf, size, poutbuf, poutsize);
  }
public:
  meminfo_vec_t old_areas;
  void *dmsg_ud;
  bool debug_debugger;

  DECLARE_UD_REPORTING(msg, dmsg_ud);
  DECLARE_UD_REPORTING(warning, dmsg_ud);
  DECLARE_UD_REPORTING(error, dmsg_ud);

  // -----------------------------------
  // Constructors and destructors
  // -----------------------------------
  debmod_t();
  virtual ~debmod_t() { cleanup(); }

  // -----------------------------------
  // Debug names methods
  // -----------------------------------
  void clear_debug_names();
  name_info_t *get_debug_names();
  void save_debug_name(ea_t ea, const char *name);

  // -----------------------------------
  // Utility methods
  // -----------------------------------
  void cleanup(void);
  char *debug_event_str(const debug_event_t *ev);
  void debdeb(const char *format, ...);
  bool deberr(const char *format, ...);
  bool same_as_oldmemcfg(const meminfo_vec_t &areas);
  void save_oldmemcfg(const meminfo_vec_t &areas);

  // -----------------------------------
  // Shared methods
  // -----------------------------------
  virtual bool check_input_file_crc32(uint32 orig_crc);
  virtual const exception_info_t *find_exception(int code);
  virtual bool get_exception_name(int code, char *buf, size_t bufsize);

  // -----------------------------------
  // Methods to be implemented
  // -----------------------------------
  virtual int idaapi dbg_init(bool _debug_debugger) = 0;
  virtual void idaapi dbg_term(void) = 0;
  virtual int  idaapi dbg_process_get_info(int n,
    const char *input,
    process_info_t *info) = 0;
  virtual int  idaapi dbg_detach_process(void) = 0;
  virtual int  idaapi dbg_start_process(const char *path,
    const char *args,
    const char *startdir,
    int flags,
    const char *input_path,
    uint32 input_file_crc32) = 0;
  virtual int  idaapi dbg_get_debug_event(debug_event_t *event, bool ida_is_idle) = 0;
  virtual int  idaapi dbg_attach_process(pid_t process_id, int event_id) = 0;
  virtual int  idaapi dbg_prepare_to_pause_process(void) = 0;
  virtual int  idaapi dbg_exit_process(void) = 0;
  virtual int  idaapi dbg_continue_after_event(const debug_event_t *event) = 0;
  virtual void idaapi dbg_set_exception_info(const exception_info_t *info, int qty);
  virtual void idaapi dbg_stopped_at_debug_event(void) = 0;
  virtual int  idaapi dbg_thread_suspend(thid_t thread_id) = 0;
  virtual int  idaapi dbg_thread_continue(thid_t thread_id) = 0;
  virtual int  idaapi dbg_thread_set_step(thid_t thread_id) = 0;
  virtual int  idaapi dbg_thread_read_registers(thid_t thread_id,
    regval_t *values,
    int count) = 0;
  virtual int  idaapi dbg_thread_write_register(thid_t thread_id,
    int reg_idx,
    const regval_t *value) = 0;
  virtual int  idaapi dbg_thread_get_sreg_base(thid_t thread_id,
    int sreg_value,
    ea_t *ea) = 0;
  virtual int  idaapi map_address(ea_t ea, const regval_t *, int /* regnum */) { return ea; }
  virtual int  idaapi dbg_get_memory_info(meminfo_vec_t &areas) = 0;
  virtual ssize_t idaapi dbg_read_memory(ea_t ea, void *buffer, size_t size) = 0;
  virtual ssize_t idaapi dbg_write_memory(ea_t ea, const void *buffer, size_t size) = 0;
  virtual int  idaapi dbg_is_ok_bpt(bpttype_t type, ea_t ea, int len) = 0;
  virtual int  idaapi dbg_add_bpt(bpttype_t type, ea_t ea, int len) = 0;
  virtual int  idaapi dbg_del_bpt(ea_t ea, const uchar *orig_bytes, int len) = 0;
  virtual int  idaapi dbg_open_file(const char *file, uint32 *fsize, bool readonly) = 0;
  virtual void idaapi dbg_close_file(int fn) = 0;
  virtual ssize_t idaapi dbg_read_file(int fn, uint32 off, void *buf, size_t size) = 0;
  virtual ssize_t idaapi dbg_write_file(int fn, uint32 off, const void *buf, size_t size) = 0;
  virtual int  idaapi handle_ioctl(int /*fn*/, const void* /*buf*/, size_t /*size*/,
                                   void** /*outbuf*/, ssize_t* /*outsize*/) { return 0; }
  virtual int  idaapi get_system_specific_errno(void) const = 0; // this code must be acceptable by winerr()
};

// some functions, per OS implemented
bool init_subsystem();
bool term_subsystem();
debmod_t *create_debug_session();

// perform an action on all existing debugger modules
struct debmod_visitor_t
{
  virtual int visit(debmod_t *debmod) = 0;
};
int for_all_debuggers(debmod_visitor_t &v);


extern bool is_dll;

#endif
