#include "tcpip.h"
#include <kernwin.hpp>

static void neterr(idarpc_stream_t *irs, const char *module)
{
  int code = irs_error(irs);
  error("%s: %s", module, winerr(code));
}

//-------------------------------------------------------------------------
#if defined(__NT__)
void NT_CDECL term_sockets(void)
{
  WSACleanup();
}

//-------------------------------------------------------------------------
#ifdef __BORLANDC__
#pragma warn -8084 // Suggest parentheses to clarify precedence
#endif

bool init_irs_layer(void)
{
  WORD wVersionRequested;
  WSADATA wsaData;
  int err;

  wVersionRequested = MAKEWORD( 2, 0 );

  err = WSAStartup( wVersionRequested, &wsaData );
  if ( err != 0 )
    return false;

  atexit(term_sockets);

  /* Confirm that the WinSock DLL supports 2.0.*/
  /* Note that if the DLL supports versions greater    */
  /* than 2.0 in addition to 2.0, it will still return */
  /* 2.0 in wVersion since that is the version we      */
  /* requested.                                        */

  if ( LOBYTE( wsaData.wVersion ) != 2 ||
       HIBYTE( wsaData.wVersion ) != 0 )
    /* Tell the user that we couldn't find a usable */
    /* WinSock DLL.                                  */
    return false;

  /* The WinSock DLL is acceptable. Proceed. */
  return true;
}
#else
void term_sockets(void) {}
bool init_irs_layer(void) { return true; }
#endif

//-------------------------------------------------------------------------
void irs_term(idarpc_stream_t *irs)
{
  SOCKET s = (SOCKET)irs;
  closesocket(s);
  term_sockets();
}

//-------------------------------------------------------------------------
ssize_t irs_send(idarpc_stream_t *irs, const void *buf, size_t n)
{
  SOCKET s = (SOCKET)irs;
  return qsend(s, buf, (int)n);
}

//-------------------------------------------------------------------------
ssize_t irs_recv(idarpc_stream_t *irs, void *buf, size_t n, int timeout)
{
  if ( timeout != 0 && !irs_ready(irs, timeout) )
    return -1; // no data
  SOCKET s = (SOCKET)irs;
  return qrecv(s, buf, (int)n);
}

//-------------------------------------------------------------------------
int irs_error(idarpc_stream_t *)
{
  return get_network_error();
}

//-------------------------------------------------------------------------
int irs_ready(idarpc_stream_t *irs, int timeout)
{
  static hit_counter_t *hc = NULL;
  if ( hc == NULL )
    hc = create_hit_counter("irs_ready");
  incrementer_t inc(*hc);

  SOCKET s = (SOCKET)irs;
  int milliseconds = timeout;
  int seconds = milliseconds / 1000;
  milliseconds %= 1000;
  struct timeval tv = { seconds, milliseconds * 1000 };
  fd_set rd;
  FD_ZERO(&rd);
  FD_SET(s, &rd);
  int code = select(int(s+1),
         &rd, NULL,
         NULL,
         seconds != -1 ? &tv : NULL);
  if ( code == 0 )
    inc.failed();
  return code;
}

//--------------------------------------------------------------------------
void setup_irs(idarpc_stream_t *irs)
{
  SOCKET socket = (SOCKET)irs;
  /* Set socket options.  We try to make the port reusable and have it
	 close as fast as possible without waiting in unnecessary wait states
	 on close.
   */
  int on = 1;
  char *const ptr = (char *)&on;
  if ( setsockopt(socket, SOL_SOCKET, SO_REUSEADDR, ptr, sizeof(on)) != 0 )
    neterr(irs, "setsockopt1");

  /* Enable TCP keep alive process. */
  if ( setsockopt(socket, SOL_SOCKET, SO_KEEPALIVE, ptr, sizeof(on)) != 0 )
    neterr(irs, "setsockopt2");

  /* Speed up the interactive response. */
  if ( setsockopt(socket, IPPROTO_TCP, TCP_NODELAY, ptr, sizeof(on)) != 0 )
    neterr(irs, "setsockopt3");
}

//-------------------------------------------------------------------------
void term_server_irs(idarpc_stream_t *irs)
{
  SOCKET s = (SOCKET)irs;
  closesocket(s);
}

//-------------------------------------------------------------------------
void term_client_irs(idarpc_stream_t *irs)
{
  term_server_irs(irs);
  term_sockets();
}

//-------------------------------------------------------------------------
static in_addr name_to_addr(const char *name)
{
  in_addr addr;
  addr.s_addr = inet_addr(name);
  if ( addr.s_addr == INADDR_NONE )
  {
    struct hostent *he = gethostbyname(name);
    if ( he != NULL )
    {
#define INADDRSZ   4
//      warning("addrtype = %d addr=%08lX", he->h_addrtype, *(uint32*)he->h_addr);
      memcpy(&addr, he->h_addr, INADDRSZ);
      return addr;
    }
  }
  return addr;
}

//-------------------------------------------------------------------------
bool name_to_sockaddr(const char *name, ushort port, sockaddr_in *sa)
{
  memset(sa, 0, sizeof(sockaddr_in));
  sa->sin_family = AF_INET;
  sa->sin_port = htons(port);
  sa->sin_addr = name_to_addr(name);
  return sa->sin_addr.s_addr != INADDR_NONE;
}

//-------------------------------------------------------------------------
idarpc_stream_t *init_client_irs(const char *hostname, int port_number)
{
  if ( hostname[0] == '\0' )
  {
    warning("AUTOHIDE NONE\n"
            "Please specify the hostname in Debugger, Process options");
    return NULL;
  }

  if ( !init_irs_layer() )
  {
    warning("AUTOHIDE NONE\n"
            "Could not initialize sockets: %s", winerr(get_network_error()));
    return NULL;
  }

  SOCKET rpc_socket = socket(AF_INET, SOCK_STREAM, 0);
  if ( rpc_socket == INVALID_SOCKET )
    neterr(NULL, "socket");

  setup_irs((idarpc_stream_t*)rpc_socket);

  struct sockaddr_in sa;
  if ( !name_to_sockaddr(hostname, (ushort)port_number, &sa) )
  {
    warning("ICON ERROR\nAUTOHIDE NONE\n"
            "Could not resolve %s: %s", hostname, winerr(get_network_error()));
    return NULL;
  }

  if ( connect(rpc_socket, (sockaddr *)&sa, sizeof(sa)) == SOCKET_ERROR )
  {
    warning("ICON ERROR\nAUTOHIDE NONE\n"
            "Could not connect to %s: %s", hostname, winerr(get_network_error()));
    return NULL;
  }
  return (idarpc_stream_t*)rpc_socket;
}
