//
// This file contains utility functions that can be used with different
// win32 debugger plugins
//

#if defined(__BORLANDC__) && __BORLANDC__ < 0x560
#define _IMAGEHLP_SOURCE_
#include "dbghelp.h"
#elif !defined(UNDER_CE)
#include <dbghelp.h>
#endif

#ifdef UNICODE
#define LookupPrivilegeValue_Name "LookupPrivilegeValueW"
#else
#define LookupPrivilegeValue_Name "LookupPrivilegeValueA"
#endif


//--------------------------------------------------------------------------
//
//      DEBUGGER INTERNAL DATA
//
//--------------------------------------------------------------------------
// dynamic linking information for ToolHelp functions and new XP/2K3 debug functions
static HMODULE th_handle;

// function prototypes
typedef HANDLE (WINAPI *CreateToolhelp32Snapshot_t)(DWORD dwFlags, DWORD th32ProcessID);
typedef BOOL   (WINAPI *Process32First_t)(HANDLE hSnapshot, LPPROCESSENTRY32 lppe);
typedef BOOL   (WINAPI *Process32Next_t)(HANDLE hSnapshot, LPPROCESSENTRY32 lppe);
typedef BOOL   (WINAPI *Module32First_t)(HANDLE hSnapshot, LPMODULEENTRY32 lpme);
typedef BOOL   (WINAPI *Module32Next_t)(HANDLE hSnapshot, LPMODULEENTRY32 lpme);
typedef BOOL   (WINAPI *DebugActiveProcessStop_t)(DWORD dwProcessID);
typedef BOOL   (WINAPI *DebugBreakProcess_t)(HANDLE Process);
typedef BOOL   (WINAPI  *CloseToolhelp32Snapshot_t)(HANDLE hSnapshot);

// functions pointers
static CreateToolhelp32Snapshot_t _CreateToolhelp32Snapshot = NULL;
static Process32First_t           _Process32First           = NULL;
static Process32Next_t            _Process32Next            = NULL;
static Module32First_t            _Module32First            = NULL;
static Module32Next_t             _Module32Next             = NULL;
static CloseToolhelp32Snapshot_t  _CloseToolhelp32Snapshot  = NULL;
static DebugActiveProcessStop_t   _DebugActiveProcessStop   = NULL;
static DebugBreakProcess_t        _DebugBreakProcess        = NULL;
static bool use_DebugBreakProcess = true;

//--------------------------------------------------------------------------
//
//      TOOLHELP FUNCTIONS
//
//--------------------------------------------------------------------------
#ifndef __WINDBG_DEBUGGER_MODULE__ // windbg doesn't use toolhelp
static void term_toolhelp(void)
{
  if (th_handle != NULL)
  {
    FreeLibrary(th_handle);
    th_handle = NULL;
  }
}

#ifdef UNDER_CE
#  define KERNEL_LIB_NAME   "coredll.dll"
#  define TOOLHELP_LIB_NAME "toolhelp.dll"
#else
#  define KERNEL_LIB_NAME   "kernel32.dll"
#  define TOOLHELP_LIB_NAME "kernel32.dll"
#endif

static const TCHAR *kernel_lib_name = TEXT(KERNEL_LIB_NAME);
static const TCHAR *toolhelp_lib_name = TEXT(TOOLHELP_LIB_NAME);

//--------------------------------------------------------------------------
static bool init_toolhelp(void)
{
  // load the library
  th_handle = LoadLibrary(toolhelp_lib_name);
  if (th_handle == NULL) return false;

  // find the needed functions
  *(FARPROC*)&_CreateToolhelp32Snapshot = GetProcAddress(th_handle, TEXT("CreateToolhelp32Snapshot"));
  *(FARPROC*)&_Process32First           = GetProcAddress(th_handle, TEXT("Process32First"));
  *(FARPROC*)&_Process32Next            = GetProcAddress(th_handle, TEXT("Process32Next"));
  *(FARPROC*)&_Module32First            = GetProcAddress(th_handle, TEXT("Module32First"));
  *(FARPROC*)&_Module32Next             = GetProcAddress(th_handle, TEXT("Module32Next"));
#ifdef UNDER_CE  
  *(FARPROC*)&_CloseToolhelp32Snapshot  = GetProcAddress(th_handle, TEXT("CloseToolhelp32Snapshot"));
#endif

  bool ok = _CreateToolhelp32Snapshot != NULL
    && _Process32First != NULL
    && _Process32Next != NULL
#ifdef UNDER_CE  
    && _CloseToolhelp32Snapshot  != NULL
#endif    
    && _Module32First != NULL
    && _Module32Next != NULL;
  if (!ok)
    term_toolhelp();

  *(FARPROC*)&_DebugActiveProcessStop = GetProcAddress(th_handle, TEXT("DebugActiveProcessStop"));
  *(FARPROC*)&_DebugBreakProcess      = GetProcAddress(th_handle, TEXT("DebugBreakProcess"));
  use_DebugBreakProcess = getenv("IDA_NO_DEBUGBREAKPROCESS") == NULL;
  return ok;
}
#endif

//--------------------------------------------------------------------------
// convert Windows protection modes to IDA protection modes
uchar win_prot_to_ida_perm(DWORD protection)
{
  uchar perm = 0;

  if (protection & PAGE_READONLY)          perm |= SEGPERM_READ;
  if (protection & PAGE_READWRITE)         perm |= SEGPERM_READ | SEGPERM_WRITE;
  if (protection & PAGE_WRITECOPY)         perm |= SEGPERM_READ | SEGPERM_WRITE;
  if (protection & PAGE_EXECUTE)           perm |=                                SEGPERM_EXEC;
  if (protection & PAGE_EXECUTE_READ)      perm |= SEGPERM_READ                 | SEGPERM_EXEC;
  if (protection & PAGE_EXECUTE_READWRITE) perm |= SEGPERM_READ | SEGPERM_WRITE | SEGPERM_EXEC;
  if (protection & PAGE_EXECUTE_WRITECOPY) perm |= SEGPERM_READ | SEGPERM_WRITE | SEGPERM_EXEC;

  return perm;
}

//--------------------------------------------------------------------------
//
//      WINDOWS VERSION
//
//--------------------------------------------------------------------------
static OSVERSIONINFO OSVersionInfo;

//--------------------------------------------------------------------------
static bool get_windows_version(void)
{
  OSVersionInfo.dwOSVersionInfoSize = sizeof(OSVersionInfo);
  return GetVersionEx(&OSVersionInfo) != NULL;
}

//--------------------------------------------------------------------------
bool is_NT(void)
{
  return OSVersionInfo.dwPlatformId == VER_PLATFORM_WIN32_NT;
}

//--------------------------------------------------------------------------
bool is_DW32(void)
{
  return OSVersionInfo.dwPlatformId == 3;
}

//--------------------------------------------------------------------------
bool is_2K(void)
{
  return OSVersionInfo.dwMajorVersion >= 5;
}

//--------------------------------------------------------------------------
//
//      DEBUG PRIVILEGE
//
//--------------------------------------------------------------------------
// dynamic linking information for Advapi functions
static HMODULE hAdvapi32;
// function prototypes
typedef BOOL (WINAPI *OpenProcessToken_t)(HANDLE ProcessHandle, DWORD DesiredAccess, PHANDLE TokenHandle);
typedef BOOL (WINAPI *LookupPrivilegeValue_t)(LPCTSTR lpSystemName, LPCTSTR lpName, PLUID lpLuid);
typedef BOOL (WINAPI *AdjustTokenPrivileges_t)(HANDLE TokenHandle, BOOL DisableAllPrivileges, PTOKEN_PRIVILEGES NewState, DWORD BufferLength, PTOKEN_PRIVILEGES PreviousState, PDWORD ReturnLength);
// functions pointers
OpenProcessToken_t      _OpenProcessToken      = NULL;
LookupPrivilegeValue_t  _LookupPrivilegeValue  = NULL;
AdjustTokenPrivileges_t _AdjustTokenPrivileges = NULL;

//--------------------------------------------------------------------------
static void term_advapi32(void)
{
  if (hAdvapi32 != NULL)
  {
    FreeLibrary(hAdvapi32);
    hAdvapi32 = NULL;
  }
}

//--------------------------------------------------------------------------
static bool init_advapi32(void)
{
  // load the library
  hAdvapi32 = LoadLibrary(TEXT("advapi32.dll"));
  if ( hAdvapi32 == NULL )
    return false;

  // find the needed functions
  *(FARPROC*)&_OpenProcessToken       = GetProcAddress(hAdvapi32, TEXT("OpenProcessToken"));
  *(FARPROC*)&_LookupPrivilegeValue   = GetProcAddress(hAdvapi32, TEXT(LookupPrivilegeValue_Name));
  *(FARPROC*)&_AdjustTokenPrivileges  = GetProcAddress(hAdvapi32, TEXT("AdjustTokenPrivileges"));

  bool ok = _OpenProcessToken      != NULL
         && _LookupPrivilegeValue  != NULL
         && _AdjustTokenPrivileges != NULL;
  if (!ok)
  {
    int code = GetLastError();
    term_advapi32();
    SetLastError(code);
  }
  return ok;
}


//--------------------------------------------------------------------------
// based on code from:
// http://support.microsoft.com/support/kb/articles/Q131/0/65.asp
bool enable_privilege(LPCTSTR privilege, bool enable)
{
  if ( !is_NT() ) // no privileges on 9X/ME
    return true;

  if ( !init_advapi32() )
    return false;

  HANDLE hToken;
  bool ok = false;
  int code = ERROR_SUCCESS;
  if (_OpenProcessToken(GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, &hToken))
  {
    LUID luid;
    if(_LookupPrivilegeValue(NULL, privilege, &luid))
    {
      TOKEN_PRIVILEGES tp;
      memset(&tp, 0, sizeof(tp));
      tp.PrivilegeCount           = 1;
      tp.Privileges[0].Luid       = luid;
      tp.Privileges[0].Attributes = enable ? SE_PRIVILEGE_ENABLED : 0;
      _AdjustTokenPrivileges(hToken, FALSE, &tp, sizeof(tp), NULL, NULL);
      code = GetLastError();
      ok = (code == ERROR_SUCCESS);
    }
    CloseHandle(hToken);
  }
  term_advapi32();
  if ( !ok )
    SetLastError(code);
  return ok;
}

//--------------------------------------------------------------------------
//
//      MINI DUMP
//
//--------------------------------------------------------------------------
#ifndef UNDER_CE
// dynamic linking information for DbgHelp functions
static HMODULE hDbgHelp;

#if defined(__BORLANDC__) && __BORLANDC__ < 0x560
#include "dbghlp_bor55.h"
#endif

// function prototypes
typedef BOOL (WINAPI *MiniDumpWriteDump_t)(HANDLE hProcess, DWORD ProcessId, HANDLE hFile, MINIDUMP_TYPE DumpType, PMINIDUMP_EXCEPTION_INFORMATION ExceptionParam, PMINIDUMP_USER_STREAM_INFORMATION UserStreamParam, PMINIDUMP_CALLBACK_INFORMATION CallbackParam);
// functions pointers
MiniDumpWriteDump_t _MiniDumpWriteDump = NULL;

static void term_dbghelp(void)
{
  if (hDbgHelp != NULL)
  {
    FreeLibrary(hDbgHelp);
    hDbgHelp = NULL;
  }
}

static bool init_dbghelp(void)
{
  // load the library
  hDbgHelp = LoadLibrary(TEXT("dbghelp.dll"));
  if (hDbgHelp == NULL) return false;

  // find the needed functions
  *(FARPROC*)&_MiniDumpWriteDump = GetProcAddress(hDbgHelp, "MiniDumpWriteDump");

  bool ok = (_MiniDumpWriteDump != NULL);
  if (!ok)
    term_dbghelp();
  return ok;
}

//--------------------------------------------------------------------------
bool create_mini_dump(void)
{
  if ( !get_windows_version()
    || !init_dbghelp()
    || !enable_privilege(SE_DEBUG_NAME, true))
      return false;

  bool ret = false;
  DWORD pid = GetCurrentProcessId();

  // get the date
  SYSTEMTIME local_time;
  GetLocalTime(&local_time);
  // dump file name is composed like this (to avoid overwriting): ida-YYYYMMDD-HHMMSS-PID.dmp
  char name[MAXSTR];
  qsnprintf(name, sizeof(name), "ida-%04d%02d%02d-%02d%02d%02d-%ld.dmp",
    local_time.wYear, local_time.wMonth,  local_time.wDay,
    local_time.wHour, local_time.wMinute, local_time.wSecond,
    pid);

  HANDLE hDump = CreateFile(name, GENERIC_READ|GENERIC_WRITE, FILE_SHARE_WRITE|FILE_SHARE_READ, 0, CREATE_NEW, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hDump != INVALID_HANDLE_VALUE)
  {
    int type =
      MiniDumpNormal                         // Include just the information necessary to capture stack traces for all existing threads in a process.
     | MiniDumpWithDataSegs                   // Include the data sections from all loaded modules. This results in the inclusion of global variables, which can make the minidump file significantly larger. For per-module control, use the ModuleWriteDataSeg enumeration value from MODULE_WRITE_FLAGS.
//     | MiniDumpWithFullMemory                 // Include all accessible memory in the process. The raw memory data is included at the end, so that the initial structures can be mapped directly without the raw memory information. This option can result in a very large file.
     | MiniDumpWithHandleData                 // Include high-level information about the operating system handles that are active when the minidump is made. Windows Me/98/95:  This value is not supported.
//     | MiniDumpFilterMemory                   // Stack and backing store memory written to the minidump file should be filtered to remove all but the pointer values necessary to reconstruct a stack trace. Typically, this removes any private information.
//     | MiniDumpScanMemory                     // Stack and backing store memory should be scanned for pointer references to modules in the module list. If a module is referenced by stack or backing store memory, the ModuleWriteFlags member of the MINIDUMP_CALLBACK_OUTPUT structure is set to ModuleReferencedByMemory.
//     | MiniDumpWithUnloadedModules            // Include information from the list of modules that were recently unloaded, if this information is maintained by the operating system. DbgHelp 5.1 and earlier:  This value is not supported.
//     | MiniDumpWithIndirectlyReferencedMemory // Include pages with data referenced by locals or other stack memory. This option can increase the size of the minidump file significantly. DbgHelp 5.1 and earlier:  This value is not supported.
//     | MiniDumpFilterModulePaths              // Filter module paths for information such as user names or important directories. This option may prevent the system from locating the image file and should be used only in special situations. DbgHelp 5.1 and earlier:  This value is not supported.
//     | MiniDumpWithProcessThreadData          // Include complete per-process and per-thread information from the operating system. DbgHelp 5.1 and earlier:  This value is not supported.
//     | MiniDumpWithPrivateReadWriteMemory     // Scan the virtual address space for other types of memory to be included. DbgHelp 5.1 and earlier:  This value is not supported.
//     | MiniDumpWithoutOptionalData            // Reduce the data that is dumped by eliminating memory regions that are not essential to meet criteria specified for the dump. This can avoid dumping memory that may contain data that is private to the user. However, it is not a guarantee that no private information will be present. DbgHelp 6.1 and earlier:  This value is not supported.
//     | MiniDumpWithFullMemoryInfo             // Include memory region information. For more information, see MINIDUMP_MEMORY_INFO_LIST. DbgHelp 6.1 and earlier:  This value is not supported.
//     | MiniDumpWithThreadInfo                 // Include thread state information. For more information, see MINIDUMP_THREAD_INFO_LIST. DbgHelp 6.1 and earlier:  This value is not supported.
//     | MiniDumpWithCodeSegs                   // Include all code and code-related sections from loaded modules to capture executable content. For per-module control, use the ModuleWriteCodeSegs enumeration value from MODULE_WRITE_FLAGS.
     ;
    const char *flags_str = getenv("IDA_MINIDUMP");
    if ( flags_str != NULL )
    {
      int type2;
      if ( sscanf(flags_str, "%x", &type2) == 1 )
        type = type2;
    }
//    MINIDUMP_EXCEPTION_INFORMATION exceptions = {0};
//    exceptions.ThreadId          = GetCurrentThreadId();
//    exceptions.ExceptionPointers = GetExceptionInformation();
//    exceptions.ClientPointers    = true;
    ret = _MiniDumpWriteDump(GetCurrentProcess(), pid, hDump, _MINIDUMP_TYPE(type), /* &exceptions */ NULL, NULL, NULL);
//    if ( ret )
//      info(
//           "IDA has encountered a problem\n"
//           "A mini dump file has been created in '%s'\n"
//           "Please send it to <support@hex-rays.com>"
//           name);
  }

  enable_privilege(SE_DEBUG_NAME, false);
  term_dbghelp();
  return ret;
}
#endif // UNDER_CE
