#include "wince_debmod.h"

#include "xscale/Breakpoint.h"
#include "rpc_hlp.h"
#include "rpc_server.h"

uchar wince_debmod_t::bpt_code[ARM_BPT_SIZE] = ARM_BPT_CODE;

int wince_debmod_t::kdstub_loaded = -1;

//--------------------------------------------------------------------------
// The only reason why we load and use kernel stub is to ignore
// hardware breakpoints in the foreign applications. If the user puts
// a breakpoint in a shared DLL, we don't want other applications
// to be aware of it - exceptions in these applications should be ignored.
bool load_kdstub(void)
{
  bool ok = false;

  if (g_global_server == NULL)
    return false;

  __try
  {
    static const char stubname[] = "\\Windows\\ida_kdstub.dll";
    // 01234567 8
    ok = g_global_server->rpc_sync_stub(stubname, &stubname[9]);
    if ( !ok )
    {
      g_global_server->dwarning("Failed to synchronize kernel debugger stub");
    }
    else
    {
      wchar_t wname[80];
      cwstr(wname, stubname, qnumber(wname));
      ok = AttachDebugger(wname);
      if ( !ok )
      {
        //Likely error codes:
        //ERROR_FILE_NOT_FOUND (2) - if LoadKernelLibrary failed
        //  This may happen if the DLL is not found or cannot be
        //  loaded. The DLL will fail to load if it is for a
        //  wrong platform or if it is linked to any other DLL.
        //ERROR_INVALID_PARAMETER (87) -if ConnectDebugger failed
        //  This may happen if IOCTL_DBG_INIT was not called
        //  by the DLL initialization routing or if some module
        //  in the system is marked as non-debuggable
        int code = GetLastError();
        g_global_server->dwarning("Failed to attach kernel debugger stub: %s", winerr(code));
      }
      else
      {
        g_global_server->dmsg("Successfully attached kernel debugger stub\n");
        //          win420_module_t wm;
        //          if ( find_module_by_name("ida_kdstub", (wince_module_t*)&wm) )
        //            msg("%x: kernel stub\n", int(wm.BasePtr)+0x1000);
      }
    }
  }
  __except( EXCEPTION_EXECUTE_HANDLER )
  {
  }
  return ok;
}


//--------------------------------------------------------------------------
wince_debmod_t::wince_debmod_t()
{
  is_xscale = false;
  databpts[0] = databpts[1] = BADADDR;
  codebpts[0] = codebpts[1] = BADADDR;
  dbcon = 0;
  kdstub_loaded = -1; // not tried to load yet
}


//--------------------------------------------------------------------------
int idaapi wince_debmod_t::dbg_is_ok_bpt(bpttype_t type, ea_t ea, int len)
{
  if ( type == BPT_SOFT )
    return BPT_OK;
  // for some reason hardware instructon breakpoints do not work
  if ( type == BPT_EXEC )
    return BPT_BAD_TYPE;
  if ( len > 4 )
    return BPT_BAD_LEN;
  bool ok = type == BPT_EXEC
    ? (codebpts[0] == BADADDR || codebpts[1] == BADADDR)
    : (databpts[0] == BADADDR || databpts[1] == BADADDR);
  return ok ? BPT_OK : BPT_TOO_MANY;
}

//--------------------------------------------------------------------------
// Set hardware breakpoint for one thread
bool wince_debmod_t::set_hwbpts(HANDLE hThread)
{
  if ( is_xscale )
  {
    uint32 d0 = databpts[0] == BADADDR ? 0 : s0tops(databpts[0]);
    uint32 d1 = databpts[1] == BADADDR ? 0 : s0tops(databpts[1]);
    uint32 c0 = codebpts[0] == BADADDR ? 0 : s0tops(codebpts[0] | 1);
    uint32 c1 = codebpts[1] == BADADDR ? 0 : s0tops(codebpts[1] | 1);
    if ( active_hwbpts() )
    {
      SetDebugControlAndStatus(DEF_GlobalDebugEnabled, DEF_GlobalDebugEnabled);
      SetDataBreakPoint(d0, d1, dbcon);
      SetCodeBreakPoint(c0, c1);
    }
    else
    {
      disable_hwbpts();
    }
    //    msg("set bpts: dcsr=%x code=%a %a data=%a %a dbcon=%a\n",
    //                SetDebugControlAndStatus(0, 0),
    //                c0, c1, d0, d1, dbcon);
    return true;
  }
  return false;
}

//--------------------------------------------------------------------------
bool wince_debmod_t::add_hwbpt(bpttype_t type, ea_t ea, int len)
{
  //  msg("add_hwbpt %d %a %d\n", type, ea, len);
  if ( !is_xscale || len > 4 )
    return false;

  if ( kdstub_loaded == -1 )
    kdstub_loaded = load_kdstub();
  if ( !kdstub_loaded )
    return false;

  if ( type == BPT_EXEC )
  {
    if ( codebpts[0] != BADADDR && codebpts[1] != BADADDR )
      return false;
    int slot = codebpts[0] != BADADDR;
    codebpts[slot] = ea;
  }
  else
  {
    if ( databpts[0] != BADADDR && databpts[1] != BADADDR )
      return false;
    int slot = databpts[0] != BADADDR;
    int bits;
    switch ( type )
    {
    case BPT_WRITE:
      bits = 1;               // store only
      break;
    case BPT_RDWR:
      bits = 2;               // load/store
      break;
      //      BPT_READ:               // load only
      //        bits = 3;
      //        break;
    default:
      return false;
    }
    databpts[slot] = ea;
    dbcon |= bits << (slot*2);
  }
  return set_hwbpts(NULL);
}

//--------------------------------------------------------------------------
bool wince_debmod_t::del_hwbpt(ea_t ea)
{
  //  msg("del_hwbpt %a\n", ea);
  if ( databpts[0] == ea )
  {
    databpts[0] = BADADDR;
    dbcon &= ~3;
  }
  else if ( databpts[1] == ea )
  {
    databpts[1] = BADADDR;
    dbcon &= ~(3<<2);
  }
  else if ( codebpts[0] == ea )
  {
    codebpts[0] = BADADDR;
  }
  else if ( codebpts[1] == ea )
  {
    codebpts[1] = BADADDR;
  }
  else
  {
    return false;
  }
  return set_hwbpts(NULL);
}

//--------------------------------------------------------------------------
ea_t wince_debmod_t::is_hwbpt_triggered(thid_t id)
{
  if ( is_xscale )
  {
    uint32 dcsr = SetDebugControlAndStatus(0, 0);
    int moe = (dcsr >> 2) & 7;  // method of entry (exception reason)
    //    msg("moe=%d\n", moe);
    switch ( moe )
    {
    case 1: // Instruction Breakpoint Hit
    case 2: // Data Breakpoint Hit
      {
        SetDebugControlAndStatus(0, 7<<2); // clean moe
        CONTEXT Context;
        Context.ContextFlags = CONTEXT_CONTROL;
        HANDLE h = get_thread_handle(id);
        if ( GetThreadContext(h, &Context) )
        {
          ea_t ea = s0tops(Context.Pc);
          if ( s0tops(codebpts[0]) == ea || s0tops(codebpts[1]) == ea )
          {
            //              msg("HARDWARE CODE BREAKPOINT!\n");
            return ea;
          }
          // This is a data breakpoint
          // Set PC to the next instruction since the data bpts always occur
          // AFTER the instruction
#define THUMB_STATE 0x0020
          Context.Pc += (Context.Psr & THUMB_STATE)? 2 : 4;
          SetThreadContext(h, &Context);
        }
        // FIXME: determine which data bpt really caused the exception
        // Currently we just return the first active bpt
        return databpts[0] != BADADDR ? databpts[0] : databpts[1];
      }
    case 0: // Processor Reset
    case 3: // BKPT Instruction Executed
    case 4: // External Debug Event (JTAG Debug Break or SOC Debug Break)
    case 5: // Vector Trap Occurred
    case 6: // Trace Buffer Full Break
    case 7: // Reserved
      break;
    }
  }
  return BADADDR;
}

//--------------------------------------------------------------------------
void wince_debmod_t::cleanup_hwbpts()
{
  databpts[0] = BADADDR;
  databpts[1] = BADADDR;
  codebpts[0] = BADADDR;
  codebpts[1] = BADADDR;
  dbcon = 0;
  // disable all bpts
  if ( is_xscale )
    disable_hwbpts();
}

//--------------------------------------------------------------------------
void wince_debmod_t::disable_hwbpts()
{
  if ( is_xscale )
  {
    SetDebugControlAndStatus(0, ~(7<<2)); // preserve moe
    SetDataBreakPoint(0, 0, 0);
    SetCodeBreakPoint(0, 0);
  }
}

//--------------------------------------------------------------------------
void wince_debmod_t::enable_hwbpts()
{
  set_hwbpts(NULL);
}

/*
//--------------------------------------------------------------------------
ea_t wince_debmod_t::pstos0(ea_t ea)        // map process slot to slot 0
{
  if ( (ea & 0xFE000000) == slot ) // redirect our process addresses
    ea  &= ~0xFE000000;            // to slot 0
  return ea;
}

//--------------------------------------------------------------------------
ea_t wince_debmod_t::s0tops(ea_t ea)        // map slot 0 to the process slot
{
  if ( (ea & 0xFE000000) == 0 )
    ea |= slot;
  return ea;
}
*/