#include "mac_debmod.h"
#include "consts.h"
#include <sys/utsname.h>
#include <mach/mach_vm.h>
#ifndef __arm__
#include <crt_externs.h>
#endif
#include "../../ldr/mach-o/common.h"

#ifdef __arm__
#define THREAD_STATE_NONE 5
#elif defined (__i386__) || defined(__x86_64__)
#define THREAD_STATE_NONE 13
#else
#error unknown platform
#endif

//#define DEBUG_MAC_DEBUGGER
#define local static  // static is used for static data items

#ifdef __arm__
#define BPT_CODE_SIZE ARM_BPT_SIZE
static const uchar dyld_opcode[BPT_CODE_SIZE] = { 0x1E, 0xFF, 0x2F, 0xE1 };
inline void cleanup_hwbpts(void) {}
#else
#define BPT_CODE_SIZE X86_BPT_SIZE
static const uchar dyld_opcode[BPT_CODE_SIZE] = { 0x55 };
#endif

mac_debmod_t::mac_debmod_t()
{
#ifdef __arm__
  // use iphone specific bpt code
  static const uchar bpt[4] = { 0x70, 0x00, 0x20, 0xE1 };
  bpt_code = bytevec_t(bpt, 4);
#endif
  exc_port = MACH_PORT_NULL;
  struct utsname name;
  is_leopard = false;
  if ( uname(&name) == 0 )
  {
//    msg("version=%s release=%s\n", name.version, name.release);
    // release: 9.2.2
    int major, minor, build;
    if ( sscanf(name.release, "%d.%d.%d", &major, &minor, &build) == 3 )
    {
//      msg("VER=%d.%d.%d\n", major, minor, build);
      is_leopard = major >= 9;
    }
  }
//  msg("is_leopard: %d\n", is_leopard);
}

mac_debmod_t::~mac_debmod_t()
{
}

local asize_t calc_image_size(const char *fname, ea_t *expected_base);

extern boolean_t exc_server(
                mach_msg_header_t *InHeadP,
                mach_msg_header_t *OutHeadP);

#define COMPLAIN_IF_FAILED(name)                        \
      if ( err != KERN_SUCCESS )                        \
        msg(name ": %s\n", mach_error_string(err))

//--------------------------------------------------------------------------
local const char *get_ptrace_name(int request)
{
  switch ( request )
  {
    case PT_TRACE_ME:    return "PT_TRACE_ME";    /* child declares it's being traced */
    case PT_READ_I:      return "PT_READ_I";      /* read word in child's I space */
    case PT_READ_D:      return "PT_READ_D";      /* read word in child's D space */
    case PT_READ_U:      return "PT_READ_U";      /* read word in child's user structure */
    case PT_WRITE_I:     return "PT_WRITE_I";     /* write word in child's I space */
    case PT_WRITE_D:     return "PT_WRITE_D";     /* write word in child's D space */
    case PT_WRITE_U:     return "PT_WRITE_U";     /* write word in child's user structure */
    case PT_CONTINUE:    return "PT_CONTINUE";    /* continue the child */
    case PT_KILL:        return "PT_KILL";        /* kill the child process */
    case PT_STEP:        return "PT_STEP";        /* single step the child */
    case PT_ATTACH:      return "PT_ATTACH";      /* trace some running process */
    case PT_DETACH:      return "PT_DETACH";      /* stop tracing a process */
    case PT_SIGEXC:      return "PT_SIGEXC";      /* signals as exceptions for current_proc */
    case PT_THUPDATE:    return "PT_THUPDATE";    /* signal for thread# */
    case PT_ATTACHEXC:   return "PT_ATTACHEXC";   /* attach to running process with signal exception */
    case PT_FORCEQUOTA:  return "PT_FORCEQUOTA";  /* Enforce quota for root */
    case PT_DENY_ATTACH: return "PT_DENY_ATTACH";
  }
  return "?";
}

//--------------------------------------------------------------------------
int32 mac_debmod_t::qptrace(int request, pid_t pid, caddr_t addr, int data)
{
  int32 code = ptrace(request, pid, addr, data);
  int saved_errno = errno;
//  if ( (request == PT_CONTINUE || request == PT_STEP) && int(addr) == 1 )
//    addr = (caddr_t)get_ip(pid);
  debdeb("%s(%u, 0x%p, 0x%X) => 0x%X", get_ptrace_name(request), pid, addr, data, code);
  if ( code == -1 )
    deberr("");
  else
    debdeb("\n");
  errno = saved_errno;
  return code;
}

//--------------------------------------------------------------------------
ida_thread_info_t *mac_debmod_t::get_thread(thid_t tid)
{
  threads_t::iterator p = threads.find(tid);
  if ( p == threads.end() )
    return NULL;
  return &p->second;
}

//--------------------------------------------------------------------------
uval_t mac_debmod_t::get_dr(thid_t tid, int idx)
{
#if !defined (__i386__) && !defined(__x86_64__)
  return 0;
#else
  machine_debug_state_t dr_regs;

#ifndef __i386__
#define dr0 __dr0
#define dr1 __dr1
#define dr2 __dr2
#define dr3 __dr3
#define dr4 __dr4
#define dr5 __dr5
#define dr6 __dr6
#define dr7 __dr7
#endif

  if ( !get_debug_state(tid, &dr_regs) )
    return 0;

  switch ( idx )
  {
    case 0:
      return dr_regs.dr0;
    case 1:
      return dr_regs.dr1;
    case 2:
      return dr_regs.dr2;
    case 3:
      return dr_regs.dr3;
    case 4:
      return dr_regs.dr4;
    case 5:
      return dr_regs.dr5;
    case 6:
      return dr_regs.dr6;
    case 7:
      return dr_regs.dr7;
  }
  return 0;
#endif
}

//--------------------------------------------------------------------------
bool mac_debmod_t::set_dr(thid_t tid, int idx, uval_t value)
{
#if !defined (__i386__) && !defined(__x86_64__)
  return false;
#else
  machine_debug_state_t dr_regs;

  if ( !get_debug_state(tid, &dr_regs) )
    return false;

  switch ( idx )
  {
    case 0:
      dr_regs.dr0 = value;
      break;
    case 1:
      dr_regs.dr1 = value;
      break;
    case 2:
      dr_regs.dr2 = value;
      break;
    case 3:
      dr_regs.dr3 = value;
      break;
    case 4:
      dr_regs.dr4 = value;
      break;
    case 5:
      dr_regs.dr5 = value;
      break;
    case 6:
      dr_regs.dr6 = value;
      break;
    case 7:
      dr_regs.dr7 = value;
      break;
  }

  return set_debug_state(tid, &dr_regs);
#undef ds
#undef dr0
#undef dr1
#undef dr2
#undef dr3
#undef dr4
#undef dr5
#undef dr6
#undef dr7
#endif
}

//--------------------------------------------------------------------------
ea_t mac_debmod_t::get_ip(thid_t tid)
{
  machine_thread_state_t state;
  if ( !get_thread_state(tid, &state) )
    return BADADDR;
#ifdef __arm__
  return state.__pc;
#else
  return state.__eip;
#endif
}

//--------------------------------------------------------------------------
local kern_return_t qthread_setsinglestep(ida_thread_info_t &ti)
{
#ifdef __arm__
  QASSERT(!ti.single_step);
  return KERN_SUCCESS;
#else

  machine_thread_state_t cpu;
  mach_port_t port = ti.port;
  mach_msg_type_number_t stateCount = IDA_THREAD_STATE_COUNT;
  kern_return_t err = thread_get_state(
                        port,
                        IDA_THREAD_STATE,
                        (thread_state_t)&cpu,
                        &stateCount);
  QASSERT(stateCount == IDA_THREAD_STATE_COUNT);
//  COMPLAIN_IF_FAILED("thread_get_state");
  if ( err != KERN_SUCCESS )
    return err;

  ti.asked_step = ti.single_step;
  int bit = ti.single_step ? EFLAGS_TRAP_FLAG : 0;
  if ( ((cpu.__eflags ^ bit) & EFLAGS_TRAP_FLAG) == 0 )
    return KERN_SUCCESS;

  if ( ti.single_step )
    cpu.__eflags |= EFLAGS_TRAP_FLAG;
  else
    cpu.__eflags &= ~EFLAGS_TRAP_FLAG;

  err = thread_set_state(port,
                         IDA_THREAD_STATE,
                         (thread_state_t)&cpu,
                         stateCount);
  QASSERT(stateCount == IDA_THREAD_STATE_COUNT);
//  COMPLAIN_IF_FAILED("thread_set_state");
  return err;
#endif
}

//--------------------------------------------------------------------------
local pid_t qwait(int *status, bool hang)
{
  int flags = hang ? 0 : WNOHANG;
  pid_t pid = waitpid(-1, status, flags);
/*  if ( (pid == -1 || pid == 0) && errno == ECHILD )
  {
    pid = waitpid(-1, status, flags | __WCLONE);
    if ( pid != -1 && pid != 0 )
      debdeb("------------ __WCLONE %d\n", pid);
  }*/
  return pid;
}

void my_mach_msg_t::display(const char *header)
{
#ifdef DEBUG_MAC_DEBUGGER
  msg("%s\n", header);
  msg("         msgh_bits       : 0x%x\n", hdr.msgh_bits);
  msg("         msgh_size       : 0x%x\n", hdr.msgh_size);
  msg("         msgh_remote_port: %d\n", hdr.msgh_remote_port);
  msg("         msgh_local_port : %d\n", hdr.msgh_local_port);
  msg("         msgh_reserved   : %d\n", hdr.msgh_reserved);
  msg("         msgh_id         : 0x%x\n", hdr.msgh_id);
  if ( hdr.msgh_size > 24 )
  {
    const uint32 *buf = ((uint32 *) this) + 6;
    msg("         data            :");
    int cnt = hdr.msgh_size / 4 - 6;
    for ( int i=0; i < cnt; i++ )
      msg(" %08x", buf[i]);
    msg("\n");
  }
#endif
}

// this function won't be called but is declared to avoid linker complaints
kern_return_t catch_exception_raise_state(
        mach_port_t exception_port,
        exception_type_t exception,
        const exception_data_t code,
        mach_msg_type_number_t codeCnt,
        int *flavor,
        const thread_state_t old_state,
        mach_msg_type_number_t old_stateCnt,
        thread_state_t new_state,
        mach_msg_type_number_t *new_stateCnt)
{
  return KERN_FAILURE;
}

// this function won't be called but is declared to avoid linker complaints
kern_return_t catch_exception_raise_state_identity(
        mach_port_t exception_port,
        mach_port_t thread,
        mach_port_t task,
        exception_type_t exception,
        exception_data_t code,
        mach_msg_type_number_t codeCnt,
        int *flavor,
        thread_state_t old_state,
        mach_msg_type_number_t old_stateCnt,
        thread_state_t new_state,
        mach_msg_type_number_t *new_stateCnt)
{
  return KERN_FAILURE;
}

// this function will be called by exc_server()
// we use exc_server() for 2 things:
//      - to decode mach message and extract exception information
//      - to actually handle the exception when we resume execution

static bool parse_mach_message;
static bool mask_exception;
static mach_exception_info_t local_exinf;

kern_return_t catch_exception_raise(mach_port_t exception_port,
                      mach_port_t thread,
                      mach_port_t task,
                      exception_type_t exception,
                      exception_data_t code_vector,
                      mach_msg_type_number_t code_count)
{
  if ( parse_mach_message )
  {
    local_exinf.task_port      = task;
    local_exinf.thread_port    = thread;
    local_exinf.exception_type = exception;
    local_exinf.exception_data = code_vector;
    local_exinf.data_count     = code_count;
    return KERN_SUCCESS;
  }

  // handle the exception for real
  if ( mask_exception )
    return KERN_SUCCESS;

  return KERN_FAILURE;
}

//--------------------------------------------------------------------------
// event->tid is filled upon entry
// returns true: created a new event in 'event'
bool mac_debmod_t::handle_signal(
        int code,
        debug_event_t *event,
        block_type_t block,
        const my_mach_msg_t *excmsg)
{
  ida_thread_info_t *ti = get_thread(event->tid);
  if ( ti == NULL )
  { // there is an improbable race condition when a thread gets created just after
    // last call to update_threads(). check it once more
    update_threads();
    ti = get_thread(event->tid);
  }
  QASSERT(ti != NULL);

  ti->block = block;
  if ( block == bl_exception )
    ti->excmsg = *excmsg;

  event->pid          = pid;
  event->handled      = false;
  event->ea           = get_ip(event->tid);
  event->eid          = EXCEPTION;
  event->exc.code     = code;
  event->exc.can_cont = true;
  event->exc.ea       = BADADDR;

  if ( code == SIGSTOP )
  {
    if ( ti->pending_sigstop )
    {
      debdeb("got pending SIGSTOP, good!\n");
      ti->pending_sigstop = false;
      return false;
    }
    if ( run_state == rs_pausing )
    {
      debdeb("successfully paused the process, good!\n");
      run_state = rs_running;
      event->eid = NO_EVENT;
    }
  }
  if ( event->eid == EXCEPTION )
  {
    bool suspend;
    const exception_info_t *ei = find_exception(code);
    if ( ei != NULL )
    {
      qsnprintf(event->exc.info, sizeof(event->exc.info), "got %s signal (%s)", ei->name.c_str(), ei->desc.c_str());
      suspend = ei->break_on();
      event->handled = ei->handle();
      if ( code == SIGKILL && run_state >= rs_exiting )
      {
        event->handled = false;
        suspend = false;
      }
    }
    else
    {
      qsnprintf(event->exc.info, sizeof(event->exc.info), "got unknown signal #%d", code);
      suspend = true;
    }
    if ( event->exc.code == SIGTRAP )
    {
      if ( ti->asked_step )
      {
        event->eid = STEP;
        code = 0;
      }
      else
      {
        ea_t bpt_ea = event->ea;
        bool hwbpt_found = false;
#if defined (__i386__) || defined(__x86_64__)
        uval_t dr6 = get_dr(event->tid, 6);
        for ( int i=0; i < MAX_BPT; i++ )
        {
          if ( dr6 & (1<<i) )  // Hardware breakpoint 'i'
          {
            if ( hwbpt_ea[i] == get_dr(event->tid, i) )
            {
              event->eid     = BREAKPOINT;
              event->bpt.hea = hwbpt_ea[i];
              event->bpt.kea = BADADDR;
              set_dr(event->tid, 6, 0); // Clear the status bits
              hwbpt_found = true;
              break;
            }
          }
        }
        // x86 returns EIP pointing to the next byte after CC. Take it into account:
        bpt_ea--;
#endif
        if ( !hwbpt_found && bpts.find(bpt_ea) != bpts.end() )
        {
          event->eid     = BREAKPOINT;
          event->bpt.hea = BADADDR;
          event->bpt.kea = BADADDR;
          event->ea = bpt_ea;
        }
        if ( event->eid == BREAKPOINT )
          code = 0;
      }
    }
    if ( event->handled )
      code = 0;
    ti->child_signum = code;
    if ( !suspend && event->eid == EXCEPTION )
    {
      my_resume_thread(*ti);
      return false;
    }
  }
  return true;
}

//--------------------------------------------------------------------------
bool mac_debmod_t::check_for_exception(
        int timeout,
        mach_exception_info_t *exinf,
        my_mach_msg_t *excmsg)
{
  if ( run_state > rs_exiting )
    return false;

  int flags = MACH_RCV_MSG;
  if ( timeout != -1 )
    flags |= MACH_RCV_TIMEOUT;
  else
    timeout = MACH_MSG_TIMEOUT_NONE;

//  msg("check for exception, timeout %d, runstate=%d\n", timeout, run_state);

  kern_return_t err = mach_msg(&excmsg->hdr,
                               flags,
                               0,               // send size
                               sizeof(my_mach_msg_t),
                               exc_port,
                               timeout,         // timeout
                               MACH_PORT_NULL); // notify port
  if ( err != MACH_MSG_SUCCESS )
    return false;
  if ( excmsg->hdr.msgh_remote_port == -1 ) // remote task alive?
    return false;
  task_suspend(task);
  excmsg->display("received an exception, details:");

  lock_begin();
  {
    my_mach_msg_t reply_msg;
    parse_mach_message = true;
    memset(&local_exinf, 0, sizeof(local_exinf));
    bool ok = exc_server(&excmsg->hdr, &reply_msg.hdr);
    QASSERT(ok);
    *exinf = local_exinf;
  }
  lock_end();
  return true;
}

//--------------------------------------------------------------------------
bool mac_debmod_t::my_resume_thread(ida_thread_info_t &ti)
{
  bool ok = true;
  // setsinglestep may fail after kill(), ignore the return code
  qthread_setsinglestep(ti);
  switch ( ti.block )
  {
    case bl_signal:
      if ( in_ptrace )
      {
        // we detach from the process and will handle the rest
        // using mach api
#if __arm__
//bool ok = task_resume(task) == KERN_SUCCESS;
        int pt = run_state >= rs_exiting ? PT_CONTINUE : PT_DETACH;
        ok = qptrace(pt, pid, caddr_t(1), 0) == 0;
        in_ptrace = false;
#else
        int pt = ti.single_step ? PT_STEP : PT_CONTINUE;
        ok = qptrace(pt, pid, caddr_t(1), ti.child_signum) == 0;
#endif
      }
      else
      {
        kern_return_t err = thread_resume(ti.tid);
        COMPLAIN_IF_FAILED("thread_resume");
      }
      break;

    case bl_exception:
      // handle the exception with exc_server
      my_mach_msg_t reply_msg;
      lock_begin();
      {
        parse_mach_message = false;
        mask_exception = ti.child_signum == 0;
        ok = exc_server(&ti.excmsg.hdr, &reply_msg.hdr);
        QASSERT(ok);
      }
      lock_end();

      kern_return_t err;
      err = mach_msg(&reply_msg.hdr,
                     MACH_SEND_MSG,
                     reply_msg.hdr.msgh_size, // send size
                     0,
                     reply_msg.hdr.msgh_remote_port,
                     0,                  // timeout
                     MACH_PORT_NULL); // notify port
      COMPLAIN_IF_FAILED("mach_msg");
      ok = (err == KERN_SUCCESS);
      task_resume(task);
      break;

    default:  // nothing to do, the process is already running
      break;
  }
  // syscalls may fail after SIGKILL, do not check the error code
  //QASSERT(ok);
  ti.block = bl_none;
  ti.single_step = false;
  return true;
}

//--------------------------------------------------------------------------
int mac_debmod_t::exception_to_signal(const mach_exception_info_t *exinf)
{
  int code = exinf->exception_data[0];
  int sig = 0;
  switch( exinf->exception_type )
  {
    case EXC_BAD_ACCESS:
      if ( code == KERN_INVALID_ADDRESS )
        sig = SIGSEGV;
      else
        sig = SIGBUS;
      break;

    case EXC_BAD_INSTRUCTION:
      sig = SIGILL;
      break;

    case EXC_ARITHMETIC:
      sig = SIGFPE;
      break;

    case EXC_EMULATION:
      sig = SIGEMT;
      break;

    case EXC_SOFTWARE:
      switch (code)
      {
//        case EXC_UNIX_BAD_SYSCALL:
//          sig = SIGSYS;
//          break;
//        case EXC_UNIX_BAD_PIPE:
//          sig = SIGPIPE;
//          break;
//        case EXC_UNIX_ABORT:
//          sig = SIGABRT;
//          break;
        case EXC_SOFT_SIGNAL:
          sig = SIGKILL;
          break;
      }
      break;

    case EXC_BREAKPOINT:
      sig = SIGTRAP;
      break;
  }
  return sig;
}

//--------------------------------------------------------------------------
// we got a signal that does not belong to our thread. find the target thread
// and store the signal in its 'pending_signals'
local void store_pending_signal(int pid, int status)
{
  struct mac_signal_storer_t : public debmod_visitor_t
  {
    int pid;
    int status;
    mac_signal_storer_t(int p, int s) : pid(p), status(s) {}
    int visit(debmod_t *debmod)
    {
      mac_debmod_t *md = (mac_debmod_t *)debmod;
      if ( md->pid == pid )
      {
        md->pending_signals.push_back(status);
        return 1; // stop
      }
      return 0; // continue
    }
  };
  mac_signal_storer_t mss(pid, status);
  for_all_debuggers(mss); // uses lock_begin(), lock_end() to protect common data
}

//--------------------------------------------------------------------------
// check if 'pending_signals' has anything
bool mac_debmod_t::retrieve_pending_signal(int *status)
{
  if ( pending_signals.empty() )
    return false;

  bool has_pending_signal;
  lock_begin();
  {
    has_pending_signal = !pending_signals.empty();
    if ( has_pending_signal )
    {
      *status = pending_signals.front();
      pending_signals.erase(pending_signals.begin());
    }
  }
  lock_end();
  return has_pending_signal;
}

//--------------------------------------------------------------------------
// timeout in milliseconds
// 0 - no timeout, return immediately
// -1 - wait forever
void mac_debmod_t::get_debug_events(int timeout)
{
//printf("waiting, block=%d numpend=%d timeout=%d...\n", block, events.size(), timeout);
//  if ( blocked() )
//    debdeb("ERROR: process is BLOCKED before qwait()!!!\n");

  int status;
  debug_event_t event;
  if ( !retrieve_pending_signal(&status) )
  {
    update_threads();

    // receive info about any exceptions in the program
    my_mach_msg_t excmsg;
    mach_exception_info_t exinf;
    while ( check_for_exception(timeout, &exinf, &excmsg) )
    {
      event.tid = exinf.thread_port;
      int sig = exception_to_signal(&exinf);
//      msg("got exception for tid=%d sig=%d %s\n", event.tid, sig, strsignal(sig));
      if ( handle_signal(sig, &event, bl_exception, &excmsg) )
        events.enqueue(event, IN_BACK);
      if ( timeout > 0 )
        timeout = 0;
    }
    if ( !events.empty() )
      return;

    // check the signals
    pid_t wpid;
    while ( true )
    {
      wpid = qwait(&status, false);
      if ( wpid == -1 || wpid == 0 )
        return;
      if ( wpid == pid )
        break;
      store_pending_signal(wpid, status);
    }
  }

  event.tid = maintid();
  if ( WIFSTOPPED(status) )
  {
    int code = WSTOPSIG(status);
//    msg("SIGNAL %d: %s (stopped)\n", code, strsignal(code));
    if ( !handle_signal(code, &event, bl_signal, NULL) )
      return;
  }
  else
  {
    if ( WIFSIGNALED(status) )
    {
//      msg("SIGNAL: %s (terminated)\n", strsignal(WSTOPSIG(status)));
      event.exit_code = WSTOPSIG(status);
    }
    else
    {
//      msg("SIGNAL: %d (exited)\n", WEXITSTATUS(status));
      event.exit_code = WEXITSTATUS(status);
    }
EXIT_EVENT:
    event.pid     = pid;
    event.ea      = BADADDR;
    event.handled = true;
    event.eid     = PROCESS_EXIT;
    run_state = rs_exited;
    // for some reason we have to call qwait once more. i guess we do something
    // very wrong, but i don't know what. anyway, mixing mach exceptions and
    // qwait is not a good idea anyway
    int wpid = qwait(&status, false);
//    msg("FINAL QWAIT RETURNED PID %d status %x\n", wpid, status);
    if ( wpid != pid )
      store_pending_signal(wpid, status);
  }
//  msg("low got event: %s\n", debug_event_str(&event));
  events.enqueue(event, IN_BACK);
}

//--------------------------------------------------------------------------
void mac_debmod_t::handle_dyld_bpt(const debug_event_t *event)
{
//  msg("handle dyld bpt, ea=%a\n", event->ea);
  update_dylib();

  machine_thread_state_t state;
  bool ok = get_thread_state(event->tid, &state);
  QASSERT(ok);

#ifdef __arm__
  // emulate bx lr. we assume the (lr & 1) == 0
  QASSERT((state.__lr & 1) == 0);
  state.__pc = state.__lr;
#else
  // emulate push ebp
  state.__esp -= addrsize;
  kern_return_t err = write_mem(state.__esp, &state.__ebp, addrsize);
  QASSERT(err == KERN_SUCCESS);
#endif

  ok = set_thread_state(event->tid, &state);
  QASSERT(ok);

  dbg_continue_after_event(event);
}

//--------------------------------------------------------------------------
int idaapi mac_debmod_t::dbg_get_debug_event(debug_event_t *event, bool ida_is_idle)
{
  if ( event == NULL )
    return false;

  while ( true )
  {
    // are there any pending events?
    if ( events.retrieve(event) )
    {
      switch ( event->eid )
      {
        // if this is dyld bpt, do not return it to ida
        case BREAKPOINT:
          if ( event->ea == dyri.dyld_notify )
          {
            handle_dyld_bpt(event);
            continue;
          }
          break;

        case PROCESS_ATTACH:
          attaching = false;        // finally attached to it
          break;

        case THREAD_EXIT:           // thread completely disappeared,
                                    // can remove it from the list
          threads.erase(event->tid);
          break;
      }
      last_event = *event;
      debdeb("GDE1: %s\n", debug_event_str(event));
      return true;
    }

    get_debug_events(ida_is_idle ? TIMEOUT : 0);
    if ( events.empty() )
      break;
  }
  return false;
}

//--------------------------------------------------------------------------
bool mac_debmod_t::suspend_all_threads(void)
{
  /* Suspend the target process */
  kern_return_t err = task_suspend(task);
  return err == KERN_SUCCESS;
}

//--------------------------------------------------------------------------
void mac_debmod_t::resume_all_threads()
{
  kern_return_t err = task_resume(task);
  QASSERT(err == KERN_SUCCESS);
}

//--------------------------------------------------------------------------
void mac_debmod_t::unblock_all_threads(void)
{
  for ( threads_t::iterator p=threads.begin(); p != threads.end(); ++p )
    my_resume_thread(p->second);
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_continue_after_event(const debug_event_t *event)
{
  if ( event == NULL )
    return 0;

  ida_thread_info_t *t = get_thread(event->tid);
  if ( debug_debugger )
  {
    debdeb("continue after event %s (%d pending, block type %d, sig#=%d)\n",
           debug_event_str(event),
           int(events.size()),
           t == NULL ? 0 : t->block,
           t == NULL ? 0 : t->child_signum);
  }

  if ( event->eid != THREAD_START
    && event->eid != THREAD_EXIT
    && event->eid != LIBRARY_LOAD
    && event->eid != LIBRARY_UNLOAD
    && (event->eid != EXCEPTION || event->handled) )
  {
    QASSERT(t != NULL);
    if ( t->child_signum != SIGKILL ) // never mask SIGKILL
    {
      debdeb("event->eid=%d, erasing child_signum\n", event->eid);
      t->child_signum = 0;
    }
  }

  if ( events.empty() && !attaching )
  {
    // if the event queue is empty, we can resume all blocked threads
    // here we resume only the threads blocked because of exceptions or signals
    // if the debugger kernel has suspended a thread for another reason, it
    // will stay suspended.
#ifdef __arm__ // if we suspended all threads, the correct way to unblock them is to
    if ( run_state == rs_suspended )
    {
      run_state = rs_running;
      resume_all_threads();
    }
#endif
    unblock_all_threads();
  }
  return 1;
}

//--------------------------------------------------------------------------
kern_return_t mac_debmod_t::read_mem(ea_t ea, void *buffer, int size)
{
  if ( ea == 0 )
    return KERN_INVALID_ADDRESS;
  mach_vm_size_t data_count = 0;
  kern_return_t err = mach_vm_read_overwrite(task, ea, size, (vm_address_t)buffer, &data_count);
  if ( err == KERN_SUCCESS && data_count != size )
    err = KERN_INVALID_ADDRESS;
  if ( err != KERN_SUCCESS )
    debdeb("vm_read %d: ea=%a size=%d => (%s)\n", task, ea, size, mach_error_string(err));
//  show_hex(buffer, size, "data:\n");
  return err;
}

//--------------------------------------------------------------------------
kern_return_t mac_debmod_t::write_mem(ea_t ea, void *buffer, int size)
{
  kern_return_t err;
/*  vm_machine_attribute_val_t flush = MATTR_VAL_CACHE_FLUSH;
printf("buffer=%x size=%x\n", buffer, size);
  err = vm_machine_attribute (mach_task_self(), (vm_offset_t)buffer, size, MATTR_CACHE, &flush);
  QASSERT(err == KERN_SUCCESS); // must succeed since it is our memory
*/
  err = mach_vm_write(task, ea, (vm_offset_t)buffer, size);
  if ( err != KERN_SUCCESS && err != KERN_PROTECTION_FAILURE )
    debdeb("vm_write %d: ea=%a, size=%d => %s\n", task, ea, size, mach_error_string(err));
  return err;
}

//--------------------------------------------------------------------------
inline bool is_shared_address(ea_t ea)
{
  ea &= GLOBAL_SHARED_SEGMENT_MASK;
  return ea == GLOBAL_SHARED_TEXT_SEGMENT || ea == GLOBAL_SHARED_DATA_SEGMENT;
}

//--------------------------------------------------------------------------
bool mac_debmod_t::xfer_page(ea_t ea, void *buffer, int size, bool write)
{
  // get old memory protection
  mach_vm_address_t r_start = ea;
  mach_vm_size_t r_size;
  mach_port_t r_object_name;
  vm_region_basic_info_data_64_t r_data;
  mach_msg_type_number_t r_info_size = VM_REGION_BASIC_INFO_COUNT_64;
  kern_return_t err = mach_vm_region(task, &r_start, &r_size,
                                VM_REGION_BASIC_INFO_64,
                                (vm_region_info_t)&r_data, &r_info_size,
                                &r_object_name);
  if ( err != KERN_SUCCESS )
  {
    // this call fails for the commpage segment
    debdeb("%" FMT_64 "x: vm_region: %s\n", r_start, mach_error_string(err));
    return false;
  }

  if ( r_start > ea )
  {
    dmsg("%a: region start is higher %" FMT_64 "x\n", ea, r_start);
    return false;
  }

//  dmsg("%a: current=%d max=%d\n", ea, r_data.protection, r_data.max_protection);
  int bit = write ? VM_PROT_WRITE : VM_PROT_READ;
  // max permissions do not allow it? fail
  // strangely enough the kernel allows us to set any protection,
  // including protections bigger than max_protection. but after that it crashes
  // we have to verify it ourselves here.
  if ( (r_data.max_protection & bit) == 0 )
    return false;

  if ( (r_data.protection & bit) == 0 )
  {
    bit |= r_data.protection;
    // set the desired bit
    err = KERN_FAILURE;
    if ( write )
    {
      if ( is_shared_address(r_start) )
      {
#ifndef __arm__
        if ( is_leopard )
          return false; // can not modify shared areas under leopard
#endif
        bit |= VM_PROT_COPY;
      }
//      dmsg("shared: %d b2=%x\n", is_shared_address(r_start), bit);
      err = mach_vm_protect(task, r_start, r_size, 0, bit);
      if ( err != KERN_SUCCESS && (bit & VM_PROT_COPY) == 0 )
      {
        bit |= VM_PROT_COPY; // if failed, make a copy of the page
        goto LASTPROT;
      }
    }
    else
    {
LASTPROT:
      err = mach_vm_protect(task, r_start, r_size, 0, bit);
    }
    if ( err != KERN_SUCCESS )
    {
      debdeb("%d: could not set %s permission at %" FMT_64 "x\n",
                        task, write ? "write" : "read", r_start);
      return false;
    }
  }

  // attempt to xfer
  if ( write )
    err = write_mem(ea, buffer, size);
  else
    err = read_mem(ea, buffer, size);

  bool ok = (err == KERN_SUCCESS);
#if 1
  if ( ok && write )
  {
    // flush the cache
//    vm_machine_attribute_val_t flush = MATTR_VAL_CACHE_FLUSH;
    vm_machine_attribute_val_t flush = MATTR_VAL_OFF;
    err = mach_vm_machine_attribute(task, r_start, r_size, MATTR_CACHE, &flush);
    if ( err != KERN_SUCCESS )
    {
      static bool complained = false;
      if ( !complained )
      {
        complained = true;
        dmsg("Unable to flush data/instruction cache ea=0x%" FMT_64 "x size=%ld: %s\n",
                        r_start, long(r_size), mach_error_string(err));
      }
    }
    else
    {
//      msg("Success cache ea=0x%" FMT_64 "x size=%d\n", r_start, r_size);
    }
  }
#endif
  // restore old memory protection
  if ( (r_data.protection & bit) == 0 )
  {
    err = mach_vm_protect(task, r_start, r_size, 0, r_data.protection);
    QASSERT(err == KERN_SUCCESS);
  }
  return ok;
}

//--------------------------------------------------------------------------
local unsigned int get_pagesize(void)
{
  static vm_size_t page = 0;
  if ( page == 0 )
  {
    kern_return_t err = host_page_size(mach_host_self(), &page);
    QASSERT(err == KERN_SUCCESS);
    QASSERT(page > 0);
    QASSERT((page & (page-1)) == 0);  // power of 2
  }
  return page;
}

//--------------------------------------------------------------------------
bool mac_debmod_t::xfer_memory(ea_t ea, void *buffer, int size, bool write)
{
return xfer_page(ea, buffer, size, write);
/*
  bool ok = true;
  char *ptr = (char *)buffer;
  int pagesize = get_pagesize();
  int delta = ea & (pagesize-1);
  ea_t base = ea - delta;
  // if the address is not page-aligned, perform i/o on the rest of the
  // first page.
  if ( delta != 0 )
  {
    static char *page = NULL;
    if ( page == NULL )
    {
      page = (char*)qalloc(pagesize);
      if ( page == NULL )
        nomem("vm_write");
    }
    ok = xfer_page(base, page, pagesize, false);
    if ( ok )
    {
      int s2 = pagesize - delta;
      int rest = qmin(s2, size);
      if ( write )
      {
        memcpy(page+delta, ptr, rest);
        ok = xfer_page(base, page, pagesize, true);
      }
      else
      {
        memcpy(ptr, page+delta, rest);
      }
      ptr += rest;
      size -= rest;
      base += pagesize;
      delta = 0;
    }
  }
  // 'base' is page aligned now, handle the remaining bytes
  if ( ok && size > 0 )
    ok = xfer_page(base, ptr, size, write);
  return ok;*/
}

//--------------------------------------------------------------------------
int mac_debmod_t::_read_memory(ea_t ea, void *buffer, int size, bool suspend)
{
  if ( exited() || pid <= 0 || size <= 0 )
    return -1;

//  debdeb("READ MEMORY %a:%d: START\n", ea, size);
  // stop all threads before accessing the process memory
  if ( suspend && !suspend_all_threads() )
    return -1;
  if ( exited() )
    return -1;

//  bool ok = xfer_memory(ea, buffer, size, false);
  kern_return_t err = read_mem(ea, buffer, size);
  bool ok = err == KERN_SUCCESS;

  if ( suspend )
    resume_all_threads();
//  debdeb("READ MEMORY %a:%d: END\n", ea, size);
  return ok ? size : 0;
}

//--------------------------------------------------------------------------
int mac_debmod_t::_write_memory(ea_t ea, const void *buffer, int size, bool suspend)
{
  if (exited() || pid <= 0 || size <= 0)
    return -1;

  // stop all threads before accessing the process memory
  if ( suspend && !suspend_all_threads() )
    return -1;
  if ( exited() )
    return -1;

  bool ok = xfer_memory(ea, (void*)buffer, size, true);
  if ( ok && size == BPT_CODE_SIZE ) // might be a breakpoint add/del
  {
    if ( memcmp(buffer, bpt_code.begin(), BPT_CODE_SIZE) == 0 )
      bpts.insert(ea);
    else
      bpts.erase(ea);
  }

  if ( suspend )
    resume_all_threads();

  return ok ? size : 0;
}

//--------------------------------------------------------------------------
ssize_t idaapi mac_debmod_t::dbg_write_memory(ea_t ea, const void *buffer, size_t size)
{
  return _write_memory(ea, buffer, size, true);
}

//--------------------------------------------------------------------------
ssize_t idaapi mac_debmod_t::dbg_read_memory(ea_t ea, void *buffer, size_t size)
{
  return _read_memory(ea, buffer, size, true);
}

//--------------------------------------------------------------------------
void mac_debmod_t::add_dll(ea_t addr, const char *fname)
{
  asize_t size = calc_image_size(fname, NULL);

  debug_event_t ev;
  ev.eid     = LIBRARY_LOAD;
  ev.pid     = pid;
  ev.tid     = maintid();
  ev.ea      = addr;
  ev.handled = true;
  qstrncpy(ev.modinfo.name, fname, sizeof(ev.modinfo.name));
  ev.modinfo.base = addr;
  ev.modinfo.size = size;
  ev.modinfo.rebase_to = BADADDR;
  if ( is_dll && stricmp(fname, input_file_path.c_str()) == 0 )
    ev.modinfo.rebase_to = addr;
  events.enqueue(ev, IN_FRONT);

  image_info_t ii(addr, size, fname);
  dlls.insert(std::make_pair(addr, ii));
  dlls_to_import.insert(addr);
}

//--------------------------------------------------------------------------
inline bool is_zeropage(const segment_command &sg)
{
  return sg.vmaddr == 0 && sg.fileoff == 0 && sg.initprot == 0;
}

//--------------------------------------------------------------------------
inline bool is_text_segment(const segment_command &sg)
{
  if ( is_zeropage(sg) )
    return false;
  const char *name = sg.segname;
  for ( int i=0; i < sizeof(sg.segname); i++, name++ )
    if ( *name != '_' )
      break;
  return strnicmp(name, "TEXT", 4) == 0;
}

//--------------------------------------------------------------------------
local asize_t calc_image_size(const char *fname, ea_t *p_base)
{
  if ( p_base != NULL )
    *p_base = BADADDR;
  linput_t *li = open_linput(fname, false);
  if ( li == NULL )
    return 0;

  asize_t size = calc_macho_image_size(li, p_base);
  close_linput(li);
  return size;
}

//--------------------------------------------------------------------------
bool mac_debmod_t::import_dll(linput_t *li, ea_t base, name_info_t &ni)
{
  struct macho_importer_t : public symbol_visitor_t
  {
    mac_debmod_t *md;
    macho_importer_t(mac_debmod_t *_md) : md(_md), symbol_visitor_t(VISIT_SYMBOLS) {}
    int visit_symbol(ea_t ea, const char *name)
    {
      if ( name[0] != '\0' )
      {
        md->save_debug_name(ea, name);
        if ( md->dylib_infos == BADADDR && strcmp(name, "_dyld_all_image_infos") == 0 )
        {
          md->dmsg("%a: address of dylib raw infos\n", ea);
          md->dylib_infos = ea;
        }
      }
      return 0;
    }
  };
  macho_importer_t mi(this);
  return parse_macho(base, li, mi, false);
}

//--------------------------------------------------------------------------
bool mac_debmod_t::import_dll_to_database(ea_t imagebase, name_info_t &ni)
{
  images_t::iterator p = dlls.find(imagebase);
  if ( p == dlls.end() )
    return false;

  const char *dllname = p->second.name.c_str();

  linput_t *li = open_linput(dllname, false);
  if ( li == NULL )
    return false;

  bool ok = import_dll(li, imagebase, ni);
  close_linput(li);
  return ok;
}

//--------------------------------------------------------------------------
ssize_t idaapi mac_debmod_t::dbg_write_file(
        int /* fn */,
        uint32 /* off */,
        const void * /* buf */,
        size_t /* size */)
{
  return 0;
}

//--------------------------------------------------------------------------
ssize_t idaapi mac_debmod_t::dbg_read_file(
        int /* fn */,
        uint32 /* off */,
        void * /* buf */,
        size_t /* size */)
{
  return 0;
}

//--------------------------------------------------------------------------
int idaapi mac_debmod_t::dbg_open_file(
        const char * /* file */,
        uint32 * /* fsize */,
        bool /* readonly */)
{
  return 0;
}

//--------------------------------------------------------------------------
void idaapi mac_debmod_t::dbg_close_file(int /* fn */)
{
}

//--------------------------------------------------------------------------
void idaapi mac_debmod_t::dbg_stopped_at_debug_event(void)
{
  // we will take advantage of this event to import information
  // about the exported functions from the loaded dlls
  name_info_t &ni = *get_debug_names();

  for (easet_t::iterator p=dlls_to_import.begin(); p != dlls_to_import.end(); )
  {
    import_dll_to_database(*p, ni);
    dlls_to_import.erase(p++);
  }
}

//--------------------------------------------------------------------------
void mac_debmod_t::cleanup(void)
{
  pid = 0;
  is_dll = false;
  run_state = rs_exited;
  dylib = BADADDR;
  dylib_infos = BADADDR;
  dyri.version = 0;  // not inited
  term_exception_ports();

  threads.clear();
  dlls.clear();
  dlls_to_import.clear();
  events.clear();
  attaching = false;
  bpts.clear();

  inherited::cleanup();
}

//--------------------------------------------------------------------------
//
//      DEBUGGER INTERFACE FUNCTIONS
//
//--------------------------------------------------------------------------
void mac_debmod_t::refresh_process_list(void)
{
  processes.clear();

  int sysControl[4];
  sysControl[0] = CTL_KERN;
  sysControl[1] = KERN_PROC;
  sysControl[2] = KERN_PROC_ALL;

  size_t length;
  sysctl(sysControl, 3, NULL, &length, NULL, 0);
  int count = (length / sizeof (struct kinfo_proc));
  if ( count <= 0 )
    return;
  length = sizeof (struct kinfo_proc) * count;

  qvector<struct kinfo_proc> info;
  info.resize(count);
  sysctl(sysControl, 3, &info[0], &length, NULL, 0);

  for ( int i=0; i < count; i++ )
  {
    mach_port_t port;
    kern_return_t result = task_for_pid(mach_task_self(), info[i].kp_proc.p_pid,  &port);
    if ( result == KERN_SUCCESS )
    {
      process_info_t pi;
      qstrncpy(pi.name, info[i].kp_proc.p_comm, sizeof(pi.name));
      pi.pid = info[i].kp_proc.p_pid;
      processes.push_back(pi);
    }
    else
    {
      debdeb("%d: %s is unavailable for debugging\n", info[i].kp_proc.p_pid, info[i].kp_proc.p_comm);
    }
  }
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
// input is valid only if n==0
int idaapi mac_debmod_t::dbg_process_get_info(int n, const char *input, process_info_t *info)
{
  if ( n == 0 )
  {
    input_file_path = input;
    refresh_process_list();
  }

  if ( n < 0 || n >= processes.size() )
    return false;

  if ( info != NULL )
    *info = processes[n];
  return true;
}

//--------------------------------------------------------------------------
local char **prepare_arguments(const char *path, const char *args)
{
  int i = 0;
  char **argv = NULL;
  while ( 1 )
  {
    qstring s;
    if ( i == 0 )
    {
      s = path;
    }
    else
    {
      while ( isspace(*args) ) args++;
      if ( *args == '\0' ) break;
      char quote = (*args == '"' || *args == '\'') ? *args++ : 0;
      while ( (quote ? *args != quote : !isspace(*args)) && *args != '\0' )
      {
        if ( *args == '\\' && args[1] != '\0' )
          args++;
        s += *args++;
      }
    }
    argv = (char **)qrealloc(argv, sizeof(char *)*(i+2));
    if ( argv == NULL )
      nomem("prepare_arguments");
    argv[i++] = s.extract();
  }
  argv[i] = NULL;
  return argv;
}

//--------------------------------------------------------------------------
// Returns the file name assciated with pid
char *get_exec_fname(int pid, char *buf, size_t bufsize)
{
  int mib[3];
  mib[0] = CTL_KERN;
  mib[1] = KERN_ARGMAX;

  int argmax = 0;
  size_t size = sizeof(argmax);

  sysctl(mib, 2, &argmax, &size, NULL, 0);
  if ( argmax <= 0 )
    argmax = QMAXPATH;

  char *args = (char *)qalloc(argmax);
  if ( args == NULL )
    nomem("get_exec_fname");

  mib[0] = CTL_KERN;
  mib[1] = KERN_PROCARGS2;
  mib[2] = pid;

  // obtain the arguments for the target process. this will
  // only work for processes that belong to the current uid,
  // so if you want it to work universally, you need to run
  // as root.
  size = argmax;
  buf[0] = '\0';
  if ( sysctl(mib, 3, args, &size, NULL, 0) != -1 )
  {
    char *ptr = args + sizeof(int);
    char *end = args + size;
//    show_hex(ptr, size, "procargs2\n");

    qstrncpy(buf, ptr, bufsize);
  }
  qfree(args);
  return buf;
}

//--------------------------------------------------------------------------
bool mac_debmod_t::thread_exit_event_planned(thid_t tid)
{
  for ( eventlist_t::iterator p=events.begin(); p != events.end(); ++p )
  {
    if ( p->eid == THREAD_EXIT && p->tid == tid )
      return true;
  }
  return false;
}

//--------------------------------------------------------------------------
bool mac_debmod_t::update_threads(void)
{
  bool generated_events = false;
  thread_act_port_array_t threadList;
  mach_msg_type_number_t threadCount;
  kern_return_t err = task_threads(task, &threadList, &threadCount);
  std::set<int> live_tids;
  if ( err == KERN_SUCCESS )
  {
    QASSERT(threadCount > 0);
    for ( int i=0; i < threadCount; i++ )
    {
      mach_port_t port = threadList[i];
      int tid = port;
      threads_t::iterator p = threads.find(tid);
      if ( p == threads.end() )
      {
        debug_event_t ev;
        ev.eid     = THREAD_START;
        ev.pid     = pid;
        ev.tid     = tid;
        ev.ea      = BADADDR;
        ev.handled = true;
        events.enqueue(ev, IN_FRONT);
        threads.insert(std::make_pair(tid, ida_thread_info_t(tid, port)));
        generated_events = true;
      }
      live_tids.insert(tid);
    }
    err = mach_vm_deallocate (mach_task_self(), (vm_address_t)threadList, threadCount * sizeof (thread_t));
    QASSERT(err == KERN_SUCCESS);
    // remove dead threads
    for ( threads_t::iterator p=threads.begin(); p != threads.end(); ++p )
    {
      thid_t tid = p->first;
      if ( live_tids.find(tid) == live_tids.end() && !thread_exit_event_planned(tid) )
      {
        debug_event_t ev;
        ev.eid     = THREAD_EXIT;
        ev.pid     = pid;
        ev.tid     = tid;
        ev.ea      = BADADDR;
        ev.handled = true;
        events.enqueue(ev, IN_BACK);
        generated_events = true;
      }
    }
  }
  return generated_events;
}

//--------------------------------------------------------------------------
thid_t mac_debmod_t::init_main_thread(void)
{
  thread_act_port_array_t threadList;
  mach_msg_type_number_t threadCount;
  kern_return_t err = task_threads(task, &threadList, &threadCount);
  QASSERT(err == KERN_SUCCESS);
  QASSERT(threadCount > 0);
  mach_port_t port = threadList[0]; // the first thread is the main thread
  thid_t tid = port;
  threads.insert(std::make_pair(tid, ida_thread_info_t(tid, port)));
  threads.begin()->second.block = bl_signal;
  err = mach_vm_deallocate(mach_task_self(), (vm_address_t)threadList, threadCount * sizeof(thread_t));
  QASSERT(err == KERN_SUCCESS);
  return tid;
}

//--------------------------------------------------------------------------
local kern_return_t save_exception_ports(task_t task, mach_exception_port_info_t *info)
{
  info->count = (sizeof (info->ports) / sizeof (info->ports[0]));
  return task_get_exception_ports(task,
                                  EXC_MASK_ALL,
                                  info->masks,
                                  &info->count,
                                  info->ports,
                                  info->behaviors,
                                  info->flavors);
}

local kern_return_t restore_exception_ports (task_t task, const mach_exception_port_info_t *info)
{
  kern_return_t err = KERN_SUCCESS;
  for ( int i = 0; i < info->count; i++ )
  {
    err = task_set_exception_ports(task,
                                   info->masks[i],
                                   info->ports[i],
                                   info->behaviors[i],
                                   info->flavors[i]);
    if ( err != KERN_SUCCESS )
      break;
  }
  return err;
}

void mac_debmod_t::init_exception_ports(void)
{
  kern_return_t err;

  // allocate a new port to receive exceptions
  err = mach_port_allocate(mach_task_self(), MACH_PORT_RIGHT_RECEIVE, &exc_port);
  QASSERT(err == KERN_SUCCESS);

  // add the 'send' right to send replies to threads
  err = mach_port_insert_right(mach_task_self(), exc_port, exc_port, MACH_MSG_TYPE_MAKE_SEND);
  QASSERT(err == KERN_SUCCESS);

  // save old exception ports
  err = save_exception_ports(task, &saved_exceptions);
  QASSERT(err == KERN_SUCCESS);

  // set new port for all exceptions
  err = task_set_exception_ports(task, EXC_MASK_ALL, exc_port, EXCEPTION_DEFAULT, THREAD_STATE_NONE);
  QASSERT(err == KERN_SUCCESS);

}

void mac_debmod_t::term_exception_ports(void)
{
  if ( exc_port != MACH_PORT_NULL )
  {
    kern_return_t err = restore_exception_ports(mach_task_self(), &saved_exceptions);
    QASSERT(err == KERN_SUCCESS);
    err = mach_port_deallocate(mach_task_self(), exc_port);
    QASSERT(err == KERN_SUCCESS);
    exc_port = MACH_PORT_NULL;
  }
}

//--------------------------------------------------------------------------
local bool is_setgid_procmod(void)
{
  gid_t gid = getegid();
  struct group *gr = getgrgid(gid);
  if ( gr == NULL )
    return false;
  bool ok = strcmp(gr->gr_name, "procmod") == 0;
  msg("Current group=%s\n", gr->gr_name);
  endgrent();
  return ok;
}

//--------------------------------------------------------------------------
bool mac_debmod_t::handle_process_start(pid_t _pid)
{
  debdeb("handle process start %d\n", _pid);
  pid = _pid;

  /* Get the mach task for the target process */
  kern_return_t err = task_for_pid(mach_task_self(), pid, &task);
  if ( err == KERN_FAILURE ) // no access?
  {
    if ( !is_setgid_procmod() )
    {
#ifdef __arm__
      const char *program = "iphone_server";
#else
      int argc = *_NSGetArgc();
      char **argv = *_NSGetArgv();
      const char *program = qbasename(argv[0]);
      if ( strstr(program, "server") == NULL )
        program = NULL; // runing local mac debugger module
#endif
      if ( program != NULL )
        dwarning("File '%s' must be setgid procmod to debug Mac OS X applications.\n"
                 "Please use the following commands to change its permissions:\n"
                 "  sudo chmod +sg %s\n"
                 "  sudo chgrp procmod %s\n", program, program, program);
      else
        dwarning("Please run idal with elevated permissons for local debugging.\n"
                 "Another solution is to run mac_server and use localhost as\n"
                 "the remote computer name");
      return false;
    }
    debdeb("could not determine process %d port: %s\n", pid, mach_error_string(err));
    return false;
  }
  QASSERT(err == KERN_SUCCESS);

  int status;
  waitpid(pid, &status, 0);
  debdeb("first waitpid on %d: %x\n", pid, status);
  if ( !WIFSTOPPED(status) )
  {
    debdeb("not stopped?\n");
    return false;
  }
  if ( WSTOPSIG(status) != SIGTRAP && WSTOPSIG(status) != SIGSTOP )
  {
    debdeb("got signal %d?\n", WSTOPSIG(status));
    return false;
  }
  in_ptrace = true;
  thid_t tid = init_main_thread();
  debdeb("initially stopped at %a pid=%d tid=%d task=%d\n", get_ip(tid), pid, tid, task);
  run_state = rs_running;

  debug_event_t ev;
  ev.eid     = PROCESS_START;
  ev.pid     = pid;
  ev.tid     = tid;
  ev.ea      = BADADDR;
  ev.handled = true;
  get_exec_fname(pid, ev.modinfo.name, sizeof(ev.modinfo.name));
  debdeb("gotexe: %s\n", ev.modinfo.name);
  ev.modinfo.size = calc_image_size(ev.modinfo.name, &ev.modinfo.base);
  ev.modinfo.rebase_to = BADADDR;

  // find the real executable base
  // also call get_memory_info() the first time
  // this should find dyld and set its bpt
  meminfo_vec_t miv;
  if ( get_memory_info(miv, false) > 0 && !is_dll )
  {
    for ( int i=0; i < miv.size(); i++ )
    {
      if ( miv[i].name == ev.modinfo.name )
      {
        ev.modinfo.rebase_to = miv[i].startEA;
        break;
      }
    }
  }
  events.enqueue(ev, IN_FRONT);

  init_exception_ports();
  return true;
}

//--------------------------------------------------------------------------
int idaapi mac_debmod_t::dbg_start_process(const char *path,
                                const char *args,
                                const char *startdir,
                                int flags,
                                const char *input_path,
                                uint32 input_file_crc32)
{
  // input file specified in the database does not exist
  if ( input_path[0] != '\0' && !qfileexist(input_path) )
    return -2;

  // temporary thing, later we will retrieve the real file name
  // based on the process id
  input_file_path = input_path;
  is_dll = (flags & DBG_PROC_IS_DLL) != 0;

  if ( !qfileexist(path) )
    return -1;

  int mismatch = 0;
  if ( !check_input_file_crc32(input_file_crc32) )
    mismatch = CRC32_MISMATCH;

  if ( startdir[0] != '\0' && chdir(startdir) == -1 )
  {
    debdeb("chdir '%s': %s\n", startdir, winerr(errno));
    return -1;
  }

  int child_pid = fork();
  if ( child_pid == -1 )
  {
    debdeb("fork: %s\n", winerr(errno));
    return -1;
  }
  if ( child_pid == 0 ) // child
  {
    if ( qptrace(PT_TRACE_ME, 0, 0, 0) == -1 )
    {
      derror("TRACEME: %s", winerr(errno));
    }
    char **argv = prepare_arguments(path, args);
    for ( char **ptr=argv; *ptr != NULL; ptr++ )
      debdeb("ARG: %s\n", *ptr);

    setpgid (0, 0);
    // drop extra privileges of the debugger server
    setgid(getgid());
    // free ida-resources
    for (int i = getdtablesize(); i > 2; i--)
      close(i);
    int code = execv(path, argv);
    qprintf("execv(%s): %s\n", path, winerr(errno));
    _exit(1);
  }
  else                  // parent
  {
    if ( !handle_process_start(child_pid) )
      return -1;
  }

  return 1 | mismatch;
}


//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_attach_process(pid_t pid, int /*event_id*/)
{
  if ( qptrace(PT_ATTACH, pid, NULL, NULL) == 0
    && handle_process_start(pid) )
  {
    // generate the attach event
    debug_event_t ev;
    ev.eid     = PROCESS_ATTACH;
    ev.pid     = pid;
    ev.tid     = maintid();
    ev.ea      = get_ip(ev.tid);
    ev.handled = true;
    get_exec_fname(pid, ev.modinfo.name, sizeof(ev.modinfo.name));
    ev.modinfo.base = BADADDR;
    ev.modinfo.size = 0;
    ev.modinfo.rebase_to = BADADDR;
    events.enqueue(ev, IN_BACK);

    // generate THREAD_START events
    update_threads();

    // block the process until all generated events are processed
    attaching = true;
    return true;
  }
  return false;
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_detach_process(void)
{
  if ( dyri.dyld_notify != 0 )
  {
    // remove the dyld breakpoint
    int size = dbg_write_memory(dyri.dyld_notify, dyld_opcode, BPT_CODE_SIZE);
    QASSERT(size == BPT_CODE_SIZE);
    dyri.dyld_notify = 0;
  }
  // cleanup exception ports
  term_exception_ports();
  if ( in_ptrace )
  {
    qptrace(PT_DETACH, pid, 0, 0);
    in_ptrace = false;
  }
  else
  {
    // let the process run
    unblock_all_threads();
  }
  debug_event_t ev;
  ev.eid     = PROCESS_DETACH;
  ev.pid     = pid;
  ev.tid     = maintid();
  ev.ea      = BADADDR;
  ev.handled = true;
  events.enqueue(ev, IN_BACK);
  return true;
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_prepare_to_pause_process(void)
{
  debdeb("remote_prepare_to_pause_process\n");
  if ( run_state >= rs_exiting )
    return false;
#ifdef __arm__
  // since we detached from ptrace, we can not send signals to inferior
  // simple suspend it and generate a fake event
  if ( !suspend_all_threads() )
    return false;
  run_state = rs_suspended;
  debug_event_t ev;
  ev.eid     = NO_EVENT;
  ev.pid     = pid;
  ev.tid     = maintid();
  ev.ea      = BADADDR;
  ev.handled = true;
  events.enqueue(ev, IN_BACK);
#else
  run_state = rs_pausing;
  kill(pid, SIGSTOP);
#endif
  return true;
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_exit_process(void)
{
  run_state = rs_exiting;
  bool ok = false;
  if ( kill(pid, SIGKILL) == 0 )
  {
    ok = true;
    unblock_all_threads();
  }
  return ok;
}

//--------------------------------------------------------------------------
// Set hardware breakpoints for one thread
bool mac_debmod_t::set_hwbpts(int hThread)
{
  bool ok = set_dr(hThread, 0, hwbpt_ea[0])
    && set_dr(hThread, 1, hwbpt_ea[1])
    && set_dr(hThread, 2, hwbpt_ea[2])
    && set_dr(hThread, 3, hwbpt_ea[3])
    && set_dr(hThread, 6, 0)
    && set_dr(hThread, 7, dr7);
//  printf("set_hwbpts: DR0=%08lX DR1=%08lX DR2=%08lX DR3=%08lX DR7=%08lX => %d\n",
//         hwbpt_ea[0],
//         hwbpt_ea[1],
//         hwbpt_ea[2],
//         hwbpt_ea[3],
//         dr7,
//         ok);
  return ok;
}

//--------------------------------------------------------------------------
bool mac_debmod_t::refresh_hwbpts(void)
{
  for ( threads_t::iterator p=threads.begin(); p != threads.end(); ++p )
    if ( !set_hwbpts(p->second.tid) )
      return false;
  return true;
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_add_bpt(bpttype_t type, ea_t ea, int len)
{
  if ( type == BPT_SOFT )
    return dbg_write_memory(ea, bpt_code.begin(), BPT_CODE_SIZE) == BPT_CODE_SIZE;

#if defined (__i386__) || defined(__x86_64__)
  int i = find_hwbpt_slot(ea);    // get slot number
  if ( i != -1 )
  {
    hwbpt_ea[i] = ea;

    int lenc;                               // length code used by the processor
    if ( len == 1 )
      lenc = 0;
    if ( len == 2 )
      lenc = 1;
    if ( len == 4 )
      lenc = 3;

    dr7 |= (1 << (i*2));            // enable local breakpoint
    dr7 |= (type << (16+(i*2)));    // set breakpoint type
    dr7 |= (lenc << (18+(i*2)));    // set breakpoint length

    return refresh_hwbpts();
  }
#endif
  return false;
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_del_bpt(ea_t ea, const uchar *orig_bytes, int len)
{
  // we update threads when we delete a breakpoint because it gives
  // better results: new threads are immediately added to the list of
  // known threads and properly suspended before "single step"
  update_threads();
  if ( orig_bytes != NULL )
    return dbg_write_memory(ea, orig_bytes, len) == len;

#if defined (__i386__) || defined(__x86_64__)
  for ( int i=0; i < MAX_BPT; i++ )
  {
    if ( hwbpt_ea[i] == ea )
    {
      hwbpt_ea[i] = BADADDR;            // clean the address
      dr7 &= ~(3 << (i*2));             // clean the enable bits
      dr7 &= ~(0xF << (i*2+16));        // clean the length and type
      return refresh_hwbpts();
    }
  }
#endif
  return false;
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_thread_get_sreg_base(thid_t tid, int sreg_value, ea_t *pea)
{
#ifdef __arm__
  return false;
#else
  // assume all segments are based on zero
  *pea = 0;
  return true;
#endif
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_thread_suspend(thid_t tid)
{
  debdeb("remote_thread_suspend %d\n", tid);
  kern_return_t err = thread_suspend(tid);
  return err == KERN_SUCCESS;
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_thread_continue(thid_t tid)
{
  debdeb("remote_thread_continue %d\n", tid);
  kern_return_t err = thread_resume(tid);
  return err == KERN_SUCCESS;
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_thread_set_step(thid_t tid)
{
#ifdef __arm__
  qnotused(tid);
  return false;
#else
  ida_thread_info_t *t = get_thread(tid);
  if ( t == NULL )
    return false;
  t->single_step = true;
  return true;
#endif
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_thread_read_registers(thid_t tid, regval_t *values, int count)
{
  if ( values == NULL )
    return false;

  machine_thread_state_t cpu;
  if ( !get_thread_state(tid, &cpu) )
    return false;

  // force null bytes at the end of floating point registers.
  // we need this to properly detect register modifications,
  // as we compare the whole regval_t structure !
  memset(values, 0, count * sizeof(regval_t));

#ifdef __arm__  // fixme - add arm logic
  values[ 0].ival = uint32(cpu.__r[ 0]);
  values[ 1].ival = uint32(cpu.__r[ 1]);
  values[ 2].ival = uint32(cpu.__r[ 2]);
  values[ 3].ival = uint32(cpu.__r[ 3]);
  values[ 4].ival = uint32(cpu.__r[ 4]);
  values[ 5].ival = uint32(cpu.__r[ 5]);
  values[ 6].ival = uint32(cpu.__r[ 6]);
  values[ 7].ival = uint32(cpu.__r[ 7]);
  values[ 8].ival = uint32(cpu.__r[ 8]);
  values[ 9].ival = uint32(cpu.__r[ 9]);
  values[10].ival = uint32(cpu.__r[10]);
  values[11].ival = uint32(cpu.__r[11]);
  values[12].ival = uint32(cpu.__r[12]);
  values[13].ival = uint32(cpu.__sp);
  values[14].ival = uint32(cpu.__lr);
  values[15].ival = uint32(cpu.__pc);
  values[16].ival = uint32(cpu.__cpsr);
#else
  values[R_EAX   ].ival = uval_t(cpu.__eax);
  values[R_EBX   ].ival = uval_t(cpu.__ebx);
  values[R_ECX   ].ival = uval_t(cpu.__ecx);
  values[R_EDX   ].ival = uval_t(cpu.__edx);
  values[R_ESI   ].ival = uval_t(cpu.__esi);
  values[R_EDI   ].ival = uval_t(cpu.__edi);
  values[R_EBP   ].ival = uval_t(cpu.__ebp);
  values[R_ESP   ].ival = uval_t(cpu.__esp);
  values[R_EIP   ].ival = uval_t(cpu.__eip);
  values[R_EFLAGS].ival = uval_t(cpu.__eflags);
  values[R_CS    ].ival = uval_t(cpu.__cs);
  values[R_FS    ].ival = uval_t(cpu.__fs);
  values[R_GS    ].ival = uval_t(cpu.__gs);
#ifdef __X64__
  values[R_DS    ].ival = 0;
  values[R_ES    ].ival = 0;
  values[R_SS    ].ival = 0;
#else
  values[R_DS    ].ival = uval_t(cpu.__ds);
  values[R_ES    ].ival = uval_t(cpu.__es);
  values[R_SS    ].ival = uval_t(cpu.__ss);
#endif

  machine_float_state_t fpu;
  if ( !get_float_state(tid, &fpu) )
    return false;

  long double *fpr = (long double*)&fpu.__fpu_stmm0;
  for (int i = 0; i < FPU_REGS_COUNT; i++)
    *(long double *)values[R_ST0+i].fval = *fpr++;

  values[R_CTRL].ival = *(ushort*)&fpu.__fpu_fcw;
  values[R_STAT].ival = *(ushort*)&fpu.__fpu_fsw;
  values[R_TAGS].ival = fpu.__fpu_ftw;
#endif
  return true;
}

//--------------------------------------------------------------------------
inline bool is_fpu_reg(int reg_idx)
{
#if defined (__i386__) || defined(__x86_64__)
  bool is_integer = reg_idx < R_ST0 || reg_idx >= R_CS;
  return !is_integer;
#else // arm
  return false;
#endif
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
static bool patch_reg_context(
        machine_thread_state_t *cpu,
        machine_float_state_t *fpu,
        int reg_idx,
        const regval_t *value)
{
  if ( value == NULL )
    return false;

#if defined (__i386__) || defined(__x86_64__)
  if ( !is_fpu_reg(reg_idx) )
  {
    switch ( reg_idx )
    {
#ifdef __X64__
      case R64_R8:   cpu->__r8     = value->ival; break;
      case R64_R9:   cpu->__r9     = value->ival; break;
      case R64_R10:  cpu->__r10    = value->ival; break;
      case R64_R11:  cpu->__r11    = value->ival; break;
      case R64_R12:  cpu->__r12    = value->ival; break;
      case R64_R13:  cpu->__r13    = value->ival; break;
      case R64_R14:  cpu->__r14    = value->ival; break;
      case R64_R15:  cpu->__r15    = value->ival; break;
      case R_DS:
      case R_ES:
      case R_SS:
        break;
#else
      case R_DS:     cpu->__ds     = value->ival; break;
      case R_ES:     cpu->__es     = value->ival; break;
      case R_SS:     cpu->__ss     = value->ival; break;
#endif
      case R_CS:     cpu->__cs     = value->ival; break;
      case R_FS:     cpu->__fs     = value->ival; break;
      case R_GS:     cpu->__gs     = value->ival; break;
      case R_EAX:    cpu->__eax    = value->ival; break;
      case R_EBX:    cpu->__ebx    = value->ival; break;
      case R_ECX:    cpu->__ecx    = value->ival; break;
      case R_EDX:    cpu->__edx    = value->ival; break;
      case R_ESI:    cpu->__esi    = value->ival; break;
      case R_EDI:    cpu->__edi    = value->ival; break;
      case R_EBP:    cpu->__ebp    = value->ival; break;
      case R_ESP:    cpu->__esp    = value->ival; break;
      case R_EIP:    cpu->__eip    = value->ival; break;
      case R_EFLAGS: cpu->__eflags = value->ival; break;
      default: return false;
    }
  }
  else // FPU related register
  {
    if ( reg_idx >= R_ST0+FPU_REGS_COUNT ) // FPU status registers
    {
      switch ( reg_idx )
      {
        case R_CTRL: *(ushort*)&fpu->__fpu_fcw = value->ival; break;
        case R_STAT: *(ushort*)&fpu->__fpu_fsw = value->ival; break;
        case R_TAGS:            fpu->__fpu_ftw = value->ival; break;
      }
    }
    else // FPU floating point register
    {
      long double *fptr = (long double*)&fpu->__fpu_stmm0;
      fptr += reg_idx - R_ST0;
      *fptr = *(long double *)&value->fval;
    }
  }
#else // arm
  switch ( reg_idx )
  {
    case  0: cpu->__r[ 0] = value->ival; break;
    case  1: cpu->__r[ 1] = value->ival; break;
    case  2: cpu->__r[ 2] = value->ival; break;
    case  3: cpu->__r[ 3] = value->ival; break;
    case  4: cpu->__r[ 4] = value->ival; break;
    case  5: cpu->__r[ 5] = value->ival; break;
    case  6: cpu->__r[ 6] = value->ival; break;
    case  7: cpu->__r[ 7] = value->ival; break;
    case  8: cpu->__r[ 8] = value->ival; break;
    case  9: cpu->__r[ 9] = value->ival; break;
    case 10: cpu->__r[10] = value->ival; break;
    case 11: cpu->__r[11] = value->ival; break;
    case 12: cpu->__r[12] = value->ival; break;
    case 13: cpu->__sp    = value->ival; break;
    case 14: cpu->__lr    = value->ival; break;
    case 15: cpu->__pc    = value->ival; break;
    case 16: cpu->__cpsr  = value->ival; break;
    default: return false;
  }
#endif
  return true;
}

//--------------------------------------------------------------------------
// 1-ok, 0-failed
int idaapi mac_debmod_t::dbg_thread_write_register(thid_t tid, int reg_idx, const regval_t *value)
{
  if ( value == NULL )
    return false;

  if ( !is_fpu_reg(reg_idx) )
  {
    machine_thread_state_t cpu;
    if ( !get_thread_state(tid, &cpu) )
      return false;

    if ( !patch_reg_context(&cpu, NULL, reg_idx, value) )
      return false;

    return set_thread_state(tid, &cpu);
  }
#if defined (__i386__) || defined(__x86_64__)
  else // fpu register
  {
    machine_float_state_t fpu;
    if ( !get_float_state(tid, &fpu) )
      return false;

    if ( !patch_reg_context(NULL, &fpu, reg_idx, value) )
      return false;

    return set_float_state(tid, &fpu);
  }
#endif
}

//--------------------------------------------------------------------------
bool idaapi mac_debmod_t::thread_write_registers(
  thid_t tid,
  int start,
  int count,
  const regval_t *values,
  const int *indices)
{
  machine_thread_state_t cpu;
  bool got_regs = false;
#if defined (__i386__) || defined(__x86_64__)
  machine_float_state_t fpu;
  bool got_i387 = false;
#endif

  for ( int i=0; i < count; i++, values++ )
  {
    int idx = indices != NULL ? indices[i] : start+i;
    if ( !is_fpu_reg(idx) )
    { // general register
      if ( !got_regs )
      {
        if ( !get_thread_state(tid, &cpu) )
          return false;
        got_regs = true;
      }
    }
#if defined (__i386__) || defined(__x86_64__)
    else
    { // fpu register
      if ( !got_i387 )
      {
        if ( !get_float_state(tid, &fpu) )
          return false;
        got_i387 = true;
      }
    }
#endif
    if ( !patch_reg_context(&cpu, &fpu, idx, values) )
      return false;
  }

  if ( got_regs && !set_thread_state(tid, &cpu) )
    return false;

#if defined (__i386__) || defined(__x86_64__)
  if ( got_i387 && !set_float_state(tid, &fpu) )
    return false;
#endif

  return true;
}

//--------------------------------------------------------------------------
static mac_debmod_t *md;
static ssize_t read_mem_helper(ea_t ea, void *buffer, int size)
{
  return md->read_mem(ea, buffer, size) == KERN_SUCCESS ? size : 0;
}

bool mac_debmod_t::is_dylib_header(ea_t base, char *filename, size_t namesize)
{
  lock_begin();
  md = this;
  bool ok = ::is_dylib_header(base, read_mem_helper, filename, namesize);
  lock_end();
  return ok;
}

//--------------------------------------------------------------------------
// find a dll in the memory information array
bool mac_debmod_t::exist_dll(const dyriv_t &riv, ea_t base)
{
  // dyld is never unloaded
  if ( base == dylib )
    return true;
  for ( int i=0; i < riv.size(); i++ )
    if ( riv[i].addr == base )
      return true;
  return false;
}

//--------------------------------------------------------------------------
void mac_debmod_t::update_dylib(void)
{
//  dmsg("start: update_dylib at %a\n", dylib_infos);
  if ( read_mem(dylib_infos, &dyri, sizeof(dyri)) == KERN_SUCCESS )
  {
    QASSERT(dyri.version >= 1);

    if ( dyri.num_info > 0 && dyri.info_array != 0 )
    {
      dyriv_t riv;
      riv.resize(dyri.num_info);
      int nbytes = dyri.num_info * sizeof(dyld_raw_info);
      if ( read_mem(dyri.info_array, &riv[0], nbytes) != KERN_SUCCESS )
        return;

//      show_hex(&riv[0], nbytes, "riv:\n");
      // remove unexisting dlls
      images_t::iterator p;
      for ( p=dlls.begin(); p != dlls.end(); )
      {
        if ( !exist_dll(riv, p->first) )
        {
          debug_event_t ev;
          ev.eid     = LIBRARY_UNLOAD;
          ev.pid     = pid;
          ev.tid     = maintid();
          ev.ea      = BADADDR;
          ev.handled = true;
          qstrncpy(ev.info, p->second.name.c_str(), sizeof(ev.info));
          events.enqueue(ev, IN_FRONT);
          dlls.erase(p++);
        }
        else
        {
          ++p;
        }
      }
      // add new dlls
      for ( int i=0; i < riv.size(); i++ )
      {
        // address zero is ignored
        if ( riv[i].addr == 0 )
          continue;
        p = dlls.find(riv[i].addr);
        if ( p == dlls.end() )
        {
          char buf[QMAXPATH];
          memset(buf, 0, sizeof(buf));
          read_mem(riv[i].name, buf, sizeof(buf)); // may fail because we don't know exact size
          buf[sizeof(buf)-1] = '\0';
//          dmsg("dll name at %a is '%s'\n", ea_t(riv[i].addr), riv[i].name, buf);
          add_dll(riv[i].addr, buf);
        }
      }
    }
  }
//  dmsg("end: update_dylib\n");
}

//--------------------------------------------------------------------------
void mac_debmod_t::init_dylib(ea_t addr, const char *fname)
{
  dmsg("%a: located dylib header and file '%s'\n", addr, fname);
  dylib = addr;

  add_dll(addr, fname);
  // immediately process it
  dbg_stopped_at_debug_event();

  // check if we just found the address of dyld raw information
  if ( dyri.version == 0 && dylib_infos != BADADDR )
  {
    read_mem(dylib_infos, &dyri, sizeof(dyri));
    if ( dyri.version > 7 )
    {
      //dwarning("dyld link version (%d) is newer than expected (7).", dyri.version);
    }
    // set a breakpoint for library loads/unloads
    dmsg("%a: setting bpt for library notifications\n", dyri.dyld_notify);
    uchar opcode[BPT_CODE_SIZE];
    read_mem(dyri.dyld_notify, opcode, sizeof(opcode));
    if ( memcmp(opcode, dyld_opcode, BPT_CODE_SIZE) != 0 )
      derror("Unexpected dyld_opcode in the debugger server (init_dylib): %x", *(uint32*)opcode);
    dbg_add_bpt(BPT_SOFT, dyri.dyld_notify, 0);
  }
}

//--------------------------------------------------------------------------
image_info_t *mac_debmod_t::get_image(ea_t addr)
{
  if ( dlls.empty() )
    return NULL;
  images_t::iterator p = dlls.lower_bound(addr);
  if ( p != dlls.end() && p->first == addr )
    return &p->second;
  if ( p == dlls.begin() )
    return NULL;
  --p;
  if ( p->first > addr )
    return NULL;
  image_info_t &im = p->second;
  if ( im.base+im.imagesize <= addr )
    return NULL;
  return &im;
}

//--------------------------------------------------------------------------
local const char *get_share_mode_name(unsigned char sm, char *buf, size_t bufsize)
{
  switch ( sm )
  {
    case SM_COW:             return "COW";
    case SM_PRIVATE:         return "PRIVATE";
    case SM_EMPTY:           return "EMPTY";
    case SM_SHARED:          return "SHARED";
    case SM_TRUESHARED:      return "TRUESHARED";
    case SM_PRIVATE_ALIASED: return "PRIV_ALIAS";
    case SM_SHARED_ALIASED:  return "SHRD_ALIAS";
  }                               // 1234567890
  qsnprintf(buf, bufsize, "%x", sm);
  return buf;
}

//--------------------------------------------------------------------------
int mac_debmod_t::get_memory_info(meminfo_vec_t &miv, bool suspend)
{
  if ( suspend && !suspend_all_threads() )
    return -1;
  if ( exited() )
    return -1;

  mach_vm_size_t size = 0;
  for ( mach_vm_address_t addr = 0; ; addr += size )
  {
    mach_port_t object_name; // unused
    vm_region_top_info_data_t info;
    mach_msg_type_number_t count = VM_REGION_TOP_INFO_COUNT;
    kern_return_t code = mach_vm_region(task, &addr, &size, VM_REGION_TOP_INFO,
                        (vm_region_info_t)&info, &count, &object_name);

//    debdeb("task=%d addr=%a size=%x err=%x\n", task, addr, size, code);
    if ( code != KERN_SUCCESS )
      break;

    // ignore segments at address 0
    if ( addr == 0 )
      continue;

    // find dylib in the memory if not found yet
    char fname[QMAXPATH];
    if ( dylib == BADADDR && is_dylib_header(addr, fname, sizeof(fname)) )
      init_dylib(addr, fname);

    mach_vm_address_t subaddr;
    mach_vm_size_t subsize = 0;
    mach_vm_address_t end = addr + size;
    for ( subaddr=addr; subaddr < end; subaddr += subsize )
    {
      natural_t depth = 1;
      vm_region_submap_info_data_64_t sinfo;
      mach_msg_type_number_t count = VM_REGION_SUBMAP_INFO_COUNT_64;
      kern_return_t code = mach_vm_region_recurse(task, &subaddr, &subsize, &depth,
                                         (vm_region_info_t)&sinfo, &count);
      if ( code != KERN_SUCCESS )
        break;

      memory_info_t &mi = miv.push_back();
      mi.startEA = subaddr;
      mi.endEA   = subaddr + subsize;
#ifdef __X64__
      if ( subaddr == 0x7FFFFFE00000 )
        mi.name = "COMMPAGE";
      mi.bitness = 2; // 64bit
#else
      mi.bitness = 1; // 32bit
#endif
      // check if we have information about this memory chunk
      image_info_t *im = get_image(subaddr);
      if ( im == NULL )
      {
        // it is not a good idea to hide any addresses because
        // they will be used by the program and the user will be
        // left blind, not seeing the executed instructions.
#if 0
        if ( is_shared_address(subaddr) )
        {
          // hide unloaded shared libraries???
          continue;
        }
#endif
      }
      else
      {
        mi.name = im->name;
      }
      mi.perm = 0;
      if ( sinfo.protection & 1 ) mi.perm |= SEGPERM_READ;
      if ( sinfo.protection & 2 ) mi.perm |= SEGPERM_WRITE;
      if ( sinfo.protection & 4 ) mi.perm |= SEGPERM_EXEC;
      char buf[40];
//      dmsg("%"FMT_64"x..%"FMT_64"x: share mode %s prot: %x name=%s\n", subaddr, subaddr+subsize, get_share_mode_name(sinfo.share_mode, buf, sizeof(buf)), sinfo.protection, mi.name.c_str());
    }
  }

#if !defined(__arm__) && !defined(__X64__)
  // add hidden dsmos data
  memory_info_t &mi = miv.push_back();
  mi.startEA = 0xFFFF0000;
  mi.endEA   = 0xFFFFF000;
  mi.bitness = 1; // 32bit
  mi.perm = 0;
  mi.name = "COMMPAGE";
#endif

  update_dylib();
  if ( suspend )
    resume_all_threads();
  return 1;
}

//--------------------------------------------------------------------------
int idaapi mac_debmod_t::dbg_get_memory_info(meminfo_vec_t &areas)
{
  int code = get_memory_info(areas, true);
  if ( code == 1 )
  {
    if ( same_as_oldmemcfg(areas) )
      code = -2;
    else
      save_oldmemcfg(areas);
  }
  return code;
}

//--------------------------------------------------------------------------
int idaapi mac_debmod_t::dbg_init(bool _debug_debugger)
{
  // remember if the input is a dll
  cleanup();
  cleanup_hwbpts();
  debug_debugger = _debug_debugger;
  return 3; // process_get_info, detach
}

//--------------------------------------------------------------------------
void idaapi mac_debmod_t::dbg_term(void)
{
  cleanup();
  cleanup_hwbpts();
}

//--------------------------------------------------------------------------
bool idaapi mac_debmod_t::thread_get_fs_base(thid_t tid, int reg_idx, ea_t *pea)
{
  qnotused(tid);
  qnotused(reg_idx);
  qnotused(pea);
  return false;
}

//--------------------------------------------------------------------------
qmutex_t g_mutex = NULL;

bool init_subsystem()
{
  if ( (g_mutex = qmutex_create()) == NULL)
    return false;

  return true;
}

//--------------------------------------------------------------------------
bool lock_begin()
{
  qmutex_lock(g_mutex);
  return true;
}

//--------------------------------------------------------------------------
bool lock_end()
{
  qmutex_unlock(g_mutex);
  return true;
}

//--------------------------------------------------------------------------
bool term_subsystem()
{
  qmutex_free(g_mutex);
  return true;
}

//--------------------------------------------------------------------------
debmod_t *create_debug_session()
{
  return new mac_debmod_t();
}
