//////////////////////////////////////////
//    Network Security Analysis Tool    //
//          (c) 1999 by Mixter          //
//                                      //
//           utility functions          //
//////////////////////////////////////////

#include "nsat.h"

#ifdef EBUG
extern ofstream dbug;
#endif

#ifdef PARANOID_CHECK
#ifndef DEBUG
#include <sys/resource.h>
#endif
#endif

extern sigjmp_buf stack;
extern ProgressIndicator pi;

int
isip (char *ip)
{
  int a, b, c, d;
  sscanf (ip, "%d.%d.%d.%d", &a, &b, &c, &d);
  if (a < 0)
    return 0;
  if (a > 255)
    return 0;
  if (b < 0)
    return 0;
  if (b > 255)
    return 0;
  if (c < 0)
    return 0;
  if (c > 255)
    return 0;
  if (d < 0)
    return 0;
  if (d > 255)
    return 0;
  return 1;
}

int 
incip (char *ip)
{
  if (strcmp (ntoa (pi.ipend), ip) == 0)
    return (0);

  int a, b, c, d;
  sscanf (ip, "%d.%d.%d.%d", &a, &b, &c, &d);
  d++;
  if (a >= 255)
    return (0);
  if (b >= 255)
    {
      b = 0;
      a++;
    }
  if (c >= 255)
    {
      c = 0;
      b++;
    }
  if (d >= 255)
    {
      d = 0;
      c++;
    }

  memset (ip, 0, strlen (ip));
  sprintf (ip, "%d.%d.%d.%d", a, b, c, d);
  return (1);
}

u_long 
resolve (char *host)
{
  struct hostent *he;
  struct sockaddr_in tmp;
  he = gethostbyname (host);
  if (he)
    {
      memcpy ((caddr_t) & tmp.sin_addr.s_addr, he->h_addr, he->h_length);
    }
  else
    {
#ifdef EBUG
      dbug << "Resolving " << host << ": " << strerror (errno) << ENTER;
#endif
      return (0);
    }
  return (tmp.sin_addr.s_addr);
}

char *
strlower (char *str)
{
  int i;
  for (i = 0; i < (int) strlen (str); i++)
    if (isupper (str[i]))
      str[i] = tolower (str[i]);
  return (str);
}

char *
strchop (char *str)
{
  int i;
  for (i = 0; i < (int) strlen (str); i++)
    if (str[i] == '\n' || str[i] == '\r')
      str[i] = '\0';
  return (str);
}

char *
strunf (char *str)
{
  int i;
  for (i = 0; i < (int) strlen (str); i++)
    if (str[i] == '\n' || str[i] == '\r')
      str[i] = ' ';
  return (str);
}

char *
strnsubst (char *str, int siz)
{
  int i;
  for (i = 0; i < (siz - 5); i++)
    if (str[i] == '\0' && str[i + 5] != '\0')
      str[i] = ' ';

  return (str);
}

u_short
cksum (u_short * buf, int nwords)
{
  unsigned long sum;
  for (sum = 0; nwords > 0; nwords--)
    sum += *buf++;
  sum = (sum >> 16) + (sum & 0xffff);
  sum += (sum >> 16);
  return ~sum;
}

// based on BASS's coward mode by Liraz Siri <liraz@bigfoot.com>

int
sysidle (int minimum)
{
  struct stat dummy;
  const char *tty = "/dev/tty", *ptyp = "/dev/ptyp", *pty = "pqrS";
  const char *order = "0123456789abcdef", *cpa, *cpb;
  char filename[strlen (tty) + 3];

  strcpy (filename, tty);
  filename[strlen (tty) + 1] = '\0';

  for (cpa = order; *cpa; cpa++)
    {
      filename[strlen (tty)] = *cpa;
      if (stat (filename, &dummy) < 0)
	continue;
      if ((time (NULL) - dummy.st_atime) < minimum)
	return (0);
    }
  filename[strlen (tty) + 2] = '\0';

  for (cpa = pty; *cpa; cpa++)
    {
      filename[strlen (tty)] = *cpa;
      for (cpb = order; *cpb; cpb++)
	{
	  filename[strlen (tty) + 1] = *cpb;
	  if (stat (filename, &dummy) < 0)
	    continue;
	  if ((time (NULL) - dummy.st_atime) < minimum)
	    return (0);
	}
    }

  strcpy (filename, ptyp);
  filename[strlen (ptyp) + 1] = '\0';
  for (cpa = order; *cpa; cpa++)
    {
      filename[strlen (ptyp)] = *cpa;
      if (stat (filename, &dummy) < 0)
	continue;
      if ((time (NULL) - dummy.st_atime) < minimum)
	return (0);
    }

  return (1);
}

void 
hook (int signum)
{
#ifdef EBUG
  dbug << "* Timeout on I/O operation (" << signum << ") - restoring stack context\n";
#endif
  alarm (0);

  siglongjmp (stack, 666);	// back to the future! :p
}

// segv/segbus evasion by guidob@synnergy.net

void
catchsegv (int signum)
{
#ifdef DEBUG
  dbug << "* Segment violation (longjmp?), resuming...\n";
#endif
  siglongjmp (stack, 666);	// siglongjmp never returns...
  exit (0);
}

void 
doh (int signum)
{
#ifdef EBUG
  dbug << "* Fatal signal " << signum << " - bailing out\n";
  dbug.flush ();
  dbug.close ();
#endif

  if (raise (SIGKILL) != 0)
    exit (-1);
}

void 
shh (int signum)
{
#ifdef EBUG
  dbug << "* Got suspicious signal " << signum << " - suspending thread\n";
#endif

  while (!sysidle (pi.idle))
    sleep (pi.idle);

  return;
}

int 
safe_fork (void)
{
  int i, p;

ugly_trick:
  p = fork ();

  if (p < 0)
    {				// oh great.
#ifdef EBUG
      dbug << "fork() failing: " << strerror (errno) << ", waiting to try again...\n";
#endif
      nice (-1);
      sleep (pi.timeout);
      goto ugly_trick;
    }

  if (p == 0)
    {
      sigset ();
      i = open ("/dev/tty", O_RDWR | O_NOCTTY);
      ioctl (i, TIOCNOTTY, NULL);
      close (i);
#ifdef PARANOID_CHECK
#ifndef DEBUG
      struct rlimit rl = {0,0};
      setrlimit (RLIMIT_CORE, &rl);
#endif
#endif
      if (pi.stealth)
	{
	  setpgrp ();
	  setsid ();
	}
    }

  return (p);
}

char *
ntoa (u_long ip)
{
  struct in_addr h;
  h.s_addr = ip;
  return (inet_ntoa (h));
}

void 
sigset (void)
{
  signal (SIGCHLD, SIG_IGN);
  signal (SIGTSTP, SIG_IGN);
  signal (SIGIO, SIG_IGN);
  signal (SIGSEGV, catchsegv);
  signal (SIGBUS, catchsegv);
  signal (SIGALRM, hook);
  signal (SIGPIPE, hook);
  signal (SIGURG, hook);
  signal (SIGHUP, doh);
  signal (SIGINT, doh);
  signal (SIGQUIT, doh);
  signal (SIGILL, doh);
  signal (SIGTRAP, doh);
  signal (SIGABRT, doh);
  signal (SIGIOT, doh);
  signal (SIGFPE, doh);
  signal (SIGTERM, doh);
  signal (SIGSTOP, doh);
  signal (SIGPROF, doh);
  signal (SIGPWR, doh);
  if (pi.stealth)
    {
      signal (SIGTTIN, shh);
      signal (SIGTTOU, shh);
      signal (SIGXCPU, shh);
      signal (SIGXFSZ, shh);
      signal (SIGVTALRM, shh);
    }
}

void 
usage (char *name)
{
  cout << "usage:\n";
  cout << "\t" << name << " [options] -h <hostname>\n";
  cout << "\t" << name << " [options] -f <input file>\n";
  cout << "\t" << name << " [options] -s <start ip> -e <end ip>\n\n";
  cout << "options:\n";
  cout << "\t-n\t\tdon't go into background and report status to console\n";
  cout << "\t-t <n>\t\tset connection timeout to n seconds\n";
  cout << "\t-m <n>\t\tset maximum process count to n\n";
  cout << "\t-l <n>\t\tset maximum seconds to spend scanning one host\n";
  cout << "\t-i <n>\t\tset idle time for coward mode to n seconds\n";
  cout << "\t-p <0/1>\tping dependence: don't scan on ping timeouts\n";
  cout << "\t-c <0/1>\tcoward mode: watch out for user activity\n";
  cout << "\t-v <0-3>\tscan intensity: select scan level (default: 3)\n";
  exit (-1);
}

void 
writepid (int pid)
{
  FILE *f = fopen (PIDFILE, "w");
  if (f == NULL)
    {
      cerr << "* cannot open pid file, DOH\n";
      cerr.flush ();
      exit (-1);
    }
  fprintf (f, "%d", pid);
  fclose (f);
}

int 
readpid (void)
{
  FILE *f = fopen (PIDFILE, "r");
  int pid;
  if (f == NULL)
    return (0);			// file not there, nsat not running

  fscanf (f, "%d", &pid);
  fclose (f);
  kill (pid, SIGCHLD);
  if (errno == ESRCH)
    {				// process doesnt exist, nsat not running

      unlink (PIDFILE);
      return (0);
    }
  return (1);			// old process exists, nsat running

}

#ifdef PARANOID_CHECK
int 
nicefile (char *name)
{
  struct stat status;
  if ((name[0] == '/') || (strstr (name, "..")))
    return (1);
  if (lstat (name, &status) != -1)
    if (S_ISLNK (status.st_mode))
      return (1);		// ouch, hax0r ;/

  return (0);
}
#endif
