//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//                  Network Security Analysis Tool                          //
//                 AuditSet.cpp - auditing functions                        //
//                                                                          //
//   Copyright (C) 1999-2000 by Mixter and 2xs ltd. <mixter@2xs.co.il>      //
//                                                                          //
// This program is free software; you can redistribute it and/or modify     //
// it under the terms of the GNU General Public License as published by     //
// the Free Software Foundation; either version 2 of the License, or        //
// (at your option) any later version.                                      //
//                                                                          //
// This program is distributed in the hope that it will be useful,          //
// but WITHOUT ANY WARRANTY; without even the implied warranty of           //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            //
// GNU General Public License for more details.                             //
//                                                                          //
// You should have received a copy of the GNU General Public License        //
// along with this program; if not, write to the Free Software              //
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#pragma implementation "auditset"
#include "AuditSet.h"
#include "mod/shellcode.h"

extern sigjmp_buf stack;
extern ProgressIndicator pi;

#ifdef EBUG
extern ofstream dbug;
#endif

// Feel free to add ports to this list
struct portlist ports[] =
{
  {1, "tcpmux", 0},
  {7, "echo", 0},
  {9, "discard", 0},
  {11, "systat", 0},
  {13, "daytime", 0},
  {15, "netstat", 0},
  {17, "qotd", 0},
  {19, "chargen", 0},
  {20, "ftp-data", 0},
  {21, "ftp", 0},
  {22, "ssh", 0},
  {23, "telnet", 0},
  {25, "smtpd", 0},
  {37, "time", 0},
  {43, "nicname", 0},
  {48, "auditd", 0},
  {49, "tacacs", 0},
  {53, "bind", 0},
  {70, "gopher", 0},
  {79, "finger", 0},
  {80, "http", 0},
  {88, "kerberos", 0},
  {98, "linuxconf", 0},
  {109, "pop2", 1},
  {110, "pop3", 0},
  {111, "portmap", 0},
  {113, "auth", 0},
  {115, "sftp", 0},
  {119, "nntp", 0},
  {123, "ntp", 0},
  {139, "netbios", 0},
  {143, "imap", 0},
  {179, "bgp", 1},
  {213, "ipx", 0},
  {261, "FW-1 AuthAgent", 0},
  {389, "ldap", 0},
  {443, "https", 0},
  {512, "rexecd", 0},
  {513, "rlogind", 0},
  {514, "rshd", 0},
  {515, "lpd", 0},
  {543, "klogin", 0},
  {540, "uucp", 0},
  {544, "kshell", 0},
  {600, "600 (common backdoor port)", 2},
  {666, "666 (common backdoor port)", 2},
  {762, "quotad", 1},
  {774, "rpasswd", 0},
  {873, "rsync", 0},
  {992, "telnets", 0},
  {993, "imaps", 0},
  {994, "ircs", 0},
  {995, "pop3s", 0},
  {1080, "SOCKS5", 0},
  {1114, "mSQL", 0},
  {1243, "1243 (subseven backdoor port)", 2},
  {1337, "1337 (common backdoor port)", 2},
  {1433, "MS-SQL", 1},
  {1524, "1524 (common backdoor port or ingreslock)", 1},
  {1525, "Oracle SQL", 0},
  {1999, "Cisco CDP (1999)", 0},
  {2000, "OpenWin", 0},
  {2049, "nfsd", 1},
  {2222, "2222 (common backdoor port)", 2},
  {2223, "mdbms", 0},
  {2224, "mdbms (fastport)", 0},
  {2301, "cim", 0},
  {2606, "netmon", 1},
  {2766, "nlpsd", 1},
  {2998, "realsecure", 0},
  {3007, "lotusmtap", 0},
  {3128, "squid", 0},
  {3279, "admind", 0},
  {3306, "mysql", 0},
  {3333, "3333 (eggdrop default port)", 0},
  {3346, "transparent proxy", 0},
  {5556, "HP-UX RemoteWatch", 0},
  {6000, "XWindows", 0},
  {6667, "ircd", 0},
  {7161, "Cisco DoS (7161)", 0},
  {7070, "RealServer DoS (7070)", 1},
  {8080, "WebProxy (8080)", 0},
  {12345, "12345 (common backdoor port)", 2},
  {22222, "22222 (common backdoor port)", 2},
  {26000, "quake", 1},
  {27655, "trinoo master (backdoor)", 1},
  {31335, "trinoo4 master (backdoor)", 1},
  {31337, "31337 (common backdoor port)", 2},
  {65535, "65535 (common backdoor port)", 2},
  {0, "", 0}
};

AuditSet::AuditSet(void)
{
  firstopen = 0;
  sl = sizeof(sin);
  memset(&sin, 0, sl);
  sin.sin_family = AF_INET;
  sin.sin_port = htons(0);
  signal(SIGALRM, hook);
  signal(SIGPIPE, hook);
}

AuditSet::~AuditSet(void)
{
  firstopen = 0;
  signal(SIGALRM, SIG_DFL);
  signal(SIGPIPE, SIG_DFL);
}

inline char *AuditSet::Hostname(void)
{
  return (ntoa(target));
}

int AuditSet::icmptest(void)
{
  int s = isock();
  fd_set st;
  struct timeval tv = {timeout * 2, 0};
  char packet[512];
  struct icmp *ich = (struct icmp *) packet;
  struct sockaddr_in sa;
  long ref;

  if (pi.vhostip)
    {
     memset(&sin, 0, sizeof(sin));
     sin.sin_family = AF_INET;
     sin.sin_port = htons(0);
     sin.sin_addr.s_addr = pi.vhostip;
     bind(s, (struct sockaddr *) &sin, sizeof(sin));
    }

  if (geteuid())
    return (1);

  memset(packet, 0, 512);
  ich->type = 8;		// echo query

  ich->id = getrandom(0, 65535);
  ich->seq = getrandom(0, 65535);
  ich->sum = 0;
  ich->sum = sum((unsigned short *) ich, (64 >> 1));
  sa.sin_family = AF_INET;
  sa.sin_port = 0;
  sa.sin_addr.s_addr = target;
  FD_ZERO(&st);
  ref = time(NULL);
  sendto(s, packet, 64, 0, (struct sockaddr *) &sa, sizeof(sa));
  FD_SET(s, &st);
  if (!select(s + 1, &st, NULL, NULL, &tv))
    {
      if (pi.VerboseLogging)
	{
	  ofstream l0g;

	  l0g.open(L_ICMP, 8);
	  l0g << I_PINGOUT << Hostname() << ENTER;
	  if (pi.Foreground)
	    cout << I_ICMP << I_PINGOUT << Hostname() << ENTER;
	  l0g.close();
	}
      SCLOSE(s);
      return (0);
    }
  else
    {
      unsigned int slen = sizeof(sa);

      recvfrom(s, &packet, 64, 0, (struct sockaddr *) &sa, (socklen_t *)&slen);
      ofstream l0g;

      l0g.open(L_ICMP, 8);
      l0g << Hostname() << I_S << (time(NULL) - ref) << "s\n";
      if (pi.Foreground)
	cout << I_ICMP << Hostname() << I_S << (time(NULL) - ref) << "s\n";
      if (pi.ScanIcmp < 2)
	{
	  l0g.close();
	  SCLOSE(s);
	  return (1);
	}

      l0g.flush();
      /* time stamp, len = 20 */
      tv.tv_sec = timeout * 2;
      tv.tv_usec = 0;
      sa.sin_family = AF_INET;
      sa.sin_port = 0;
      sa.sin_addr.s_addr = target;
      memset(packet, 0, 512);
      ich->seq = getrandom(0, 65535);
      ich->sum = 0;
      ich->type = 13;		// timestamp query

      ich->code = 0;
      gettimeofday((struct timeval *) (ich + 8), NULL);
      ich->sum = sum((unsigned short *) ich, (64 >> 1));
      FD_ZERO(&st);
      FD_SET(s, &st);
      /* the recv stuff is not checked, but it works 99% of the time */
      sendto(s, packet, 21, 0, (struct sockaddr *) &sa, sizeof(sa));
      if (!select(s + 1, &st, NULL, NULL, &tv))
	{
	  l0g.close();
	  SCLOSE(s);
	  return (1);
	}
      if (recvfrom(s, &packet, 500, 0, (struct sockaddr *) &sa, (socklen_t *)&slen) > 0)
	l0g << Hostname() << I_ICTIME;
      l0g.flush();

      /* mask request, len = 12 */
      sa.sin_family = AF_INET;
      sa.sin_port = 0;
      sa.sin_addr.s_addr = target;
      memset(packet, 0, 512);
      ich->seq = getrandom(0, 65535);
      ich->sum = 0;
      ich->type = 17;		// netmask query

      ich->code = 0;
      *((char *) (ich + 8)) = 255;
      ich->sum = sum((unsigned short *) ich, (64 >> 1));
      FD_ZERO(&st);
      FD_SET(s, &st);
      sendto(s, packet, 13, 0, (struct sockaddr *) &sa, sizeof(sa));
      if (!select(s + 1, &st, NULL, NULL, &tv))
	{
	  l0g.close();
	  return (1);
	}
      if (recvfrom(s, &packet, 500, 0, (struct sockaddr *) &sa, (socklen_t *)&slen) > 0)
	l0g << Hostname() << I_ICMASK;
      l0g.close();
    }

  SCLOSE(s);
  return (1);
}

void AuditSet::plog(int s)
{
  if (getpeername(s, (struct sockaddr *) &sin, (socklen_t *)&sl) == -1)
    {
#ifdef EBUG
      dbug << "getpeername failed: " << strerror(errno) << ENTER;
#endif
      SCLOSE(s);
      return;
    }
  char *desc = NULL;
  int vuln = 0;

  for (int n = 0; (ports[n].port != 65535); n++)
    if (ports[n].port == ntohs(sin.sin_port))
      {
	desc = ports[n].info;
	vuln = ports[n].vuln;
      }

  log.open(L_PORTS, 8);
  if (desc == NULL)
    {
      log << Hostname() << I_S << ntohs(sin.sin_port) << (char *) (vuln ? M_DANGEROUS_SVC : "") << ENTER;
      if (pi.Foreground)
	cout << I_PORTL << Hostname() << I_S << ntohs(sin.sin_port) << (char *) (vuln ? M_DANGEROUS_SVC : "") << ENTER;
    }
  else
    {
      log << Hostname() << I_S << desc << (char *) (vuln ? M_DANGEROUS_SVC : "") << ENTER;
      if (pi.Foreground)
	cout << I_PORTL << Hostname() << I_S << desc << (char *) (vuln ? M_DANGEROUS_SVC : "") << ENTER;
    }

  log.close();

  if ((vuln == 2) && (pi.ScanBackdoor))
    backdoor(s, sin.sin_port);
  else
    SCLOSE(s);

  return;
}

void AuditSet::nstat(int s)
{
  log.open(L_PORTS, 8);
  log << Hostname() << I_NETSTAT;
  if (pi.Foreground)
    cout << I_PORTL << Hostname() << I_NETSTAT;
  log.close();
  if (pi.ScanNetstat < 2)
    {
      SCLOSE(s);
      return;
    }

  log.open(L_NETSTAT, 8);
  log << M_LBANNER << Hostname() << M_RBANNER;
  if (pi.Foreground)
    cout << I_NSTAT << M_LBANNER << Hostname() << M_RBANNER;

  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  alarm(timeout);
  if (sigsetjmp(stack, 1) == 666)
    {
      SCLOSE(s);
      log.close();
      alarm(0);
      return;
    }
  read(s, b, BUFSIZE - 1);
  alarm(0);
  log << b << ENTER;
  if (pi.Foreground)
    cout << b << ENTER;
  log.close();
  SCLOSE(s);
}

void AuditSet::ftp(int s)
{
  alarm(timeout * 5);
  if (sigsetjmp(stack, 1) != 666)
    {
      log.open(L_FTP, 8);
      char b[BUFSIZE];
      int l = 1;

      memset(b, 0, BUFSIZE);
      while (strstr(b, ENTER) == NULL && l > 0)
	l = read(s, b + strlen(b), BUFSIZE - strlen(b) - 1);
      log << Hostname() << I_S << b;
      if (pi.Foreground)
	cout << I_FTP << Hostname() << I_S << b;
      if (pi.ScanFtp > 1)
	ftptest(s, Hostname());
    }
  siginterrupt(SIGALRM, 1);
  log.close();
  alarm(0);
  SCLOSE(s);
}

void AuditSet::ssh(int s)
{
  log.open(L_SSH, 8);
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  alarm(timeout);
  if (sigsetjmp(stack, 1) == 666)
    {
      SCLOSE(s);
      alarm(0);
      log.close();
      return;
    }
  read(s, b, BUFSIZE - 1);
  strchop(b);
  alarm(0);
  log << Hostname() << I_S << b << ENTER;
  if (pi.Foreground)
    cout << I_SSH << Hostname() << I_S << b << ENTER;
  log.close();
  SCLOSE(s);
}

void AuditSet::telnet(int s)
{
  log.open(L_TELNET, 8);
  char b[BUFSIZE * 2];

  if (pi.ScanTelnet < 2)
    {
      log << I_TELNET << Hostname() << ENTER;
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << ENTER;
      log.close();
      SCLOSE(s);
      return;
    }

  alarm(timeout * 10);
  if (sigsetjmp(stack, 1) == 666)
    {
      log.close();
      SCLOSE(s);
      siginterrupt(SIGALRM, 1);
      return;
    }

  int len = 0;

  while (len != 1337)
    {
      memset(b, 0, sizeof(b));
      if ((len = read(s, b, 1)) <= 0)
	break;
      if (*b == '\xff')
	{
	  read(s, (b + 1), 2);
	  if (*(b + 1) == '\xfd' && !*(b + 2));
	  else if (*(b + 1) == '\xfd')
	    {
	      *(b + 1) = '\xfc';
	      write(s, b, 3);
	    }
	}
      else
	{
	  if (*b != 0)
	    {
	      memset(b, 0, sizeof(b));
	      read(s, b, sizeof(b));
	      len = 1337;
	    }
	}
    }

  alarm(0);
  SCLOSE(s);
  my_strnsubst(b, sizeof(b));
  strlower(b);

  if (strlen(b) < 2)
    {
      log << Hostname() << " - ALL:ALL in hosts.deny?\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - ALL:ALL in hosts.deny?\n";
      log.close();
      return;
    }
  if (strstr(b, "irix"))
    {
      log << Hostname() << " - IRIX\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - IRIX\n";
      log.close();
      return;
    }
  if (strstr(b, "system v"))
    {
      log << Hostname() << " - System V banner\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - System V banner\n";
      log.close();
      return;
    }
  if (strstr(b, "solaris"))
    {
      log << Hostname() << " - Solaris\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - Solaris\n";
      log.close();
      return;
    }
  if (strstr(b, "sunos"))
    {
      log << Hostname() << " - SunOS\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - SunOS\n";
      log.close();
      return;
    }
  if (strstr(b, "wingate"))
    {
      log << Hostname() << " - Open WinGate\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - Open WinGate\n";
      log.close();
      return;
    }
  if (strstr(b, "red hat"))
    {
      log << Hostname() << " - RedHat Linux\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - RedHat Linux\n";
      log.close();
      return;
    }
  if (strstr(b, "suse"))
    {
      log << Hostname() << " - SuSE Linux\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - SuSE Linux\n";
      log.close();
      return;
    }
  if (strstr(b, "slackware"))
    {
      log << Hostname() << " - Slackware Linux\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - Slackware Linux\n";
      log.close();
      return;
    }
  if (strstr(b, "debian"))
    {
      log << Hostname() << " - Debian Linux\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - Debian Linux\n";
      log.close();
      return;
    }
  if (strstr(b, "linux"))
    {
      log << Hostname() << " - Linux\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - Linux\n";
      log.close();
      return;
    }
  if (strstr(b, "freebsd"))
    {
      log << Hostname() << " - FreeBSD\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - FreeBSD\n";
      log.close();
      return;
    }
  if (strstr(b, "netbsd"))
    {
      log << Hostname() << " - NetBSD\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - NetBSD\n";
      log.close();
      return;
    }
  if (strstr(b, "openbsd"))
    {
      log << Hostname() << " - OpenBSD\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - OpenBSD\n";
      log.close();
      return;
    }
  if (strstr(b, "bsdi"))
    {
      log << Hostname() << " - BSDi\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - BSDi\n";
      log.close();
      return;
    }
  if (strstr(b, "bsd"))
    {
      log << Hostname() << " - BSD\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - BSD\n";
      log.close();
      return;
    }
  if (strstr(b, "cisco"))
    {
      log << Hostname() << " - CISCO IOS\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - CISCO IOS\n";
      log.close();
      return;
    }
  if (strstr(b, "sco"))
    {
      log << Hostname() << " - SCO UNIX\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - SCO UNIX\n";
      log.close();
      return;
    }
  if (strstr(b, "vms"))
    {
      log << Hostname() << " - OpenVMS/VAX\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - OpenVMS/VAX\n";
      log.close();
      return;
    }
  if (strstr(b, "hp"))
    {
      log << Hostname() << " - HP-UNIX\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - HP-UNIX\n";
      log.close();
      return;
    }
  if (strstr(b, "next"))
    {
      log << Hostname() << " - NeXT\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - NeXT\n";
      log.close();
      return;
    }
  if (strstr(b, "dg"))
    {
      log << Hostname() << " - DG-UNIX\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - DG-UNIX\n";
      log.close();
      return;
    }
  if (strstr(b, "user access"))
    {
      log << Hostname() << " - running radius / router\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - running radius / router\n";
      log.close();
      return;
    }
  if (strstr(b, "ascend"))
    {
      log << Hostname() << " - insecure ascend router\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - insecure ascend router\n";
      log.close();
      return;
    }
  if (strstr(b, "cisco"))
    {
      log << Hostname() << " - insecure cisco router\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - insecure cisco router\n";
      log.close();
      return;
    }
  if (strstr(b, "password"))
    {
      log << Hostname() << " - system password (probably router)\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - system password (probably router)\n";
      log.close();
      return;
    }
  if (strstr(b, "telnet"))
    {
      log << Hostname() << " - unknown gateway/proxy\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - unknown gateway/proxy\n";
      log.close();
      return;
    }
  if (strstr(b, "gate"))
    {
      log << Hostname() << " - unknown gateway/proxy\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - unknown gateway/proxy\n";
      log.close();
      return;
    }
  if (strstr(b, "login"))
    {
      log << Hostname() << " - unknown login\n";
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << " - unknown login\n";
      log.close();
      return;
    }
  if (pi.VerboseLogging)
    {
      log << Hostname() << I_UNKNOWN;
      if (pi.Foreground)
	cout << I_TELNET << Hostname() << I_UNKNOWN;
    }
  if (pi.ScanTelnet == 3)
    {
      log << Hostname() << b << ENTER;
      if (pi.Foreground)
	cout << Hostname() << b << ENTER;
    }
  log.close();
  return;
}

void AuditSet::smtp(int s)
{
  log.open(L_SMTP, 8);
  int siz = BUFSIZE * 5;
  char b[siz + 1];

  if (sigsetjmp(stack, 1) == 666)
    {
      log.close();
      siginterrupt(SIGALRM, 1);
      SCLOSE(s);
      return;
    }
  alarm((timeout) * 10);
  memset(b, 0, siz);
  while (strstr(b, ENTER) == NULL)
    read(s, b + strlen(b), siz - strlen(b));

  log << Hostname() << I_S << b;
  if (pi.Foreground)
    cout << I_SMTP << Hostname() << I_S << b;

  if (pi.ScanSmtp < 2)
    {
      alarm(0);
      log.close();
      sleep(1);
      SCLOSE(s);
      return;
    }

  if (strstr(b, "8.9") != NULL)
    smcheck = 1;
  else
    smcheck = 0;

  write(s, SMTP_HELO, strlen(SMTP_HELO));
  memset(b, 0, siz);
  while (strstr(b, ENTER) == NULL)
    read(s, b + strlen(b), siz - strlen(b));

  memset(b, 0, siz);
  write(s, SMTP_EXPN, strlen(SMTP_EXPN));
  while (strstr(b, ENTER) == NULL)
    read(s, b + strlen(b), siz - strlen(b));
  if (strstr(b, "250"))
    {
      log << Hostname() << I_EXPN;
      if (pi.Foreground)
	cout << I_SMTP << Hostname() << I_EXPN;
      log.flush();
    }
  else
    {
      write(s, SMTP_VRFY, strlen(SMTP_VRFY));
      memset(b, 0, siz);
      while (strstr(b, ENTER) == NULL)
	read(s, b + strlen(b), siz - strlen(b));
      if (strstr(b, "250"))
	log << Hostname() << I_VRFY;
      if (pi.Foreground)
	cout << I_SMTP << Hostname() << I_VRFY;
      if (pi.ScanSmtp < 3)
	{
	  write(s, SMTP_QUIT, strlen(SMTP_QUIT));
	  alarm(0);
	  log.close();
	  sleep(1);
	  SCLOSE(s);
	  return;
	}
      else
	log.flush();
    }

  write(s, SMTP_FROM, strlen(SMTP_FROM));
  memset(b, 0, siz);
  while (strstr(b, ENTER) == NULL)
    read(s, b + strlen(b), siz - strlen(b));
  write(s, SMTP_TO, strlen(SMTP_TO));
  memset(b, 0, siz);
  while (strstr(b, ENTER) == NULL)
    read(s, b + strlen(b), siz - strlen(b));
  write(s, SMTP_DATA, strlen(SMTP_DATA));
  strunf(b);
  while (strstr(b, ENTER) == NULL)
    read(s, b + strlen(b), siz - strlen(b));
  write(s, SMTP_END, strlen(SMTP_END));
  strunf(b);
  while (strstr(b, ENTER) == NULL)
    read(s, b + strlen(b), siz - strlen(b));
  if (strstr(b, "250"))
    {
      log << Hostname() << I_FAKEMAIL;
      if (pi.Foreground)
	cout << I_SMTP << Hostname() << I_FAKEMAIL;
    }
  if (strstr(b, "354"))
    {
      log << Hostname() << I_SPAM;
      if (pi.Foreground)
	cout << I_SMTP << Hostname() << I_SPAM;
    }
  if ((pi.ScanSmtp == 4) && (smcheck))
    {
      memset(b, 0, siz);
      write(s, SMTP_FROM, strlen(SMTP_FROM));
      memset(b, 0, siz);
      while (strstr(b, ENTER) == NULL)
	read(s, b + strlen(b), siz - strlen(b));
      write(s, SMTP_TO89, strlen(SMTP_TO89));
      memset(b, 0, siz);
      while (strstr(b, ENTER) == NULL)
	read(s, b + strlen(b), siz - strlen(b));
      if (strstr(b, "250"))
	{
	  log << Hostname() << I_RPMMAIL;
	  if (pi.Foreground)
	    cout << I_SMTP << Hostname() << I_RPMMAIL;
	}
    }
  write(s, SMTP_QUIT, strlen(SMTP_QUIT));
  strunf(b);
  while (strstr(b, ENTER) == NULL)
    read(s, b + strlen(b), siz - strlen(b));
  alarm(0);

  log.close();
  sleep(1);			/* avoid ugly sendmail syslog errors... */
  SCLOSE(s);
}

void AuditSet::nbind(int s)
{
  SCLOSE(s);
  if (sigsetjmp(stack, 1) != 666)
    probe_bind(sin, timeout);
  siginterrupt(SIGALRM, 1);
}

void AuditSet::finger(int s)
{
  if (sigsetjmp(stack, 1) == 666)
    {
      log.close();
      SCLOSE(s);
      siginterrupt(SIGALRM, 1);
      return;
    }

  if (pi.ScanFinger < 2)
    {
      SCLOSE(s);
      log.open(L_PORTS, 8);
      log << Hostname() << I_FINGER;
      if (pi.Foreground)
	cout << I_PORTL << Hostname() << I_FINGER;
      log.close();
      return;
    }

  alarm(timeout * 3);

  if (write(s, FINGER_HELLCODE, sizeof(FINGER_HELLCODE)) == -1)
    {
#ifdef EBUG
      dbug << "write failed: " << strerror(errno) << ENTER;
#endif
      alarm(0);
      SCLOSE(s);
      return;
    }
  log.open(L_PORTS, 8);
  log << Hostname() << I_FINGER;
  if (pi.Foreground)
    cout << I_PORTL << Hostname() << I_FINGER;
  log.close();
  char b[BUFSIZE * 2];

  memset(b, 0, BUFSIZE * 2);
  int i = read(s, b, (BUFSIZE * 2) - 1);

  alarm(0);
  if (strlen(my_strnsubst(b, i + 1)) > 2)
    {
      log.open(L_FINGER, 8);
      log << M_LBANNER << Hostname() << M_RBANNER;
      log << b << ENTER;
      if (pi.Foreground)
	{
	  cout << I_FING << M_LBANNER << Hostname() << M_RBANNER;
	  cout << b << ENTER;
	}
      log.close();
    }
  SCLOSE(s);
}

void AuditSet::http(int s)
{
  if (sigsetjmp(stack, 1) != 666)
    webscan(s);
  siginterrupt(SIGALRM, 1);
  SCLOSE(s);
}

void AuditSet::pop2(int s)
{
  alarm(timeout);
  if (sigsetjmp(stack, 1) == 666)
    {
      log.close();
      SCLOSE(s);
      siginterrupt(SIGALRM, 1);
      return;
    }
  log.open(L_POP2, 8);
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  read(s, b, BUFSIZE - 1);
  alarm(0);
  log << Hostname() << I_S << b << ENTER;
  if (pi.Foreground)
    cout << I_POP2 << Hostname() << I_S << b << ENTER;
  log.close();
  SCLOSE(s);
}

void AuditSet::pop3(int s)
{
  alarm(timeout);
  if (sigsetjmp(stack, 1) == 666)
    {
      log.close();
      SCLOSE(s);
      siginterrupt(SIGALRM, 1);
      return;
    }
  log.open(L_POP3, 8);
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  read(s, b, BUFSIZE - 1);
  alarm(0);
  strchop(b);
  log << Hostname() << I_S << b << ENTER;
  if (pi.Foreground)
    cout << I_POP3 << Hostname() << I_S << b << ENTER;
  log.close();
  SCLOSE(s);
}

void AuditSet::pmap(int s)
{
  SCLOSE(s);
  log.open(L_PORTS, 8);
  log << Hostname() << I_PMAP;
  if (pi.Foreground)
    cout << I_PORTL << Hostname() << I_PMAP;
  log.close();
  if (pi.ScanRpc > 1)
    {
      alarm(timeout);
      if (sigsetjmp(stack, 1) == 666)
	{
	  siginterrupt(SIGALRM, 1);
	  return;
	}
      rpccheck(Hostname());
    }
}

void AuditSet::nntp(int s)
{
  alarm(timeout * 3);
  if (sigsetjmp(stack, 1) == 666)
    {
      log.close();
      SCLOSE(s);
      siginterrupt(SIGALRM, 1);
      return;
    }
  log.open(L_NNTP, 8);
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  read(s, b, BUFSIZE - 1);
  alarm(0);
  log << Hostname() << I_S << b << ENTER;
  if (pi.Foreground)
    cout << I_NNTP << Hostname() << I_S << b << ENTER;
  log.close();
  SCLOSE(s);
}

void AuditSet::smb(int s)
{
  SCLOSE(s);
  int pid = fork();

  if (pid == 0)
    {
      if (pi.Foreground)
	execlp(I_SMBEXEC, I_SMBEXEC, Hostname(), "1", NULL);
      else
	execlp(I_SMBEXEC, I_SMBEXEC, Hostname(), NULL);
      raise(SIGKILL);
    }
  else
    {
      sleep(timeout * 5);
      kill(pid, SIGKILL);
    }
}

void AuditSet::imap(int s)
{
  alarm(timeout * 3);
  if (sigsetjmp(stack, 1) == 666)
    {
      log.close();
      SCLOSE(s);
      siginterrupt(SIGALRM, 1);
      return;
    }
  log.open(L_IMAP, 8);
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  read(s, b, BUFSIZE - 1);
  alarm(0);
  log << Hostname() << I_S << b << ENTER;
  if (pi.Foreground)
    cout << I_IMAP << Hostname() << I_S << b << ENTER;
  log.close();
  SCLOSE(s);
}

void AuditSet::backdoor(int s, int pt)
{
  alarm(timeout * 2);
  if (sigsetjmp(stack, 1) == 666)
    {
      SCLOSE(s);
      siginterrupt(SIGALRM, 1);
      return;
    }
  if (send(s, ROOTINFO, sizeof(ROOTINFO), 0) == -1)
    {
#ifdef EBUG
      dbug << "send failed: " << strerror(errno) << ENTER;
#endif
      alarm(0);
      SCLOSE(s);
      return;
    }
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  recv(s, b, BUFSIZE - 1, 0);
  alarm(0);
  if (strlen(b) > 3)
    {
      log.open(L_BACKDOOR, 8);
      log << M_LBANNER << Hostname() << I_PORT << ntohs(pt) << M_RBANNER;
      log << b << ENTER;
      if (pi.Foreground)
	{
	  cout << I_BACKDOOR << M_LBANNER << Hostname() << I_PORT << ntohs(pt) << M_RBANNER;
	  cout << b << ENTER;
	}
      log.close();
    }
  SCLOSE(s);
}

void AuditSet::nlps(int s)
{
  log.open(L_PORTS, 8);
  log << Hostname() << I_NLPS;
  if (pi.Foreground)
    cout << I_PORTL << Hostname() << I_NLPS;
  log.close();
  if (pi.ScanNlps < 2)
    {
      SCLOSE(s);
      return;
    }
  if (sigsetjmp(stack, 1) == 666)
    {
      SCLOSE(s);
      siginterrupt(SIGALRM, 1);
      return;
    }
  alarm(timeout);
  write(s, nlps_x86, sizeof(nlps_x86));
  write(s, ROOTINFO, sizeof(ROOTINFO));
  char b[BUFSIZE];

  memset(b, 0, BUFSIZE);
  read(s, b, BUFSIZE - 1);
  alarm(0);
  if (strlen(b) > 2)
    {
      log.open(L_NLPS, 8);
      log << I_NLPL << M_LBANNER << Hostname() << M_RBANNER << b << ENTER;
      if (pi.Foreground)
	cout << M_LBANNER << Hostname() << M_RBANNER << b << ENTER;
      log.close();
    }
  SCLOSE(s);
}

void AuditSet::ircd(int s)
{
  log.open(L_PORTS, 8);
  log << Hostname() << I_IRCD;
  if (pi.Foreground)
    cout << I_PORTL << Hostname() << I_IRCD;
  log.close();
  if (pi.ScanIrcd < 2)
    {
      SCLOSE(s);
      return;
    }
  char b[BUFSIZE * 3];

  if (sigsetjmp(stack, 1) == 666)
    {
      SCLOSE(s);
      siginterrupt(SIGALRM, 1);
      return;
    }
  alarm((timeout) * 10);
  write(s, IRC_QUERY, sizeof(IRC_QUERY));
  while (!strstr(b, " 351 "))
    {
      memset(b, 0, (BUFSIZE * 3));
      if (!read(s, b, (BUFSIZE * 3) - 1))
	return;
    }
  alarm(0);
  log.open(L_IRCD, 8);
  log << M_LBANNER << Hostname() << M_RBANNER << b << ENTER;
  if (pi.Foreground)
    cout << I_IRC << M_LBANNER << Hostname() << M_RBANNER << b << ENTER;
  log.close();
  SCLOSE(s);
}

void AuditSet::xwin(int s)
{
  log.open(L_PORTS, 8);
  log << Hostname() << I_XWIN;
  if (pi.Foreground)
    cout << I_PORTL << Hostname() << I_XWIN;
  log.close();
  if (pi.ScanXWin < 2)
    {
      SCLOSE(s);
      return;
    }

#ifdef HAVE_X11_XLIB_H
  char disp[BUFSIZE];

  snprintf(disp, BUFSIZE - 1, "%s:0", Hostname());
  struct Display *d;

  if ((d = XOpenDisplay(disp)) != NULL)
    {
      log.open(L_XWIN, 8);
      log << Hostname() << I_XWIN2;
      if (pi.Foreground)
	cout << I_XWINL << Hostname() << I_XWIN2;
      log.close();
      XCloseDisplay(d);
    }

  SCLOSE(s);
#endif
}

void AuditSet::nbname(void)
{
/* raw nbname session from nbname.c (c) by mynock */
  const char *nbquery =
  "\x03\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x20\x43\x4b\x41\x41"
  "\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41"
  "\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x41\x00\x00\x21\x00\x00";
  int sockfd, i, n;
  struct sockaddr_in servaddr, myaddr;
  char buf[1024];
  fd_set readfs;
  struct timeval tv;
  bzero(&servaddr, sizeof(struct sockaddr_in));

  servaddr.sin_family = AF_INET;
  servaddr.sin_port = htons(137);
  servaddr.sin_addr.s_addr = target;
  sockfd = socket(AF_INET, SOCK_DGRAM, 0);
  memset(&myaddr, 0, sizeof(myaddr));
  myaddr.sin_family = AF_INET;
  myaddr.sin_port = htons(137);
  myaddr.sin_addr.s_addr = htonl(INADDR_ANY);
  bind(sockfd, (struct sockaddr *) &myaddr, sizeof(myaddr));
  while (1)
    {
      sendto(sockfd, nbquery, 50, 0, (struct sockaddr *) &servaddr, sizeof(struct sockaddr_in));

      FD_ZERO(&readfs);
      FD_SET(sockfd, &readfs);
      tv.tv_sec = 5;
      tv.tv_usec = 0;
      n = select(sockfd + 1, &readfs, NULL, NULL, &tv);
      switch (n)
	{
	case 1:
	  if ((i = recvfrom(sockfd, buf, 800, 0, NULL, NULL)) < 0)
	    {
	      SCLOSE(sockfd);
	      return;
	    }
	  break;
	case 0:
	  SCLOSE(sockfd);
	  return;
	  break;
	default:
	  SCLOSE(sockfd);
	  return;
	  break;
	}
      if (n == 1)
	break;
    }

  if (strlen(buf + 58) < 2)
    {
      SCLOSE(sockfd);
      return;
    }

  if (strlen(buf + 57) >= 1)
    {
      log.open(L_NETBIOS, 8);
      log << Hostname() << I_S;
      if (pi.Foreground)
	cout << I_NBIOS << Hostname() << I_S;
      for (n = 57; (n < i) && (buf[n] != ' '); n++)
	{
	  log << buf[n];
	  if (pi.Foreground)
	    cout << buf[n];
	}
      if (pi.Foreground)
	cout << ENTER;
      log << ENTER;
      log.close();
    }

  SCLOSE(sockfd);
  return;
}

void AuditSet::bo()
{
  int s = usock();

  memset(&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons(0);
  if (pi.vhostip)
    sin.sin_addr.s_addr = pi.vhostip;

  bind(s, (struct sockaddr *) &sin, sizeof(sin));

  alarm(timeout * 5);
  if (sigsetjmp(stack, 1) == 666)
    {
      SCLOSE(s);
      return;
    }

  if (!sendpacket(target, 31337, s))
    getinput(s);
#ifdef EBUG
  else
    dbug << "sendping failed: " << Hostname() << ENTER;
#endif

  SCLOSE(s);
}

void AuditSet::os(void)
{
  alarm(timeout * 5);
  if (sigsetjmp(stack, 1) == 666)
    return;

  if (firstopen)
    tcprobe(target, firstopen, timeout);
  else if (pi.VerboseLogging > 2)
    tcprobe(target, 113, timeout);

  alarm(0);
}

void AuditSet::snmp(void)
{
  alarm(timeout * 5);
  if (sigsetjmp(stack, 1) == 666)
    return;
  snmpwalk(target);
  alarm(0);
}
