//////////////////////////////////////////
//    Network Security Analysis Tool    //
//          (c) 2000 by Mixter          //
//                                      //
//           utility functions          //
//////////////////////////////////////////

#include "nsat.h"

#ifdef EBUG
extern ofstream dbug;
#endif

extern sigjmp_buf stack;
extern ProgressIndicator pi;

char *
strunf(char *str)
{
  int i;

  for (i = 0; i < (int) strlen(str); i++)
    if (str[i] == '\n' || str[i] == '\r')
      str[i] = ' ';
  return (str);
}

char *
my_strnsubst(char *str, int siz)
{
  int i;

  for (i = 0; i < (siz - 5); i++)
    if (str[i] == '\0' && str[i + 5] != '\0')
      str[i] = ' ';

  return (str);
}

// based on BASS's coward mode by Liraz Siri <liraz@bigfoot.com>

int
sysidle(int minimum)
{
  struct stat dummy;
  const char *tty = "/dev/tty", *ptyp = "/dev/ptyp", *pty = "pqrS";
  const char *order = "0123456789abcdef", *cpa, *cpb;
  char filename[strlen(tty) + 3];

  strcpy(filename, tty);
  filename[strlen(tty) + 1] = '\0';

  for (cpa = order; *cpa; cpa++)
    {
      filename[strlen(tty)] = *cpa;
      if (stat(filename, &dummy) < 0)
	continue;
      if ((time(NULL) - dummy.st_atime) < minimum)
	return (0);
    }
  filename[strlen(tty) + 2] = '\0';

  for (cpa = pty; *cpa; cpa++)
    {
      filename[strlen(tty)] = *cpa;
      for (cpb = order; *cpb; cpb++)
	{
	  filename[strlen(tty) + 1] = *cpb;
	  if (stat(filename, &dummy) < 0)
	    continue;
	  if ((time(NULL) - dummy.st_atime) < minimum)
	    return (0);
	}
    }

  strcpy(filename, ptyp);
  filename[strlen(ptyp) + 1] = '\0';
  for (cpa = order; *cpa; cpa++)
    {
      filename[strlen(ptyp)] = *cpa;
      if (stat(filename, &dummy) < 0)
	continue;
      if ((time(NULL) - dummy.st_atime) < minimum)
	return (0);
    }

  return (1);
}

void
hook(int signum)
{
#ifdef EBUG
  dbug << "* Timeout on I/O operation (" << signum << ") - restoring stack context\n";
#endif
  alarm(0);

  siglongjmp(stack, 666);	// back to the future! :p

}

// segv/segbus evasion by guidob@synnergy.net

void
catchsegv(int signum)
{
#ifdef EBUG
  dbug << "* Segment violation (longjmp?), resuming...\n";
#endif
  siglongjmp(stack, 666);	// siglongjmp never returns...

  exit(0);
}

void
doh(int signum)
{
#ifdef EBUG
  dbug << "* Fatal signal " << signum << " - bailing out\n";
  dbug.flush();
  dbug.close();
#endif

  if (raise(SIGKILL) != 0)
    exit(-1);
}

void
shh(int signum)
{
#ifdef EBUG
  dbug << "* Got suspicious signal " << signum << " - suspending thread\n";
#endif

  while (!sysidle(pi.idle))
    sleep(pi.idle);

  return;
}

int
safe_fork(void)
{
  int i, p;

ugly_trick:
  p = fork();

  if (p < 0)
    {				// oh great.
#ifdef EBUG
      dbug << "fork() failing: " << strerror(errno) << ", waiting to try again...\n";
#endif
      nice(-1);
      sleep(pi.timeout);
      goto ugly_trick;
    }

  if (p == 0)
    {
      sigset();
      i = open("/dev/tty", O_RDWR | O_NOCTTY);
      ioctl(i, TIOCNOTTY, NULL);
      close(i);
#ifdef PARANOID_CHECK
#ifndef EBUG
      struct rlimit rl =
      {0, 0};

      setrlimit(RLIMIT_CORE, &rl);
#endif
#endif
      if (pi.stealth)
	{
#ifdef HAVE_LINUX
	  setpgrp();
#endif
	  setsid();
	}
    }

  return (p);
}

int 
tsock(void)
{
  int fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);

  if (fd < 0)
    fd = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);

  if (pi.vhostip)
    {
      struct sockaddr_in sin;

      memset(&sin, 0, sizeof(sin));
      sin.sin_family = AF_INET;
      sin.sin_port = htons(0);
      sin.sin_addr.s_addr = pi.vhostip;
      (void) bind(fd, (struct sockaddr *) &sin, sizeof(sin));
    }

  return fd;
}

int 
usock(void)
{
  return socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
}

int 
isock(void)
{
  return socket(AF_INET, SOCK_RAW, IPPROTO_ICMP);
}

void
sigset(void)
{
  signal(SIGCHLD, SIG_IGN);
  signal(SIGTSTP, SIG_IGN);
  signal(SIGIO, SIG_IGN);
  signal(SIGSEGV, catchsegv);
  signal(SIGBUS, catchsegv);
  signal(SIGALRM, hook);
  signal(SIGPIPE, hook);
  signal(SIGURG, hook);
  signal(SIGHUP, doh);
  signal(SIGINT, doh);
  signal(SIGQUIT, doh);
  signal(SIGILL, doh);
  signal(SIGTRAP, doh);
  signal(SIGABRT, doh);
  signal(SIGIOT, doh);
  signal(SIGFPE, doh);
  signal(SIGTERM, doh);
  signal(SIGSTOP, doh);
  signal(SIGPROF, doh);
#ifdef SIGPWR
  signal(SIGPWR, doh);
#endif
  if (pi.stealth)
    {
      signal(SIGTTIN, shh);
      signal(SIGTTOU, shh);
      signal(SIGXCPU, shh);
      signal(SIGXFSZ, shh);
      signal(SIGVTALRM, shh);
    }
}

void
usage(char *name)
{
  cout << " usage:\n";
  cout << "\t" << name << " [options] -h <hostname>\n";
  cout << "\t" << name << " [options] -f <input file>\n";
  cout << "\t" << name << " [options] -s <start ip> -e <end ip>\n\n";
  cout << " options:\n";
  cout << "\t-C <file>\tspecify a custom configuration file\n";
  cout << "\t-V <host>\tspecify a virtual host or IP address to scan from\n";
  cout << "\t-n\t\tdon't go into background and report status to console\n";
  cout << "\t-t <n>\t\tset connection timeout to n seconds\n";
  cout << "\t-m <n>\t\tset maximum process count to n\n";
  cout << "\t-l <n>\t\tset maximum seconds to spend scanning one host\n";
  cout << "\t-i <n>\t\tset idle time for coward mode to n seconds\n";
  cout << "\t-p <0/1>\tping dependence: don't scan on ping timeouts\n";
  cout << "\t-c <0/1>\tcoward mode: watch out for local user activity\n";
  cout << "\t-v <0-3>\told-style scan intensity (0-3). (deprecated)\n";
  cout << "\tPlease read the manpage for nsat(8) for more information.\n\n";
  exit(-1);
}

void
writepid(int pid)
{
  FILE *f = fopen(PIDFILE, "w");

  if (f == NULL)
    {
      cerr << "* cannot open pid file, DOH\n";
      cerr.flush();
      exit(-1);
    }
  fprintf(f, "%d", pid);
  fclose(f);
}

int
readpid(void)
{
  FILE *f = fopen(PIDFILE, "r");
  int pid;

  if (f == NULL)
    return (0);			// file not there, nsat not running

  fscanf(f, "%d", &pid);
  fclose(f);
  kill(pid, SIGCHLD);
  if (errno == ESRCH)
    {				// process doesnt exist, nsat not running

      unlink(PIDFILE);
      return (0);
    }
  return (1);			// old process exists, nsat running

}

void 
fdmax(void)
{
  const char *inode_max = "12288", *file_max = "8192";
  int fd = 0;

  if ((fd = open("/proc/sys/kernel/inode-max", O_WRONLY)) >= 0)
    {
      write(fd, inode_max, strlen(inode_max));
      close(fd);
    }
  if ((fd = open("/proc/sys/kernel/file-max", O_WRONLY)) >= 0)
    {
      write(fd, file_max, strlen(file_max));
      close(fd);
    }
  if ((fd = open("/proc/sys/fs/inode-max", O_WRONLY)) >= 0)
    {
      write(fd, inode_max, strlen(inode_max));
      close(fd);
    }
  if ((fd = open("/proc/sys/fs/file-max", O_WRONLY)) >= 0)
    {
      write(fd, file_max, strlen(file_max));
      close(fd);
    }
}

void 
rlmax(void)
{
  struct rlimit rl =
  {8192, 8192};

  setrlimit(RLIMIT_NPROC, &rl);
  setrlimit(RLIMIT_NOFILE, &rl);
}

#ifdef PARANOID_CHECK
int
nicefile(char *name)
{
  struct stat status;

  if ((name[0] == '/') || (strstr(name, "..")))
    return (1);
  if (lstat(name, &status) != -1)
    if (S_ISLNK(status.st_mode))
      return (1);		// ouch, hax0r ;/

  return (0);
}
#endif
