/*
 * SpiderNet - remote host-based intrusion detection
 * 12/99 by Mixter <mixter@newyorkoffice.com>
 *
 * spidermon.c - logging / verification daemon
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "spider.h"
#include "aes.h"
#include <setjmp.h>

sigjmp_buf env;
char hostbuf[512];

void fatal (char *);
void timeout (int);
void alert (char *);
int secure_session (unsigned long);

int
main (int argc, char **argv)
{
  char buf[512], buf2[512];
  int i = 0, verbose = 0, total = 0;
  struct
    {
      char sum[35];
      unsigned long host;
      long rtime, rtime_o;
      char rsum[35];
      int promisc;
    }
  check[MAX_HOSTS];
  FILE *config = fopen (SPIDER, "r");

  if (config == NULL)
    {
      fprintf (stderr, "Unable to open config file '%s'!\n", SPIDER);
      fatal ("cannot open configuration file");
    }

  for (i = argc; i > 0; i--)
    if (!strcmp (argv[argc - 1], "-v"))
      verbose++;

  printf ("spidernet loghost monitor v%s starting\n", SVER);

  while (fgets (buf, 512, config) != NULL)
    {
      if (buf[0] == '#')
	continue;
      if (buf[0] == '\n')
	continue;
      if (buf[0] == '\r')
	continue;
      TrimBuf (buf);
      sscanf (buf, "%s %s", buf2, check[i].sum);
      check[i].host = resolve (buf2);
      if (check[i].host != 0)
	i++;
      memset (buf, 0, 512);
    }

  total = i;

  printf ("\t - monitoring %d remote hosts\n", total);

  if (!total)
    exit (0);

  if (verbose)
    for (i = 0; i < total; i++)
      printf ("Host: %30s\nChecksum: %s\n", ntoa (check[i].host), check[i].sum);

#ifndef DEBUG
  if (fork ())
    exit (0);
#endif

  signal (SIGCHLD, SIG_IGN);
  signal (SIGALRM, timeout);
  while (1)
  {
  sleep (MIN_TIME);

  for (i = 0; i < total; i++)
    {
      int x;
      if (!(x = secure_session (check[i].host)))
	{
	  sprintf (buf2, "Unable to connect to / receive from host %s\n", ntoa (check[i].host));
	  alert (buf2);
	}
      else
	{
	  aes_setkey (check[i].sum);
	  memset (buf, 0, 512);
          decode64 (hostbuf, buf, x);
	  if (sscanf (buf, "%ld\n%d\n%32s\n", &check[i].rtime, &check[i].promisc, check[i].rsum) != 3)
	    {
	      sprintf (buf2, "One or more files have been altered on %s - unreadable input\n", ntoa (check[i].host));
	      alert (buf2);
	      continue;
	    }
	  if (strcmp (check[i].sum, check[i].rsum) != 0)
	    {
	      sprintf (buf2, "One or more files have been altered on %s - checksums don't match\n", ntoa (check[i].host));
	      alert (buf2);
	      continue;
	    }
	  if (check[i].promisc)
	    {
	      sprintf (buf2, "Network interface(s) of host %s are being in promiscuous mode - sniffing suspected\n", ntoa (check[i].host));
	      alert (buf2);
	      continue;
	    }
	  if (check[i].rtime == check[i].rtime_o)
	    {
	      sprintf (buf2, "Found cloned time stamps - replay attack suspected on host %s\n", ntoa (check[i].host));
	      alert (buf2);
	      continue;
	    }
	  check[i].rtime_o = check[i].rtime;
	  #ifdef DEBUG
	  printf ("HOST: %s DATA: %ld %s %d\n", ntoa (check[i].host), check[i].rtime, check[i].rsum, check[i].promisc);
	  #endif
	}
    }
  }
  return 0;
}

int 
secure_session (unsigned long ip)
{
  int i = 0, s = socket (AF_INET, SOCK_STREAM, 0);
  struct sockaddr_in sin;
  sin.sin_family = AF_INET;
  sin.sin_port = htons (SPIDER_PORT);
  sin.sin_addr.s_addr = ip;
  alarm (60);
  siginterrupt (SIGALRM, 1);
  if (sigsetjmp (env, 1))
    return 0;
  if (connect (s, (struct sockaddr *) &sin, sizeof (struct sockaddr)) < 0)
      return 0;
  memset (hostbuf, 0, 512);
  i = read (s, hostbuf, 510);
  alarm (0);
  close (s);
  return i;
}

void
fatal (char *reason)
{
  log ("spidermon", reason);
  exit (0);
}

void
timeout (int signum)
{
  alarm (0);
  siglongjmp (env, 1);
}

void
alert (char *msg)
{
  char tmp[200];
  log ("spidermon", msg);
#ifdef EMAIL_NOTIFY
  snprintf (tmp, 200, "echo '%s' | mail -s 'Security Notice from SPIDERNET' %s", msg, EMAIL_NOTIFY);
  system (tmp);
#endif
}
