/*
 *  $Id: libipg_lookup.c,v 1.3 2003/12/31 20:18:51 mike Exp $
 *
 *  libipgeo
 *  libipgeo_lookup.c - lookup routines
 *
 *  Copyright (c) 1998 - 2005 Mike D. Schiffman <stolencreditcard@gmail.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/**
 * @file libipgeo_lookup.c
 * @ingroup sourcefiles
 * 
 * @author Mike Schiffman
 *
 * @brief ipgeo db lookup routines
 *
 */

#if (HAVE_CONFIG_H)
#include "../include/config.h"
#endif

#include "./libipgeo.h"

u_int32_t
ipgeo_ipa2ipn(char *ip)
{
    char *p = ip;
    u_int32_t o1, o2, o3, o4;

    /** currently we don't deal with hostnames; ip has to be dots n decimals */
    if (!isdigit(ip[0]))
    {
        return (0);
    }

    /** compute first byte (256^3 * o1) */
    o1 = 16777216 * atol(strsep(&p, "."));

    /** compute second byte (256^2 * o2) */
    o2 = 65536 * atol(strsep(&p, "."));

    /** compute third byte (256 * o3) */
    o3 = 256 * atol(strsep(&p, "."));

    /** compute fourth byte (o4) */
    o4 = atol(strsep(&p, "."));

    return (o1 + o2 + o3 + o4);
}

int
ipgeo_lookup(u_int32_t ipn, u_int8_t flags, ipgeo_t *ipg)
{
    int n;
    struct stat sb;
    off_t start, end, mid;
    char *p, *q;
    u_int32_t min, max;
    char buf[IPGEO_BUF_SIZE];

    if (ipg == NULL)
    {
        return (-1);
    }

#if (LIBIPG_LIL_ENDIAN)
    ipn = htonl(ipn);
#endif

    /** we need to know how big the db file is */
    if (fstat(fileno(ipg->db), &sb) == -1)
    {
        snprintf(ipg->err_buf, IPGEO_ERRBUF_SIZE, strerror(errno));
        return (-1);
    }

    start = 0;
    end   = sb.st_size;

    /** binary search, divide and conquer */ 
    while (end > start)
    {
        /** cut the remaining list in half and move the file pointer there */
        mid = (start + end) / 2;
        if (fseek(ipg->db, mid, SEEK_SET) == -1)
        {
            snprintf(ipg->err_buf, IPGEO_ERRBUF_SIZE, strerror(errno));
            return (-1);
        }

        /** first do a "dummy" fgets() to read up to the first \n */
        /** XXX this will lose an entry every so often though in practice I
         ** don't think it will be an issue. */
        if (fgets(buf, IPGEO_BUF_SIZE - 1, ipg->db) == NULL)
        {
            snprintf(ipg->err_buf, IPGEO_ERRBUF_SIZE, 
                    "ipgeo_lookup(): dummy fgets() failed\n");
            return (-1);
        }
        /** read in the line */
        if (fgets(buf, IPGEO_BUF_SIZE - 1, ipg->db) == NULL)
        {
            snprintf(ipg->err_buf, IPGEO_ERRBUF_SIZE, 
                    "ipgeo_lookup(): real fgets() failed\n");
        }
        p = buf;

        /* step over quote */
        p += 1;
        min = strtoul(strsep(&p, ","), (char **)NULL, 10);

        /* step over quote */
        p += 1;
        max = strtoul(strsep(&p, ","), (char **)NULL, 10);

        ipg->comparisons++;

        /** found a match */
        if (ipn >= min && ipn <= max)
        {
            p += 1;
            q = strsep(&p, ",");
            for (n = 0; n < sizeof(ipg->cc) - 1; n++)
            {
                if (q[n] == '"')
                {
                    ipg->cc[n] = (char)NULL;
                    break;
                }
                ipg->cc[n] = q[n];
            }

            /* get cc */
            p += 1;
            q = strsep(&p, ",");
            for (n = 0; n < sizeof(ipg->country) - 1; n++)
            {
                if (q[n] == '"')
                {
                    ipg->country[n] = (char)NULL;
                    break;
                }
                ipg->country[n] = q[n];
            }

            /* get country */
            p += 1;
            q = strsep(&p, ",");
            for (n = 0; n < sizeof(ipg->region) - 1; n++)
            {
                if (q[n] == '"')
                {
                    ipg->region[n] = (char)NULL;
                    break;
                }
                ipg->region[n] = q[n];
            }

            /* get city */
            p += 1;
            q = strsep(&p, ",");
            for (n = 0; n < sizeof(ipg->city) - 1; n++)
            {
                if (q[n] == '"')
                {
                    ipg->city[n] = (char)NULL;
                    break;
                }
                ipg->city[n] = q[n];
            }

            /* get latitude */
            p += 1;
            ipg->latitude = strtod(strsep(&p, ","), (char **)NULL);

            /* get longitude */
            p += 1;
            ipg->longitude = strtod(strsep(&p, ","), (char **)NULL);

            /* get isp */
            p += 1;
            q = strsep(&p, ",");
            for (n = 0; n < sizeof(ipg->isp) - 1; n++)
            {
                if (q[n] == '"')
                {
                    ipg->isp[n] = (char)NULL;
                    break;
                }
                ipg->isp[n] = q[n];
            }

            rewind(ipg->db);
            return (1);
        }
        /** did not find a match, we need to re-divide */
        if (ipn < min)
        {
            /** our target is in the first half of the list */
            end = mid;
        }
        else
        {
            /** our target is in the second half of the list */
            start = mid;
        }
    }
    /* no match */
    return (0);
}

char *
ipgeo_get_cc(ipgeo_t *ipg)
{
    if (ipg->cc[0] == '-')
    {
        return("UNKNOWN");
    }
    return (ipg->cc);
}

char *
ipgeo_get_country(ipgeo_t *ipg)
{
    if (ipg->country[0] == '-')
    {
        return("UNKNOWN");
    }
    return (ipg->country);
}

char *
ipgeo_get_region(ipgeo_t *ipg)
{
    if (ipg->region[0] == '-')
    {
        return("UNKNOWN");
    }
    return (ipg->region);
}

char *
ipgeo_get_city(ipgeo_t *ipg)
{
    if (ipg->city[0] == '-')
    {
        return("UNKNOWN");
    }
    return (ipg->city);
}

char *
ipgeo_get_isp(ipgeo_t *ipg)
{
    if (ipg->isp[0] == '-')
    {
        return("UNKNOWN");
    }
    return (ipg->isp);
}

double
ipgeo_get_lat(ipgeo_t *ipg)
{
    return (ipg->latitude);
}

double
ipgeo_get_long(ipgeo_t *ipg)
{
    return (ipg->longitude);
}

/* EOF */
