// SpoofSocket.cpp : implementation file
//
/*
 *
 *
 *  Copyright (c) 2000 Barak Weichselbaum <barak@komodia.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *
 * Contact info:
 * Site: http://www.komodia.com
 * Email: barak@komodia.com
 */

#include "stdafx.h"
#include "BinaryTree.h"
#include "SpoofSocket.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//Message handler
LRESULT CALLBACK SocketMessageHandler(HWND hwnd,      // handle to window
							  		  UINT uMsg,      // message identifier
									  WPARAM wParam,  // first message parameter
									  LPARAM lParam)   // second message parameter
													
{
	//first get the socket
	CSpoofSocket* cSock;

	cSock=CSpoofSocket::GetSocketByID((int)wParam);

	if (cSock)
		//Socket exists
		switch (uMsg)
		{
		case WM_SOCKET_GENERAL:
			if (WSAGETSELECTEVENT(lParam) == FD_READ)
				return cSock->OnSocketReceive(WSAGETSELECTERROR(lParam));
			else if (WSAGETSELECTEVENT(lParam) == FD_WRITE)
				return cSock->OnSocketWrite(WSAGETSELECTERROR(lParam));
			else if (WSAGETSELECTEVENT(lParam) == FD_OOB)
				return cSock->OnSocketOOB(WSAGETSELECTERROR(lParam));
			else if (WSAGETSELECTEVENT(lParam) == FD_CLOSE)
				return cSock->OnSocketClose(WSAGETSELECTERROR(lParam));
			break;
		case WM_SOCKET_CONNECT:
			if (WSAGETSELECTEVENT(lParam) == FD_CONNECT)
				return cSock->OnSocketConnect(WSAGETSELECTERROR(lParam));
			break;
		case WM_SOCKET_ACCEPT:
			if (WSAGETSELECTEVENT(lParam) == FD_ACCEPT)
				return cSock->OnSocketAccept(WSAGETSELECTERROR(lParam));
			break;
		case WM_TIMER:
			//Inform the socket
			return cSock->OnSocketTimeout();
		default:                       /* Passes it on if unproccessed    */
			return (int)(DefWindowProc(hwnd, uMsg, wParam, lParam));
		}
	else
		return (int)(DefWindowProc(hwnd, uMsg, wParam, lParam));

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// CSpoofSocket

CSpoofSocket::CSpoofSocket()
{
	InitializeIP();
}

CSpoofSocket::CSpoofSocket(SOCKET sok)
{
	AssignSocket(sok);
}

CSpoofSocket::~CSpoofSocket()
{
	//Delete options
	SetOptions(FALSE);

	Close();
}

/////////////////////////////////////////////////////////////////////////////
// CSpoofSocket member functions

BOOL CSpoofSocket::Create(int iProtocol)
{
	//Here we create the raw socket
	if (m_Raw || iProtocol==IPPROTO_ICMP)
		m_SpoofSocket=socket(AF_INET,SOCK_RAW,iProtocol);//iProtocol);
	else
		if (iProtocol==IPPROTO_TCP)
			m_SpoofSocket=socket(AF_INET,SOCK_STREAM,iProtocol);
		else if (iProtocol==IPPROTO_UDP)
			m_SpoofSocket=socket(AF_INET,SOCK_DGRAM,iProtocol);

	//Check for socket validity
	if (m_SpoofSocket==INVALID_SOCKET)
	{
		//Error
		m_LastError=WSAGetLastError();
		return FALSE;
	}

	if (m_Raw)
	{
		//Set that the application will send the IP header
		unsigned int iTrue=1;

		if(setsockopt(m_SpoofSocket,IPPROTO_IP,IP_HDRINCL,(char*)&iTrue,sizeof(iTrue))==SOCKET_ERROR)
		{
			//Check for options error
			m_LastError=WSAGetLastError();
			return FALSE;
		}
		if (iProtocol==IPPROTO_ICMP)
			SetHandlers();
	}
	else
		SetHandlers();
	
	//Well no errors
	m_LastError=0;

	//Add socket to list
	AddSocketToList();

	return TRUE;
}

BOOL CSpoofSocket::Send(LPCSTR lpDestinationAddress,char* buf,int bufLength)
{
	//Quit if not ok
	if (!CheckSocketValid())
		return FALSE;

	//Define the target address
	sockaddr_in m_TargetAddress;
	memset(&m_TargetAddress,0,sizeof(m_TargetAddress));

	m_TargetAddress.sin_family=AF_INET;
	m_TargetAddress.sin_addr.s_addr=inet_addr(lpDestinationAddress);
	m_TargetAddress.sin_port=0;

	//packet send status ?
	int iResult;

	//Only if allowing raw headers !!
	if (m_Raw)
	{
		//Header length
		unsigned char ucHeaderLength=IpHeaderLength;

		if (m_Options)
			ucHeaderLength+=m_IPOptions->GetBufferLength();
		
		//First construct the packet
		LPIpHeader lpHead=ConstructIPHeader(m_Protocol,IpFragFlag_DONT_FRAG,m_TTL,GetCurrentProcessId(),ucHeaderLength);

		//Set the address
		SetIPHeaderAddress(lpHead,m_SourceAddress,lpDestinationAddress);

		//Now add some more options
		int iTotalLength;
		iTotalLength=ucHeaderLength+bufLength;

		//Set the header
		lpHead->TotalLength=htons(iTotalLength);

		//Need to construct a new packet
		char* newBuf=new char[iTotalLength];
		//Copy two buffers
		memcpy(newBuf,lpHead,IpHeaderLength);

		//Do we need to copy options ?
		if (m_Options)
			memcpy(newBuf+IpHeaderLength,m_IPOptions->GetBuffer(),m_IPOptions->GetBufferLength());

		//Only if not null
		if (buf)
			memcpy(newBuf+ucHeaderLength,buf,bufLength);
		
		//Calculate the checksum
		lpHead->CheckSum=CalculateChecksum((unsigned short*)newBuf,iTotalLength);

		//Recopy the ip
		memcpy(newBuf,lpHead,IpHeaderLength);
		
		//Send the data
		iResult=sendto(getHandle(),(const char*)newBuf,iTotalLength,0,(sockaddr*)&m_TargetAddress,sizeof(m_TargetAddress));

		//Dispose of the buffer
		delete newBuf;

		//Dispose the header
		delete lpHead;
	}
	else
	{
		iResult=!SOCKET_ERROR;

		//Insert options
		if (m_Options)
			if (setsockopt(getHandle(),IPPROTO_IP,IP_OPTIONS,m_IPOptions->GetBuffer(),m_IPOptions->GetBufferLength()==SOCKET_ERROR))
				//Error
				iResult=SOCKET_ERROR;
			else
				;
		else
			//No options
			//iResult=setsockopt(getHandle(),IPPROTO_IP,IP_OPTIONS,NULL,0);
			;

		//Check if we had an error
		if (iResult!=SOCKET_ERROR)
			//Use regular send !!!
			iResult=sendto(getHandle(),(const char*)buf,bufLength,0,(sockaddr*)&m_TargetAddress,sizeof(m_TargetAddress));
	}

	if (iResult==SOCKET_ERROR)
		//Set the error
		SetLastError();
	else
		m_LastError=0;

	return iResult!=SOCKET_ERROR;
}

LPIpHeader CSpoofSocket::ConstructIPHeader(unsigned char  ucProtocol,
										   unsigned short usFragmentationFlags,
										   unsigned char  ucTTL,
										   unsigned short usIdentification,
										   unsigned char  ucHeaderLength)
{
	//Need to construct the IP header
	LPIpHeader lpHead=new _IpHeader;

	//Header length (in 32 bits)
	lpHead->HeaderLength_Version=ucHeaderLength/4 + IpVersion*16;

	//Protocol
	lpHead->Protocol=ucProtocol;

	//Fragmentation flags
	lpHead->FragmentationFlags=htons(usFragmentationFlags);

	//Time to live
	lpHead->TTL=ucTTL;

	//Checksum - set to 0
	lpHead->CheckSum=0;

	//Identification
	lpHead->Identification=htons(usIdentification);

	//Precedence
	lpHead->TypeOfService=IpService_ROUTINE;

	//Return IP to user
	return lpHead;
}

void CSpoofSocket::SetIPHeaderAddress(LPIpHeader lpHead, LPCSTR lpSourceAddress, LPCSTR lpDestinationAddress)
{
	//We need to place the header
	
	//If source is NULL then we need to use default source
	if (!lpSourceAddress)
	{
		//We will implement it
	}
	else
		//Use sockets2
		lpHead->sourceIPAddress=inet_addr(lpSourceAddress);

	//Place destination address
	lpHead->destIPAddress=inet_addr(lpDestinationAddress);

	//Done
}

void CSpoofSocket::SetLastError()
{
	m_LastError=WSAGetLastError();
}

int CSpoofSocket::GetLastError()
{
	return m_LastError;
}

BOOL CSpoofSocket::ValidSocket()
{
	return m_SpoofSocket!=INVALID_SOCKET;
}

unsigned short CSpoofSocket::CalculateChecksum(unsigned short *usBuf, int iSize)
{
	unsigned long usChksum=0;

	//Calculate the checksum
	while (iSize>1)
	{
		usChksum+=*usBuf++;
		iSize-=sizeof(unsigned short);
	}

	//If we have one char left
	if (iSize)
		usChksum+=*(unsigned char*)usBuf;

	//Complete the calculations
	usChksum=(usChksum >> 16) + (usChksum & 0xffff);
	usChksum+=(usChksum >> 16);

	//Return the value (inversed)
	return (unsigned short)(~usChksum);
}


BOOL CSpoofSocket::Bind(LPCSTR lpSourceAddress,int iPort)
{
	//Quit if not ok
	if (!CheckSocketValid())
		return FALSE;

	//Create the local address
	sockaddr_in soSrc;
	//Set to 0
	memset(&soSrc,0,sizeof(soSrc));
	soSrc.sin_family=AF_INET;
	soSrc.sin_addr.s_addr=inet_addr(lpSourceAddress);
	soSrc.sin_port=htons(iPort);

	//Now we need to bind it
	if (bind(getHandle(),(sockaddr*)&soSrc,sizeof(soSrc)))
	{
		//Error
		m_LastError=WSAGetLastError();
		return FALSE;
	}

	m_LastError=0;

	//If already has a source address then don't change it
	if (!m_SourceAddress)
		m_SourceAddress=lpSourceAddress;

	return TRUE;
}

SOCKET CSpoofSocket::getHandle()
{
	return m_SpoofSocket;
}

BOOL CSpoofSocket::CheckSocketValid()
{
	//Check if socket is invalid
	if (!ValidSocket())
	{
		m_LastError=WSAESHUTDOWN;
		return FALSE;
	}

	//OK
	return TRUE;
}

BOOL CSpoofSocket::Close()
{
	//Close the socket
	//Quit if not ok
	if (!CheckSocketValid())
		return FALSE;

	//Close it
	if (closesocket(getHandle())==SOCKET_ERROR)
	{
		//Error in closing ?
		m_LastError=WSAGetLastError();
		return FALSE;
	}

	//Set the socket to invalid
	m_SpoofSocket=INVALID_SOCKET;

	//Remove from tree
	RemoveSocketFromList();

	return TRUE;
}


BOOL CSpoofSocket::InitializeSockets()
{
	//Initialize the sockets
	WORD wVersionRequested;
	WSADATA wsaData;
	int err;
 
	wVersionRequested = MAKEWORD( 2, 2 );
 
	err = WSAStartup( wVersionRequested, &wsaData );
	if (err!=0)
		/* Tell the user that we could not find a usable */
		/* WinSock DLL.                                  */
		return FALSE;
 
	/* Confirm that the WinSock DLL supports 2.2.*/
	/* Note that if the DLL supports versions greater    */
	/* than 2.2 in addition to 2.2, it will still return */
	/* 2.2 in wVersion since that is the version we      */
	/* requested.                                        */
 
	if (LOBYTE(wsaData.wVersion)!=2 || HIBYTE(wsaData.wVersion)!=2)
	{
		/* Tell the user that we could not find a usable */
		/* WinSock DLL.                                  */
		WSACleanup();
		return FALSE;
	}

	//OK
	//Intialize the tree
	if (!m_SocketTree)
		m_SocketTree=new CBinaryTree;

	return TRUE;
}
 

void CSpoofSocket::SetProtocol(int iProtocol)
{
	m_Protocol=iProtocol;
}

void CSpoofSocket::SetSourceAddress(LPCSTR lpSourceAddress)
{
	//Set the source address, in case we want to spoof it
	m_SourceAddress=lpSourceAddress;
}

unsigned short CSpoofSocket::CalculatePseudoChecksum(char *buf, int BufLength,LPCSTR lpDestinationAddress,int iPacketLength)
{
	//Calculate the checksum
	LPPseudoHeader lpPseudo;
	lpPseudo=new PseudoHeader;

	lpPseudo->DestinationAddress=inet_addr(lpDestinationAddress);
	lpPseudo->SourceAddress=inet_addr(m_SourceAddress);
	lpPseudo->Zeros=0;
	lpPseudo->PTCL=m_Protocol;
	lpPseudo->Length=htons(iPacketLength);

	//Calculate checksum of all
	int iTotalLength;
	iTotalLength=PseudoHeaderLength+BufLength;

	char* tmpBuf;
	tmpBuf=new char[iTotalLength];

	//Copy pseudo
	memcpy(tmpBuf,lpPseudo,PseudoHeaderLength);

	//Copy header
	memcpy(tmpBuf+PseudoHeaderLength,buf,BufLength);

	//Calculate the checksum
	unsigned short usChecksum;
	usChecksum=CalculateChecksum((unsigned short*)tmpBuf,iTotalLength);

	//Delete all
	delete tmpBuf;
	delete lpPseudo;

	//Return checksum
	return usChecksum;
}

void CSpoofSocket::SetTTL(unsigned char ucTTL)
{
	if (m_Raw)
	{
		//Set the ttl
		m_TTL=ucTTL;
	}
	else
		setsockopt(getHandle(),IPPROTO_IP,IP_TTL,(const char*)&ucTTL,sizeof(ucTTL));

}

BOOL CSpoofSocket::Listen(int iBackLog)
{
	int iResult;
	iResult=listen(getHandle(),iBackLog);

	if (iResult)
		SetLastError();

	return !iResult;
}

BOOL CSpoofSocket::ShutdownSockets()
{
	//Clear windows
	BOOL bHandlers;
	bHandlers=RemoveHandlers();

	//Delete tree
	delete m_SocketTree;
	m_SocketTree=NULL;

	if (WSACleanup()==SOCKET_ERROR)
	{
		SetLastError();
		return FALSE;
	}
	else
		return bHandlers;
}

void CSpoofSocket::SetRaw(BOOL bRaw)
{
	//Do we want to create raw socket (YES!!)
	m_Raw=bRaw;
}

void CSpoofSocket::SetOptions(BOOL bOptions)
{
	//Do we want options, normaly not
	m_Options=bOptions;

	if (m_IPOptions)
	{
		delete m_IPOptions;
		m_IPOptions=NULL;
	}

	if (bOptions)
		m_IPOptions=new CIPOptions;

}

CIPOptions::CIPOptions()
{
	//Initialize our buffer
	m_Buffer=new char[IPOption_SIZE];

	//Set our buffer to nothing
	Reset();

	//Our buffer length
	m_BufferLength=0;

	//Set auto pad
	m_AutoPAD=TRUE;
}

CIPOptions::~CIPOptions()
{
	delete m_Buffer;
}

void CIPOptions::AddOption_Nothing()
{
	//Add option do nothing
	tOptionType OT;

	//Get the option
	OT=GetOption(IPOption_DONT_COPY,IPOption_CONTROL,IPOption_NO_OPERATION);

	//Add it to buffer
	AddToBuffer((char*)&OT,sizeof(OT));
}

tOptionType CIPOptions::GetOption(unsigned char CopyFlag, unsigned char ClassFlag, unsigned char TypeFlag)
{
	//Return a single option type
	return CopyFlag | ClassFlag | TypeFlag;
}

void CIPOptions::AddToBuffer(char *buf, int BufLength)
{
	if (m_BufferLength<IPOption_SIZE)
	{
		//Add our option to the buffer
		memcpy(m_Buffer+m_BufferLength,buf,BufLength);
		m_BufferLength+=BufLength;
	}
}

const char* CIPOptions::GetBuffer()
{
	return m_Buffer;
}

int CIPOptions::GetBufferLength()
{
	//Check if auto pad or not
	if (m_AutoPAD)
		if (m_BufferLength/IPOption_WRAPSIZE==(m_BufferLength/IPOption_WRAPSIZE)*IPOption_WRAPSIZE && m_BufferLength>=IPOption_WRAPSIZE)
			return m_BufferLength;
		else
			return int((float)m_BufferLength/IPOption_WRAPSIZE+1)*IPOption_WRAPSIZE;
	else
		return m_BufferLength;
}

void CIPOptions::AddOption_ENDLIST()
{
	//End the list of options
	tOptionType OT;

	//Get the option
	OT=GetOption(IPOption_DONT_COPY,IPOption_CONTROL,IPOption_END_OPTION);

	//Add it to buffer
	AddToBuffer((char*)&OT,sizeof(OT));
}

void CIPOptions::SetAutoPad(BOOL bAutoPAD)
{
	m_AutoPAD=bAutoPAD;
}

CIPOptions* CSpoofSocket::GetOptions()
{
	return m_IPOptions;
}

void CIPOptions::Reset()
{
	//Set all to zeros
	memset(m_Buffer,0,IPOption_SIZE);
}

void CIPOptions::AddOption_Security(unsigned short usType)
{
	//Add option security
	tOptionType OT;

	//Get the option
	OT=GetOption(IPOption_COPY,IPOption_CONTROL,IPOption_SECURITY);

	//Add it to buffer
	AddToBuffer((char*)&OT,sizeof(OT));

	//Add length
	OT=IPOption_SECURITY_LENGTH;
	AddToBuffer((char*)&OT,sizeof(OT));

	//Add options
	AddToBuffer((char*)&usType,sizeof(usType));

	//Add zeros
	unsigned short usZeros=0;
	unsigned char ucZeros=0;

	//A hacker would enumarate these values, according to the RFC
	//Compartments
	AddToBuffer((char*)&usZeros,sizeof(usZeros));

	//Handling restrictions
	AddToBuffer((char*)&usZeros,sizeof(usZeros));


	//Transmition control code (TCC)
	AddToBuffer((char*)&usZeros,sizeof(usZeros));
	AddToBuffer((char*)&ucZeros,sizeof(ucZeros));

	//Done
}

void CIPOptions::AddOption_Stream(unsigned short usStreamID)
{
		//Add option security
	tOptionType OT;

	//Get the option
	OT=GetOption(IPOption_COPY,IPOption_CONTROL,IPOption_STREAM);

	//Add it to buffer
	AddToBuffer((char*)&OT,sizeof(OT));

	//Add length
	OT=IPOption_STREAM_LENGTH;
	AddToBuffer((char*)&OT,sizeof(OT));

	//Add options
	unsigned short usnStreamID;
	usnStreamID=htons(usStreamID);

	AddToBuffer((char*)&usnStreamID,sizeof(usnStreamID));
}

void CIPOptions::AddOption_StrictRoute(tRouting tRoute)
{
	AddOption_Route(IPOption_STRICT_ROUTING,tRoute);
}

void CIPOptions::AddOption_RecordRoute(int iMaxRoutes)
{
	//Option for strict routine
	//Add option strict route
	tOptionType OT;

	//Get the option
	OT=GetOption(IPOption_DONT_COPY,IPOption_CONTROL,IPOption_RECORD_ROUTE);

	//Add it to buffer
	AddToBuffer((char*)&OT,sizeof(OT));

	//Add the length
	OT=iMaxRoutes*4+IPOption_STRICT_ROUTING_LENGTH;
	AddToBuffer((char*)&OT,sizeof(OT));
	
	//Add the pointer
	OT=IPOption_STRICT_ROUTING_POINTER;
	AddToBuffer((char*)&OT,sizeof(OT));
	
	char cNothing[IPOption_SIZE]="";
	AddToBuffer(cNothing,iMaxRoutes*4);
}

void CIPOptions::AddOption_Route(tOptionType tRouteType,tRouting tRoute)
{
	//Option for strict routine
	//Add option strict route
	tOptionType OT;

	//Get the option
	OT=GetOption(IPOption_COPY,IPOption_CONTROL,tRouteType);

	//Add it to buffer
	AddToBuffer((char*)&OT,sizeof(OT));

	//Add the length
	OT=tRoute.iRoutes*4+IPOption_STRICT_ROUTING_LENGTH;
	AddToBuffer((char*)&OT,sizeof(OT));
	
	//Add the pointer
	OT=IPOption_STRICT_ROUTING_POINTER;
	AddToBuffer((char*)&OT,sizeof(OT));
	
	//Add the routing table
	AddToBuffer((char*)tRoute.ulRoutes,tRoute.iRoutes*4);
}

void CIPOptions::AddOption_LooseRoute(tRouting tRoute)
{
	AddOption_Route(IPOption_LOOSE_ROUTING,tRoute);
}

void CIPOptions::AddOption_Timestamp(tOptionType tFlags, int iMaxStamps)
{
	//Add option for timestamp
	tOptionType OT;

	//Get the option
	OT=GetOption(IPOption_DONT_COPY,IPOption_DEBUGGING,IPOption_TIMESTAMP);

	//Add it to buffer
	AddToBuffer((char*)&OT,sizeof(OT));

	//Add the length
	OT=iMaxStamps*IPOption_TIMESTAMP_SIZE+IPOption_TIMESTAMP_LENGTH-1;
	AddToBuffer((char*)&OT,sizeof(OT));
	
	//Add the pointer
	OT=IPOption_TIMESTAMP_LENGTH;
	AddToBuffer((char*)&OT,sizeof(OT));

	//Add the flags
	AddToBuffer((char*)&tFlags,sizeof(tFlags));

	//Add the empty buffer
	char cNothing[IPOption_SIZE]="";
	AddToBuffer(cNothing,iMaxStamps*IPOption_TIMESTAMP_SIZE);
}

BOOL CSpoofSocket::SetHandlers()
{
	//Check if we allow async sockets
	if (!m_Async)
		return TRUE;

	//First create the window class
	if (!m_Window)
		if (!RegisterWindow())
		{
			//Error
			m_LastError=::GetLastError();
			return FALSE;
		}
		else
		{
			m_WindowHandle=CreateWindowEx(0,CSpoofSocket_Class,"Socket notification sink",
										  WS_OVERLAPPED,0,0,0,0,0,NULL,GetInstance(),NULL);
			//Check the value of the window
			if (!m_WindowHandle)
			{
				//Error
				m_LastError=::GetLastError();
				return FALSE;
			}
			else
				//We have a window
				m_Window=TRUE;
		}

	//Created !!
	//Success
	return TRUE;
}

HINSTANCE CSpoofSocket::GetInstance()
{
	//Returns the instance of the application, must be overided
	return m_Instance;
}

BOOL CSpoofSocket::OnSocketReceive(int iErrorCode)
{
	//Must override!
	return TRUE;
}

BOOL CSpoofSocket::OnSocketWrite(int iErrorCode)
{
	return TRUE;
}

BOOL CSpoofSocket::OnSocketOOB(int iErrorCode)
{
	return TRUE;
}

BOOL CSpoofSocket::OnSocketClose(int iErrorCode)
{
	return TRUE;
}

BOOL CSpoofSocket::OnSocketAccept(int iErrorCode)
{
	return TRUE;
}

BOOL CSpoofSocket::OnSocketConnect(int iErrorCode)
{
	return TRUE;
}

int CSpoofSocket::GetSocketID()
{
	return m_SocketID;
}

CSpoofSocket* CSpoofSocket::GetSocketByID(int iSockID)
{
	//Get the socket
	CBinaryTree* retVal;

	retVal=m_SocketTree->getNode(iSockID);

	//Check if valid
	if (retVal)
		//Got socket
		return (CSpoofSocket*)retVal->getData();
	else
		//Nothing
		return NULL;
}

BOOL CSpoofSocket::RegisterWindow()
{
	WNDCLASS wc;

	/* Fill in window class structure with parameters that describe the       */
    /* main window.                                                           */

    wc.style = 0;										  /* Class style(s).                    */
    wc.lpfnWndProc = (WNDPROC)SocketMessageHandler;       /* Function to retrieve messages for  */
                                        /* windows of this class.             */
    wc.cbClsExtra = 0;                  /* No per-class extra data.           */
    wc.cbWndExtra = 0;                  /* No per-window extra data.          */
    wc.hIcon = NULL;				    /* Icon name from .RC        */
    wc.hInstance = GetInstance();          /* Application that owns the class.   */
    wc.hCursor = NULL;
    wc.hbrBackground = NULL;
    wc.lpszMenuName =  NULL;   /* Name of menu resource in .RC file. */
    wc.lpszClassName = CSpoofSocket_Class ; /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */

    return (RegisterClass(&wc));
}

//Static members
CBinaryTree* CSpoofSocket::m_SocketTree=NULL;
BOOL CSpoofSocket::m_Window=FALSE;
HWND CSpoofSocket::m_WindowHandle=0;

void CSpoofSocket::SetInstance(HINSTANCE hInst)
{
	m_Instance=hInst;
}

BOOL CSpoofSocket::isRaw()
{
	return m_Raw;
}

BOOL CSpoofSocket::RemoveHandlers()
{
	//First shut down the windows
	if (m_Window)
	{
		if (!DestroyWindow(m_WindowHandle))
			return FALSE;

		if (!UnregisterClass(CSpoofSocket_Class,GetInstance()))
			return FALSE;
	}

	m_Window=FALSE;
	m_WindowHandle=NULL;

	return TRUE;
}

HWND CSpoofSocket::getWindowHandle()
{
	return m_WindowHandle;
}

void CSpoofSocket::InitializeIP()
{
	//Invalid the socket
	m_SpoofSocket=INVALID_SOCKET;

	//More invalids
	m_SourceAddress=NULL;

	//Some defaults
	m_TTL=IP_DEF_TTL;

	//We don't want raw header (so it can work on win 98/NT)
	m_Raw=FALSE;

	//Set our options
	m_IPOptions=NULL;

	//Set options to false
	SetOptions(FALSE);

	//We want async socket
	SetBlocking(FALSE);
}

void CSpoofSocket::AssignSocket(SOCKET sok)
{
	//Binds to a socket
	m_SpoofSocket=sok;

	//Set non raw
	SetRaw(FALSE);

	//Set the handlers
	SetHandlers();

	//Append socket to list
	AddSocketToList();
}

void CSpoofSocket::AddSocketToList()
{
	//Add socket to list
	m_SocketID=m_SpoofSocket;
	m_SocketTree->newNode(GetSocketID())->setData(this);
}

void CSpoofSocket::RemoveSocketFromList()
{
	m_SocketTree->deleteNode(m_SocketID);
}

int CSpoofSocket::Receive(char *buf, int bufLen)
{
	if (!ValidSocket())
		return SOCKET_ERROR;

	//Receive data
	int iResult;

	//Receive
	if (m_Protocol!=IPPROTO_TCP)
		iResult=recvfrom(getHandle(),buf,bufLen,NULL,NULL,NULL);
	else
		iResult=recv(getHandle(),buf,bufLen,NULL);

	//Check if error
	if (iResult==SOCKET_ERROR)
		//Error
		SetLastError();

	//Number of bytes received
	return iResult;
}

char FAR * CSpoofSocket::LongToString(long lAddr)
{
	//First create the address
	in_addr addr;

	//Assign it
	addr.S_un.S_addr=lAddr;

	//Return the value
	return inet_ntoa(addr);
}

BOOL CSpoofSocket::OnSocketTimeout()
{
	//Indicating timeout proccessed
	return FALSE;
}

BOOL CSpoofSocket::SetTimeout(int iMs)
{
	if (!m_Window)
		return FALSE;

	//Set the timer
	return SetTimer(getWindowHandle(),getHandle(),iMs,NULL);
}

BOOL CSpoofSocket::KillTimer()
{
	if (!m_Window)
		return FALSE;

	return ::KillTimer(getWindowHandle(),getHandle());
}

BOOL CSpoofSocket::ValidAddress(LPCTSTR lpAddress)
{
	return inet_addr(lpAddress)!=INADDR_NONE;
}

void CSpoofSocket::SetBlocking(BOOL bBlock)
{
	//Do we want blocking socket
	m_Async=!bBlock;
}
