// TCPSocket.cpp: implementation of the CTCPSocket class.
//
//////////////////////////////////////////////////////////////////////
/*
 *
 *
 *  Copyright (c) 2000 Barak Weichselbaum <barak@komodia.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *
 * Contact info:
 * Site: http://www.komodia.com
 * Email: barak@komodia.com
 */

#include "stdafx.h"
#include "TCPSocket.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CTCPSocket::CTCPSocket() : CSpoofSocket()
{
	InitializeTCP();
}

CTCPSocket::CTCPSocket(SOCKET sok) : CSpoofSocket(sok)
{
	InitializeTCP();
}

CTCPSocket::~CTCPSocket() 
{
	CSpoofSocket::~CSpoofSocket();
}

BOOL CTCPSocket::Create()
{
	SetProtocol(IPPROTO_TCP);
	return CSpoofSocket::Create(IPPROTO_IP);
}

BOOL CTCPSocket::Connect(int iSourcePort, LPCSTR lpDestinationAddress, int iDestinationPort)
{
	if (isRaw())
	{
		//Let's try our first attack
		LPTCPHeader lpHead;

		//Header length
		int iHeaderLength;
		iHeaderLength=TCPHeaderLength;

		//If we have TCP options
		if (m_Options)
			iHeaderLength+=m_TCPOptions->GetBufferLength();

		//Create the header
		lpHead=ConstructTCPHeader(iSourcePort,iDestinationPort,iHeaderLength);

		//Set the flags
		SetHeaderFlag(lpHead,TCPFlag_ACK);
		
		//Result 
		BOOL bResult;

		//Construct diffrently if we have options
		if (m_Options)
		{
			char* buf;
			buf=new char[iHeaderLength];

			//Copy header
			memcpy(buf,lpHead,TCPHeaderLength);
		
			//Copy options
			memcpy(buf+TCPHeaderLength,m_TCPOptions->GetBuffer(),m_TCPOptions->GetBufferLength());

			//Checksum it
			lpHead->Checksum=CalculatePseudoChecksum(buf,iHeaderLength,lpDestinationAddress,iHeaderLength);
			
			//Recopy header
			memcpy(buf,lpHead,TCPHeaderLength);

			//Send the data
			bResult=CSpoofSocket::Send(lpDestinationAddress,buf,iHeaderLength);

			//Dispose
			delete buf;
		}
		else
		{
			lpHead->Checksum=CalculatePseudoChecksum((char*)lpHead,TCPHeaderLength,lpDestinationAddress,TCPHeaderLength);

			//Send the data
			bResult=CSpoofSocket::Send(lpDestinationAddress,(char*)lpHead,TCPHeaderLength);
		}

		//Dispose the header
		delete lpHead;

		return bResult;
	}
	else
	{
		//Set async notification
		int iResult;

		iResult=WSAAsyncSelect(getHandle(),getWindowHandle(),WM_SOCKET_CONNECT,FD_CONNECT);
		if (iResult)
		{
			SetLastError();
			return FALSE;
		}

		//Create the address
		sockaddr_in soSrc;
	
		//Set to 0
		memset(&soSrc,0,sizeof(soSrc));
		soSrc.sin_family=AF_INET;
		soSrc.sin_addr.s_addr=inet_addr(lpDestinationAddress);
		soSrc.sin_port=htons(iDestinationPort);

		iResult=connect(getHandle(),(sockaddr*)&soSrc,sizeof(soSrc));

		int iTmp;
		iTmp=WSAGetLastError();

		return iResult!=SOCKET_ERROR;
	}
}

LPTCPHeader CTCPSocket::ConstructTCPHeader(int iSourcePort, int iDestinationPort,int iHeaderLength)
{
	//Construct the header
	LPTCPHeader lpHead=new _TCPHeader;
	
	//Set source and destination port
	lpHead->SourcePort=htons(iSourcePort);
	lpHead->DestinationPort=htons(iDestinationPort);

	//No checksums yet
	lpHead->Checksum=0;

	//Set windows to 3.0k
	lpHead->Windows=htons(512);

	//Set the packet number
	lpHead->AcknowledgeNumber=0;

	//And the sequence
	lpHead->SequenceNumber=htonl(m_Sequence++);

	//Data offset
	lpHead->DataOffset=(iHeaderLength/4) << 4;

	//Flags
	lpHead->Flags=0;

	//Urgent pointer
	lpHead->UrgentPointer=0;

	//Return it to the user
	return lpHead;
}

unsigned int CTCPSocket::m_Sequence=(unsigned int)GetCurrentProcessId();

void CTCPSocket::SetHeaderFlag(LPTCPHeader lpHead, int iFlag)
{
	//Logical or
	lpHead->Flags|=iFlag;	
}

void CTCPOptions::Reset()
{
	CIPOptions::Reset();
}

void CTCPOptions::SetAutoPad(BOOL bAutoPAD)
{
	CIPOptions::SetAutoPad(bAutoPAD);
}

void CTCPOptions::AddOption_ENDLIST()
{
	//Add option end list
	tOptionType OT;

	//Get the option
	OT=TCPOptions_END;

	//Add it to buffer
	AddToBuffer((char*)&OT,sizeof(OT));
}

int CTCPOptions::GetBufferLength()
{
	return CIPOptions::GetBufferLength();
}

const char* CTCPOptions::GetBuffer()
{
	return CIPOptions::GetBuffer();
}

void CTCPOptions::AddOption_Nothing()
{
	//Add option do nothing
	tOptionType OT;

	//Get the option
	OT=TCPOptions_NO_OPERATION;

	//Add it to buffer
	AddToBuffer((char*)&OT,sizeof(OT));
}

CTCPOptions::CTCPOptions() : CIPOptions()
{
}

CTCPOptions::~CTCPOptions()
{
	CIPOptions::~CIPOptions();
}

void CTCPOptions::AddOption_SegmentSize(unsigned short usMax)
{
	//Add option Max segment
	tOptionType OT;

	//Get the option
	OT=TCPOptions_MAX_Segment;

	//Add it to buffer
	AddToBuffer((char*)&OT,sizeof(OT));

	//Add length
	OT=TCPOptions_MAX_Segment_Length;
	AddToBuffer((char*)&OT,sizeof(OT));

	//Add segment size
	unsigned short usOT;
	usOT=htons(usMax);

	AddToBuffer((char*)&usOT,sizeof(usOT));

}

void CTCPSocket::SetTCPOptions(BOOL bOptions)
{
	//Do we want options, normaly not
	m_Options=bOptions;

	if (m_TCPOptions)
	{
		delete m_TCPOptions;
		m_TCPOptions=NULL;
	}

	if (bOptions)
		m_TCPOptions=new CTCPOptions;

}

CTCPOptions* CTCPSocket::GetTCPOptions()
{
	return m_TCPOptions;
}

BOOL CTCPSocket::CreateRegular()
{
	SetProtocol(IPPROTO_TCP);
	return CSpoofSocket::Create(IPPROTO_TCP);
}

BOOL CTCPSocket::Listen(int iBackLog)
{
	int iResult;
	
	iResult=WSAAsyncSelect(getHandle(),getWindowHandle(),WM_SOCKET_ACCEPT,FD_ACCEPT);
	if (iResult)
	{
		SetLastError();
		return FALSE;
	}

	return CSpoofSocket::Listen(iBackLog);
}

BOOL CTCPSocket::SetAsync()
{
	//Set event to read / write / close / oob
	int iResult;

	iResult=WSAAsyncSelect(getHandle(),getWindowHandle(),WM_SOCKET_GENERAL,FD_WRITE | FD_READ | FD_CLOSE | FD_OOB);
	if (iResult)
	{
		SetLastError();
		return FALSE;
	}
	
	return TRUE;
}

BOOL CTCPSocket::OnSocketConnect(int iErrorCode)
{
	//First set async again
	return SetAsync();
}

CTCPSocket* CTCPSocket::Accept()
{
	//First accept the socket
	SOCKET sok;

	int iTmp;

	sok=accept(getHandle(),(sockaddr*)&m_ConnectedTo,&iTmp);

	if (sok!=INVALID_SOCKET)
	{
		//Create the new tcp socket
		CTCPSocket* tSok;
		tSok=new CTCPSocket(sok);

		//Set the address
		tSok->m_ConnectedTo=m_ConnectedTo;

		if (tSok->SetAsync())
			return tSok;
		else
		{
			//Error
			delete tSok;
			return NULL;
		}
	}
	else
		//Error
		return NULL;
}

void CTCPSocket::InitializeTCP()
{
	//No options
	m_TCPOptions=NULL;

	SetTCPOptions(FALSE);
}

BOOL CTCPSocket::Accept(CTCPSocket *tSok)
{
	//First accept the socket
	SOCKET sok;
	memset(&m_ConnectedTo,0,sizeof(m_ConnectedTo));

	int iTmp;

	sok=accept(getHandle(),(sockaddr*)&m_ConnectedTo,&iTmp);

	if (sok!=INVALID_SOCKET)
	{
		tSok->m_ConnectedTo=m_ConnectedTo;

		tSok->AssignSocket(sok);
		tSok->SetAsync();
		return TRUE;
	}
	else
		return FALSE;
}

BOOL CTCPSocket::Send(char *buf, char bufLen)
{
	//Send the data
	int iResult;

	//And send it
	iResult=send(getHandle(),buf,bufLen,NULL);

	return iResult;
}

long CTCPSocket::GetPeerAddress()
{
	//Get the address we are connected to
	return m_ConnectedTo.sin_addr.S_un.S_addr;
}
