/*
 *  Copyright (c) 2000-2003 Barak Weichselbaum <barak@komodia.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Contact info:
 * -------------
 *
 * Site:					http://www.komodia.com
 * Main contact:			barak@komodia.com
 * For custom projects, 
 * consulting, or other
 * paid services:			sales@komodia.com
 */

#include "stdafx.h"
#include "RealIPResolver.h"

#include "ErrorHandlerMacros.h"
#include "Array_ptr.h"

#include "TCPSocket.h"

#include <iphlpapi.h>

#ifdef _MEMORY_DEBUG 
	#define new	   DEBUG_NEW  
	#define malloc DEBUG_MALLOC  
    static char THIS_FILE[] = __FILE__;  
#endif

KOMODIA_NAMESPACE_START

#define CRealIPResolver_Class "CRealIPResolver"

CRealIPResolver::CRealIPResolver() : CErrorHandler()
{
	try
	{
		//Set our name
		SetName(CRealIPResolver_Class);
	}
	ERROR_HANDLER_RETHROW("CRealIPResolver")
}

CRealIPResolver::~CRealIPResolver()
{
}

IP CRealIPResolver::Resolve(IP aInternetIP,
						    const std::string& rTestURL,
							unsigned short usPort)
{
	try
	{
		//Get it
		DWORD dwResult;
		DWORD dwIndex;
		dwResult=GetBestInterface(aInternetIP,
								  &dwIndex);

		//Did we manage?
		if (dwResult!=NO_ERROR)
		{
			//Report it
			ReportErrorOS("Resolve","Failed to call routing command! (GetBestInterface)");

			//Exit
			return 0;
		}

		//Get the length of data
		DWORD dwLength;
		dwLength=0;
		dwResult=GetInterfaceInfo(NULL,
								  &dwLength);
		
		if (dwResult!=NO_ERROR &&
			dwResult!=ERROR_INSUFFICIENT_BUFFER)
		{
			//Report it
			ReportErrorOS("Resolve","Failed to call routing command! (GetInterfaceInfo, NULL)");

			//Exit
			return 0;
		}

		//Allocate the data
		char* pData;
		pData=new char[dwLength];

		//Protect it
		CArray_ptr<char> pProtection(pData);

		//Recall
		if (GetInterfaceInfo((PIP_INTERFACE_INFO)pData,
							 &dwLength)!=NO_ERROR)
		{
			//Report it
			ReportErrorOS("Resolve","Failed to call routing command! (GetInterfaceInfo)");

			//Exit
			return 0;
		}

		//The Interface name
		WCHAR pName[MAX_ADAPTER_NAME];
		pName[0]=0;
		pName[1]=0;

		//Get the data
		PIP_INTERFACE_INFO pInfo;
		pInfo=(PIP_INTERFACE_INFO)pData;

		//Start to iterate the interfaces
		for (int iCount=0;
			 iCount<pInfo->NumAdapters;
			 ++iCount)
			if (pInfo->Adapter[iCount].Index==dwIndex)
			{
				//Copy the data
				wcscpy(pName,
					   pInfo->Adapter[iCount].Name);

				//Exit
				break;
			}

		//Do we have it?
		if (!pName[0])
			//Nop
			return 0;

		//Further iterations
		//Now get adapters information
		dwLength=0;
		dwResult=GetAdaptersInfo(NULL,
								 &dwLength);

		if (dwResult!=NO_ERROR &&
			dwResult!=ERROR_BUFFER_OVERFLOW)
		{
			//Report it
			ReportErrorOS("Resolve","Failed to call routing command! (GetAdaptersInfo, NULL)");

			//Exit
			return 0;
		}

		//Reallocate the data
		char* pAdapterRawInfo;
		pAdapterRawInfo=new char[dwLength];

		//Protect it
		CArray_ptr<char> pProtection2(pAdapterRawInfo);

		//And get it
		if (GetAdaptersInfo((PIP_ADAPTER_INFO)pAdapterRawInfo,
							&dwLength)!=NO_ERROR)
		{
			//Report it
			ReportErrorOS("Resolve","Failed to call routing command! (GetAdaptersInfo)");

			//Exit
			return 0;
		}

		//Get the name we want
		std::string sSymbolicToFind;
		sSymbolicToFind=ConvertSymbolic((const char*)pName);

		//Our info
		PIP_ADAPTER_INFO pAdapterInfo;
		pAdapterInfo=(PIP_ADAPTER_INFO)pAdapterRawInfo;

		//The IP we want
		IPVector aVector;

		do
		{
			//Get the symbolic
			std::string sSymbolic;
			sSymbolic=pAdapterInfo->AdapterName;

			//Is it?
			if (sSymbolic==sSymbolicToFind)
			{
				//Get the IPs
				aVector=ExtractIP((const void*)pAdapterInfo);

				//Exit the loop
				break;
			}

			//Next
			pAdapterInfo=pAdapterInfo->Next;
		} while (pInfo);

		//Do we have addresses?
		if (aVector.empty())
			return 0;

		//Do we have only one address?
		if (aVector.size()==1)
			//Do we need to test it?
			if (!rTestURL.empty())
				if (TestAddress(aVector[0],
							    CSocketBase::StringToLong(rTestURL),
								usPort))
					return aVector[0];
				else
					return 0;
			else
				return aVector[0];

		//We need to traverse the vetor
		return 0;

	}
	ERROR_HANDLER_RETURN("Resolve",0)
}

std::string CRealIPResolver::ConvertSymbolic(const char* pName)const
{
	try
	{
		//Our data
		std::string sData;

		//Our position
		int iPosition=0;

		//Are we in copy mode?
		BOOL bCopy;
		bCopy=FALSE;

		//Iterate the name
		while (pName[iPosition] ||
			   pName[iPosition+1])
		{
			//Do we need it?
			if (pName[iPosition]=='{')
				bCopy=TRUE;

			//Do we copy?
			if (bCopy)
				sData+=pName[iPosition];

			//Next
			iPosition+=2;
		}

		//Done
		return sData;
	}
	ERROR_HANDLER_RETURN("ConvertSymbolic","")
}

CRealIPResolver::IPVector CRealIPResolver::ExtractIP(const void* pData)const
{
	//Blank list
	IPVector aBlank;

	try
	{
		//Our list
		IPVector aVector;

		//Get the data we need
		PIP_ADAPTER_INFO pInfo;
		pInfo=(PIP_ADAPTER_INFO)pData;

		//Get the address list
		PIP_ADDR_STRING pAddress;
		pAddress=(PIP_ADDR_STRING)&(pInfo->IpAddressList);

		//Iterate it
		do
		{
			//Get the address
			IP aAddress;
			aAddress=CSocketBase::StringToLong(std::string((char*)&(pAddress->IpAddress)));

			//Add it
			aVector.push_back(aAddress);

			//Next
			pAddress=pAddress->Next;
		} while (pAddress);

		//The IP
		return aVector;
	}
	ERROR_HANDLER_RETURN("ExtractIP",aBlank)
}

BOOL CRealIPResolver::IsNATAddress(IP aIP)const
{
	try
	{
		//Our NAT addresses data
		static const iNatCount=4;
		static const std::string sNat[iNatCount*2]={"10.0.0.0","10.255.255.255",
											        "127.0.0.0","127.255.255.255",
											        "172.16.0.0","172.31.255.255",
											        "192.168.0.0","192.168.255.255"};

		//Our address
		IP aTestIP;
		aTestIP=htonl(aIP);

		//Run a test
		for (int iCount=0;
			 iCount<iNatCount;
			 ++iCount)
		{
			//Translate the addresses
			IP aFromIP;
			aFromIP=htonl(CSocketBase::StringToLong(sNat[iCount*2]));

			//To address
			IP aToIP;
			aToIP=htonl(CSocketBase::StringToLong(sNat[iCount*2+1]));

			//Is it NAT
			if (aFromIP<=aTestIP &&
				aToIP>=aTestIP)
				return TRUE;
		}

		//Not NAT
		return FALSE;
	}
	ERROR_HANDLER_RETURN("IsNATAddress",FALSE)
}

BOOL CRealIPResolver::TestAddress(IP aAddress,
								  IP aTestAddress,
								  unsigned short usPort)const
{
	try
	{
		//Do we missing an address?
		if (!aAddress ||
			!aTestAddress)
		{
			//Report it
			ReportError("TestAddress","Atleast one address is zero!");

			//Exit
			return FALSE;
		}

		//Create the socket
		CTCPSocket aSocket(FALSE);
		if (!aSocket.Create())
		{
			//Report it
			ReportError("TestAddress","Failed to create socket!");

			//Exit
			return FALSE;
		}

		//Try to bind the socket
		if (!aSocket.Bind(aAddress,
						  0))
		{
			//Report it
			ReportError("TestAddress","Failed to bind socket!");

			//Exit
			return FALSE;
		}

		//Try to connect
		if (!aSocket.Connect(aTestAddress,
							 usPort))
			//Failed to connect
			return FALSE;
		else
			return TRUE;
	}
	ERROR_HANDLER_RETURN("TestAddress",FALSE)
}

KOMODIA_NAMESPACE_END
