/*
 *  Copyright (c) 2000-2003 Barak Weichselbaum <barak@komodia.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Contact info:
 * -------------
 *
 * Site:					http://www.komodia.com
 * Main contact:			barak@komodia.com
 * For custom projects, 
 * consulting, or other
 * paid services:			sales@komodia.com
 */

#include "stdafx.h"
#include "TCPRelay.h"

#include "ErrorHandlerMacros.h"
#include "GenericCriticalSection.h"
#include "OSManager.h"

#ifdef _MEMORY_DEBUG 
	#define new	   DEBUG_NEW  
	#define malloc DEBUG_MALLOC  
    static char THIS_FILE[] = __FILE__;  
#endif

KOMODIA_NAMESPACE_START

// ----------------------------- CAcceptSocket start ---------------------

#define CAcceptSocket_Class "CAcceptSocket"

CTCPRelay::CAcceptSocket::CAcceptSocket(CTCPRelay* pFather) : CTCPSocketAsync(FALSE),
															  m_pFather(pFather)
{
	try
	{
		//Set our name
		SetName(CAcceptSocket_Class);
	}
	ERROR_HANDLER("CAcceptSocket")
}

CTCPRelay::CAcceptSocket::~CAcceptSocket()
{
}

BOOL CTCPRelay::CAcceptSocket::OnSocketAccept(int iErrorCode)
{
	try
	{
		//Do we have an error
		if (iErrorCode)
		{
			//Report it
			ReportError("OnSocketAccept","Had an error code!",iErrorCode);

			//Done
			return FALSE;
		}

		//Accept a new socket
		CClientSocket* pSocket;
		pSocket=new CClientSocket(m_pFather,
								  NULL);

		//Protect if
		std::auto_ptr<CClientSocket> pProtection(pSocket);

		//Try to accept
		if (!Accept(pSocket))
		{
			//Report it
			ReportErrorOS("OnSocketAccept","Failed to accept socket!");

			//Exit
			return FALSE;
		}

		//Create a new socket for it (opposite socket)
		CClientSocket* pOutgoingSocket;
		pOutgoingSocket=new CClientSocket(m_pFather,
										  pSocket);

		//Protect if
		std::auto_ptr<CClientSocket> pProtection2(pOutgoingSocket);

		//Try to create it
		if (!pOutgoingSocket->Create())
		{
			//Report it
			ReportErrorOS("OnSocketAccept","Failed to create socket!");

			//Exit
			return FALSE;
		}

		//Set the other socket
		pSocket->SetSocket(pOutgoingSocket);

		//Release protections
		pProtection.release();
		pProtection2.release();

		//Connect the socket
		pSocket->Connect();

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("OnSocketAccept",FALSE)
}

// ----------------------------- CAcceptSocket end ---------------------

// ----------------------------- CClientSocket start ---------------------

#define CClientSocket_Class "CClientSocket"

CTCPRelay::CClientSocket::CClientSocket(CTCPRelay* pFather,
										CClientSocket* pSocket) : CTCPSocketAsync(FALSE),
																  m_pFather(pFather),
																  m_pSocket(pSocket),
																  m_bIncoming(pSocket==NULL),
																  m_pCSection(NULL),
																  m_aConnectionID(0)
{
	try
	{
		//Set our name
		SetName(CClientSocket_Class);

		//Create only if incoming
		if (m_bIncoming)
			m_pCSection=COSManager::CreateCriticalSection();
	}
	ERROR_HANDLER("CClientSocket")
}

CTCPRelay::CClientSocket::~CClientSocket()
{
	try
	{
		delete m_pCSection;
	}
	ERROR_HANDLER("~CClientSocket")
}

void CTCPRelay::CClientSocket::Connect()
{
	try
	{
		//Try to connect the other socket
		((CTCPSocketAsync*)m_pSocket)->Connect(0,
											   m_pFather->GetTarget(),
											   m_pFather->GetTargetPort(),
											   FALSE,
											   TRUE);
	}
	ERROR_HANDLER("Connect")
}

int CTCPRelay::CClientSocket::Send(const char* pBuffer,
								   unsigned long ulBufferLength)
{
	try
	{
		//Do we have a timer
		if (m_bIncoming &&
			m_pFather->GetTimeout())
			//Set it
			if (!SetTimeout(m_pFather->GetTimeout()))
				//Report it
				ReportError("Send","Failed to set timeout!");

		//Delegate call
		return CTCPSocketAsync::Send(pBuffer,
									 ulBufferLength);
	}
	ERROR_HANDLER_RETURN("Send",GetErrorCode())
}

BOOL CTCPRelay::CClientSocket::OnSocketConnect(int iErrorCode)
{
	try
	{
		//Do we have an error
		if (iErrorCode)
		{
			//Delete the other socket
			m_pSocket->DeleteSocketFromThread();
			m_pSocket=NULL;

			//Delete ourself from our thread
			DeleteSocketFromThread();

			//Done
			return FALSE;
		}

		//We are connected
		//Create the connection
		m_aConnectionID=m_pFather->NewConnection(m_pSocket,
												 this);

		//Set the other socket's ID
		m_pSocket->SetConnectionID(m_aConnectionID);

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("OnSocketConnect",FALSE)
}

BOOL CTCPRelay::CClientSocket::OnSocketClose(int iErrorCode)
{
	try
	{
		//Remove the connection
		m_pFather->RemoveConnection(m_aConnectionID);

		//Delete other socket
		m_pSocket->DeleteSocketFromThread();
		m_pSocket=NULL;

		//Delete ourselves
		DeleteSocketFromThread();

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("OnSocketClose",FALSE)
}

BOOL CTCPRelay::CClientSocket::OnSocketTimeout()
{
	try
	{
		//Stop the other socket
		m_pSocket->Stop();

		//Remove ourselves
		m_pFather->RemoveConnection(m_aConnectionID);

		//Delete ourselves
		delete this;

		//Done
		return FALSE;
	}
	ERROR_HANDLER_RETURN("OnSocketTimeout",FALSE)
}

BOOL CTCPRelay::CClientSocket::OnSocketReceive(int iErrorCode)
{
	try
	{
		if (iErrorCode)
		{
			//Report it
			ReportError("OnSocketReceive","Received an error code!",iErrorCode);

			//Done
			return FALSE;
		}
			
		//Do we have a timer
		if (m_bIncoming &&
			m_pFather->GetTimeout())
			//Set it
			if (!SetTimeout(m_pFather->GetTimeout()))
				//Report it
				ReportError("OnSocketReceive","Failed to set timeout!");

		//Do we have a CS
		if (m_pCSection)
		{
			//Lock
			CCriticalAutoRelease aRelease(m_pCSection);

			//Are we connected
			if (m_aConnectionID)
			{
				//No more events
				m_bEvent=FALSE;

				//Save CS
				CGenericCriticalSection* pCS;
				pCS=m_pCSection;
				m_pCSection=NULL;

				//Exit the CS
				aRelease.Exit();

				//We can delete the CS
				delete pCS;
			}
			else
			{
				//Set we have an event
				m_bEvent=TRUE;

				//Exit
				return TRUE;
			}
		}

		char cBuffer[2000];

		//Read the data
		int iResult;
		iResult=Receive(cBuffer,
						sizeof(cBuffer)-1);

		//Try to modify it
		if (iResult>=0)
		{
			//Terminate the string
			cBuffer[iResult]=0;

			//Get the modified string
			std::string sModifiedString;
			sModifiedString=m_pFather->ModifyReceiveString(m_bIncoming,
														   cBuffer,
														   iResult);

			//Is it modified?
			if (!sModifiedString.empty())
			{
				//Replace the buffer
				strcpy(cBuffer,
					   sModifiedString.c_str());

				//Set the size
				iResult=sModifiedString.size();
			}
		}

		//Save the socket
		CClientSocket* pSocket;
		pSocket=m_pSocket;

		//Do we have anything
		if (iResult>0)
			if (pSocket &&
				pSocket->Send(cBuffer,
							  iResult)<=0)
			{
				//Report it
				ReportErrorOS("OnSocketReceive","Failed to send data!");

				//Exit
				return FALSE;
			}
			else
				;
		else if (iResult==GetErrorCode())
		{
			//Report it
			ReportErrorOS("OnSocketReceive","Failed to receive data!");

			//Exit
			return FALSE;
		}

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("OnSocketReceive",FALSE)
}

void CTCPRelay::CClientSocket::SetSocket(CClientSocket* pSocket)
{
	//Save the socket
	m_pSocket=pSocket;
}	

void CTCPRelay::CClientSocket::SetConnectionID(ConnectionID aConnectionID)
{
	try
	{
		//Do we have the CS
		if (m_pCSection)
		{
			//Lock
			CCriticalAutoRelease aRelease(m_pCSection);

			//Set it
			m_aConnectionID=aConnectionID;

			//Do we have an event?
			if (m_bEvent)
				//Force an event
				ForceReceiveEvent();
		}
		else
			//Set it
			m_aConnectionID=aConnectionID;
	}
	ERROR_HANDLER("SetConnectionID")
}

void CTCPRelay::CClientSocket::Stop()
{
	try
	{
		//Which socket are we
		if (m_bIncoming)
		{
			//Take the socket
			CClientSocket* pSocket;
			pSocket=m_pSocket;
			m_pSocket=NULL;

			//Stop the other socket
			pSocket->Stop();
		}
		else
			m_pSocket=NULL;

		//Post a delete
		DeleteSocketFromThread();
	}
	ERROR_HANDLER("Stop")
}

// ----------------------------- CClientSocket end ---------------------

#define CTCPRelay_Class "CTCPRelay"

CTCPRelay::CTCPRelay() : CRelay(),
						 m_pCSection(NULL),
						 m_pSocket(NULL),
						 m_aRunningID(0),
						 m_bCreated(FALSE)
{
	try
	{
		//Set our name
		SetName(CTCPRelay_Class);

		//Create the CS
		m_pCSection=COSManager::CreateCriticalSection();
	}
	ERROR_HANDLER("CTCPRelay")
}

CTCPRelay::~CTCPRelay()
{
	try
	{
		//Are we created
		if (m_bCreated)
			Stop();

		//Delete the CS
		delete m_pCSection;
	}
	ERROR_HANDLER("~CTCPRelay")
}

CTCPRelay::ConnectionID CTCPRelay::NewConnection(CClientSocket* pIncomingConnection,
												 CClientSocket* pOutgoingConnection)
{
	try
	{
		//The structure
		SocketData aData;
		aData.pIncomingConnection=pIncomingConnection;
		aData.pOutgoingConnection=pOutgoingConnection;

		//Lock the data
		CCriticalAutoRelease aRelease(m_pCSection);

		//Get the ID
		ConnectionID aID;
		aID=++m_aRunningID;

		//Add to the map
		m_aClientMap.insert(ClientMap::value_type(aID,aData));

		//Done
		return aID;
	}
	ERROR_HANDLER_RETURN("NewConnection",0)
}

void CTCPRelay::RemoveConnection(ConnectionID aConnectionID)
{
	try
	{
		//Lock the data
		CCriticalAutoRelease aRelease(m_pCSection);

		//Erase from the map
		m_aClientMap.erase(aConnectionID);

		//Done
	}
	ERROR_HANDLER("RemoveConnection")
}

IP CTCPRelay::GetTarget()const
{
	return m_aTarget;
}

unsigned short CTCPRelay::GetTargetPort()const
{
	return m_usTargetPort;
}

IP CTCPRelay::GetBindAddress()const
{
	return m_aBindAddress;
}

BOOL CTCPRelay::Relay(IP aBindAddress,
					  unsigned short usBindPort,
					  IP aDestinationAddress,
					  unsigned short usDestinationPort)
{
	try
	{
		//Are we created?
		if (m_bCreated)
		{
			//Report it
			ReportError("Relay","Already created!");

			//Exit
			return FALSE;
		}

		//Create the socket
		m_pSocket=new CAcceptSocket(this);

		//Protect it
		std::auto_ptr<CAcceptSocket> pProtection(m_pSocket);

		//Try to create it
		if (!m_pSocket->Create())
		{
			//Report it
			ReportError("Relay","Failed to create socket!");

			//Exit
			return FALSE;
		}

		//Try to bind it
		if (!m_pSocket->Bind(aBindAddress,
							 usBindPort))
		{
			//Report it
			ReportError("Relay","Failed to create socket!");

			//Exit
			return FALSE;
		}

		//Save the data
		m_aTarget=aDestinationAddress;
		m_usTargetPort=usDestinationPort;
		m_aBindAddress=aBindAddress;

		//Try to listen
		if (!m_pSocket->Listen())
		{
			//Report it
			ReportError("Relay","Failed to listen!");

			//Exit
			return FALSE;
		}

		//Remove protection
		pProtection.release();

		//We are done
		m_bCreated=TRUE;

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("Relay",FALSE)
}

BOOL CTCPRelay::Relay(const std::string& rBindAddress,
					  unsigned short usBindPort,
					  const std::string& rDestinationAddress,
					  unsigned short usDestinationPort)
{
	try
	{
		return Relay(CSocketBase::StringToLong(rBindAddress),
					 usBindPort,
					 CSocketBase::StringToLong(rDestinationAddress),
					 usDestinationPort);
	}
	ERROR_HANDLER_RETURN("Relay",FALSE)
}

BOOL CTCPRelay::IsRunning()const
{
	return m_bCreated;
}

BOOL CTCPRelay::Stop()
{
	try
	{
		//Are we created?
		if (!m_bCreated)
			return TRUE;

		//Delete the listener
		m_pSocket->DeleteSocketFromThread();
		m_pSocket=NULL;

		{
			//Lock
			CCriticalAutoRelease aRelease(m_pCSection);

			//Iterate the data
			ClientMap::iterator aIterator;
			aIterator=m_aClientMap.begin();
			while (aIterator!=m_aClientMap.end())
			{
				//Signal the socket
				aIterator->second.pIncomingConnection->Stop();

				//Erase
				aIterator=m_aClientMap.erase(aIterator);
			}
		}

		//We are not created
		m_bCreated=FALSE;

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("Stop",FALSE)
}

std::string CTCPRelay::ModifyReceiveString(BOOL bIncoming,
										   const char* pBuffer,
										   unsigned short usBufferSize)const
{
	return "";
}

KOMODIA_NAMESPACE_END
