/*
 *  Copyright (c) 2000-2003 Barak Weichselbaum <barak@komodia.com>
 *  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Contact info:
 * -------------
 *
 * Site:					http://www.komodia.com
 * Main contact:			barak@komodia.com
 * For custom projects, 
 * consulting, or other
 * paid services:			sales@komodia.com
 */

#include "stdafx.h"
#include "WhoisSocket.h"

#include "ErrorHandlerMacros.h"

#ifdef _MEMORY_DEBUG 
	#define new	   DEBUG_NEW  
	#define malloc DEBUG_MALLOC  
    static char THIS_FILE[] = __FILE__;  
#endif

KOMODIA_NAMESPACE_START

#define CWhoisSocket_Class "CWhoisSocket"
#define FINISH_CHAR "FINISH"

std::string CWhoisSocket::m_sEmpty="";

std::string CWhoisSocket::m_sData[]={"COM","whois.crsnic.net",
									 "NET","whois.crsnic.net",
									 "ORG","whois.crsnic.net",
									 "EDU","whois.crsnic.net",
									 "UK","whois.nic.uk",
									 "NU","whois.nic.nu",
									 "CC","whois.nic.cc",
									 "AU","whois.aunic.net",
									 "MIL","whois.nic.mil",
									 "IL","whois.isoc.org.il",
									 "AC","whois.nic.ac",
									 "AG","whois.nic.ag",
									 "AS","whois.nic.as",
									 "AT","whois.nic.at",
									 "BE","whois.dns.be",
									 "BG","whois.digsys.bg",
									 "BR","whois.nic.br",
									 "BV","whois.norid.no",
									 "SJ","whois.norid.no",
									 "CC","whois.nic.cc",
									 "CH","whois.nic.ch",
									 "CD","whois.nic.cd",
									 "CK","whois.ck-nic.org.ck",
									 "CL","whois.nic.cl",
									 "CX","whois.nic.cx",
									 "DE","whois.denic.de",
									 "CZ","whois.nic.cz",
									 "EE","whois.eenet.ee",
									 "FJ","whois.usp.ac.fj",
									 "FR","whois.nic.fr",
									 "GS","whois.adamsnames.tc",
									 "HU","nic.hu",
									 "IN","whois.ncst.ernet.in",
									 "IE","whois.domainregistry.ie",
									 "IR","whois.nic.ir",
									 "IS","whois.isnet.is",
									 "IT","whois.nic.it",
									 "JP","whois.nic.ad.jp",
									 "KR","whois.krnic.net",
									 "LI","whois.nic.li",
									 "LU","whois.dns.lu",
									 "TC","whois.adamsnames.tc",
									 "MX","whois.nic.mx",
									 "NL","whois.domain-registry.nl",
									 "NO","whois.norid.no",
									 "NU","whois.nic.nu",
									 "PL","whois.nask.pl",
									 "PW","whois.nic.pw",
									 "RE","whois.nic.fr",
									 "RU","whois.ripn.net",
									 "RO","whois.rnc.ro",
									 "SE","whois.nic-se.se",
									 "SG","whois.nic.net.sg",
									 "SH","whois.nic.sh",
									 "SI","whois.arnes.si",
									 "SZ","whois.adamsnames.tc",
									 "TC","whois.adamsnames.tc",
									 "TH","whois.thnic.net",
									 "TJ","whois.nic.tj",
									 "TO","whois.tonic.to",
									 "TR","whois.metu.edu.tr",
									 "TW","whois.twnic.net",
									 "UA","whois.net.ua",
									 "VG","whois.adamsnames.tc",
									 "WF","whois.nic.fr",
									 "YT","whois.nic.fr",
									 FINISH_CHAR};

std::string CWhoisSocket::m_aIPWhois[wsLast]={"",
											  "whois.arin.net",
											  "whois.apnic.net",
											  "whois.ripe.net"};

std::string CWhoisSocket::m_aWhoisNotFound[wsLast]={"",
													"These addresses have been further assigned",
													"No entries found",
													"This address space is assigned at various other places in"};
CWhoisSocket::WhoisMAP CWhoisSocket::m_aData;

CWhoisSocket CWhoisSocket::m_aSocket;

const std::string CWhoisSocket::m_sWhoisClause="Whois Server: ";

CWhoisSocket::CWhoisSocket() : CTCPSocketAsync(FALSE)
{
	try
	{
		//Set our name
		SetName(CWhoisSocket_Class);

		//Build the map
		PopulateMap();
	}
	ERROR_HANDLER("CWhoisSocket")
}

CWhoisSocket::CWhoisSocket(BOOL bAsync) : CTCPSocketAsync(FALSE),
										  m_bAsync(bAsync),
										  m_bDone(TRUE),
										  m_bError(FALSE),
										  m_aServer(wsNone),
										  m_dwTimeout(5000)
{
	try
	{
		//Set our name
		SetName(CWhoisSocket_Class);
	}
	ERROR_HANDLER("CWhoisSocket")
}

CWhoisSocket::~CWhoisSocket()
{
}

void CWhoisSocket::QueryDone(BOOL bError,
							 BOOL bReallocate)
{
	try
	{
		//Set the data
		m_bError=bError;

		//Do we have an error
		if (bReallocate)
		{
			//Close the socket
			if (!Close())
				ReportError("QueryDone","Failed to close socket!");

			//Recreate it
			if (!Create())
				ReportError("QueryDone","Failed to create socket!");
		}

		//What are we
		if (m_bURL)
		{
			m_bDone=TRUE;

			//If error reset the string
			if (bError)
				m_sWhoisData="";
			//Do we need to get the additional server
			else if (m_bAdditionalServer)
				m_sAdditionalWhoisServer=ExtractAdditionalWhoisServer(m_sWhoisData);
			else
				m_sAdditionalWhoisServer="";

			//Call user routing
			WhoisDone(bError);
		}
		else if (m_aProgress!=wsNone)
		{
			//Check the text
			//Try to analyze it
			if (!bError &&
				m_sWhoisData.find(m_aWhoisNotFound[m_aProgress])==std::string::npos)
			{
				//We have it
				m_aServer=m_aProgress;

				//We are done
				m_bDone=TRUE;

				//Call user routing
				WhoisDone(FALSE);
			}
			else
			{
				//Decrease the progress
				m_aProgress=(WhoisServer)((int)m_aProgress-1);

				//Where are we
				if (m_aProgress==wsNone)
				{
					//We are done
					m_bDone=TRUE;
					m_bError=TRUE;
		
					//Call user routing
					WhoisDone(TRUE);
				}
				else if (!WhoisURL(m_sDataToSend,m_aIPWhois[m_aProgress]))
				{
					//Report it
					ReportError("QueryDone","Failed to connect!");

					//Try to move to the next one
					//Decrease the progress
					m_aProgress=(WhoisServer)((int)m_aProgress-1);
		
					//Call user routing
					QueryDone(TRUE,
							  TRUE);
				}
			}
		}
		else
			//We are done
			WhoisDone(TRUE);
	}
	ERROR_HANDLER("QueryDone")
}

void CWhoisSocket::WhoisDone(BOOL bError)
{
}

BOOL CWhoisSocket::OnSocketTimeout()
{
	try
	{
		//Kill the timer
		KillTimer();

		//We are done
		QueryDone(TRUE,
				  TRUE);

		//Exit
		return FALSE;
	}
	ERROR_HANDLER_RETURN("OnSocketTimeout",FALSE)
}

BOOL CWhoisSocket::OnSocketConnect(int iErrorCode)
{
	try
	{
		//Do we have an error
		if (iErrorCode)
		{
			//Report it
			ReportError("OnSocketConnect",iErrorCode);

			//We are done
			QueryDone(TRUE,
					  TRUE);

			//Exit
			return FALSE;
		}

		//Send the URL
		if (!SendURL(m_sDataToSend))
		{
			//Report it
			ReportError("OnSocketConnect","Failed to send data!");

			//We are done
			QueryDone(TRUE,
					  TRUE);

			//Exit
			return FALSE;
		}

		//Set the timeout
		SetTimeout(m_dwTimeout);

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("OnSocketConnect",FALSE)
}

BOOL CWhoisSocket::OnSocketClose(int iErrorCode)
{
	try
	{
		//Kill the timeout
		KillTimer();

		//Close the socket
		if (!Close())
			ReportError("OnSocketClose","Failed to close socket!");

		//Recreate it
		if (!Create())
		{
			//Report it
			ReportError("OnSocketClose","Failed to create socket!");

			//Exit
			return FALSE;
		}

		//We are done
		QueryDone(FALSE,
				  FALSE);

		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("OnSocketClose",FALSE)
}

BOOL CWhoisSocket::OnSocketReceive(int iErrorCode)
{
	try
	{
		//Kill the timer
		KillTimer();

		//Do we have an error
		if (iErrorCode)
		{
			//Report it
			ReportError("OnSocketReceive",iErrorCode);

			//We are done
			QueryDone(TRUE,
					  TRUE);

			//Exit
			return FALSE;
		}

		//Receive the data
		char aData[4000];

		//Get the data
		int iSize;
		iSize=Receive(aData,sizeof(aData)/2-1);

		//Is an error
		if (iSize==GetErrorCode())
		{
			//We are done
			QueryDone(TRUE,
					  TRUE);

			//Exit
			return FALSE;
		}

		//Remove nulls
		for (int iCounter=0;
			 iCounter<iSize;
			 iCounter++)
			if (!aData[iCounter])
				aData[iCounter]=10;

		//Set the terminating sero
		aData[iSize]=0;

		//Modify it
		ModifyBuffer(aData,iSize+1);

		//Add it to the string
		m_sWhoisData+=aData;

		//Reset the timeout
		SetTimeout(m_dwTimeout);
		
		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("OnSocketReceive",FALSE)
}

const std::string& CWhoisSocket::GetWhoisServer(const std::string& rURL)const
{
	try
	{
		//Get the last part of the address
		std::string::size_type aPos;
		aPos=rURL.find_last_of('.');

		//Do we have it
		if (aPos==std::string::npos)
			return m_sEmpty;

		//Check the size of the extension
		if (rURL.length()-aPos>4)
			return m_sEmpty;

		//Take it
		std::string::size_type aCopied;
		char aExtension[10];
		aCopied=rURL.copy(aExtension,rURL.length()-aPos-1,aPos+1);
		aExtension[aCopied]=0;

		//Convert it to upper case
		strupr(aExtension);

		//Copy it into a string
		std::string sExtension;
		sExtension=aExtension;

		//Look in the map
		WhoisMAP::const_iterator aIterator;
		aIterator=m_aData.find(sExtension);

		//Do we have it
		if (aIterator!=m_aData.end())
			return aIterator->second;
		else
			//Not found
			return m_sEmpty;
	}
	ERROR_HANDLER_RETURN("GetWhoisServer",m_sEmpty)
}

BOOL CWhoisSocket::Create()
{
	try
	{
		//Try to create it
		if (CTCPSocketAsync::Create())
		{
			if (!m_bAsync)
			{
				if (!ReBlock())
				{
					//Report it
					ReportError("CreateRegular","Failed to reblock!");

					//Exit
					return FALSE;
				}

				//Set it on the receive
				if (!SetReceiveTimeout(m_dwTimeout))
				{
					//Report it
					ReportError("CreateRegular","Failed to set timeout!");

					//Exit
					return FALSE;
				}
			}

			//Done
			return TRUE;
		}
		else
			return FALSE;
	}
	ERROR_HANDLER_RETURN("Create",FALSE)
}

BOOL CWhoisSocket::WhoisURL(const std::string& rURL)
{
	try
	{
		//Get the host to connect to
		std::string sHost;
		sHost=GetWhoisServer(rURL);

		//URL trace
		m_bURL=TRUE;

		//Is it valid
		if (!sHost.length())
		{
			//Report it
			ReportError("WhoisURL","Failed to find host to connect to!");

			//Exit
			return FALSE;
		}
		else
			return WhoisURL(rURL,sHost);
	}
	ERROR_HANDLER_RETURN("WhoisURL",FALSE)
}

BOOL CWhoisSocket::WhoisURL(const std::string& rURL,
							const std::string& rWhoisServer)
{
	try
	{
		//Try to resolve the URL
		IP aAddress;
		aAddress=ResolveDNS(rWhoisServer.c_str());

		//Do we have it
		if (!aAddress)
		{
			//Report it
			ReportError("WhoisURL","Failed to resolve whois server!");

			//Exit
			return FALSE;
		}

		//Set some data
		m_bDone=FALSE;
		m_bError=FALSE;
		m_sWhoisData="";
		m_sDataToSend=rURL;
		m_sAdditionalWhoisServer="";
		m_bAdditionalServer=rWhoisServer=="whois.crsnic.net";

		//Try to connect
		if (!Connect(aAddress,43))
		{
			//Report it
			ReportError("WhoisURL","Failed to connect!");

			//Not done
			m_bDone=TRUE;

			//We have errors
			m_bError=TRUE;

			//Exit
			return FALSE;
		}

		//Are we async ?
		if (!m_bAsync)
			if (!SendURL(rURL))
			{
				//Report it
				ReportError("WhoisURL","Failed to send data!");

				//Not done
				m_bDone=TRUE;

				//We have errors
				m_bError=TRUE;

				//Exit
				return FALSE;
			}
			else
				return ReceiveWhois();
		else
			return TRUE;
	}
	ERROR_HANDLER_RETURN("WhoisURL",FALSE)
}

BOOL CWhoisSocket::IsWhoisDone()const
{
	return m_bDone;
}

BOOL CWhoisSocket::IsError()const
{
	return m_bError;
}

BOOL CWhoisSocket::ReceiveWhois()
{
	try
	{
		//Receive the data
		char aData[4000];

		//Infinite loop
		while (1)
		{
			//Get it
			int iSize;
			iSize=Receive(aData,sizeof(aData)/2-1);

			//How much size
			if (!iSize)
			{
				//Close the socket
				if (!Close())
					ReportError("ReceiveWhois","Failed to close socket!");

				//Recreate it
				if (!Create())
				{
					//Report it
					ReportError("ReceiveWhois","Failed to create socket!");

					//Exit
					return FALSE;
				}

				//Do we need to get the additional server
				if (m_bAdditionalServer)
					m_sAdditionalWhoisServer=ExtractAdditionalWhoisServer(m_sWhoisData);

				//Done
				return TRUE;
			}

			//Is an error
			if (iSize==GetErrorCode())
				return FALSE;

			//Remove nulls
			for (int iCounter=0;
				 iCounter<iSize;
				 iCounter++)
				if (!aData[iCounter])
					aData[iCounter]=10;

			//Set the terminating sero
			aData[iSize]=0;

			//Modify it
			ModifyBuffer(aData,iSize+1);

			//Add it to the string
			m_sWhoisData+=aData;
		}
		
		//Done
		return TRUE;
	}
	ERROR_HANDLER_RETURN("ReceiveWhois",FALSE)
}

BOOL CWhoisSocket::SendURL(const std::string& rURL)
{
	try
	{
		//Create the URL buffer
		char* pBuffer;
		pBuffer=new char[rURL.length()+10];

		//Copy it
		rURL.copy(pBuffer,rURL.length(),0);
		
		//Set the enter
		pBuffer[rURL.length()]=10;

		//Send it
		int iSendSize;
		iSendSize=Send(pBuffer,rURL.length()+1);

		//Delete the string
		delete [] pBuffer;

		//Did we send it
		return iSendSize!=GetErrorCode();
	}
	ERROR_HANDLER_RETURN("SendURL",FALSE)
}

void CWhoisSocket::ModifyString(std::string& rString)const
{
	try
	{
		//The iterator 
		std::string::iterator aIterator;
		aIterator=rString.begin();

		//String to replace with
		std::string sReplace;
		char aChar;
		aChar=13;

		//Run it
		while (aIterator<rString.end())
		{
			//Is it our character
			if (*aIterator==10)
			{
				//Copy it
				std::string::iterator aBackupIterator;
				aBackupIterator=aIterator;

				//Move on
				++aIterator;

				//Change the data
				rString.insert(aBackupIterator,aChar);
			}
			else
				++aIterator;
		}
	}
	ERROR_HANDLER("ModifyString")
}

const std::string& CWhoisSocket::GetWhoisData()const
{
	return m_sWhoisData;
}

BOOL CWhoisSocket::WhoisIP(const std::string& rIP)
{
	try
	{
		//Convert it and send
		return WhoisIP(StringToLong(rIP));
	}
	ERROR_HANDLER_RETURN("WhoisIP",FALSE)
}

BOOL CWhoisSocket::WhoisIP(IP aAddress)
{
	try
	{
		//No server yet
		m_aServer=wsNone;

		//Are we async
		if (!m_bAsync)
		{
			//Try to get the data
			for (int iCounter=wsRipe;
				 iCounter>wsNone;
				 --iCounter)
			{
				//Try to get the data
				if (!WhoisIP(aAddress,m_aIPWhois[iCounter]))
				{
					//Error
					ReportError("WhoisIP","Failed getting data from server!");

					//Exit
					return FALSE;
				}

				//Try to analyze it
				if (m_sWhoisData.find(m_aWhoisNotFound[iCounter])==std::string::npos)
				{
					//We have it
					m_aServer=(WhoisServer)iCounter;

					//Exit
					return TRUE;
				}
			}

			//Didn't found it
			return FALSE;
		}
		else
		{
			//Set the current server
			m_aProgress=wsRipe;

			//Connect
			return WhoisIP(aAddress,m_aIPWhois[m_aProgress]);
		}
	}
	ERROR_HANDLER_RETURN("WhoisIP",FALSE)
}

BOOL CWhoisSocket::WhoisIP(IP aAddress,
						   const std::string& rWhoisServer)
{
	try
	{
		//IP trace
		m_bURL=FALSE;

		//Trace it
		return WhoisURL(LongToStdString(aAddress),rWhoisServer);
	}
	ERROR_HANDLER_RETURN("WhoisIP",FALSE)
}

void CWhoisSocket::SetWhoisTimeout(DWORD dwMS)
{
	try
	{
		//Set it
		m_dwTimeout=dwMS;

		//Set it on the receive
		SetReceiveTimeout(dwMS);
	}
	ERROR_HANDLER("SetWhoisTimeout")
}

BOOL CWhoisSocket::IsIPTrace()const
{
	return !m_bURL;
}

void CWhoisSocket::ModifyBuffer(char* pBuffer,
								int iBufferSize)const
{
	try
	{
		//Allocate a big buffer
		char* pNewBuffer;
		pNewBuffer=new char[iBufferSize*2];

		//Buffer pos
		int iBufferPos;
		iBufferPos=0;

		//Start iterating it
		for (int iCounter=0;iCounter<iBufferSize;++iCounter)
		{
			//Copy the data
			pNewBuffer[iBufferPos++]=pBuffer[iCounter];

			//Check it
			if (pBuffer[iCounter]==10)
				pNewBuffer[iBufferPos++]=13;
		}

		//Copy it back
		memcpy(pBuffer,pNewBuffer,iBufferPos);

		//Delete it
		delete [] pNewBuffer;
	}
	ERROR_HANDLER("ModifyBuffer")
}

void CWhoisSocket::PopulateMap()
{
	try
	{
		//Start looking for it
		int iPos;
		iPos=0;

		while (m_sData[iPos]!=FINISH_CHAR)
		{
			//Add it
			m_aData.insert(WhoisMAP::value_type(m_sData[iPos],m_sData[iPos+1]));

			//Continue
			iPos+=2;
		}
	}
	ERROR_HANDLER_STATIC(CWhoisSocket_Class,"PopulateMap")
}

CWhoisSocket::AddressVector CWhoisSocket::AnalyzeAddress(const std::string& rAddress)
{
	//Our vector
	AddressVector aVector;

	//Do we have data in the string
	if (!rAddress.length())
	{
		//Report it
		ReportStaticError(CWhoisSocket_Class,"AnalyzeAddress","Recieved empty string!");

		//Exit
		return aVector;
	}

	try
	{
		//Copy the string to a tmp buffer
		std::string sTmp;
		sTmp=rAddress;

		//Get the last part of the address
		std::string::size_type aPos;
		aPos=sTmp.find_last_of('.');

		//Do we have it
		while (aPos!=std::string::npos)
		{
			//How much to copy
			int iCopySize;
			iCopySize=sTmp.length()-aPos-1;

			//Allocate the buffer
			char* pExtension;
			pExtension=new char[iCopySize+1];

			//Take it
			std::string::size_type aCopied;
			aCopied=sTmp.copy(pExtension,iCopySize,aPos+1);
			pExtension[aCopied]=0;

			//Convert it to upper case
			strupr(pExtension);

			//Put it in a string
			std::string sData;
			sData=pExtension;

			//Put inside the vector
			aVector.push_back(sData);

			//Done with the buffer
			delete [] pExtension;

			//Cut the string
			sTmp=sTmp.substr(0,aPos);

			//Get the last part of the address
			aPos=sTmp.find_last_of('.');
		}

		//Insert this string
		aVector.push_back(sTmp);

		//Done
		return aVector;
	}
	ERROR_HANDLER_STATIC_RETURN(CWhoisSocket_Class,"AnalyzeAddress",aVector)
}

std::string CWhoisSocket::GetAddressForWhois(const std::string& rAddress)
{
	try
	{
		//Get the vector
		AddressVector aVector;
		aVector=AnalyzeAddress(rAddress);

		//Do we have data
		if (aVector.begin()==aVector.end())
			return "";

		//Get the iterator
		AddressVector::const_iterator aIterator;
		aIterator=aVector.begin();

		//Our address
		std::string sAddress;
		std::string sAddAddress;

		//Get the first one
		sAddress+=*aIterator;

		//Do we have the second one ?
		++aIterator;
		if (aIterator==aVector.end())
			return "";

		//Add it
		sAddAddress=*aIterator;
		sAddAddress+='.';
		sAddAddress+=sAddress;

		//What type is it
		if (!(*aIterator=="COM" ||
			  *aIterator=="NET" ||
			  *aIterator=="ORG" ||
			  *aIterator=="MIL" ||
			  *aIterator=="EDU" ||
			  *aIterator=="CO") ||
			  sAddress.length()==3)
			//Done
			return sAddAddress;

		//Another sub address
		++aIterator;
		if (aIterator==aVector.end())
			return "";

		//This is the final address
		sAddress=*aIterator;
		sAddress+='.';
		sAddress+=sAddAddress;

		//Done
		return sAddress;
	}
	ERROR_HANDLER_STATIC_RETURN(CWhoisSocket_Class,"GetAddressForWhois","")
}

const std::string& CWhoisSocket::GetAdditionalWhoisServer()const
{
	return m_sAdditionalWhoisServer;
}

std::string CWhoisSocket::ExtractAdditionalWhoisServer(const std::string rWhoisQuery)const
{
	try
	{
		if (!rWhoisQuery.length())
			return "";

		//Get the last part of the address
		std::string::size_type aPos;
		aPos=rWhoisQuery.find(m_sWhoisClause);

		//Do we have it
		if (aPos==std::string::npos)
			return "";

		//Char to find
		char aChar;
		aChar=10;

		//Find the end of line
		std::string::size_type aLastPos;
		aLastPos=rWhoisQuery.find_first_of(aChar,aPos);

		//Can't be npos but check anyway
		if (aLastPos==std::string::npos)
			return "";

		//We don't need the header
		aPos+=m_sWhoisClause.length();

		//Get that string
		return rWhoisQuery.substr(aPos,aLastPos-aPos);
	}
	ERROR_HANDLER_RETURN("ExtractAdditionalWhoisServer","")
}

const std::string& CWhoisSocket::GetLastTarget()const
{
	return m_sDataToSend;
}

KOMODIA_NAMESPACE_END
