


#include <arpa/inet.h>
#include <errno.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <unistd.h>
#include "Flitz.h"
#include "net.h"
#include "parse.h"
#include "xwrapper.h"



/* Socket states. */
#define UNUSED          0x00
#define LISTENING       0x01
#define CONNECTED       0x02
#define CLOSE           0x03



/* Connection types. */
#define SLAVE           0x00
#define MASTER          0x01



static struct {
    char state;
    char type;
    int socket;
    int auth;
    time_t time_rd;
    char *inbuf;
    char *outbuf;
    unsigned long who;
} socklist[MAXSOCKS];



void net_init()
{
    int i;

    for (i = 0; i < MAXSOCKS; ++i) {
        memset(&socklist[i], 0x00, sizeof(socklist[i]));
    }
    return;
}



void net_close(int i)
{
    socklist[i].state = CLOSE;
    return;
}



int net_assign()
{
    int i;

    for (i = 0; i < MAXSOCKS; ++i) {
        if (socklist[i].state == UNUSED) {
            return(i);
        }
    }
    return(-1);
}



void net_reset(int i)
{
    close(socklist[i].socket);
    if (socklist[i].inbuf != NULL) {
        free(socklist[i].inbuf);
    }
    if (socklist[i].outbuf != NULL) {
        free(socklist[i].outbuf);
    }
    memset(&socklist[i], 0x00, sizeof(socklist[i]));
    return;
}



void net_accept(int s)
{
    int n, new, len;
    struct sockaddr who;

    memset(&who, 0x00, sizeof(struct sockaddr));
    if ((new = accept(s, (struct sockaddr *) &who, &len)) > 0) {
        if ((n = net_assign()) > -1) {
            socklist[n].state = CONNECTED;
            socklist[n].type = MASTER;
            socklist[n].socket = new;
            time(&socklist[n].time_rd);
        }
    }
    return;
}



int net_nonblock(int s)
{
    int fdflag;

    fdflag = fcntl(s, F_GETFL, 0);
    fcntl(s, F_SETFL, fdflag | O_NONBLOCK);
    return(s);
}



int net_tcplisten(int port, int nr)
{
    int s, arg;
    struct sockaddr_in addr;

    memset(&addr, 0x00, sizeof(struct sockaddr_in));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = inet_addr("0.0.0.0");
    addr.sin_port = htons(port);

    if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
        return(-1);
    }

    arg = 1;
    setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, &arg, sizeof(arg));
    arg = 0;
    setsockopt(s, SOL_SOCKET, SO_LINGER, &arg, sizeof(arg));
    arg = 1;
    setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &arg, sizeof(arg));

    if ((s = net_nonblock(s)) < 0) {
        close(s);
        return(-1);
    }

    if (bind(s, (struct sockaddr *) &addr, sizeof(addr)) < 0 ||
        listen(s, nr) < 0) {
        close(s);
        return(-1);
    }
    return(s);
}



int net_udplisten(int port)
{
    int s, arg;
    struct sockaddr_in addr;

    memset(&addr, 0x00, sizeof(struct sockaddr_in));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(port);

    if ((s = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
        return(-1);
    }

    arg = 1;
    setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, &arg, sizeof(arg));
    arg = 0;
    setsockopt(s, SOL_SOCKET, SO_LINGER, &arg, sizeof(arg));
    arg = 1;
    setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &arg, sizeof(arg));

    if (bind(s, (struct sockaddr *) &addr, sizeof(struct sockaddr)) < 0) {
        close(s);
        return(-1);
    }
    return(s);
}



void net_send(int i, char *data, ...)
{
    char str[1024], *str_ptr;
    va_list line;

    if (socklist[i].state == CONNECTED &&
        socklist[i].type == MASTER) {

        memset(&line, 0x00, sizeof(line));
        va_start(line, data);
        vsnprintf(str, 1024, data, line);
        va_end(line);

        if (socklist[i].outbuf == NULL) {
            socklist[i].outbuf = (char *) xmalloc(strlen(str) + 1);
            strcpy(socklist[i].outbuf, str);
        }
        else {
            str_ptr = socklist[i].outbuf;
            socklist[i].outbuf = (char *) xmalloc(strlen(str_ptr) +
                                                  strlen(str) + 1);
            strcpy(socklist[i].outbuf, str_ptr);
            strcat(socklist[i].outbuf, str);
            free(str_ptr);
        }
    }
    return;
}



void net_fsend(int nr, char *line1, char *line2, ...)
{
    char str[256];
    va_list line;

    memset(&line, 0x00, sizeof(line));
    va_start(line, line2);
    vsnprintf(str, 256, line2, line);
    va_end(line);

    net_send(nr, "%-32s: %s\n", line1, str);
    return;
}



void net_wall(char *data, ...)
{
    int i;
    char str[1024], *str_ptr;
    va_list line;

    for (i = 0; i < MAXSOCKS; ++i) {
        if (socklist[i].state == CONNECTED &&
            socklist[i].type == MASTER &&
            socklist[i].auth == 1) {

            memset(&line, 0x00, sizeof(line));
            va_start(line, data);
            vsnprintf(str, 1024, data, line);
            va_end(line);

            if (socklist[i].outbuf == NULL) {
                socklist[i].outbuf = (char *) xmalloc(strlen(str) + 1);
                strcpy(socklist[i].outbuf, str);
            }
            else {
                str_ptr = socklist[i].outbuf;
                socklist[i].outbuf = (char *) xmalloc(strlen(str_ptr) +
                                                      strlen(str) + 1);
                strcpy(socklist[i].outbuf, str_ptr);
                strcat(socklist[i].outbuf, str);
                free(str_ptr);
            }
        }
    }
    return;
}



void net_read()
{
    int i, re, len;
    fd_set set;
    char data[512], *str_ptr;
    struct timeval tm;
    struct sockaddr_in th_addr;

    memset(&th_addr, 0x00, sizeof(struct sockaddr_in));
    tm.tv_sec = 1;
    tm.tv_usec = 0;
    FD_ZERO(&set);

    for (i = 0; i < MAXSOCKS; ++i) {
        if (socklist[i].state == CONNECTED ||
            socklist[i].state == LISTENING) {
            FD_SET(socklist[i].socket, &set);
        }
    }

    re = select(FD_SETSIZE, &set, NULL, NULL, &tm);

    if (re > 0) {
        for (i = 0; i < MAXSOCKS; ++i) {
            if ((socklist[i].state == CONNECTED ||
                socklist[i].state == LISTENING) &&
                FD_ISSET(socklist[i].socket, &set)) {

                if (socklist[i].state == LISTENING &&
                    socklist[i].type == MASTER) {
                    net_accept(socklist[i].socket);
                    continue;
                }

                memset(&data, 0x00, sizeof(data));
                re = recvfrom(socklist[i].socket, data, 511, 0,
                              (struct sockaddr *) &th_addr, &len);

                socklist[i].who = th_addr.sin_addr.s_addr;

                if (re < 0 && errno != EAGAIN) {
                    net_reset(i);
                }
                else if (re == 0) {
                    net_reset(i);
                }
                else if (re > 0) {
                    if (socklist[i].inbuf == NULL) {
                        socklist[i].inbuf = (char *) xmalloc(strlen(data) + 1);
                        strcpy(socklist[i].inbuf, data);
                    }
                    else {
                        str_ptr = socklist[i].inbuf;
                        socklist[i].inbuf = (char *) xmalloc(strlen(str_ptr) +
                                                             strlen(data) + 1);
                        strcpy(socklist[i].inbuf, str_ptr);
                        strcat(socklist[i].inbuf, data);
                        free(str_ptr);
                    }
                    time(&socklist[i].time_rd);
                }
            }
        }
    }
    return;
}



void net_write()
{
    int i, wr, tmp1, tmp2, size;
    char *temp;

    for (i = 0; i < MAXSOCKS; ++i) {
        if (socklist[i].state == CONNECTED || 
            socklist[i].state == CLOSE) {
            if (socklist[i].outbuf != NULL) {
                size = strlen(socklist[i].outbuf);
                wr = write(socklist[i].socket, socklist[i].outbuf, size);
                if (wr < 0 && errno != EAGAIN) {
                    net_reset(i);
                }
                else if (wr == strlen(socklist[i].outbuf)) {
                    free(socklist[i].outbuf);
                    socklist[i].outbuf = NULL;
                    if (socklist[i].state == CLOSE) {
                        net_reset(i);
                    }
                }
                else if (wr < strlen(socklist[i].outbuf)) {
                    if (wr == 0) {
                        continue;
                    }
                    tmp1 = 0;
                    tmp2 = wr;
                    temp = socklist[i].outbuf;
                    while ((temp[tmp1] = socklist[i].outbuf[tmp2]) != '\0') {
                        ++tmp1;
                        ++tmp2;
                    }
                    temp[tmp1] = '\0';
                    free(socklist[i].outbuf);
                    socklist[i].outbuf = (char *) xmalloc(strlen(temp) + 1);
                    strcpy(socklist[i].outbuf, temp);
                }
            }
            else if (socklist[i].state == CLOSE) { /* Output buffer empty. */
                net_reset(i);
            }
        }
    }
    return;
}



void net_time()
{
    int i;
    time_t now;

    time(&now);
    for (i = 0; i < MAXSOCKS; ++i) {
        if (socklist[i].state == CONNECTED) {
            if (socklist[i].type == MASTER) {
                if ((now - socklist[i].time_rd) > TIMEOUT) {
                    net_reset(i);
                }
            }
        }
    }
    return;
}



void net_parse()
{
    int i, d1, d2, c;
    char temp[1024];

    for (i = 0; i < MAXSOCKS; ++i) {
        if ((socklist[i].state == CONNECTED && socklist[i].type == MASTER) ||
            (socklist[i].state == LISTENING && socklist[i].type == SLAVE)) {
            if (socklist[i].inbuf != NULL) {
                if (strlen(socklist[i].inbuf) > 1023) {
                    net_send(i, "A input line is to long, emptying buffer.\n");
                    free(socklist[i].inbuf);
                    socklist[i].inbuf = NULL;
                    continue;
                }
                d1 = 0;
                d2 = 0;
                c = 0;
                while (socklist[i].inbuf[d1] != '\0') {
                    if ((socklist[i].inbuf[d1] != '\n') &&
                        (socklist[i].inbuf[d1] != '\r')) {
                        temp[d2] = socklist[i].inbuf[d1];
                    }
                    else if (socklist[i].inbuf[d1] == '\r') {
                        --d2;
                    }
                    else if (socklist[i].inbuf[d1] == '\n') {
                        temp[d2] = '\0';
                        if (socklist[i].type == MASTER) {
                            socklist[i].auth =
                                parse_master(i, socklist[i].auth, temp);
                            if (socklist[i].auth == 0) {
                                net_close(i);
                            }
                        }
                        else if (socklist[i].type == SLAVE) {
                            parse_slave(socklist[i].who, temp);
                        }
                        c = d1;
                        ++c;
                        if (socklist[i].inbuf[c] == '\0') {
                            free(socklist[i].inbuf);
                            socklist[i].inbuf = NULL;
                            return;
                        }
                        d2 = -1;
                    }
                    ++d1;
                    ++d2;
                }
                temp[d2] = '\0';
                free(socklist[i].inbuf);
                socklist[i].inbuf = (char *) xmalloc(strlen(temp) + 1);
                strcpy(socklist[i].inbuf, temp);
            }
        }
    }
    return;
}



int net_setuptcp()
{
    int tcp;

    tcp = net_tcplisten(MASTER_PORT, 5);

    if (tcp < 0) {
        return(-1);
    }

    socklist[0].state = LISTENING;
    socklist[0].type = MASTER;
    socklist[0].socket = tcp;
    time(&socklist[0].time_rd);
    return(1);
}



int net_setupudp()
{
    int udp;

    udp = net_udplisten(PING_PORT);

    if (udp < 0) {
        return(-1);
    }

    socklist[1].state = LISTENING;
    socklist[1].type = SLAVE;
    socklist[1].socket = udp;
    time(&socklist[1].time_rd);
    return(1);
}



void net_writeslave(unsigned long ip, int port, char *data)
{
    int s;
    struct sockaddr_in addr;

    memset(&addr, 0x00, sizeof(struct sockaddr_in));
    s = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = ip;
    addr.sin_port = htons(port);

    sendto(s, data, strlen(data), 0, (struct sockaddr *) &addr, sizeof(addr));
    close(s);
    return;
}
