# this module manages the blacklist of the hosts which are currently blocked
# and also invokes the predefined commands for counter measures.

package SM::Blacklist;
use SM::Conf;
use SM::GV qw(%log_record); # Global Variable
use Data::Dumper;
use warnings;

# the blacklist datastructur,
# containing all ips that are currenctly blocked.
my %bl = ();


# this functions returns the level on which
# the current ip already is or zero if not on
# blacklist
sub is_on_blacklist {
	my ($ip) = @_;
	if(exists($SM::Blacklist::bl{$ip})) {
		if($SM::Blacklist::bl{$ip} eq "LEVEL1") {
			return 1;
		} elsif ($SM::Blacklist::bl{$ip} eq "LEVEL2") {
			return 2;
		}
	}
	return 0;
}

# make the blacklist over the pipe available to the gui.
sub write_blacklist {

	open(PIPE, "> " . $SM::Conf::PIPE_HANDLE_S2G) || 
		SM::Tools::cleanup("Couldn't open pipe");

	my $index=0;

	# each lines contains an index, the ip address and two timestamps
	# when the record was added and when it get removed
	foreach $ip (keys %bl) {
		print(PIPE $index++, 
					" $ip ", 
					($bl{$ip}{remove_at}-$log_record{timestamp}),
					" " , $bl{$ip}{type}, "\n");
	}

	close(PIPE);	

}

# this function is called from time to time to clean
# up the blacklist datastructure.
sub clean_up {
	# get the current time
	my ($now) = @_;
	# go thru each entry in the blacklist
	foreach $ip (keys %bl) {

		# remove the entry if it was 
		# banned long enough
		if($bl{$ip}{remove_at} < $now) {
			
			# remove the entry
			# from the list
			unban($ip, 0);
		}
		
	}
}

# this function is called to add an offender to the blacklist
# and execute the predefined commands for each offender/attacker detected
sub ban {
	# get the arguments, depending from the caller (gui/scrutinizer)
	my ($ip, $time, $type, $from_gui, $rec) = @_;

	# check if predefined commands exist
	if(exists($SM::Conf::ALERT_CMD{BAN}{$type}{CMDS})) {
		# loop thru all ban-commands
		foreach my $cmd_desc (keys %{$SM::Conf::ALERT_CMD{BAN}{$type}{CMDS}})
		{
			# get the current command
			my $cmd = $SM::Conf::ALERT_CMD{BAN}{$type}{CMDS}{$cmd_desc};

			# get the duration in seconds from the argument
			# or if nothing was defined take the default duration
			my $timeout = $time ||
				 $SM::Conf::ALERT_CMD{BAN}{$type}{DEFAULT_TIMEOUT};
			# replace the placeholders from the commands
			# with the current elements from the data structure
			$cmd=~s/\%s/$timeout/g;	# the duration
			$cmd=~s/\%i/$ip/g;		# the ip address
			$cmd=~s/\%t/$log_record{ts}/g;	# the timestamp

			# here the difference between
			# calling by the user interface
			# or the scrutinizer itself
			if($from_gui) {
				$cmd=~s/\%d//g;
			# add additional informations
			# if it was called by the
			# scrutinizer itself
			} else {
				my $debug_output=Data::Dumper->Dump([$rec], [qw(*node)]);
				$cmd=~s/\%d/$debug_output/g;
			}

			# print debug informations
			print "BLOCK: $cmd_desc -> $cmd \n" if ($SM::Conf::DEBUG>1);

			# add the host to the blacklist
			# including all timing informations
			$bl{$ip}{type} = $type;
			$bl{$ip}{added_at}  = $log_record{timestamp};
			$bl{$ip}{remove_at} = $log_record{timestamp}+$timeout;
			$bl{$ip}{timeout}	= $timeout;


			# execute the predefined command but only if not in training mode
			# and DO_ALERT_CMD is turned on
			system($cmd) if (!$SM::Conf::DO_TRAINING && $DO_ALERT_CMD);	

		}

		# send a positive acknowledgment
		# back to the user interface
		`echo ACK > $SM::Conf::PIPE_HANDLE_S2G` if ($from_gui);

	} else {

		# send a negative acknowledgment
		# back to the user interface
		`echo NAK > $SM::Conf::PIPE_HANDLE_S2G` if ($from_gui);

		# print a warning if
		# no command is predefined
		warn "ban: don't know what to do...."
	}
}

# this function is called to
# remove a host from the blacklist
sub unban {

	# get the arguments
	my ($ip, $from_gui) = @_;

	# if such a host is in the blacklist
	if(exists($bl{$ip})) {

		# get the blocking level
		my $type = $bl{$ip}{type};

		# remove the host from the list
		delete($bl{$ip});	

		# check if some unban commands are predefined
		if(exists($SM::Conf::ALERT_CMD{UNBAN}{$type}{CMDS})) {
			# loop thru each command
			foreach my $cmd_desc (keys %{$SM::Conf::ALERT_CMD{UNBAN}{$type}{CMDS}})
			{
				# get the current command
				my $cmd = $SM::Conf::ALERT_CMD{UNBAN}{$type}{CMDS}{$cmd_desc};
	
				# replace the placeholders
				$cmd=~s/\%i/$ip/g;	# the ip address
				$cmd=~s/\%t/$log_record{ts}/g; # the timestamp

				# print some debug informations
				print "UNBLOCK: $cmd_desc -> $cmd \n" if ($SM::Conf::DEBUG>1);

				# execute the predefined command but only if not in training mode
				# and DO_ALERT_CMD is turned on
				system($cmd) if (!$SM::Conf::DO_TRAINING && $DO_ALERT_CMD);	
		
				# send a positive acknownledgment
				# back to the user interface
				`echo ACK > $SM::Conf::PIPE_HANDLE_S2G` if ($from_gui);
			}
		} else {

			# send a negative acknowledgment
			# back to the user interface
			`echo NAK > $SM::Conf::PIPE_HANDLE_S2G` if ($from_gui);

			# print a warning if
			# no command is predefined
			warn "unban: don't know what to do...."
		}

	} else {
		
		# send a negative acknowledgment
		# back to the user interface
		`echo NAK > $SM::Conf::PIPE_HANDLE_S2G` if ($from_gui);

		# print a warning if the
		# host wasn't on the list
		warn "unable to unban $ip, cause it isn't banned.... (?)";
	}

}

1;
