# this module generates the main datastructure which
# will hold all ip records that are currently active
# in the system

package SM::HashTable;

use SM::Defs;
use SM::Conf;
use SM::Statistic;
use strict;
use warnings;

# the base datastructure to hola all the records.
our(@htbl); 

# allocate space for the hashtable
$#htbl=$SM::Conf::HASH_TBL_SIZE; 

# this function removes an entry from the
# datastructure and will link together the
# previous with the next entry
sub del_entry {
	# get the arguments
	my ($nr, $cur) = @_;
	# create variables for
	# temporary storing pre-
	# and successor entries
	my ($prev, $next);

	# If statistics are enables and the current entry was at least for
	# one second active (not only one request transmit) add it to the
	# statistics (and only then).
	if($SM::Conf::DO_STATISTICS && $cur->[VAL]->{time_periodes} > 0) {
		# calculate the moving average of the request rate
		$SM::Statistic::stat{ma}[int($cur->[VAL]->{max_ma})]++;
		# calculate the active period of the session
		$SM::Statistic::stat{periode}[$cur->[VAL]->{time_periodes}]++;
		# if the hit ratio exceed the value 10
		if (int($cur->[VAL]->{max_hit_ratio})>10) { 
			# add the current entry to the hit ratio statistic
			$SM::Statistic::stat{hit_ratio}[int($cur->[VAL]->{max_hit_ratio})]++;
		}

		# chi values of 0 aren't taken into account for doing
		# statistics because every request that makes just in
		# one periode requests, will produce a chi value of 0.
		# if we would count them too, the rating function
		# would produce unwanted results. 
		if($cur->[VAL]->{chi} > 0) {
			# add the entries with enough data to the
			# chi value statistic
			$SM::Statistic::stat{chi}[int($cur->[VAL]->{chi}*100)]++
		}

		# get the number of files of each filetype
	    my $f0=$cur->[VAL]{filetypes}{ DYNAMIC."" } || 0;
        my $f1=$cur->[VAL]{filetypes}{ STATIC."" }  || 0;
        my $f2=$cur->[VAL]{filetypes}{ PICTURE."" } || 0;

		# calculate the ratio between the different filetypes
		 $SM::Statistic::stat{filetype}[int($f0/($f0+$f1+$f2)*100)]++
			# but only if they're
			# reconized correctly
			if($f0 || $f1 || $f2);

	}


	# remember the previous and the next element 
	$prev = $cur->[PREV];
	$next = $cur->[NEXT];
		
	# forget about the current element
	undef $cur;


	# relink the predesessor and the successor	
	$prev->[NEXT] = $next;
	$next->[PREV] = $prev;

	# we got an element less in our LL ..
	$htbl[$nr][VAL]--;

	# and one less in the hole system 
	$SM::GV::nodes_in_system--;

	# return the new current entry
	return $next;
} 

# add an entry to the datastructure
sub add_entry {
	# get the arguments
	my ($nr, $record) = @_;
	# create a new entry
	my ($entry);


	# get the the first element of the list
	$entry = [$htbl[$nr][NEXT], $htbl[$nr], $record];

	# move the old first one position back
	$htbl[$nr][NEXT]->[PREV] = $entry;
	# add the new entry in front of all
	$htbl[$nr][NEXT] = $entry;


	# one node more in the system
	$SM::GV::nodes_in_system++;

	# if statistics are enabled
	if($SM::Conf::DO_STATISTICS > 0) {
		# increment the counter of the number of nodes in system
		$SM::Statistic::stat{nbr_nodes}[$SM::GV::nodes_in_system]++;
	}

	# one more in the LL
	return ++$htbl[$nr][VAL];
}

# put an entry to the pole position
sub move_entry {
	# get the arguments
	my ($nr, $cur) = @_;
	# create placeholders for the entries
	my ($prev, $next, $entry);

	# remember the previous and the next element 
	$prev = $cur->[PREV];
	$next = $cur->[NEXT];
		
	# relink the predesessor and the successor	
	$prev->[NEXT] = $next;
	$next->[PREV] = $prev;

	# get the first element of the list
	$entry = [$htbl[$nr][NEXT], $htbl[$nr], $cur->[VAL]];

	# move the old first one position back
	$htbl[$nr][NEXT]->[PREV] = $entry;
	# add the new entry in front of all
	$htbl[$nr][NEXT] = $entry;

}

1;
