# this module defines the different graphs which get
# created by gnuplot. if the training mode is used,
# gnuplot tries to find a function (exponential)
# which fits the messured data 

package SM::Plot;

use SM::Conf;
use SM::PP;
use SM::Statistic;
use SM::Parameters;
use Data::Dumper;
use File::Copy;
use POSIX qw(strftime);
use strict;
use warnings;

# define the header information for all plots.
# if you want for example add general informations
# or change the image size, you can do it here
my $header=<<EOT;
set size 1.0,1.0
set terminal png small color
EOT

# the run function will generate all the statistics. 
# if you ran the scrutinizer in non training mode
# and you see that the data didn't fit to the 
# current rating function, you can configure 
# in SM/Conf.pm the $DO_TRAINING=1 mode and
# just run 
# $ perl -e 'use SM::Plot; SM::Plot::run()'
# and the system gets trained with the already
# created statistic files. don't forget to switch
# off the training mode. 
 
sub run {

	# call the different plotting functions
	request_ratio();
	periodes();
	alert_level();
	page_code(); 
	filetype();
	time_distribution();
	uri_spreading();
	nodes_in_system(); 

	if($SM::Conf::DO_TRAINING) {

		# first to a backup of the current Parameters.pm file
		File::Copy::copy("SM/Parameters.pm",
			 "SM/Parameters.pm." . strftime("%b_%e_%H:%M:%S_%Y", localtime)) ||
				warn "couldn't create backup file of Parametrs.pm: $!";
	

		# write the new Parameters.pm file if we are in trainings mode

		open(FH, ">SM/Parameters.pm") || cleanup("can't write parameter file $!");
		print FH "package SM::Parameters;\n\n";
		print FH Data::Dumper->Dump(
					[\%SM::Parameters::param_alert],
					[qw(*param_alert)]
				 );
		print FH "\n1;\n";
		close(FH);

	}

	# set the training mode to 0 that the functions
	# doesn't get fitted again if if we are in
	# training mode .... (yes i know, not the nicest solution)
	$SM::Conf::DO_TRAINING=0;

	# call the same plotting functions again, but this
	# time with the xrange limited, that the important
	# part of the graph is surely be visible
	request_ratio({ xrange=>"0:40", output=>"stat_requests_xrange_limit.png" });
	periodes({ xrange=>"0:100", output=>"stat_periode_xrange_limit.png" });
	alert_level({ xrange=>"0:100", output=>"stat_alert_xrange_limit.png" });
	uri_spreading({ xrange=>"0:100", output=>"stat_uri_spreading_xrange_limit.png" });

}

# graph on how many requests are done per second
sub request_ratio {

	# check if arguments are given
	my $add = shift() || {};

	# define the properties of this graphic
	my %prop = (
	title 	=> "Requets Per Second Rating Function",
	xlabel 	=> "requets per second",
	ylabel 	=> "probability",
	xrange	=> "0:*",
	output 	=> "stat_requests.png",
	logfile	=> "stat_ma.log",
	plot	=> 	[
					['1:3', 'probability', 'w p'],
					['1:4', 'cdf', 'w p'],
				],
	fx		=>	"f(x) w l",
	other	=> 	["set grid"],
	);

	# add the additional arguments to
	# the properties
	foreach my $key (keys %{$add}) {
		$prop{$key} = $add->{$key}; 
	}


	# in training mode add the function
	# which should get fitted
	if($SM::Conf::DO_TRAINING) {
		$prop{func}		=	"f(x)=1-a*(x-b)**c";
		$prop{vars}		=	['a','b','c'];
		$prop{fitrow}	=	"1:4";
		$prop{varfile}	=	"var.log";

		# plot the graph ...
		plot(\%prop);

		# ... and read the variables back from gnuplot
		read_vars(\%prop, \%{$SM::Parameters::param_alert{request_ratio}});

	} else {

		# add the function which was already fitted
		push(@{$prop{other}},
				"f(x)=1-" .
				$SM::Parameters::param_alert{request_ratio}{a} . "*(x-" .
				$SM::Parameters::param_alert{request_ratio}{b} . ")**" .
				$SM::Parameters::param_alert{request_ratio}{c}
		); 
		plot(\%prop);

	}
}

# graph of the active periodes
sub periodes {

	my $add = shift() || {};
	
	my %prop = (
	title 	=> "Active Periode Of A Client",
	xrange 	=> "0:*",
	xlabel 	=> "active time",
	yrange 	=> "*:*",
	ylabel 	=> "probability",
	output 	=> "stat_periode.png",
	logfile	=> "stat_periode.log",
	plot	=> 	[
					['1:3', 'probability', 'w l'],
					['1:4', 'alert', 'w p'],
				],
	fx		=>	"f(x) w l",
	other	=> 	["set grid"],
	);

	foreach my $key (keys %{$add}) {
		$prop{$key} = $add->{$key}; 
	}

	SM::PP::pretty_print(\%prop);

	if($SM::Conf::DO_TRAINING) {
		$prop{func}		=	"f(x)=1-a*(x-b)**c";
		$prop{vars}		=	['a','b','c'];
		$prop{fitrow}	=	"1:4";
		$prop{varfile}	=	"var.log";

		plot(\%prop);
		read_vars(\%prop, \%{$SM::Parameters::param_alert{periodes}});

	} else {
		push(@{$prop{other}},
				"f(x)=1-" .
				$SM::Parameters::param_alert{periodes}{a} . "*(x-" .
				$SM::Parameters::param_alert{periodes}{b} . ")**" .
				$SM::Parameters::param_alert{periodes}{c}
		); 
		plot(\%prop);

	}

}

# the time spread of the different requests of a client
sub time_distribution {
	my %prop = (
	title 	=> "Time Distribution Of Requests using Chi^2 value",
	xrange 	=> "0:25",
	xlabel 	=> "chi^2 normed value",
	yrange 	=> "*:*",
	ylabel 	=> "alert value",
	output 	=> "stat_time_dist.png",
	logfile	=> "stat_chi.log",
	plot	=> 	[
					['($1/100):3 axes x1y2', 'probability', 'w p pt 1 ps 0.5'],
					['($1/100):(1-$4)', '1-cdf', 'w p ps 0.4'],
				],
	fx		=>	"f(x) w l lw 1.5",
	other	=> [
			"set ytics nomirror",
			"set grid",
			"set y2label \"probatility\"",
			"set y2range [0:0.01]",
			"set y2tics 0, 0.001"],
	);

	if($SM::Conf::DO_TRAINING) {
		$prop{func}		=	"f(x)=d*(e**x)";
		$prop{vars}		=	['d','e'];
		$prop{fitrow}	=	'($1/100):(1-$4)';
		$prop{varfile}	=	"var.log";

		plot(\%prop);
		read_vars(\%prop, \%{$SM::Parameters::param_alert{time_dist}});

	} else {
		push(@{$prop{other}},
				"f(x)=" .
				$SM::Parameters::param_alert{time_dist}{d} . "*(" .
				$SM::Parameters::param_alert{time_dist}{e} . "**x)"
		);
		plot(\%prop);

	}
}

# graph on how much the same resource gets requested
sub uri_spreading {
	
	my $add = shift() || {};

	my %prop = (
	title 	=> "Spreading Of The URI Requests",
	yrange 	=> "0:1",
	xlabel 	=> "hit_ratio (how many times hit to the same resource)",
	ylabel 	=> "probability / alert value for hit_ratio",
	output 	=> "stat_uri_spreading.png",
	logfile	=> "stat_hit_ratio.log",
	plot	=> 	[
					['($1-10):3', 'probability', 'w p'],
					['($1-10):4', 'alert', 'w p'],
				],
	fx		=>	"f(x) w l",
	other	=> 	["set grid"],
	);

	foreach my $key (keys %{$add}) {
		$prop{$key} = $add->{$key}; 
	}

	if($SM::Conf::DO_TRAINING) {
		$prop{func}		=	"f(x)=1-a*(x-b)**c";
		$prop{vars}		=	['a','b','c'];
		$prop{fitrow}	=	'($1-10):4';
		$prop{varfile}	=	"var.log";

		plot(\%prop);
		read_vars(\%prop, \%{$SM::Parameters::param_alert{uri_spreading}});

	} else {
		push(@{$prop{other}},
				"f(x)=1-" .
				$SM::Parameters::param_alert{uri_spreading}{a} . "*((x)-" .
				$SM::Parameters::param_alert{uri_spreading}{b} . ")**" .
				$SM::Parameters::param_alert{uri_spreading}{c}
		); 
		plot(\%prop);

	}


}


# the alert level distribution
sub alert_level {

	my $add = shift() || {};

	my %prop = (
	title 	=> "Alert Level Distribution",
	xrange 	=> "0:*",
	xlabel 	=> "alert level %",
	ylabel 	=> "cumulated probability",
	output 	=> "stat_alert.png",
	logfile	=> "stat_alert_dist.log",
	plot	=> 	[
					['1:3 axes x1y2', 'alert level probability', 'w p'],
					['1:4', 'alert level cdf', 'w l'],
				],
	other	=> [
			"set ytics nomirror",
			"set grid",
			"set y2label \"probatility\"",
			"set y2range [0:0.1]",
			"set y2tics 0, 0.01"],

	);

	foreach my $key (keys %{$add}) {
		$prop{$key} = $add->{$key}; 
	}

	plot(\%prop);
}

# graph of the apache return codes
sub page_code {
	my %prop = (
	title 	=> "Apache Return Code Distribution per ip-session",
	xrange 	=> "*:100",
	xlabel 	=> "rate of (2xx+4xx)/(2xx+3xx+4xx)",
	ylabel 	=> "probability",
	output 	=> "stat_code.png",
	logfile	=> "stat_alert_code.log",
	plot	=> 	[
					['1:3', 'probability', 'w l'],
					['1:4', 'cdf', 'w l'],
				],
	other	=>	["set grid"],
	);

	plot(\%prop);
}

# graph of the different filetypes that get requested
sub filetype {
	my %prop = (
	title 	=> "What Resources Get Requested?",
	xrange 	=> "0:100",
	yrange 	=> "0:1",
	xlabel 	=> "rate of (DYNAMIC)/(DYNAMIC+STATIC+PICTURES) in %",
	ylabel 	=> "probability",
	output 	=> "stat_filetypes.png",
	logfile	=> "stat_filetype.log",
	plot	=> 	[
					['1:3', 'probability', 'w l'],
					['1:4', 'cdf', 'w l'],
				],
	other	=>	["set grid"],
	);

	plot(\%prop);
}

# graph of how many nodes are in our system at at time
sub nodes_in_system {
	my %prop = (
	title 	=> "Concurrent Nodes In System",
	xrange 	=> "0:200",
	yrange 	=> "0:*",
	xlabel 	=> "nodes in system",
	ylabel 	=> "probability",
	output 	=> "stat_nodes.png",
	logfile	=> "stat_nbr_nodes.log",
	fx		=>	"f(x) w l",
	plot	=> 	[
					['1:3', 'probability', 'w p'],
					['1:4', 'cdf', 'w l'],
				],
	other	=>	["set grid"],
	);

	if($SM::Conf::DO_TRAINING) {
		$prop{func}			=	"f(x)=1-g**x";
		$prop{vars}		=	['g'];
		$prop{fitrow}	=	'1:4';
		$prop{varfile}	=	"var.log";

		plot(\%prop);
		read_vars(\%prop, \%{$SM::Parameters::param_alert{nodes_in_system}});

	} else {
		push(@{$prop{other}},
				"f(x)=1-" .
				$SM::Parameters::param_alert{nodes_in_system}{g} . "**x"
		); 
		plot(\%prop);

	}
	plot(\%prop);
}


# plot a graphic based on given properties 
sub plot {

	my ($prop) = @_;
	my ($cmd, $fit_plot) = ("", 0);

	SM::PP::pretty_print($prop);

	$cmd = $header;
	$cmd.= "set title  \"" . $prop->{title} . "\"\n";
	$cmd.= "set xrange [" . $prop->{xrange} . "]\n" if (exists($prop->{xrange}));
	$cmd.= "set yrange [" . $prop->{yrange} . "]\n" if (exists($prop->{yrange}));
	$cmd.= "set xlabel \"" . $prop->{xlabel} . "\"\n" if (exists($prop->{xlabel}));
	$cmd.= "set ylabel \"" . $prop->{ylabel} . "\"\n" if (exists($prop->{ylabel}));
	$cmd.= "set output \"" . $prop->{output} . "\"\n";

	if(exists($prop->{other})) {
		foreach my $opt (@{$prop->{other}}) {
			$cmd.=$opt . "\n";
		}
	}

	# do we have to fit a function?
	if(exists($prop->{func})) { #
		$cmd.=$prop->{func}."\n";
		$cmd.="fit f(x) \"" . $SM::Conf::LOG_DIR.$prop->{logfile} .
			  "\" using " . $prop->{fitrow} . " ";
		$cmd.="via " . join(",",@{$prop->{vars}}) . "\n";
		if(exists($prop->{vars})) {
			$cmd.="save var '". $prop->{varfile}. "'\n"
		}
		$fit_plot=1;
	}	

	if(exists($prop->{plot})) {
		$cmd.="plot ";
		foreach my $line (@{$prop->{plot}}) {
			$cmd.="\t\"" . $SM::Conf::LOG_DIR.$prop->{logfile} .
				  "\" using " . $line->[0] . " ";
			if(exists($line->[1])) {
				$cmd.="title \"" . $line->[1] . "\" ";
			}
			if(exists($line->[2])) {
				$cmd.=$line->[2];
			}
			$cmd.=",\\\n";
		}

		if(exists($prop->{fx})) {
			$cmd.="\t" . $prop->{fx} . "\n";
		} else {
			$cmd=~s/..$//;
		} 
	}
	$cmd.="\n";

	print STDERR $cmd;

	# open a pipe to gnuplot and tell it what do 
	open(PROG, "| $SM::Conf::CMD{gnuplot}") || cleanup("$!");
	print PROG $cmd;
	close(PROG);

}

# read the values from the varfile and store them in $ref
sub read_vars {

	my ($prop,$ref) = @_;
	
	# open the var file
	open(FH, $prop->{varfile}) || cleanup("could not open $prop->{varfile}"); 

	# read all lines
	while(<FH>) {
		# skip the ones that are commented out
		next if(m/^#/);
		# use the var name ($1) and its value ($2)
		if(m/^([^ ]) = (.*)$/) {
			# add it to the hash
			$ref->{$1}=$2;
		}
	}
	close(FH);
}

1;
