# this module contains miscellaneous functions used by different
# parts of the system.

package SM::Tools;
use SM::GV qw(%log_record);
use SM::Conf;
use SM::Defs;
use SM::Statistic;
use SM::HashTable;
use Time::Local;
use Time::HiRes qw(gettimeofday);
use warnings;

# convert an ip address to a number.
# this number is used as key in our hashtable.
sub ip_to_nr {
	# get the arguments
	my($host) = @_;
	# return zero if a character
	# is found in the address
	return 0 if($host=~m/[a-zA-Z]/);

	# create an array of the octets
	my @octets = split(/\./, $host);
	my $ip_number = 0;
	# for each octet
	foreach my $octet (@octets) {
		# shift the number 8 bits
		# to the right
		$ip_number <<= 8;
		# xor the number with the
		# current octet
		$ip_number |= $octet;
	}
	# return the generated number
	return $ip_number;
}

# this function is called when a SIGUSR1 is recieved. the pipe
# to the gui gets opened and the command, what do to, is read.
sub handle_signal {
	# open the command pipe
	open(PIPE, $SM::Conf::PIPE_HANDLE_G2S) || cleanup("Couldn't open pipe");
	# get the command
	my $cmd=<PIPE>;
	# close the pipe
	close(PIPE);

	# truncate the command
	chomp($cmd);

	# stat -> send statistical data
	# to the user interface
	if($cmd eq "stat") {
		SM::Tools::write_gui_stat();
	# bl -> send blacklist data
	# to the user interface
	} elsif ($cmd eq "bl") {
		SM::Blacklist::write_blacklist();
	# ip -> send ip details
	# to the user interface
	} elsif ($cmd=~m/^ip (.*)/) {
		SM::Tools::write_gui_stat($1);
	# ban -> put this ip on the blacklist
	# ip, duration and ban level
	# is sent by the user interface
	} elsif ($cmd=~m/^ban ([^ ]+) ([^ ]+) ([^ ]+)/) {
		SM::Blacklist::ban($1,$2,$3,1);
	# unban -> remove this ip from the blacklist
	# ip is sent by the user interface
	} elsif ($cmd=~m/^unban ([^ ]+)/) {
		SM::Blacklist::unban($1,1);
	# this command is unknown
	} else {
		print STDERR "unknown command read from pipe! -> $cmd\n";
	}
}

# this function prints the statistical values
# into the pipe to the user interface
sub write_gui_stat {
	# get the arguments
	my ($ip) = @_;

	# open the pipe
	open(PIPE, ">", $SM::Conf::PIPE_HANDLE_S2G) || cleanup("Couldn't open pipe");

	# for each entry of the hast table
	for(my $i=0,my $index=1;$i<$SM::Conf::HASH_TBL_SIZE;$i++) {

		# skip empty ones
		next if($SM::HashTable::htbl[$i][VAL] == 0);

		# get the current entry
		my $cur=\@{$SM::HashTable::htbl[$i][NEXT]};

		# for each element of the linked list
		for(my $j=0;$j<$SM::HashTable::htbl[$i][VAL];$j++) {

			# if a value is contained by this element
			if(defined($cur->[VAL]{last_ts}) &&
				# and the value is not outdated
				 $log_record{timestamp}-$cur->[VAL]{last_ts}<$SM::Conf::TIMEOUT_URI)
			{

				# if an ip address is provided
				# by the caller
				if(defined($ip)) {
					# only print the ip detail
					# informations
					if($ip eq $cur->[VAL]->{ip}) {
						# loop thru all uri records
						foreach my $uri (keys(%{$cur->[VAL]{e}})) {
							# print the uri informations
							# into the pipe
							print PIPE "$uri " . 
								$cur->[VAL]{e}{$uri}{hits} . " " .
								scalar(keys %{$cur->[VAL]{e}{$uri}{param}}) . " " .
								$cur->[VAL]{e}{$uri}{last_ts} . "\n";
						}
					}
				} else {
					# print only the statistical
					# information about all current
					# clients contained in the
					# systems data structure
					# into the pipe
					print PIPE $index++ . " " .	
						$cur->[VAL]{ip} . " " .
						$cur->[VAL]{alert} . " " .  
						$cur->[VAL]{req_st} . " " . 
						$cur->[VAL]{req_lt} . " " . 
						$cur->[VAL]{time_periodes} . " " . 
						$cur->[VAL]{count_sum} . " " .  
						$cur->[VAL]{last_ts} . "\n";

				}
			}
			# get the next entry
			$cur=$cur->[NEXT];
		}
	}
	# close the pipe
	close(PIPE);
}

# when an input line couldn't be parsed, this function
# gets called which prints out an error containing the
# attribut that didn't match.
sub pattern_match_error {
	my ($what, $line) = @_;
	print STDERR "\nERROR couldn't parse logfile ";
	print STDERR "$what wasn't found in:\n$line\n";

	return -1;
}

# parse the logfile. currently the combined format is matched
# but you can just comment out the last two matches and add 
# $SM::GV::log_record{browser}=$SM::GV::log_record{referer}="";
# and you will have the common format
sub parse_line {
	$_ = shift(); # get line

	# start benchmakr 'parse'
	$SM::GV::perf{parse}{start} = gettimeofday if ($SM::Conf::BENCHMARK);

	# remove quotings
	# (dirty hack if you have unclean logfiles)
	# $_=~s/\\"//g;

	# match the ip address	
	if(m/^(\S+) /g)	{ 
		$SM::GV::log_record{ip} = $1;
	 } else { return pattern_match_error('ip',$_); }

	# match the logname
	if(m/\G(\S+) /g)		{
		$SM::GV::log_record{logname} = $1;
	 } else { return pattern_match_error('logname', $_); }

	# mtach the user
	if(m/\G(\S+) /g)		{ 
		my $user=$1;
		chop($user);
		$SM::GV::log_record{user} = $user; 
	} else { return pattern_match_error('user',$_); }

	# match and split up the timestamp
	if(m/\G\[(\d{1,2})\/(\w{3})\/(\d{4}):(\d{2}):(\d{2}):(\d{2}).{0,6}\]\s/g) {
		$SM::GV::log_record{timestamp} =
			timelocal($6, $5, $4, $1, $SM::GV::month{$2}, $3);

		$SM::GV::log_record{ts}="[" . ($SM::GV::month{$2}+1) . "/$1 $4:$5:$6]";
	} else { return pattern_match_error('time',$_); }

	# parse the request
	if(m/\G\"\s*([^ "]+)\s*([^? "]*)([^ "]*)[^"]*\"/g)	{ 
		$SM::GV::log_record{method} = $1;
		$SM::GV::log_record{request} = $2;
		$SM::GV::log_record{param} = $3;
	} else { return pattern_match_error('request',$_); }

	# match the return code by apache
	if(m/\G\s(\S+) /g)	{ 
		$SM::GV::log_record{code} = $1; 
	} else { 
		return pattern_match_error('code',$_);
	 }

	# the transfered bytes
	if(m/\G(\S+)/g)		{ 
		$SM::GV::log_record{bytes} = $1; 
	} else { return pattern_match_error('bytes',$_); }


	if($SM::Conf::LOG_FORMAT==0) {
		$SM::GV::log_record{referer}=$SM::GV::log_record{browser}="";
		return 0;
	} # else -> log format is combined...


	# the referer
	if(m/\G \"(.*)\"\s\"/g) { 
		$SM::GV::log_record{referer} = $1; 
	} else { return pattern_match_error('referer',$_); }

	# the browser that sent the request
	if(m/\G(.*)\"\s/)	   { 
		$SM::GV::log_record{browser} = $1; 
	} else { return pattern_match_error('browser',$_); }

	# stop benchmark 'parse'
	$SM::GV::perf{parse}{sum}+=gettimeofday-$SM::GV::perf{parse}{start}
		if ($SM::Conf::BENCHMARK);	

	return 0; # OK
}


# write the pid file
sub write_pid_file {
	# if fork succeeds
	if((my $pid=fork())==0) {
		# get the parent pid
		# of the forked child
		# (pid of scrutinizer)
		my $ppid=getppid();
		# create the pid file
		if(!open(FH, ">", $SM::Conf::PID_FILE)) {
			warn "couldn't write pid file: $!\n";
			# kill the parent process
			kill 9, $ppid;
			# kill myself
			cleanup("PID_FILE: $!");
		} 
		# save the pid of
		# scrutinizer into
		# the pid file
		print FH "$ppid\n";
		# close the pid file
		close(FH);
		# exit the function
		exit(0);
	} elsif ($pid == -1) {
		# print an error and
		# terminate if fork fails
		cleanup("fork failed: $!");
	}
}

# remove the pid file
sub remove_pid_file {
	# delete the pid file
	unlink $SM::Conf::PID_FILE || die "where is the pid file?";
	exit(0);

}

# print out the benchmark results
sub benchmark_results {
	# create local variable for
	# storing the result time
	my($total)=(0);
	# loop thru all benchmark results
	foreach my $key (keys %SM::GV::perf) {
		# print each benchmark result to stderr
   		print STDERR "$key:\t took " . $SM::GV::perf{$key}{sum} . " secs\n";
		# increment the total time
		$total+=$SM::GV::perf{$key}{sum};
	}
	# print out the total time
	print STDERR "Total:\n$total\n";
}

# simple clean up function
sub cleanup {
	remove_pid_file();
	die shift();	
}
1;
