/*	blacklist.c
	rothm4 && ryter

	helper program for mod_scrutinizer. blacklist.c maintains the linked list which contais
	the bad ip adresses of potential attackers. mod_scrutinizer asks blacklist.c via ipc
	messages if the requesting ip is a malicious one. scrutinizer.pl sends ips via named
	pipe to the blacklist.c.

	26.10.04	-	created
	27.10.04	-	added the function to send back ack or nak
	28.10.04	-	delete the ips after some seconds
	29.10.04	-	added cleanup & pid-file for fork-check
	02.11.04	-	need some command line args (from apache)
	03.11.04	-	added time argument for blocking
*/

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <error.h>
#include <time.h>
#include <signal.h>

#define PIPE_FILE "/tmp/bad_ip"	//cmd line arg!
#define BUF_SIZE 1024

#define MSG_SIZE	20
#define MSG_KEY_SEND  2810
#define MSG_KEY_RECV  1028

#define PID_PATH "/var/run"		//cmd line arg!

#define DEBUG 0

// define elements of the dll
struct node {
        char ip[MSG_SIZE];	// the address
		time_t since;		// timestamp of entry
		int    duration;	// how long to block?
        struct node *next;	// next elem
        struct node *prev;	// prev elem
};

typedef struct node ip_rec;

// messagebuffer for ipc messages
struct msgbuf {
	long    mtype;			// type of message
	char    mtext[MSG_SIZE];// the message
} sndbuf, rcvbuf;

// define the functions
int add_ip(char *new_ip, int dur);
int del_ip(char *del_ip);
int chk_ip(char *ip);
int chk_old();
int print_list();
void cleanup(int sig);

// head points to the first list element
ip_rec *head;

// the pipe file descriptor
int pipe_fd;
// variables used for ipc
int msgid_send, msgid_recv;
// the pid-file descriptor
int pid_fd;
char *pid_path;
char *pipe_file;

int main(int argc, char *argv[])
// argv[0] = app.name, argv[1] = path2pid, argv[2] = path+fifo_file
{
	if (DEBUG) printf("args: '%s', '%s', '%s'.\n", argv[0], argv[1], argv[2]);
	// multi purpose variables
	int res, bytes_read = 0;

	// variables used for pipe
	char pipe_buf[BUF_SIZE];
	int open_mode = 0;

	// variables used for strtok
	char *token, *next, *token2, *next2, *tmp_token;

	// variable for loop-control
	int loops = 0;
	int do_output = 0;

	// save the pid
	pid_t pid;
	char s_pid[10];
	int pid_flags = 0;

	// register the function cleanup (signal handling)
	(void) signal(SIGTERM, cleanup);
	(void) signal(SIGINT, cleanup);
	(void) signal(SIGKILL, cleanup);

	// save the pid into the file /var/run/argv[0].pid
	pid = getpid();
	// prepare the flags for file i/o
	pid_flags = O_WRONLY | O_CREAT;

	// prepare the filename/path
	pid_path = (char *)malloc(strlen(argv[1]) + strlen(argv[0]) + 6);

	// copy the parts
	// check for cmdline argument
	if (argv[1] == NULL) {
		strcat(pid_path, PID_PATH);
	} else {
		strcat(pid_path, argv[1]);
	}
	strcat(pid_path, "/");
	strcat(pid_path, argv[0]);
	strcat(pid_path, ".pid");

	// open the file for saving the pid
	if (DEBUG) printf("open/create file '%s' for saving pid '%d'\n", pid_path, (int)pid);
	pid_fd = open(pid_path, pid_flags);
	if (pid_fd == -1) {
		fprintf(stderr, "Could not create file '%s'.\n", pid_path);
		exit(EXIT_FAILURE);
	}

	// write the pid into the file
	sprintf(s_pid, "%d\n", (int)pid);
	res = write(pid_fd, s_pid, strlen(s_pid));
	if (res == -1) {
		fprintf(stderr, "Could not write into pid-file.\n");
		exit(EXIT_FAILURE);
	}
	if (DEBUG) printf("wrote '%s' (%d bytes) into '%s'.\n", s_pid, res, pid_path);
	close(pid_fd);

	// prepare the fifo filename/path
	pipe_file = (char *)malloc(strlen(argv[2]) + 1);

	// copy the parts
	// check for cmdline argument
	if (argv[2] == NULL) {
		pipe_file = PIPE_FILE;
	} else {
		pipe_file = strdup(argv[2]);
	}

	// check if FIFO-file is existent
	// if not, create a new one.
	if (access(pipe_file, F_OK) == -1) {
		res = mkfifo(pipe_file, 0777);
		if (res != 0) {
			fprintf(stderr, "Could not create fifo %s\n", pipe_file);
			exit(EXIT_FAILURE);
		}
	}

	// prepare the list
	if (DEBUG) printf("size of one element: %d\n", sizeof(ip_rec));
	head = (ip_rec *)malloc(sizeof(ip_rec));
	if (head == NULL) {
		fprintf(stderr, "Could not allocate memory for header\n");
		exit(EXIT_FAILURE);
	} else {
		head->next = (ip_rec *) head;
		head->prev = (ip_rec *) head;
	}

	// open the input pipe (non-blocking read)
	open_mode = O_RDONLY | O_NONBLOCK;
	if (DEBUG) printf("Process %d opening FIFO (%s)\n", getpid(), pipe_file);
	pipe_fd = open(pipe_file, open_mode);
	if (pipe_fd == -1) {
		fprintf(stderr, "Could not open FIFO (%s)\n", pipe_file);
		exit(EXIT_FAILURE);
	} else {
		// print the result (of opening -> not reading)
		if (DEBUG) printf("Process %d result %d\n", getpid(), pipe_fd);
	}
	
	// create the message queues
	msgid_recv = msgget((key_t)MSG_KEY_RECV, 0666 | IPC_CREAT);
	msgid_send = msgget((key_t)MSG_KEY_SEND, 0666 | IPC_CREAT);
	if (msgid_recv == -1) {
		fprintf(stderr, "Could not create Recieve Message Queue\n");
		exit(EXIT_FAILURE);
	} else if (msgid_send == -1) {
		fprintf(stderr, "Could not create Send Message Queue\n");
		exit(EXIT_FAILURE);
	} else {
		if (DEBUG) printf("Message Queues created: msgid_recv = %d, msgid_send = %d\n",
							 msgid_recv, msgid_send);
	}

	// read the pipe and add/remove the addresses 
	while (1) {
		// increment the loop counter
		loops++;
		// all hundred loops, check for old entries (~every second)
		if (loops > 1000) {
			res = chk_old();
			if (DEBUG) printf("cleared %d entries, because they stayed long enough in list.\n", res);
			// reset loop counter
			loops = 0;
			// enable output
			do_output = 1;
		} else {
			// disable output
			do_output = 0;
		}
		// read from the pipe
		bytes_read = read(pipe_fd, pipe_buf, BUF_SIZE);
		// terminate the string in the buffer
		pipe_buf[bytes_read] = '\0';

		// the ip addresses must each be terminated 
		// with a '\n'
		if (bytes_read > 0) {
			if (DEBUG) printf("read %d bytes: '%s'\n", bytes_read, pipe_buf);
			// if multiple lines were in the file
			// tokenize them into single ip adresses ('\n')
			next = pipe_buf;
			while ((token = strtok(next, "\n")) != NULL) {
				if (DEBUG) printf("found token: %s\n", token);
				// get address and duration
				// separated by ' '
				next2 = token;
				while ((token2 = strtok(next2, " ")) != NULL) {
					if (DEBUG) printf("found token '%s'.\n", token2);
					tmp_token = token2;
					next2 = NULL;
				}
				if (chk_ip(token)) {
					if (del_ip(token) == 1) {
						if (DEBUG) printf("token %s deleted from list!\n", token);
					} else {
						if (DEBUG) printf("NOT DELTED!!\n");
					}
				} else {
					if (DEBUG) printf("try to add '%s' with '%s'.\n", token, tmp_token);
					if (add_ip(token, atoi(tmp_token)) == 1) {
						 if (DEBUG) printf("token %s added to list\n", token);
					} else {
						if (DEBUG) printf("NOT ADDED!!\n");
					}
				}
				next = NULL;
			}
		} else {
			if (do_output && DEBUG) printf("nothing to read in pipe :(\n");
			usleep(1);
		}

		// read from the message queue
		res = msgrcv(msgid_recv, (void*)&rcvbuf, MSG_SIZE, 0, IPC_NOWAIT);
		if (res == -1) {
			if (do_output && DEBUG) printf("nothing to read in Message Queue :(\n");
		} else {
			if (DEBUG) printf("Read %d bytes of Message: '%s'\n", res, rcvbuf.mtext);
			// check the list
			if (chk_ip(rcvbuf.mtext)) {
				if (DEBUG) printf("yep this ip is in the list!\n");
				strcpy(sndbuf.mtext, "ACK");
				sndbuf.mtype = 1;
				res = msgsnd(msgid_send, (void *)&sndbuf, MSG_SIZE, 0);
				if (res == -1) {
					perror("msgsnd error ");
					fprintf(stderr, "Could not send '%s' Message\n", sndbuf.mtext);
					exit(EXIT_FAILURE);
				}
				if (DEBUG) printf("sent %d bytes successfully.\n", res);
			} else {
				if (DEBUG) printf("nope, this ip is not in the list.\n");
				strcpy(sndbuf.mtext, "NAK");
				sndbuf.mtype = 1;
				res = msgsnd(msgid_send, (void *)&sndbuf, MSG_SIZE, 0);
				if (res == -1) {
					perror("msgsnd error ");
					fprintf(stderr, "Could not send '%s' Message\n", sndbuf.mtext);
					exit(EXIT_FAILURE);
				}
				if (DEBUG) printf("send %d bytes successfully.\n", res);
			}
		}
		if (do_output && DEBUG) print_list();
	}

	// close the pipe if it was successfully openend
	if (pipe_fd != -1) (void)close(res);
	if (DEBUG) printf("Process %d finished\n", getpid());
	exit(EXIT_SUCCESS);
}

int add_ip(char *new_ip, int dur)
	// add specified ip to the list
	// return 1 if ip is added successfully
{
	time_t t;

        // get the last element
        ip_rec *tmp;
	tmp = head->prev;

        // allocate memory for new element
	ip_rec *new = malloc(sizeof(ip_rec));
	if (new == NULL) {
		fprintf(stderr, "Could not allocate Memory for new Node\n");
		return -1;
	}

        // copy ip adress into it
        strcpy(new->ip, new_ip);

	// store a timestamp
	(void) time(&t);
	new->since = t;

	// set the duration limit
	new->duration = dur;

        // link it into the list
        head->prev = (ip_rec *) new;
        tmp->next  = (ip_rec *) new;
        new->next  = (ip_rec *) head;
        new->prev  = (ip_rec *) tmp;
        return 1;
}

int del_ip(char *del_ip)
	// delete specified ip from list
	// return 1 if ip is deleted
	// return -1 if ip is not in list
{
        ip_rec *i = head->next;
        ip_rec *found = NULL;

        // cycle thru the list until element is found
        // or the head is reached
        while (i != head || found == NULL) {
                // check if ip is in the list
		if (DEBUG) printf("-> comparing %s with %s\n", i->ip, del_ip);
                if (strcmp(i->ip, del_ip) == 0) {
                        found = i;
                        break;
		}
                // if not, go the next list element
                i = i->next;
        }

        if (found != NULL) {
                // unlink the element
                i = found->prev;
                i->next = found->next;
                i = found->next;
                i->prev = found->prev;
		// free the memory
		free(found);
                return 1;
        }
        return -1;
}

int chk_ip(char *ip)
	// check if the specified ip is in the list
	// return 1 if yes, 0 if no
{
        // get the first element in list
        ip_rec *i = head->next;

        // for all ip's in ll: check if actual req_ip is in ll,
        while(i != head) {
                if(!strcmp(i->ip, ip))
                        return 1;
                i = i->next;
        }
        return 0;
}

int chk_old()
	// check all members of the list
	// if on of them is older than
	// 200 seconds, remove it.
	// returns the number of removed elements
{
	time_t t;
	ip_rec *i = head->next;
	ip_rec *next_node;
	int removed = 0;

	// get the current time
	(void) time(&t);

	// for all members
	while(i != head) {
		// store next node at beginning
		// i->next can be destroyed by del_ip
		next_node = i->next;
		if (DEBUG) printf("entry %s is %d seconds (of %d second max) old.\n",
							 i->ip, (int) (t - i->since), i->duration);
		if(((int) (t - i->since)) > i->duration) {
			del_ip(i->ip);
			removed++;
		}
		i = next_node;
	}
	return removed;
}

int print_list()
{
	ip_rec *curr = head->next;
	printf("[head]");
	while (curr != head) {
		printf("->%s(since: %i)", curr->ip, (int)curr->since);
		curr = curr->next;
	}
	printf("\n");
	return 1;
}

int free_list()
{
	ip_rec *curr = head->next;
	while (curr != head) {
		curr = curr->next;
		free(curr->prev);
	}
	free(head);
	return 1;
}

void cleanup(int sig)
	// cleaning up the environment
{
	if (DEBUG) printf("Caught signal %d - cleaning up.\n", sig);

	// close the pipe 
	close(pipe_fd);	//or delete it:  remove(pipe_file)
	if (DEBUG) printf("Closed the pipe.\n");

	// close the message queues
	if (msgctl(msgid_send, IPC_RMID, 0) == -1) {
		fprintf(stderr, "Could not delete Send Message Queue.\n");
		exit(EXIT_FAILURE);
	}
	if (msgctl(msgid_recv, IPC_RMID, 0) == -1) {
		fprintf(stderr, "Could not delete Receive Message Queue.\n");
		exit(EXIT_FAILURE);
	}
	if (DEBUG) printf("Deleted the Message Queues (send: %d & recv: %d).\n", msgid_send, msgid_recv);

	// delete the pid-file
	if (remove(pid_path) == -1) {
		fprintf(stderr, "Could not delete the file '%s'.\n", pid_path);
		exit(EXIT_FAILURE);
	}
	if (DEBUG) printf("deleted successfully the file '%s'.\n", pid_path);

	// delete the pipe
	if (remove(pipe_file) == -1) {
		fprintf(stderr, "Could not delete the pipe '%s'.\n", pipe_file);
		exit(EXIT_FAILURE);
	}

	// free the memory
	free_list();
	exit(EXIT_SUCCESS);
}
