/*	mod_scrutinizer
	rothm4 && ryter
	
	helper module for (distributed) denial of service
	scrutinizer for web services
	-> it generates a 420 message if the requesting
	   ip is on the black list.

	20.10.04	-	created
	22.10.04	-	addedd named pipe & doubly linked list
	27.10.04	-	connected with blacklist.c (no pipes -> ipc msg)
	28.10.04	-	addedd logging informations (error_log)
	29.10.04	-	addedd cleanup & execl for blacklist
	02.11.04	-	add configuration handling & init function for fork
*/

#include "httpd.h"
#include "http_config.h"
#include "http_log.h"
#include "http_request.h"
#include "http_protocol.h"
#include "ap_config.h"

#include <sys/types.h>
#include <sys/ipc.h>
//#include <sys/msg.h>

#include <unistd.h>

// define our own error code
#define HTTP_SCRUTINIZED 420

// define some config defaults
#define DEFAULT_PATH	"/usr/local/scrutinizer/mod_scrutinizer"	// path to helper
#define DEFAULT_APP		"blacklist"									// helper's executable
#define DEFAULT_PIPE	"/tmp/bad_ip"								// fifo file
#define DEFAULT_PID		"/var/run"									// path to pid file:

// defines for ipc
#define MSG_SIZE	20
#define MSG_KEY_SEND  1028
#define MSG_KEY_RECV  2810

// messagebuffer for ipc messages
struct msgbuf {
        long    mtype;
        char    mtext[MSG_SIZE];
} sndbuf, rcvbuf;

// message queue identifiers
int msgid_send, msgid_recv;

// pid of blacklist.c
pid_t black_pid;

// make some things customizable
typedef struct {
	char *app_path;
	char *app_name;
	char *pipe_file;
	char *pid_path;
} scrutinizer_config;

// declare myselft to the http core
module AP_MODULE_DECLARE_DATA scrutinizer_module;

static apr_status_t kill_blacklist_process(int pid)
{
	return kill(pid, SIGTERM);
}

static void *create_scrutinizer_config(apr_pool_t *p, server_rec *dummy)
{
	// create the configuration structure
	scrutinizer_config *newcfg;

	// allocate some memory
	newcfg = (scrutinizer_config *) apr_pcalloc(p, sizeof(scrutinizer_config));

	// set the strings to the default values
	newcfg->app_path  = DEFAULT_PATH;
	newcfg->app_name  = DEFAULT_APP;
	newcfg->pipe_file = DEFAULT_PIPE;
	newcfg->pid_path  = DEFAULT_PID;

	return (void *) newcfg;
}

static int scrutinizer_checker(request_rec *r)
{
	// copy the requesting ip into the send buffer
	strcpy(sndbuf.mtext, r->connection->remote_ip);
	// send the ip to blacklist.c for checking the blacklist
	// set message type to a value > 0
	sndbuf.mtype = 1;
	// send the message to the message queue
	if(msgsnd(msgid_send, (void *)&sndbuf, MSG_SIZE, 0) == -1) {
		// write error_log and return 500
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, 
		   "mod_scrutinizer: Could not send Message to blacklist!");
		return HTTP_INTERNAL_SERVER_ERROR;
	}
	// check if blacklist is running
	// if not, don't try to scrutinize and return DECLINED
	// (but write to error_log the	first time).

	// -> do this here!	

	// wait for a response of blacklist.c
	if(msgrcv(msgid_recv, (void *)&rcvbuf, MSG_SIZE, 0, 0) == -1) {
		// write error_log and return 500
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, 
		   "mod_scrutinizer: Could not receive Message from blacklist!");
		return HTTP_INTERNAL_SERVER_ERROR;
	}
	// if it's in the list, scrutinize it!
	if(strcmp(rcvbuf.mtext, "ACK") == 0) {
		// write to access_log for feedback
		return HTTP_SCRUTINIZED;
	}
	// otherwise decline the request to the next access module
	return DECLINED;
}

static int scrutinizer_init(apr_pool_t *p, apr_pool_t *plog,
								 apr_pool_t *ptemp, server_rec *s)
{
	// get the module configuration
	scrutinizer_config *s_cfg = 
			ap_get_module_config(s->module_config, &scrutinizer_module);

	// variables for module initialisation
	pid_t pid;
	char *pid_file	= (char *)apr_palloc(p, 
				strlen(s_cfg->pid_path) + strlen(s_cfg->app_name) + 6);
	char *app		= (char *)apr_palloc(p,
				 strlen(s_cfg->app_path) + strlen(s_cfg->app_name) + 2);

	// create the complete path (incl. filename) to the pid-file
	pid_file = strcat(pid_file, s_cfg->pid_path);
	pid_file = strcat(pid_file, "/");
	pid_file = strcat(pid_file, s_cfg->app_name);
	pid_file = strcat(pid_file, ".pid");

	// create the complete path to the application
	app	 = strcat(app, s_cfg->app_path);
	app	 = strcat(app, "/");
	app	 = strcat(app, s_cfg->app_name);

	// check if not already a blacklist is running
	// if yes, dont fork and dont start blacklist.
	if (access(pid_file, F_OK) == -1) {
		// fork and start blacklist.c
		pid = fork();
		switch(pid)
		{
		// the child
		case 0:	
			execl(app, s_cfg->app_name, s_cfg->pid_path, s_cfg->pipe_file, 0);
		// the parent
		default:
			// save child's pid
			black_pid = pid;
		}
	}

	// register the cleanup for blacklist
	// kill blacklist when i get killed
	apr_pool_cleanup_register(p, (void *)black_pid, kill_blacklist_process,
									 apr_pool_cleanup_null);

	// create the message queues
	// iff they aren't already here
	msgid_send = msgget((key_t)MSG_KEY_SEND, 0666 | IPC_CREAT);
    msgid_recv = msgget((key_t)MSG_KEY_RECV, 0666 | IPC_CREAT);

	return OK;
}

static void scrutinizer_register_hooks(apr_pool_t *p)
{
	// call for each request scrutinizer_checher as (not very) first access_check function
	ap_hook_access_checker(scrutinizer_checker, NULL, NULL, APR_HOOK_REALLY_FIRST);
	// init the module after parsing the configuration file
	ap_hook_post_config(scrutinizer_init, NULL, NULL, APR_HOOK_MIDDLE);
}

static const char *set_pipe_file(cmd_parms *cmd, void *mconfig, const char *arg)
{
	scrutinizer_config *s_cfg = ap_get_module_config(
			cmd->server->module_config, &scrutinizer_module);

	// the module don't need the pipe, but the
	// helper application does
	// it's not needed, that the file is existant
	// because the helper application creates
	// a new one if needed.
	s_cfg->pipe_file = (char *) arg;
	
	return (char *)0;
}

static const char *set_pid_path(cmd_parms *cmd, void *mconfig, const char *arg)
{
	scrutinizer_config *s_cfg = ap_get_module_config(
			cmd->server->module_config, &scrutinizer_module);

	// check if pid path is existant
	if (access(arg, R_OK) == 0) {
		s_cfg->pid_path = (char *) arg;
	} else {
		return "Invalid scrutinizer pid path in apache configuration!";
	}
	
	return (char *)0;
}

static const char *set_app(cmd_parms *cmd, void *mconfig, const char *arg)
{
	scrutinizer_config *s_cfg = ap_get_module_config(
			cmd->server->module_config, &scrutinizer_module);

	// variables for tokenizer
	char *token, *next, *last_token;
	char *argument = strdup(arg);
	char path[strlen(arg)];

	// check if the application is executable
	if (access(arg, X_OK) == 0) {
		// parse path & application name
		next = argument;
		// get all parts of the path & name
		while ((token = strtok(next, "/")) != NULL) {
			last_token = token;
			next = NULL;
		}
			
		// copy the path-part
		strncpy(path, arg, strlen(arg)-strlen(last_token) - 1);	
		path[strlen(arg)-strlen(last_token) - 1] = '\0';

		// allocate conf-pool memory for saving the new configuration
		s_cfg->app_name = (char *) apr_pcalloc(cmd->server->process->pconf,
														 	strlen(last_token));
		s_cfg->app_path = (char *) apr_pcalloc(cmd->server->process->pconf,
																 strlen(path));

		// save the name & path in the configuration
		strcpy(s_cfg->app_name, last_token);
		strcpy(s_cfg->app_path, path);
	} else {
		return "Invalid scrutinizer application in apache configuration!";
	}

	return (char *)0;
}

static const command_rec mod_scrutinizer_cmds[] =
{
	AP_INIT_TAKE1("ScrutinizerPipeFile", set_pipe_file, NULL,
				 RSRC_CONF,"The named pipe (e.g., /tmp/bad_ip."),
	AP_INIT_TAKE1("ScrutinizerPidPath", set_pid_path, NULL,
		 RSRC_CONF,"The path to the pid file (e.g., '/var/run' ."),
	AP_INIT_TAKE1("ScrutinizerApplication", set_app, NULL, RSRC_CONF,
		"The full path to the helper application (e.g., '/usr/local/scrutinizer/blacklist' ."),
	{NULL}
};

module AP_MODULE_DECLARE_DATA scrutinizer_module = {
	STANDARD20_MODULE_STUFF, 
	NULL,
	NULL,
	create_scrutinizer_config,
	NULL,			
	mod_scrutinizer_cmds,		
	scrutinizer_register_hooks
};
