/*
 * $Id: FsUtils.java,v 1.5 2004/04/06 20:38:20 wrossi Exp $
 
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: FsUtils.java,v $
 * Revision 1.5  2004/04/06 20:38:20  wrossi
 * Improving javadocs
 *
 * Revision 1.4  2003/11/11 01:14:02  wrossi
 * Added getU64 and getS64 for 64 bit types.
 *
 * Revision 1.3  2003/11/04 12:09:38  wrossi
 * Bug fix for little endian case.  Was using wrong byte offset
 *
 * Revision 1.2  2003/10/31 12:51:02  wrossi
 * Moved endian into one central location
 *
 * Revision 1.1  2003/10/27 11:57:33  wrossi
 * Added FsException and FsUtils.  Added code to ReiserSuperBlock to parse
 * a buffer and do some sanity checks.
 *
 * Revision 1.1  2003/10/22 23:20:11  wrossi
 * Initial checkin
 *
 */

package rossi.fstools.fs;

/**
 * Utility class for converting numbers from on-disk format.
 */

public class FsUtils
{
  /** Controls endian-ness.  */
  public static boolean bigEndian = false;   // Controls endian-ness

  /**
   * Read a 64 bit unsigned number off the disk.  Note that Java doesn't support
   * unsigned 64 bit integers, so this will not work correctly for very big numbers.
   *
   * @param buffer   a byte[] containing the data
   * @param offset   a int offset into the buffer
   * @return a long read from the buffer
   */
  public static long getU64(byte[] buffer, int offset)
  {
    // Java doesnt have an unsigned 64 bit type, so we'll just use a 
    // signed 64 bit type and hope for the best
    long r = 0;
    int pos, inc;

    inc = bigEndian ? 1 : -1;
    pos = bigEndian ? offset : offset+7;

    r = r | ((long)buffer[pos] & 0xff) << 56;
    pos += inc;
    r = r | ((long)buffer[pos] & 0xff) << 48;
    pos += inc;
    r = r | ((long)buffer[pos] & 0xff) << 40;
    pos += inc;
    r = r | ((long)buffer[pos] & 0xff) << 32;
    pos += inc;
    r = r | ((long)buffer[pos] & 0xff) << 24;
    pos += inc;
    r = r | ((long)buffer[pos] & 0xff) << 16;
    pos += inc;
    r = r | ((long)buffer[pos] & 0xff) << 8;
    pos += inc;
    r = r | ((long)buffer[pos] & 0xff);

    return r;
  }

  /**
   * This is the same as getU64().
   *
   * @see #getU64(byte[],int)
   * @param buffer   a byte[] containing the data
   * @param offset   a int offset into the buffer
   * @return a long read from the buffer
   */
  public static long getS64(byte[] buffer, int offset)
  {
    return getU64(buffer, offset);
  }

  /**
   * Read a 32 bit unsigned number off the disk.  Note that Java doesn't support
   * unsigned integers, so this will be returned in a 64 bit value.
   *
   * @param buffer   a byte[] containing the data
   * @param offset   a int offset into the buffer
   * @return a long read from the buffer
   */
  public static long getU32(byte[] buffer, int offset)
  {
    long r = 0;
    int pos, inc;

    inc = bigEndian ? 1 : -1;
    pos = bigEndian ? offset : offset+3;
 
    r = r | ((long)buffer[pos] & 0xff) << 24;
    pos += inc;
    r = r | ((long)buffer[pos] & 0xff) << 16;
    pos += inc;
    r = r | ((long)buffer[pos] & 0xff) << 8;
    pos += inc;
    r = r | ((long)buffer[pos] & 0xff);

    return r;
  }

  /**
   * Read a 32 bit signed number off the disk.  Note that Java doesn't support
   * unsigned integers, so this will be returned in a 64 bit value.
   *
   * @param buffer   a byte[] containing the data
   * @param offset   a int offset into the buffer
   * @return a long read from the buffer
   */
  public static long getS32(byte[] buffer, int offset)
  {
    long r = getU32(buffer, offset);

    if ( (r & 0x80000000) != 0)
      r |= 0xffffffff00000000L;

    return r;
  }

  /**
   * Read a 16 bit unsigned number off the disk.  Note that Java doesn't support
   * unsigned integers, so this will be returned in a 32 bit value.
   *
   * @param buffer   a byte[] containing the data
   * @param offset   a int offset into the buffer
   * @return a long read from the buffer
   */
  public static int getU16(byte[] buffer, int offset)
  {
    int r = 0;
    int pos, inc;

    inc = bigEndian ? 1 : -1;
    pos = bigEndian ? offset : offset+1;
 
    r = r | ((int)buffer[pos] & 0xff) << 8;
    pos += inc;
    r = r | ((int)buffer[pos] & 0xff);

    return r;
  }

  /**
   * Read a 16 bit signed number off the disk.  Note that Java doesn't support
   * unsigned integers, so this will be returned in a 32 bit value.
   *
   * @param buffer   a byte[] containing the data
   * @param offset   a int offset into the buffer
   * @return a long read from the buffer
   */
  public static int getS16(byte[] buffer, int offset)
  {
    int r = getU16(buffer, offset);

    if ( (r & 0x8000) != 0)
      r |= 0xffff0000L;

    return r;
  }

}
