/*
 * $Id: Ext2File.java,v 1.2 2004/04/09 21:11:25 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: Ext2File.java,v $
 * Revision 1.2  2004/04/09 21:11:25  wrossi
 * Adding javadocs
 *
 * Revision 1.1  2004/04/02 21:35:10  wrossi
 * Ext2 file implementation
 *
 *
 */

package rossi.fstools.fs.ext2fs;

import rossi.fstools.fs.File;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.io.BlockPointer;
import rossi.fstools.io.BlockReader;
import rossi.fstools.io.BlockPointerInputStream;

import java.util.List;
import java.util.ArrayList;

import java.io.IOException;

/**
 * Representation of a file.
 */

public class Ext2File implements File
{
  protected List blkPointerList;
  protected List slackList;
  protected long size;
  protected long sizeRemaining;
  protected BlockReader reader;
  protected Ext2SuperBlock sb;

  /**
   * Create a file.
   *
   * @param sb   a Ext2SuperBlock
   * @param fileSize   The length of the file from its Inode.
   * @param reader   a BlockReader to read the disk.
   */
  public Ext2File(Ext2SuperBlock sb, long fileSize, BlockReader reader)
  {
    blkPointerList = new ArrayList();
    slackList = new ArrayList();
    this.size = fileSize;
    this.reader = reader;
    this.sb = sb;
    sizeRemaining = size;
  }

  /**
   * Adds the specified block to the file.
   * @param blocknum block number of the block to be added.
   */
  public void addDirectBlock(long blocknum)
  {
    BlockPointer bp;
    BlockPointer lastBp = null;
    int blksize;
    int curSize;
    int lastend = 0;
    int lastendoffset = 0;

    blksize = (int) sb.getBlockSize();

    curSize = (sizeRemaining < blksize) ? (int)sizeRemaining : blksize;

    if (blkPointerList.size() != 0)
    {
      lastBp = (BlockPointer) blkPointerList.get(blkPointerList.size() - 1);

      lastend = lastBp.getBlockNum() + ((lastBp.getOffset()+lastBp.getLength()) / blksize);
      lastendoffset = ((lastBp.getOffset()+lastBp.getLength()) % blksize);
    }

    if (lastBp != null && lastend == blocknum && lastendoffset == 0)
    {
      /* Add this block onto the last block pointer since it is a consecutive block */
      lastBp.setLength(lastBp.getLength() + curSize);
    }
    else
    {
      /* Create a new block pointer */

      bp = new BlockPointer();
      bp.setBlockNum((int) blocknum);
      bp.setOffset(0);
      bp.setLength(curSize);
      bp.setBlockSize(blksize);
      blkPointerList.add(bp);
    }

    /* record slack */
    if (curSize != blksize)
    {
      bp = new BlockPointer();
      bp.setBlockNum((int) blocknum);
      bp.setOffset(curSize);
      bp.setLength(blksize - curSize);
      bp.setBlockSize(blksize);
      slackList.add(bp);
    }

    sizeRemaining = sizeRemaining - curSize;
  }

  /**
   *  @param blocknum the block number of the (single, double, tripply) indirect block
   *  @param level of indirection -- 0 = direct, 1 = first indirect...
   */
  public void addIndirectBlock(long blocknum, int level) throws IOException
  {
    int blksize = (int) sb.getBlockSize();
    byte buffer[];

    if (level == 0)
    {
      addDirectBlock(blocknum);
      return;
    }

    buffer = reader.getBlock((int) blocknum);
    
    for (int i=0; i<blksize/4; i++)
    {
      if (sizeRemaining == 0)
        break;

      addIndirectBlock(FsUtils.getU32(buffer, i*4), level-1);
    }
  }

  public BlockPointerInputStream getData()
  {
    return new BlockPointerInputStream(reader, blkPointerList);
  }

  public BlockPointerInputStream getSlack()
  {
    return new BlockPointerInputStream(reader, slackList);
  }
}
