/*
 * $Id: Ext2FileSystem.java,v 1.8 2004/04/09 21:11:25 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: Ext2FileSystem.java,v $
 * Revision 1.8  2004/04/09 21:11:25  wrossi
 * Adding javadocs
 *
 * Revision 1.7  2004/04/03 19:48:08  wrossi
 * Added sockets and fifos
 *
 * Revision 1.6  2004/04/03 19:44:27  wrossi
 * Added devices and symlinks
 *
 * Revision 1.5  2004/04/03 17:55:36  wrossi
 * Adding directory support
 *
 * Revision 1.4  2004/04/03 16:51:43  wrossi
 * Can extract a file given the inode
 *
 * Revision 1.3  2004/04/01 21:40:33  wrossi
 * Bias inode number by one when reading inode.
 *
 * Revision 1.2  2004/04/01 11:59:57  wrossi
 * Wrote getInode function
 *
 * Revision 1.1  2004/03/31 21:45:33  wrossi
 * Starting file system
 *
 *
 */

package rossi.fstools.fs.ext2fs;

import rossi.fstools.fs.FileSystem;
import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.InodePtr;
import rossi.fstools.fs.Inode;
import rossi.fstools.fs.File;
import rossi.fstools.fs.Directory;
import rossi.fstools.fs.Device;
import rossi.fstools.fs.SymLink;
import rossi.fstools.fs.Socket;
import rossi.fstools.fs.Fifo;
import rossi.fstools.fs.FsObject;
import rossi.fstools.fs.FsException;

import rossi.fstools.io.BlockReader;
import rossi.fstools.io.BlockCache;

import java.io.IOException;

/** 
 * Implementation of the EXT2 and EXT3 file systems.
 */
public class Ext2FileSystem implements FileSystem
{
  private BlockReader blockReader;
  private Ext2SuperBlock sb;

  /** Inode number of the bad blocks file. */
  public final static long EXT2_BAD_INO = 1;
  /** Inode number of the root directory. */
  public final static long EXT2_ROOT_INO = 2;

  public SuperBlock open(BlockReader br) throws FsException, IOException
  {
    blockReader = br;

    br.setBlockSize(1024);  // 1KB blocks  -- for starters...
    sb = new Ext2SuperBlock();

    sb.loadFromBuffer(br.getBlock(1), 0);

    br.setBlockSize((int) sb.getBlockSize());
    return sb;
  }

  public void close() throws IOException
  {
    blockReader.close();
  }

  public InodePtr getRootDir()
  {
    return new Ext2InodePtr(EXT2_ROOT_INO);   
  }

  /** Gets an inode.  Uses the inode pointer supplied to load an inode. */
  public Inode getInode(InodePtr ptr) throws FsException, IOException
  {
    long inodesPerGroup;     // num of inodes per group
    long groupNum;           // group containing this inode
    long groupOffset;        // inode within the group 
    long inodeNumber;        // number of the desired inode
    long inodeTableStart;    // First block number of the group's inode table
    long inodesPerBlock;     // number of inodes per block
    long inodeTableBlock;    // block within the inode table we need;
    long inodeTableBlockOffset;   // inode within the block we need
    byte[] buffer;
    Ext2Inode inode;
    GroupDesc groupDesc;

    /* First, figure out what group the inode is in, and the offset within the group.
       Note:  The first inode is numbered 1, not 0.  */
    inodeNumber = ((Ext2InodePtr)ptr).getNumber() - 1;

    /* an inode number of -1 is invalid.  */
    if (inodeNumber < 0)
      return null;

    inodesPerGroup = sb.getInodesPerGroup();
    groupNum = inodeNumber / inodesPerGroup;
    groupOffset = inodeNumber % inodesPerGroup;

    /* Get the group descriptor for the group containing the inode */
    groupDesc = getGroupDescriptor((int)groupNum);
    
    /* Find the inode table for the group */
    inodeTableStart = groupDesc.getInodeTable();

    /* Figure out which block of the inode table we need */
    inodesPerBlock = sb.getBlockSize() / sb.getInodeSize();
    inodeTableBlock = (groupOffset / inodesPerBlock) + inodeTableStart;

    /* Which inode within the block */
    inodeTableBlockOffset = groupOffset % inodesPerBlock;

    /* Read in the block containing the inode */
    buffer = blockReader.getBlock((int) inodeTableBlock);
    
    /* Load our inode */
    inode = new Ext2Inode();
    inode.loadFromBuffer(buffer, (int)(inodeTableBlockOffset * sb.getInodeSize()));

    return inode;
  }

  /** 
   *  Gets a group descriptor.
   *
   *  The table of group descriptors is located just behind the superblock 
   *
   *  @param groupno the group number
   *  @return the group descriptor
   *  @exception IOException in case of an IO error
   *  @exception FSException in case of a FS format error
   */
  private GroupDesc getGroupDescriptor(int groupno) throws FsException, IOException
  {
    int firstGroupBlock;
    int groupsPerBlock;
    int groupSize;
    int groupBlock;
    int groupOffset;
    byte buffer[];
    GroupDesc gd;

    /* First group block is in the block following the block containing the superblock.
       Since the superblock is fixed at offset 1024 bytes from the beginnint, this can
       be block number 1 or block number 2.   It is only block 2 in the case where blocksize
       is 1024.  Any other blocksize puts the superblock in block 0 */

    firstGroupBlock = (sb.getBlockSize() == 1024) ? 2 : 1;

    gd = new GroupDesc();
    groupSize = gd.getDataSize();


    /* Compute the number of group descriptors per block */
    groupsPerBlock = (int) sb.getBlockSize() / groupSize;

    groupBlock = groupno / groupsPerBlock;
    groupOffset = groupno % groupsPerBlock;

    /* Read the desired block into memory */
    buffer = blockReader.getBlock(groupBlock + firstGroupBlock);

    /* Load the desired group descriptor */
    gd.loadFromBuffer(buffer, groupOffset * groupSize);

    return gd;
  }

  /** 
   *  Creates a file from an inode
   *
   *  @param inode the inode
   *  @return the file
   *  @exception IOException in case of an IO error
   *  @exception FSException in case of a FS format error
   */
  public File createExt2File(Ext2Inode inode) throws FsException, IOException
  {
    Ext2File file;
    long fileSize;
    long blockno;

    fileSize = inode.getSize();

    file = new Ext2File(sb, fileSize, blockReader);

    for (int i=0; i<Ext2Inode.EXT2_NDIR_BLOCKS; i++)
    {
      blockno = inode.getDirectBlock(i);
      if (blockno == 0)
        break;

      file.addDirectBlock(blockno);
    }

    blockno = inode.getIndirectBlock();
    if (blockno != 0)
      file.addIndirectBlock(blockno, 1);

    blockno = inode.getDoubleIndirectBlock();
    if (blockno != 0)
      file.addIndirectBlock(blockno, 2);

    blockno = inode.getTripleIndirectBlock();
    if (blockno != 0)
      file.addIndirectBlock(blockno, 3);

    return file;
  }

  /**
   * Create a directory.
   * @param inode   a Ext2Inode
   * @return Directory
   * @exception FsException
   * @exception IOException
   */
  public Directory createExt2Directory(Ext2Inode inode) throws FsException, IOException
  {
    return new Ext2Directory(createExt2File(inode));
  }

  /**
   * Create a device.
   * @param inode   a Ext2Inode
   * @return Device
   */
  public Device createExt2Device(Ext2Inode inode)
  {
    return new Ext2Device(inode);
  }

  /**
   * Create a symlink.  Note the blocks field of the inode controls whether 
   * it is a fast or slow symlink.
   * @param inode   a Ext2Inode
   * @return SymLink
   * @exception FsException
   * @exception IOException
   */
  public SymLink createExt2SymLink(Ext2Inode inode) throws FsException, IOException
  {
    if (inode.getBlocks() == 0)  // fast symlink
      return new Ext2FastSymLink(inode);
    return new Ext2SlowSymLink(createExt2File(inode));
  }

  public FsObject getObject(Inode inode) throws FsException, IOException
  {
    int fileType;

    fileType = inode.getMode() & Inode.FILETYPE_MASK;

    switch (fileType)
    {
      case Inode.DIRECTORY_MODE:  return createExt2Directory((Ext2Inode)inode);
      case Inode.FILE_MODE: return createExt2File((Ext2Inode)inode);    
      case Inode.SYMLINK_MODE: return createExt2SymLink((Ext2Inode)inode);    
      case Inode.BLOCKDEV_MODE: return createExt2Device((Ext2Inode)inode);    
      case Inode.CHARDEV_MODE: return createExt2Device((Ext2Inode)inode);    
      case Inode.SOCKET_MODE: return new Ext2Socket();
      case Inode.FIFO_MODE: return new Ext2Fifo();
    }

    throw new FsException("Unsupported file type "+fileType);
  }
}
