/*
 * $Id: Ext2Inode.java,v 1.8 2004/04/09 21:11:25 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: Ext2Inode.java,v $
 * Revision 1.8  2004/04/09 21:11:25  wrossi
 * Adding javadocs
 *
 * Revision 1.7  2004/04/03 19:44:27  wrossi
 * Added devices and symlinks
 *
 * Revision 1.6  2004/04/03 16:51:43  wrossi
 * Can extract a file given the inode
 *
 * Revision 1.5  2004/04/02 21:35:40  wrossi
 * fixed offsets in inode
 *
 * Revision 1.4  2004/03/31 12:33:08  wrossi
 * Added comments and inode pointer class
 *
 * Revision 1.3  2004/03/30 21:38:40  wrossi
 * Filled out marshalling code
 *
 * Revision 1.2  2004/03/29 21:37:32  wrossi
 * Work in progress
 *
 * Revision 1.1  2004/03/29 21:13:59  wrossi
 * adding inode
 *
 *
 */

package rossi.fstools.fs.ext2fs;

import rossi.fstools.fs.DiskStructure;
import rossi.fstools.fs.Inode;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;

/** 
 * Representation of ext2/3 inode.
 *
 * <pre>
 *  On disk the superblock looks like this:
 *
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |  Mode         |     UID            |
 *        --------------------------------------
 *  0004  |          Size                      |
 *        --------------------------------------
 *  0008  |          ATime                     |
 *        --------------------------------------
 *  000C  |          CTime                     |
 *        --------------------------------------
 *  0010  |          MTime                     |
 *        --------------------------------------
 *  0014  |          DTime                     |
 *        --------------------------------------
 *  0018  |  GID         |  Num Links          |
 *        --------------------------------------
 *  001C  |          Block Count               |
 *        --------------------------------------
 *  0020  |          Flags                     |
 *        --------------------------------------
 *  0024  |          Reserved                  |
 *        --------------------------------------
 *  0028  |          Direct Block 0            |
 *        --------------------------------------
 *                        :                     
 *        --------------------------------------
 *  0054  |          Direct Block 11           |
 *        --------------------------------------
 *  0058  |          Indirect Block            |
 *        --------------------------------------
 *  005C  |      Double Indirect Block         |
 *        --------------------------------------
 *  0060  |      Triple Indirect Block         |
 *        --------------------------------------
 *  0064  |      File version                  |
 *        --------------------------------------
 *  0068  |      File ACL                      |
 *        --------------------------------------
 *  006C  |      Dir ACL                       |
 *        --------------------------------------
 *  0070  |      Fragment address              |
 *        --------------------------------------
 *  0074  | Frag | fsize   |    pad            |
 *        --------------------------------------
 *  0078  |   UID high     |    GID High       |
 *        --------------------------------------
 *  007C  |          Reserved                  |
 *        --------------------------------------
 * </pre>
 */

public class Ext2Inode implements DiskStructure, Inode
{
  protected int mode;
  protected int uidLow;
  protected long sizeLow;
  protected long accessTime;
  protected long modifyTime;
  protected long changedTime;
  protected long deletedTime;
  protected int gidLow;
  protected int numLinks;
  protected long blocks;   // number of 512-byte unit allocated (not blocks)
  protected long flags;
  protected long[] directBlocks;
  protected long indirectBlock;
  protected long doubleIndirectBlock;
  protected long tripleIndirectBlock;
  protected byte[] blockData;
  protected long fileVersion;
  protected long fileACL;   // block num access control list
  protected long dirACL;    // High 32 bits of file size
  protected long fragAddr;
  protected int fragNum;
  protected int fragSize;
  protected int uidHigh;
  protected int gidHigh;

  public static int EXT2_NDIR_BLOCKS = 12;  // number of direct blocks
  public static int EXT2_N_BLOCKS = 15;     // number of blocks
  public static int EXT2_GOOD_OLD_INODE_SIZE = 128;

  public Ext2Inode()
  {
  }

  public int getDataSize() { return 0x20; }

  /**
   * Get the mode of the file.  This includes permission and type information.
   * @return mode
   */
  public int getMode() { return mode; }
  public void setMode(int aMode) { mode=aMode; }

  /**
   * Get the low 16 bits of the user id. 
   * @return userid
   */
  public int getUidLow() { return uidLow; }
  public void setUidLow(int aUidLow) { uidLow=aUidLow; }

  /**
   * Get the low 32 bits of the file size.
   * @return filesize
   */
  public long getSizeLow() { return sizeLow; }
  public void setSizeLow(long aSizeLow) { sizeLow=aSizeLow; }

  public long getAccessTime() { return accessTime; }
  public void setAccessTime(long aAccessTime) { accessTime=aAccessTime; }

  public long getModifyTime() { return modifyTime; }
  public void setModifyTime(long aModifyTime) { modifyTime=aModifyTime; }

  public long getChangedTime() { return changedTime; }
  public void setChangedTime(long aChangedTime) { changedTime=aChangedTime; }

  public long getDeletedTime() { return deletedTime; }
  public void setDeletedTime(long aDeletedTime) { deletedTime=aDeletedTime; }

  public int getGidLow() { return gidLow; }
  public void setGidLow(int aGidLow) { gidLow=aGidLow; }

  public long getNumLinks() { return numLinks; }
  public void setNumLinks(long aNumLinks) { numLinks=(int)aNumLinks; }

  /** This field is really the number of SECTORS the file takes up on disk.  That is the
   *  number of 512 byte sectors.  This is completely unrelated to the blocksize set in the
   *  superblock.  Also, this field is tested to determine if a symlink is a "fast symlink" or
   *  not. 
   *  @return number of sectors.
   */
  public long getBlocks() { return blocks; }
  public void setBlocks(long aBlocks) { blocks=aBlocks; }

  public long getFlags() { return flags; }
  public void setFlags(long aFlags) { flags=aFlags; }

  /** 
   * Get the direct block pointers as an array.
   * @see #getDirectBlock(int)
   * @return long[] for direct pointers.
   */
  public long[] getDirectBlocks() { return directBlocks; }
  public void setDirectBlocks(long[] aDirectBlocks) { directBlocks=aDirectBlocks; }

  /**
   * Get a direct block given its index.
   * @param blocknum  which of the 12 direct blocks to return
   * @return long the block pointer
   */
  public long getDirectBlock(int blocknum) 
  {
    return getDirectBlocks()[blocknum];
  }

  /**
   * Get the indirect block pointer.  This is a pointer to a block of pointers.
   * @return long
   */
  public long getIndirectBlock() { return indirectBlock; }
  public void setIndirectBlock(long aIndirectBlock) { indirectBlock=aIndirectBlock; }

  /**
   * Get the double indirect block pointer. This is a pointer to a block of pointers that
   * point to a block of pointers, which finally point to data blocks.
   * @return long
   */
  public long getDoubleIndirectBlock() { return doubleIndirectBlock; }
  public void setDoubleIndirectBlock(long aDoubleIndirectBlock) { doubleIndirectBlock=aDoubleIndirectBlock; }

  /**
   * Get the triple indirect block pointer.
   * @return long
   */
  public long getTripleIndirectBlock() { return tripleIndirectBlock; }
  public void setTripleIndirectBlock(long aTripleIndirectBlock) { tripleIndirectBlock=aTripleIndirectBlock; }

  /**
   * Get the block data as a byte array.  Sometime data is stored in here such as
   * fast symlinks.
   * @return byte[] overlaid on the block pointers.
   */
  public byte[] getBlockData() { return blockData; }
  public void setBlockData(byte[] aBlockData) { blockData=aBlockData; }

  public long getFileVersion() { return fileVersion; }
  public void setFileVersion(long aFileVersion) { fileVersion=aFileVersion; }

  public long getFileACL() { return fileACL; }
  public void setFileACL(long aFileACL) { fileACL=aFileACL; }

  /** This field is overloaded so that it contains the most significant 32 bits of the file size
   *  when mode == FILE_MODE 
   *  @return directory ACL or high 32 bits of file size
   */
  public long getDirACL() { return dirACL; }
  public void setDirACL(long aDirACL) { dirACL=aDirACL; }

  public long getFragAddr() { return fragAddr; }
  public void setFragAddr(long aFragAddr) { fragAddr=aFragAddr; }

  public int getFragNum() { return fragNum; }
  public void setFragNum(int aFragNum) { fragNum=aFragNum; }

  public int getFragSize() { return fragSize; }
  public void setFragSize(int aFragSize) { fragSize=aFragSize; }

  public int getUidHigh() { return uidHigh; }
  public void setUidHigh(int aUidHigh) { uidHigh=aUidHigh; }

  public int getGidHigh() { return gidHigh; }
  public void setGidHigh(int aGidHigh) { gidHigh=aGidHigh; }

  /* synthesized methods */

  public long getUserId()
  {
    return getUidHigh() << 16 | getUidLow();
  }

  public void setUserId(long uid)
  {
    setUidHigh((int) (uid >> 16) & 0xffff);
    setUidLow((int) uid & 0xffff);
  }

  public void setGroupId(long gid)
  {
    setGidHigh((int) (gid >> 16) & 0xffff);
    setGidLow((int) gid & 0xffff);
  }

  public long getGroupId()
  {
    return getGidHigh() << 16 | getGidLow();
  }

  public long getSize()
  {
    if ((getMode() & FILE_MODE) == FILE_MODE)
      return getDirACL() << 32 | getSizeLow();
    else
      return getSizeLow();
  }

  public void setSize(long s)
  {
    if ((getMode() & FILE_MODE) == FILE_MODE)
      setDirACL( (s >> 32) & 0xffffffff);

    setSizeLow( s & 0xffffffff);
  }

  public void loadFromBuffer(byte buffer[], int offset) throws FsException
  {
    setMode(FsUtils.getU16(buffer, offset+0x0));
    setUidLow(FsUtils.getU16(buffer, offset+0x2));
    setSizeLow(FsUtils.getU32(buffer, offset+0x4));
    setAccessTime(FsUtils.getU32(buffer, offset+0x8));
    setChangedTime(FsUtils.getU32(buffer, offset+0xC));
    setModifyTime(FsUtils.getU32(buffer, offset+0x10));
    setDeletedTime(FsUtils.getU32(buffer, offset+0x14));
    setGidLow(FsUtils.getU16(buffer, offset+0x18));
    setNumLinks(FsUtils.getU16(buffer, offset+0x1A));
    setBlocks(FsUtils.getU32(buffer, offset+0x1C));
    setFlags(FsUtils.getU32(buffer, offset+0x20));

    directBlocks = new long[EXT2_NDIR_BLOCKS];
    for (int i=0; i<EXT2_NDIR_BLOCKS; i++)
      directBlocks[i] = FsUtils.getU32(buffer, offset+0x28+i*4);

    blockData = new byte[EXT2_N_BLOCKS * 4];
    System.arraycopy(buffer, offset+0x28, blockData, 0, (EXT2_N_BLOCKS * 4));

    setIndirectBlock(FsUtils.getU32(buffer, offset+0x58));
    setDoubleIndirectBlock(FsUtils.getU32(buffer, offset+0x5C));
    setTripleIndirectBlock(FsUtils.getU32(buffer, offset+0x60));
    setFileVersion(FsUtils.getU32(buffer, offset+0x64));
    setFileACL(FsUtils.getU32(buffer, offset+0x68));
    setDirACL(FsUtils.getU32(buffer, offset+0x6C));
    setFragAddr(FsUtils.getU32(buffer, offset+0x70));
    setFragNum(buffer[offset+0x74]);
    setFragSize(buffer[offset+0x75]);
    setUidHigh(FsUtils.getU16(buffer, offset+0x78));
    setGidHigh(FsUtils.getU16(buffer, offset+0x7A));
  }
}
