/*
 * $Id: GroupDesc.java,v 1.3 2004/04/09 21:41:05 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: GroupDesc.java,v $
 * Revision 1.3  2004/04/09 21:41:05  wrossi
 * Added javadocs.
 *
 * Revision 1.2  2004/03/29 21:13:59  wrossi
 * adding inode
 *
 * Revision 1.1  2004/03/26 21:35:52  wrossi
 * Directory entry and group descriptor
 *
 * Revision 1.6  2004/03/25 21:14:44  wrossi
 *
 */

package rossi.fstools.fs.ext2fs;

import rossi.fstools.fs.DiskStructure;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;

/** 
 * Representation of ext2/3 group descriptor.
 * <pre>
 *  On disk the superblock looks like this:
 *
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |        Block Bitmap Block          |
 *        --------------------------------------
 *  0004  |        Inode Bitmap Block          |
 *        --------------------------------------
 *  0008  |        Inode table                 |
 *        --------------------------------------
 *  000C  | Free Blocks   |  Free Inodes       |
 *        --------------------------------------
 *  0010  |  Used Dirs    |  Pad               |
 *        --------------------------------------
 *  0014  |        Reserved                    |
 *        --------------------------------------
 *  0018  |        Reserved                    |
 *        --------------------------------------
 *  001C  |        Reserved                    |
 *        --------------------------------------
 */

public class GroupDesc implements DiskStructure
{
  protected long blockBitmap;  // block address of block bitmap block
  protected long inodeBitmap;  // block address of inode bitmap block
  protected long inodeTable;   // block address of inode table
  protected int freeBlocks;    // number of free blocks in this group
  protected int freeInodes;    // number of free inodes in this group
  protected int usedDirs;      // number of directories

  public int getDataSize() { return 0x20; }

  /**
   * Get the location of the block bitmap block for this group.
   * @return long
   */
  public long getBlockBitmap() { return blockBitmap; }
  public void setBlockBitmap(long aBlockBitmap) { blockBitmap=aBlockBitmap; }

  /**
   * Get the location of the inode bitmap block for this group.
   * @return long
   */
  public long getInodeBitmap() { return inodeBitmap; }
  public void setInodeBitmap(long aInodeBitmap) { inodeBitmap=aInodeBitmap; }

  /**
   * Get the location of the inode table for this group.
   * @return long
   */
  public long getInodeTable() { return inodeTable; }
  public void setInodeTable(long aInodeTable) { inodeTable=aInodeTable; }

  /**
   * Get the number of free blocks in this group.
   * @return long
   */
  public int getFreeBlocks() { return freeBlocks; }
  public void setFreeBlocks(int aFreeBlocks) { freeBlocks=aFreeBlocks; }

  /**
   * Get the number of free inodes in this group.
   * @return long
   */
  public int getFreeInodes() { return freeInodes; }
  public void setFreeInodes(int aFreeInodes) { freeInodes=aFreeInodes; }

  /**
   * Get the number of directories in this group.
   * @return long
   */
  public int getUsedDirs() { return usedDirs; }
  public void setUsedDirs(int aUsedDirs) { usedDirs=aUsedDirs; }

  public void loadFromBuffer(byte buffer[], int offset) throws FsException
  {
    setBlockBitmap(FsUtils.getU32(buffer, offset+0x0));
    setInodeBitmap(FsUtils.getU32(buffer, offset+0x4));
    setInodeTable(FsUtils.getU32(buffer, offset+0x8));
    setFreeBlocks(FsUtils.getU16(buffer, offset+0xA));
    setFreeInodes(FsUtils.getU16(buffer, offset+0xC));
    setUsedDirs(FsUtils.getU16(buffer, offset+0x10));
  }
}
